// ===========================================================================
//
//                            PUBLIC DOMAIN NOTICE
//            National Center for Biotechnology Information (NCBI)
//
//  This software/database is a "United States Government Work" under the
//  terms of the United States Copyright Act. It was written as part of
//  the author's official duties as a United States Government employee and
//  thus cannot be copyrighted. This software/database is freely available
//  to the public for use. The National Library of Medicine and the U.S.
//  Government do not place any restriction on its use or reproduction.
//  We would, however, appreciate having the NCBI and the author cited in
//  any work or product based on this material.
//
//  Although all reasonable efforts have been taken to ensure the accuracy
//  and reliability of the software and data, the NLM and the U.S.
//  Government do not and cannot warrant the performance or results that
//  may be obtained by using this software or data. The NLM and the U.S.
//  Government disclaim all warranties, express or implied, including
//  warranties of performance, merchantability or fitness for any particular
//  purpose.
//
// ===========================================================================
//
// File Name:  transmute.go
//
// Author:  Jonathan Kans
//
// ==========================================================================

package main

import (
	"bufio"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"github.com/gedex/inflector"
	"github.com/klauspost/cpuid"
	"github.com/pbnjay/memory"
	"html"
	"io"
	"io/ioutil"
	"net/url"
	"os"
	"runtime"
	"runtime/debug"
	"runtime/pprof"
	"strconv"
	"strings"
	"sync"
	"unicode"
	"unicode/utf8"
)

// TRANSMUTE HELP MESSAGE TEXT

const transmuteHelp = `
Pretty-Printing

 Reformat XML

  -x2p

 Reformat JSON

  -j2p

 Table column alignment

  -align

    -a    Column alignment codes:

            l left
            c center
            r right
            n numeric align on decimal point
            N trailing zero-pad decimals
            z leading zero-pad integers

    -g    Spacing between columns
    -h    Indent before columns

Data Conversion

 JSON stream to XML

  -j2x

    -set setWrapper
    -rec recordWrapper
    -nest [flat|recurse|plural|depth]

 ASN.1 stream to XML

  -a2x

    -set setWrapper
    -rec recordWrapper

 Tab-delimited table to XML

  -t2x

    -set setWrapper
    -rec recordWrapper
    -skip linesToSkip
    -header
    -lower | -upper
    -indent | -flush

      XML object names per column

 Comma-separated values file to XML

  -c2x

    -set setWrapper
    -rec recordWrapper
    -skip linesToSkip
    -header
    -lower | -upper
    -indent | -flush

      XML object names per column

 GenBank/GenPept flatfile to INSDSeq XML

  -g2x

Sequence Comparison

  -diff        Compare two aligned files for point differences

Sequence Editing

  -revcomp     Reverse complement nucleotide sequence

  -remove      Trim at ends of sequence

    -first       Delete first N bases
    -last        Delete last N bases

  -retain      Save either end of sequence

    -leading     Keep first N bases
    -trailing    Keep last N bases

  -replace     Apply base or residue substitution

    -offset      skip ahead by 0-based count (SPDI), or
    -column      move just before 1-based position (HGVS)

    -delete      Delete N bases
    -insert      Insert given sequence

  -extract     Use xtract -insd feat_location instructions

Sequence Processing

  -cds2prot    Translate coding region into protein

    -code        Genetic code
    -frame       Offset in sequence
    -stop        Include stop residue
    -trim        Remove trailing Xs
    -part5       CDS partial at 5' end
    -part3       CDS extends past 3' end
    -every       Translate all codons

  -molwt       Calculate molecular weight of peptide

    -met         Do not cleave leading methionine

Variation Processing

  -hgvs        Convert HGVS variation format to XML

String Transformations

 XML

  -encodeXML
  -decodeXML

  -plainXML

 URL

  -encodeURL
  -decodeURL

 Base64

  -encode64
  -decode64

 Protein

  -aa1to3
  -aa3to1

Customized XML Reformatting

  -format [copy|compact|flush|indent|expand]

    -xml
    -doctype
    -comment
    -cdata
    -separate
    -self
    -unicode [fuse|space|period|brackets|markdown|slash|tag]
    -script [brackets|markdown]
    -mathml [terse]

XML Modification

  -filter Object
            [retain|remove|encode|decode|shrink|expand|accent]
              [content|cdata|comment|object|attributes|container]

EFetch XML Normalization

  -normalize [database]

Examples

  -j2x -set - -rec GeneRec

  -t2x -set Set -rec Rec -skip 1 Code Name

  -filter ExpXml decode content

  -filter LocationHist remove object

  -normalize pubmed

  -wrp PubmedArticleSet -pattern PubmedArticle -format

Sequence Substitution

  echo ATGAAACCCGGGTTTTAG |
  transmute -replace -offset 5 -delete 1 -insert G

Protein Translation

  echo "CTAAAACCCGGGTTTCAT" |
  transmute -revcomp |
  transmute -cds2prot

Variation Extraction

  echo "NP_000504.1:p.Glu41Lys,NP_000504.1:p.P43Leu,NP_000504.1:p.Trp142Ter" |
  transmute -hgvs | transmute -format

Sequence Comparison

  transmute -diff <( echo "MKPGSQPVIY" ) <( echo "-KPGFQ*VIY" )

Translation of Coding Regions

  efetch -db nuccore -id U54469 -format gb |
  transmute -g2x |
  xtract -insd CDS sub_sequence |
  cut -f 2 |
  while read seq
  do
    echo "$seq" |
    transmute -cds2prot
    echo ""
  done

Mitochondrial Mistranslation

  efetch -db nuccore -id NC_012920 -format gb |
  transmute -g2x |
  xtract -insd CDS gene product protein_id translation sub_sequence |
  while IFS=$'\t' read acc gene prod prid prot seq
  do
    mito=$( echo "$seq" | transmute -cds2prot -code 2 -stop )
    norm=$( echo "$seq" | transmute -cds2prot -code 1 -stop )
    if [ "$mito" != "$norm" ]
    then
      echo ">$acc $gene $prid $prod"
      transmute -diff <( echo "$mito" ) <( echo "$norm" )
      echo ""
    fi
  done

Systematic Mutations

  echo ATGAAACCCGGGTTTTAG |
  while read seq
  do
    for (( i=0; i<${#seq}; i++ ))
    do
      ch="${seq:$i:1}"
      for sub in A C G T
      do
        echo "$seq" |
        transmute -replace -offset "$i" -delete "$ch" -insert "$sub"
      done
    done
  done |
  while read seq
  do
    tns=$( echo "$seq" | transmute -cds2prot )
    mwt=$( echo "$tns" | transmute -molwt )
    echo -e "${seq}\t${tns}\t${mwt}"
  done
`

const transmuteExtra = `
Mismatch Detection (RefSeq Proteins with 3 Residue Differences from RefSeq Genome)

  esearch -db gene -query "DMD [GENE] AND human [ORGN]" |
  efetch -format docsum |
  xtract -pattern DocumentSummary -block GenomicInfoType \
    -tab "\n" -element ChrAccVer,ChrStart,ChrStop |
  xargs -n 3 sh -c 'efetch -db nuccore -format gbc \
    -id "$0" -chr_start "$1" -chr_stop "$2"' > dystrophin.xml

  cat dystrophin.xml |
  xtract -insd CDS gene product translation sub_sequence > dystrophin.txt

  cat dystrophin.txt |
  while IFS=$'\t' read acc gene prod prot seq
  do
    trans=$( echo "$seq" | transmute -cds2prot )
    if [ "$prot" != "$trans" ]
    then
      echo ">$acc $gene $prod"
      transmute -diff <( echo "$prot" ) <( echo "$trans" )
      echo ""
    fi
  done > failures.txt
`

// UTILITIES

func ParseMarkup(str, cmd string) MarkupPolicy {

	switch str {
	case "fuse", "fused":
		return FUSE
	case "space", "spaces":
		return SPACE
	case "period", "periods":
		return PERIOD
	case "bracket", "brackets":
		return BRACKETS
	case "markdown":
		return MARKDOWN
	case "slash":
		return SLASH
	case "tag", "tags":
		return TAGS
	case "terse":
		return TERSE
	default:
		if str != "" {
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized %s value '%s'\n", cmd, str)
			os.Exit(1)
		}
	}
	return NOMARKUP
}

// XML FORMATTING FUNCTIONS

// CreateFormatters does concurrent reformatting, using flush-left to remove leading spaces
func CreateFormatters(parent string, inp <-chan Extract) <-chan Extract {

	if inp == nil {
		return nil
	}

	out := make(chan Extract, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create formatter channel\n")
		os.Exit(1)
	}

	// xmlFormatter reads partitioned XML from channel and formats on a per-record basis
	xmlFormatter := func(wg *sync.WaitGroup, inp <-chan Extract, out chan<- Extract) {

		// report when this formatter has no more records to process
		defer wg.Done()

		// clean and reformat one record, flush left
		doFormat := func(text string) string {

			var buffer strings.Builder

			// print attributes
			printAttributes := func(attr string) {

				if attr == "" {
					return
				}
				attr = strings.TrimSpace(attr)
				attr = CompressRunsOfSpaces(attr)

				if DeAccent {
					if IsNotASCII(attr) {
						attr = DoAccentTransform(attr)
					}
				}
				if DoASCII {
					if IsNotASCII(attr) {
						attr = UnicodeToASCII(attr)
					}
				}

				buffer.WriteString(" ")
				buffer.WriteString(attr)
			}

			ret := "\n"
			pfx := ""
			skip := 0

			doCleanup := func(tkn Token, nxtTag TagType, nxtName string) {

				if skip > 0 {
					skip--
					return
				}

				name := tkn.Name

				switch tkn.Tag {
				case STARTTAG:
					// convert start-stop to self-closing tag if attributes are present, otherwise skip
					if nxtTag == STOPTAG && nxtName == name {
						if tkn.Attr != "" {
							buffer.WriteString(pfx)
							buffer.WriteString("<")
							buffer.WriteString(name)
							printAttributes(tkn.Attr)
							buffer.WriteString("/>")
							pfx = ret
						}
						skip++
						return
					}
					buffer.WriteString(pfx)
					buffer.WriteString("<")
					buffer.WriteString(name)
					printAttributes(tkn.Attr)
					buffer.WriteString(">")
					pfx = ret
				case SELFTAG:
					if tkn.Attr != "" {
						buffer.WriteString(pfx)
						buffer.WriteString("<")
						buffer.WriteString(name)
						printAttributes(tkn.Attr)
						buffer.WriteString("/>")
						pfx = ret
					}
				case STOPTAG:
					buffer.WriteString(pfx)
					buffer.WriteString("</")
					buffer.WriteString(name)
					buffer.WriteString(">")
					if DoMixed && nxtTag == CONTENTTAG {
						buffer.WriteString(" ")
					}
					pfx = ret
				case CONTENTTAG:
					if nxtTag == STARTTAG || nxtTag == SELFTAG {
						if DoStrict {
							fmt.Fprintf(os.Stderr, "ERROR: UNRECOGNIZED MIXED CONTENT <%s> in <%s>\n", nxtName, name)
						} else if !DoMixed {
							fmt.Fprintf(os.Stderr, "ERROR: UNEXPECTED MIXED CONTENT <%s> in <%s>\n", nxtName, name)
						}
					}
					if len(name) > 0 && IsNotJustWhitespace(name) {
						// support for all content processing flags
						if DoStrict || DoMixed || DoCompress {
							ctype := tkn.Cont
							name = CleanupContents(name, (ctype&ASCII) != 0, (ctype&AMPER) != 0, (ctype&MIXED) != 0)
						}
						buffer.WriteString(name)
					}
					if DoMixed && nxtTag == STARTTAG {
						buffer.WriteString(" ")
					}
					pfx = ""
				case CDATATAG, COMMENTTAG:
					// ignore
				case DOCTYPETAG:
				case NOTAG:
				case ISCLOSED:
					// now handled at end of calling function
				default:
					buffer.WriteString(pfx)
					pfx = ""
				}
			}

			var prev Token
			primed := false

			// track adjacent pairs to give look-ahead at next token
			doPair := func(tkn Token) {

				if primed {
					doCleanup(prev, tkn.Tag, tkn.Name)
				}

				prev = Token{tkn.Tag, tkn.Cont, tkn.Name, tkn.Attr, tkn.Index, tkn.Line}
				primed = true
			}

			ParseXML(text, parent, nil, doPair, nil, nil)

			// isclosed tag
			if primed {
				buffer.WriteString(pfx)
			}

			txt := buffer.String()

			if DoMixed {
				// clean up reconstructed mixed content
				txt = DoTrimFlankingHTML(txt)
				if HasBadSpace(txt) {
					txt = CleanupBadSpaces(txt)
				}
				if HasAdjacentSpaces(txt) {
					txt = CompressRunsOfSpaces(txt)
				}
				if NeedsTightening(txt) {
					txt = TightenParentheses(txt)
				}
			}

			return txt
		}

		// read partitioned XML from producer channel
		for ext := range inp {

			idx := ext.Index
			text := ext.Text

			if text == "" {
				// should never see empty input data
				out <- Extract{idx, "", text, nil}
				continue
			}

			str := doFormat(text[:])

			// send even if empty to get all record counts for reordering
			out <- Extract{idx, "", str, nil}
		}
	}

	var wg sync.WaitGroup

	// launch multiple formatter goroutines
	for i := 0; i < NumServe; i++ {
		wg.Add(1)
		go xmlFormatter(&wg, inp, out)
	}

	// launch separate anonymous goroutine to wait until all formatters are done
	go func() {
		wg.Wait()
		close(out)
	}()

	return out
}

// ProcessTokens shows individual tokens in stream (undocumented)
func ProcessTokens(rdr <-chan string) {

	if rdr == nil {
		return
	}

	tknq := CreateTokenizer(rdr)

	if tknq == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create debug tokenizer\n")
		os.Exit(1)
	}

	var buffer strings.Builder

	count := 0
	indent := 0

	for tkn := range tknq {

		tag := tkn.Tag
		name := tkn.Name
		attr := tkn.Attr

		switch tag {
		case STARTTAG:
			buffer.WriteString("ST: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
			if attr != "" {
				buffer.WriteString("AT: ")
				for i := 0; i < indent; i++ {
					buffer.WriteString("  ")
				}
				buffer.WriteString(attr)
				buffer.WriteString("\n")
			}
			indent++
		case SELFTAG:
			buffer.WriteString("SL: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("/")
			buffer.WriteString("\n")
			if attr != "" {
				buffer.WriteString("AT: ")
				for i := 0; i < indent; i++ {
					buffer.WriteString("  ")
				}
				buffer.WriteString(attr)
				buffer.WriteString("\n")
			}
		case STOPTAG:
			indent--
			buffer.WriteString("SP: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("/")
			buffer.WriteString("\n")
		case CONTENTTAG:
			ctype := tkn.Cont
			if (ctype & LFTSPACE) != 0 {
				if (ctype & RGTSPACE) != 0 {
					buffer.WriteString("FL: ")
				} else {
					buffer.WriteString("LF: ")
				}
			} else if (ctype & RGTSPACE) != 0 {
				buffer.WriteString("RT: ")
			} else {
				buffer.WriteString("VL: ")
			}
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case CDATATAG:
			buffer.WriteString("CD: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case COMMENTTAG:
			buffer.WriteString("CO: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case DOCTYPETAG:
			buffer.WriteString("DC: ")
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case NOTAG:
			buffer.WriteString("NO:")
			if indent != 0 {
				buffer.WriteString(" (indent ")
				buffer.WriteString(strconv.Itoa(indent))
				buffer.WriteString(")")
			}
			buffer.WriteString("\n")
		case ISCLOSED:
			buffer.WriteString("CL:")
			if indent != 0 {
				buffer.WriteString(" (indent ")
				buffer.WriteString(strconv.Itoa(indent))
				buffer.WriteString(")")
			}
			buffer.WriteString("\n")
			txt := buffer.String()
			if txt != "" {
				// print final buffer
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			return
		default:
			buffer.WriteString("UNKONWN:")
			if indent != 0 {
				buffer.WriteString(" (indent ")
				buffer.WriteString(strconv.Itoa(indent))
				buffer.WriteString(")")
			}
			buffer.WriteString("\n")
		}

		count++
		if count > 1000 {
			count = 0
			txt := buffer.String()
			if txt != "" {
				// print current buffered output
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			buffer.Reset()
		}
	}
}

// ProcessFormat reformats XML for ease of reading
func ProcessFormat(rdr <-chan string, args []string, useTimer bool) int {

	if rdr == nil || args == nil {
		return 0
	}

	var buffer strings.Builder
	count := 0
	maxLine := 0

	// skip past command name
	args = args[1:]

	copyRecrd := false
	compRecrd := false
	flushLeft := false
	wrapAttrs := false
	ret := "\n"

	xml := ""
	customDoctype := false
	doctype := ""

	doComment := false
	doCdata := false

	// look for [copy|compact|flush|indent|expand] specification
	if len(args) > 0 {
		inSwitch := true

		switch args[0] {
		case "compact", "compacted", "compress", "compressed", "terse", "*":
			// compress to one record per line
			compRecrd = true
			ret = ""
		case "flush", "flushed", "left":
			// suppress line indentation
			flushLeft = true
		case "expand", "expanded", "extend", "extended", "verbose", "@":
			// each attribute on its own line
			wrapAttrs = true
		case "indent", "indented", "normal", "default":
			// default behavior
		case "copy":
			// fast block copy
			copyRecrd = true
		default:
			// if not any of the controls, will check later for -xml and -doctype arguments
			inSwitch = false
		}

		if inSwitch {
			// skip past first argument
			args = args[1:]
		}
	}

	// fast block copy, supporting only -spaces processing flag
	if copyRecrd {

		for str := range rdr {
			if str == "" {
				break
			}

			os.Stdout.WriteString(str)
		}
		os.Stdout.WriteString("\n")
		return 0
	}

	unicodePolicy := ""
	scriptPolicy := ""
	mathmlPolicy := ""

	fuseTopSets := true
	keepEmptySelfClosing := false

	// look for -xml and -doctype arguments (undocumented)
	for len(args) > 0 {

		switch args[0] {
		case "-xml":
			args = args[1:]
			// -xml argument must be followed by value to use in xml line
			if len(args) < 1 || strings.HasPrefix(args[0], "-") {
				fmt.Fprintf(os.Stderr, "\nERROR: -xml argument is missing\n")
				os.Exit(1)
			}
			xml = args[0]
			args = args[1:]
		case "-doctype":
			customDoctype = true
			args = args[1:]
			if len(args) > 0 && !strings.HasPrefix(args[0], "-") {
				// if -doctype argument followed by value, use instead of DOCTYPE line
				doctype = args[0]
				args = args[1:]
			}

		// allow setting of unicode, script, and mathml flags within -format
		case "-unicode":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Unicode argument is missing\n")
				os.Exit(1)
			}
			unicodePolicy = args[1]
			args = args[2:]
		case "-script":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Script argument is missing\n")
				os.Exit(1)
			}
			scriptPolicy = args[1]
			args = args[2:]
		case "-mathml":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: MathML argument is missing\n")
				os.Exit(1)
			}
			mathmlPolicy = args[1]
			args = args[2:]

		// also allow setting additional processing flags within -format (undocumented)
		case "-strict":
			// can set -strict within -format
			DoStrict = true
			args = args[1:]
		case "-mixed":
			// can set -mixed within -format
			DoMixed = true
			args = args[1:]
		case "-compress", "-compressed":
			DoCompress = true
			args = args[1:]
		case "-accent":
			DeAccent = true
			args = args[1:]
		case "-ascii":
			DoASCII = true
			args = args[1:]
		case "-separate", "-separated":
			fuseTopSets = false
			args = args[1:]
		case "-self", "-self-closing":
			keepEmptySelfClosing = true
			args = args[1:]
		case "-comment":
			doComment = true
			args = args[1:]
		case "-cdata":
			doCdata = true
			args = args[1:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -format command\n")
			os.Exit(1)
		}
	}

	UnicodeFix = ParseMarkup(unicodePolicy, "-unicode")
	ScriptFix = ParseMarkup(scriptPolicy, "-script")
	MathMLFix = ParseMarkup(mathmlPolicy, "-mathml")

	if UnicodeFix != NOMARKUP {
		DoUnicode = true
	}

	if ScriptFix != NOMARKUP {
		DoScript = true
	}

	if MathMLFix != NOMARKUP {
		DoMathML = true
	}

	CountLines = DoMixed || useTimer
	AllowEmbed = DoStrict || DoMixed
	ContentMods = AllowEmbed || DoCompress || DoUnicode || DoScript || DoMathML || DeAccent || DoASCII

	// array to speed up indentation
	indentSpaces := []string{
		"",
		"  ",
		"    ",
		"      ",
		"        ",
		"          ",
		"            ",
		"              ",
		"                ",
		"                  ",
	}

	indent := 0

	// indent a specified number of spaces
	doIndent := func(indt int) {
		if compRecrd || flushLeft {
			return
		}
		i := indt
		for i > 9 {
			buffer.WriteString("                    ")
			i -= 10
		}
		if i < 0 {
			return
		}
		buffer.WriteString(indentSpaces[i])
	}

	badAttributeSpacing := func(attr string) bool {

		if len(attr) < 2 {
			return false
		}

		var prev rune

		for _, ch := range attr {
			if ch == '=' && prev == ' ' {
				return true
			}
			if ch == ' ' && prev == '=' {
				return true
			}
			prev = ch
		}

		return false
	}

	// print attributes
	printAttributes := func(attr string) {

		if attr == "" {
			return
		}
		attr = strings.TrimSpace(attr)
		attr = CompressRunsOfSpaces(attr)

		if DeAccent {
			if IsNotASCII(attr) {
				attr = DoAccentTransform(attr)
			}
		}
		if DoASCII {
			if IsNotASCII(attr) {
				attr = UnicodeToASCII(attr)
			}
		}

		if wrapAttrs {

			start := 0
			idx := 0
			inQuote := false

			attlen := len(attr)

			for idx < attlen {
				ch := attr[idx]
				if ch == '=' && !inQuote {
					inQuote = true
					str := strings.TrimSpace(attr[start:idx])
					buffer.WriteString("\n")
					doIndent(indent)
					buffer.WriteString(" ")
					buffer.WriteString(str)
					// skip past equal sign
					idx++
					ch = attr[idx]
					if ch != '"' && ch != '\'' {
						// "
						// skip past unexpected blanks
						for InBlank[ch] {
							idx++
							ch = attr[idx]
						}
					}
					// skip past leading double quote
					idx++
					start = idx
				} else if ch == '"' || ch == '\'' {
					// "
					inQuote = !inQuote
					str := strings.TrimSpace(attr[start:idx])
					buffer.WriteString("=\"")
					buffer.WriteString(str)
					buffer.WriteString("\"")
					// skip past trailing double quote and (possible) space
					idx += 2
					start = idx
				} else {
					idx++
				}
			}

			buffer.WriteString("\n")
			doIndent(indent)

		} else if badAttributeSpacing(attr) {

			buffer.WriteString(" ")

			start := 0
			idx := 0

			attlen := len(attr)

			for idx < attlen {
				ch := attr[idx]
				if ch == '=' {
					str := strings.TrimSpace(attr[start:idx])
					buffer.WriteString(str)
					// skip past equal sign
					idx++
					ch = attr[idx]
					if ch != '"' && ch != '\'' {
						// "
						// skip past unexpected blanks
						for InBlank[ch] {
							idx++
							ch = attr[idx]
						}
					}
					// skip past leading double quote
					idx++
					start = idx
				} else if ch == '"' || ch == '\'' {
					// "
					str := strings.TrimSpace(attr[start:idx])
					buffer.WriteString("=\"")
					buffer.WriteString(str)
					buffer.WriteString("\"")
					// skip past trailing double quote and (possible) space
					idx += 2
					start = idx
				} else {
					idx++
				}
			}

		} else {

			buffer.WriteString(" ")
			buffer.WriteString(attr)
		}
	}

	parent := ""
	pfx := ""
	skip := 0
	okIndent := true

	printXMLAndDoctype := func(xml, doctype, parent string) {

		// check for xml line explicitly set in argument
		if xml != "" {
			xml = strings.TrimSpace(xml)
			if strings.HasPrefix(xml, "<") {
				xml = xml[1:]
			}
			if strings.HasPrefix(xml, "?") {
				xml = xml[1:]
			}
			if strings.HasPrefix(xml, "xml") {
				xml = xml[3:]
			}
			if strings.HasPrefix(xml, " ") {
				xml = xml[1:]
			}
			if strings.HasSuffix(xml, "?>") {
				xlen := len(xml)
				xml = xml[:xlen-2]
			}
			xml = strings.TrimSpace(xml)

			buffer.WriteString("<?xml ")
			buffer.WriteString(xml)
			buffer.WriteString(" ?>")
		} else {
			buffer.WriteString("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>")
		}

		buffer.WriteString("\n")

		// check for doctype taken from XML file or explicitly set in argument
		if doctype != "" {
			doctype = strings.TrimSpace(doctype)
			if strings.HasPrefix(doctype, "<") {
				doctype = doctype[1:]
			}
			if strings.HasPrefix(doctype, "!") {
				doctype = doctype[1:]
			}
			if strings.HasPrefix(doctype, "DOCTYPE") {
				doctype = doctype[7:]
			}
			if strings.HasPrefix(doctype, " ") {
				doctype = doctype[1:]
			}
			doctype = strings.TrimSuffix(doctype, ">")
			doctype = strings.TrimSpace(doctype)

			buffer.WriteString("<!DOCTYPE ")
			buffer.WriteString(doctype)
			buffer.WriteString(">")
		} else {
			buffer.WriteString("<!DOCTYPE ")
			buffer.WriteString(parent)
			buffer.WriteString(">")
		}

		buffer.WriteString("\n")
	}

	cleanupMixed := func(txt string) string {

		txt = DoTrimFlankingHTML(txt)
		if HasBadSpace(txt) {
			txt = CleanupBadSpaces(txt)
		}
		if HasAdjacentSpaces(txt) {
			txt = CompressRunsOfSpaces(txt)
		}
		if NeedsTightening(txt) {
			txt = TightenParentheses(txt)
		}

		return txt
	}

	doCleanup := func(tkn Token, nxtTag TagType, nxtName, nxtAttr string) {

		if skip > 0 {
			skip--
			return
		}

		name := tkn.Name

		switch tkn.Tag {
		case STARTTAG:
			// detect first start tag, print xml and doctype parent
			if indent == 0 && parent == "" {
				parent = name
				printXMLAndDoctype(xml, doctype, parent)
				// do not fuse <opt> or <anon> top-level objects (converted from JSON)
				if parent == "opt" || parent == "anon" {
					fuseTopSets = false
				}
			}
			// convert start-stop to self-closing tag if attributes are present, otherwise skip
			if nxtTag == STOPTAG && nxtName == name {
				if tkn.Attr != "" || keepEmptySelfClosing {
					buffer.WriteString(pfx)
					doIndent(indent)
					buffer.WriteString("<")
					buffer.WriteString(name)
					printAttributes(tkn.Attr)
					buffer.WriteString("/>")
					pfx = ret
					okIndent = true
				}
				skip++
				return
			}
			buffer.WriteString(pfx)
			doIndent(indent)
			indent++
			buffer.WriteString("<")
			buffer.WriteString(name)
			printAttributes(tkn.Attr)
			buffer.WriteString(">")
			pfx = ret
			okIndent = true
			if compRecrd && indent == 1 {
				buffer.WriteString("\n")
			}
		case SELFTAG:
			if tkn.Attr != "" || keepEmptySelfClosing {
				buffer.WriteString(pfx)
				doIndent(indent)
				buffer.WriteString("<")
				buffer.WriteString(name)
				printAttributes(tkn.Attr)
				buffer.WriteString("/>")
				pfx = ret
				okIndent = true
			}
		case STOPTAG:
			// skip internal copies of top-level </parent><parent> tags (to fuse multiple chunks returned by efetch)
			// do not skip if attributes are present, unless pattern ends in "Set" (e.g., <DocumentSummarySet status="OK">)
			if nxtTag == STARTTAG && indent == 1 && fuseTopSets && nxtName == parent {
				if nxtAttr == "" || (strings.HasSuffix(parent, "Set") && len(parent) > 3) {
					skip++
					return
				}
			}
			buffer.WriteString(pfx)
			indent--
			if okIndent {
				doIndent(indent)
			}
			buffer.WriteString("</")
			buffer.WriteString(name)
			buffer.WriteString(">")
			if DoMixed && nxtTag == CONTENTTAG {
				buffer.WriteString(" ")
			}
			pfx = ret
			okIndent = true
			if compRecrd && indent < 2 {
				buffer.WriteString("\n")
			}
		case CONTENTTAG:
			if nxtTag == STARTTAG || nxtTag == SELFTAG {
				if DoStrict {
					fmt.Fprintf(os.Stderr, "ERROR: UNRECOGNIZED MIXED CONTENT <%s> in <%s>\n", nxtName, name)
				} else if !DoMixed {
					fmt.Fprintf(os.Stderr, "ERROR: UNEXPECTED MIXED CONTENT <%s> in <%s>\n", nxtName, name)
				}
			}
			if len(name) > 0 && IsNotJustWhitespace(name) {
				// support for all content processing flags
				if DoStrict || DoMixed || DoCompress {
					ctype := tkn.Cont
					name = CleanupContents(name, (ctype&ASCII) != 0, (ctype&AMPER) != 0, (ctype&MIXED) != 0)
				}
				buffer.WriteString(name)
			}
			if DoMixed && nxtTag == STARTTAG {
				buffer.WriteString(" ")
			}
			pfx = ""
			okIndent = false
		case CDATATAG:
			if doCdata {
				buffer.WriteString(pfx)
				doIndent(indent)
				buffer.WriteString("<![CDATA[")
				buffer.WriteString(name)
				buffer.WriteString("]]>")
				pfx = ret
				okIndent = true
			}
		case COMMENTTAG:
			if doComment {
				buffer.WriteString(pfx)
				doIndent(indent)
				buffer.WriteString("<!--")
				buffer.WriteString(name)
				buffer.WriteString("-->")
				pfx = ret
				okIndent = true
			}
		case DOCTYPETAG:
			if customDoctype && doctype == "" {
				doctype = name
			}
		case NOTAG:
		case ISCLOSED:
			// now handled at end of calling function
		default:
			buffer.WriteString(pfx)
			pfx = ""
			okIndent = false
		}

		count++
		if count > 1000 {
			count = 0
			txt := buffer.String()
			if DoMixed {
				txt = cleanupMixed(txt)
			}
			if txt != "" {
				// print current buffered output
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			buffer.Reset()
		}
	}

	var prev Token
	primed := false
	skipDoctype := false

	// track adjacent pairs to give look-ahead at next token
	doPair := func(tkn Token) {

		if tkn.Tag == NOTAG {
			return
		}

		maxLine = tkn.Line

		if tkn.Tag == DOCTYPETAG {
			if skipDoctype {
				return
			}
			skipDoctype = true
		}

		if primed {
			doCleanup(prev, tkn.Tag, tkn.Name, tkn.Attr)
		}

		prev = Token{tkn.Tag, tkn.Cont, tkn.Name, tkn.Attr, tkn.Index, tkn.Line}
		primed = true
	}

	ParseXML("", "", rdr, doPair, nil, nil)

	// handle isclosed tag
	if primed {
		buffer.WriteString(pfx)
		txt := buffer.String()
		if DoMixed {
			txt = cleanupMixed(txt)
		}
		if txt != "" {
			// print final buffer
			fmt.Fprintf(os.Stdout, "%s", txt)
		}
	}

	return maxLine
}

// ProcessOutline displays outline of XML structure
func ProcessOutline(rdr <-chan string) {

	if rdr == nil {
		return
	}

	tknq := CreateTokenizer(rdr)

	if tknq == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create outline tokenizer\n")
		os.Exit(1)
	}

	var buffer strings.Builder

	count := 0
	indent := 0

	for tkn := range tknq {

		tag := tkn.Tag
		name := tkn.Name

		switch tag {
		case STARTTAG:
			if name == "eSummaryResult" ||
				name == "eLinkResult" ||
				name == "eInfoResult" ||
				name == "PubmedArticleSet" ||
				name == "DocumentSummarySet" ||
				name == "INSDSet" ||
				name == "Entrezgene-Set" ||
				name == "TaxaSet" {
				break
			}
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
			indent++
		case SELFTAG:
			for i := 0; i < indent; i++ {
				buffer.WriteString("  ")
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case STOPTAG:
			indent--
		case DOCTYPETAG:
		case NOTAG:
		case ISCLOSED:
			txt := buffer.String()
			if txt != "" {
				// print final buffer
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			return
		default:
		}

		count++
		if count > 1000 {
			count = 0
			txt := buffer.String()
			if txt != "" {
				// print current buffered output
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			buffer.Reset()
		}
	}
}

// ProcessSynopsis displays paths to XML elements
func ProcessSynopsis(rdr <-chan string, leaf bool, delim string) {

	if rdr == nil {
		return
	}

	tknq := CreateTokenizer(rdr)

	if tknq == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create synopsis tokenizer\n")
		os.Exit(1)
	}

	var buffer strings.Builder
	count := 0

	// synopsisLevel recursive definition
	var synopsisLevel func(string) bool

	synopsisLevel = func(parent string) bool {

		for tkn := range tknq {

			tag := tkn.Tag
			name := tkn.Name

			switch tag {
			case STARTTAG:
				if name == "eSummaryResult" ||
					name == "eLinkResult" ||
					name == "eInfoResult" ||
					name == "PubmedArticleSet" ||
					name == "DocumentSummarySet" ||
					name == "INSDSet" ||
					name == "Entrezgene-Set" ||
					name == "TaxaSet" {
					break
				}
				if leaf {
					if name == "root" ||
						name == "opt" ||
						name == "anon" {
						break
					}
				}
				if !leaf {
					// show all paths, including container objects
					if parent != "" {
						buffer.WriteString(parent)
						buffer.WriteString(delim)
					}
					buffer.WriteString(name)
					buffer.WriteString("\n")
				}
				path := parent
				if path != "" {
					path += delim
				}
				path += name
				if synopsisLevel(path) {
					return true
				}
			case SELFTAG:
				if parent != "" {
					buffer.WriteString(parent)
					buffer.WriteString(delim)
				}
				buffer.WriteString(name)
				buffer.WriteString("\n")
			case STOPTAG:
				// break recursion
				return false
			case CONTENTTAG:
				if leaf {
					// only show endpoint paths
					if parent != "" {
						buffer.WriteString(parent)
						buffer.WriteString("\n")
					}
				}
			case DOCTYPETAG:
			case NOTAG:
			case ISCLOSED:
				txt := buffer.String()
				if txt != "" {
					// print final buffer
					fmt.Fprintf(os.Stdout, "%s", txt)
				}
				return true
			default:
			}

			count++
			if count > 1000 {
				count = 0
				txt := buffer.String()
				if txt != "" {
					// print current buffered output
					fmt.Fprintf(os.Stdout, "%s", txt)
				}
				buffer.Reset()
			}
		}
		return true
	}

	for {
		// may have concatenated XMLs, loop through all
		if synopsisLevel("") {
			return
		}
	}
}

// ProcessFilter modifies XML content, comments, or CDATA
func ProcessFilter(rdr <-chan string, args []string) {

	if rdr == nil || args == nil {
		return
	}

	tknq := CreateTokenizer(rdr)

	if tknq == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create filter tokenizer\n")
		os.Exit(1)
	}

	var buffer strings.Builder

	count := 0

	// skip past command name
	args = args[1:]

	max := len(args)
	if max < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: Insufficient command-line arguments supplied to transmute -filter\n")
		os.Exit(1)
	}

	pttrn := args[0]

	args = args[1:]
	max--

	if max < 2 {
		fmt.Fprintf(os.Stderr, "\nERROR: No object name supplied to transmute -filter\n")
		os.Exit(1)
	}

	type ActionType int

	const (
		NOACTION ActionType = iota
		DORETAIN
		DOREMOVE
		DOENCODE
		DODECODE
		DOSHRINK
		DOEXPAND
		DOACCENT
	)

	action := args[0]

	what := NOACTION
	switch action {
	case "retain":
		what = DORETAIN
	case "remove":
		what = DOREMOVE
	case "encode":
		what = DOENCODE
	case "decode":
		what = DODECODE
	case "shrink":
		what = DOSHRINK
	case "expand":
		what = DOEXPAND
	case "accent":
		what = DOACCENT
	default:
		fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized action '%s' supplied to transmute -filter\n", action)
		os.Exit(1)
	}

	trget := args[1]

	which := NOTAG
	switch trget {
	case "attribute", "attributes":
		which = ATTRIBTAG
	case "content", "contents":
		which = CONTENTTAG
	case "cdata", "CDATA":
		which = CDATATAG
	case "comment", "comments":
		which = COMMENTTAG
	case "object":
		// object normally retained
		which = OBJECTTAG
	case "container":
		which = CONTAINERTAG
	default:
		fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized target '%s' supplied to transmute -filter\n", trget)
		os.Exit(1)
	}

	inPattern := false
	prevName := ""

	for tkn := range tknq {

		tag := tkn.Tag
		name := tkn.Name
		attr := tkn.Attr

		switch tag {
		case STARTTAG:
			prevName = name
			if name == pttrn {
				inPattern = true
				if which == CONTAINERTAG && what == DOREMOVE {
					continue
				}
			}
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			buffer.WriteString("<")
			buffer.WriteString(name)
			if attr != "" {
				if which != ATTRIBTAG || what != DOREMOVE {
					attr = strings.TrimSpace(attr)
					attr = CompressRunsOfSpaces(attr)
					buffer.WriteString(" ")
					buffer.WriteString(attr)
				}
			}
			buffer.WriteString(">\n")
		case SELFTAG:
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			buffer.WriteString("<")
			buffer.WriteString(name)
			if attr != "" {
				if which != ATTRIBTAG || what != DOREMOVE {
					attr = strings.TrimSpace(attr)
					attr = CompressRunsOfSpaces(attr)
					buffer.WriteString(" ")
					buffer.WriteString(attr)
				}
			}
			buffer.WriteString("/>\n")
		case STOPTAG:
			if name == pttrn {
				inPattern = false
				if which == OBJECTTAG && what == DOREMOVE {
					continue
				}
				if which == CONTAINERTAG && what == DOREMOVE {
					continue
				}
			}
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			buffer.WriteString("</")
			buffer.WriteString(name)
			buffer.WriteString(">\n")
		case CONTENTTAG:
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			if inPattern && which == CONTENTTAG && what == DOEXPAND {
				var words []string
				if strings.Contains(name, "|") {
					words = strings.FieldsFunc(name, func(c rune) bool {
						return c == '|'
					})
				} else if strings.Contains(name, ",") {
					words = strings.FieldsFunc(name, func(c rune) bool {
						return c == ','
					})
				} else {
					words = strings.Fields(name)
				}
				between := ""
				for _, item := range words {
					max := len(item)
					for max > 1 {
						ch := item[max-1]
						if ch != '.' && ch != ',' && ch != ':' && ch != ';' {
							break
						}
						// trim trailing punctuation
						item = item[:max-1]
						// continue checking for runs of punctuation at end
						max--
					}
					if HasFlankingSpace(item) {
						item = strings.TrimSpace(item)
					}
					if item != "" {
						if between != "" {
							buffer.WriteString(between)
						}
						buffer.WriteString(item)
						buffer.WriteString("\n")
						between = "</" + prevName + ">\n<" + prevName + ">\n"
					}
				}
				continue
			}
			if inPattern && which == tag {
				switch what {
				case DORETAIN:
					// default behavior for content - can use -filter X retain content as a no-op
				case DOREMOVE:
					continue
				case DOENCODE:
					name = html.EscapeString(name)
				case DODECODE:
					name = html.UnescapeString(name)
				case DOSHRINK:
					name = CompressRunsOfSpaces(name)
				case DOACCENT:
					if IsNotASCII(name) {
						name = DoAccentTransform(name)
					}
				default:
					continue
				}
			}
			// content normally printed
			if HasFlankingSpace(name) {
				name = strings.TrimSpace(name)
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case CDATATAG:
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			if inPattern && which == tag {
				switch what {
				case DORETAIN:
					// cdata requires explicit retain command
				case DOREMOVE:
					continue
				case DOENCODE:
					name = html.EscapeString(name)
				case DODECODE:
					name = html.UnescapeString(name)
				case DOSHRINK:
					name = CompressRunsOfSpaces(name)
				case DOACCENT:
					if IsNotASCII(name) {
						name = DoAccentTransform(name)
					}
				default:
					continue
				}
				// cdata normally removed
				if HasFlankingSpace(name) {
					name = strings.TrimSpace(name)
				}
				buffer.WriteString(name)
				buffer.WriteString("\n")
			}
		case COMMENTTAG:
			if inPattern && which == OBJECTTAG && what == DOREMOVE {
				continue
			}
			if inPattern && which == tag {
				switch what {
				case DORETAIN:
					// comment requires explicit retain command
				case DOREMOVE:
					continue
				case DOENCODE:
					name = html.EscapeString(name)
				case DODECODE:
					name = html.UnescapeString(name)
				case DOSHRINK:
					name = CompressRunsOfSpaces(name)
				case DOACCENT:
					if IsNotASCII(name) {
						name = DoAccentTransform(name)
					}
				default:
					continue
				}
				// comment normally removed
				if HasFlankingSpace(name) {
					name = strings.TrimSpace(name)
				}
				buffer.WriteString(name)
				buffer.WriteString("\n")
			}
		case DOCTYPETAG:
		case NOTAG:
		case ISCLOSED:
			txt := buffer.String()
			if txt != "" {
				// print final buffer
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			return
		default:
		}

		count++
		if count > 1000 {
			count = 0
			txt := buffer.String()
			if txt != "" {
				// print current buffered output
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			buffer.Reset()
		}
	}
}

// ProcessNormalize adjusts XML fields to conform to common conventions
func ProcessNormalize(rdr <-chan string, args []string) {

	if rdr == nil || args == nil {
		return
	}

	tknq := CreateTokenizer(rdr)

	if tknq == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create normalize tokenizer\n")
		os.Exit(1)
	}

	var buffer strings.Builder

	count := 0

	// e.g., transmute -normalize pubmed

	// skip past command name
	args = args[1:]

	max := len(args)
	if max < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: No database name supplied to transmute -normalize\n")
		os.Exit(1)
	}

	db := args[0]

	// force -strict cleanup flag
	switch db {
	case "bioc", "biocollections", "clinvar", "dbvar", "gap", "gapplus", "grasp", "pccompound", "pcsubstance":
	default:
		DoStrict = true
	}
	AllowEmbed = true
	ContentMods = true

	uid := ""
	isDocsum := false
	prevName := ""

	// removes mixed content tags
	mfix := strings.NewReplacer(
		"<b>", "",
		"<i>", "",
		"<u>", "",
		"</b>", "",
		"</i>", "",
		"</u>", "",
		"<b/>", "",
		"<i/>", "",
		"<u/>", "",
		"<sup>", "",
		"<sub>", "",
		"</sup>", "",
		"</sub>", "",
		"<sup/>", "",
		"<sub/>", "",
	)

	// reencodes < and > to &lt and &gt
	rfix := strings.NewReplacer(
		"<", "&lt;",
		">", "&gt;",
	)

	for tkn := range tknq {

		tag := tkn.Tag
		name := tkn.Name
		attr := tkn.Attr

		switch tag {
		case STARTTAG:
			if name == "Id" && uid != "" {
				uid = ""
			}
			if uid != "" {
				// if object after DocumentSummary is not already Id, create Id from rescued attribute
				buffer.WriteString("<Id>\n")
				buffer.WriteString(uid)
				buffer.WriteString("</Id>\n")
				// clear until next docsum
				uid = ""
			}
			if name == "DocumentSummary" {
				isDocsum = true
				atts := ParseAttributes(attr)
				for i := 0; i < len(atts)-1; i += 2 {
					if atts[i] == "uid" {
						// store uid from DocumentSummary
						uid = atts[i+1]
						// if uid found, remove all attributes
						attr = ""
					}
				}
			}
			buffer.WriteString("<")
			buffer.WriteString(name)
			if attr != "" {
				attr = strings.TrimSpace(attr)
				attr = CompressRunsOfSpaces(attr)
				buffer.WriteString(" ")
				buffer.WriteString(attr)
			}
			buffer.WriteString(">\n")
			prevName = name
		case SELFTAG:
			buffer.WriteString("<")
			buffer.WriteString(name)
			if attr != "" {
				attr = strings.TrimSpace(attr)
				attr = CompressRunsOfSpaces(attr)
				buffer.WriteString(" ")
				buffer.WriteString(attr)
			}
			buffer.WriteString("/>\n")
		case STOPTAG:
			buffer.WriteString("</")
			buffer.WriteString(name)
			buffer.WriteString(">\n")
		case CONTENTTAG:
			if isDocsum {
				if db == "pubmed" && prevName == "Title" {
					if strings.Contains(name, "&") ||
						strings.Contains(name, "<") ||
						strings.Contains(name, ">") {
						ctype := tkn.Cont
						name = CleanupContents(name, (ctype&ASCII) != 0, (ctype&AMPER) != 0, (ctype&MIXED) != 0)
						if HasFlankingSpace(name) {
							name = strings.TrimSpace(name)
						}
						name = html.UnescapeString(name)
						// remove mixed content tags
						name = mfix.Replace(name)
						// reencode < and > to avoid breaking XML
						if strings.Contains(name, "<") || strings.Contains(name, ">") {
							name = rfix.Replace(name)
						}
					}
				} else if db == "gene" && prevName == "Summary" {
					if strings.Contains(name, "&amp;") {
						if HasFlankingSpace(name) {
							name = strings.TrimSpace(name)
						}
						name = html.UnescapeString(name)
						// reencode < and > to avoid breaking XML
						if strings.Contains(name, "<") || strings.Contains(name, ">") {
							name = rfix.Replace(name)
						}
					}
				} else if (db == "biosample" && prevName == "SampleData") ||
					(db == "medgen" && prevName == "ConceptMeta") ||
					(db == "sra" && prevName == "ExpXml") ||
					(db == "sra" && prevName == "Runs") {
					if strings.Contains(name, "&lt;") && strings.Contains(name, "&gt;") {
						if HasFlankingSpace(name) {
							name = strings.TrimSpace(name)
						}
						name = html.UnescapeString(name)
					}
				}
			} else {
				if db == "pubmed" {
					ctype := tkn.Cont
					name = CleanupContents(name, (ctype&ASCII) != 0, (ctype&AMPER) != 0, (ctype&MIXED) != 0)
					if HasFlankingSpace(name) {
						name = strings.TrimSpace(name)
					}
				} else if db == "bioc" {
					name = CleanupContents(name, true, true, true)
					if HasFlankingSpace(name) {
						name = strings.TrimSpace(name)
					}
				}
			}
			// content normally printed
			if HasFlankingSpace(name) {
				name = strings.TrimSpace(name)
			}
			buffer.WriteString(name)
			buffer.WriteString("\n")
		case CDATATAG:
			if isDocsum {
				if db == "assembly" && prevName == "Meta" {
					if strings.Contains(name, "<") && strings.Contains(name, ">") {
						// if CDATA contains embedded XML, simply remove CDATA wrapper
						if HasFlankingSpace(name) {
							name = strings.TrimSpace(name)
						}
						buffer.WriteString(name)
						buffer.WriteString("\n")
					}
				} else if db == "gtr" && prevName == "Extra" {
					// remove entire CDATA contents
				}
			}
		case COMMENTTAG:
			if !isDocsum {
				if db == "sra" {
					if strings.Contains(name, "<") && strings.Contains(name, ">") {
						// if comment contains embedded XML, remove comment wrapper and trim to leading < bracket
						pos := strings.Index(name, "<")
						if pos > 0 {
							name = name[pos:]
						}
						if HasFlankingSpace(name) {
							name = strings.TrimSpace(name)
						}
						buffer.WriteString(name)
						buffer.WriteString("\n")
					}
				}
			}
		case DOCTYPETAG:
			doctype := strings.TrimSpace(name)
			if strings.HasPrefix(doctype, "<") {
				doctype = doctype[1:]
			}
			if strings.HasPrefix(doctype, "!") {
				doctype = doctype[1:]
			}
			if strings.HasPrefix(doctype, "DOCTYPE") {
				doctype = doctype[7:]
			}
			if strings.HasPrefix(doctype, " ") {
				doctype = doctype[1:]
			}
			doctype = strings.TrimSuffix(doctype, ">")
			doctype = strings.TrimSpace(doctype)

			buffer.WriteString("<!DOCTYPE ")
			buffer.WriteString(doctype)
			buffer.WriteString(">")
		case NOTAG:
		case ISCLOSED:
			txt := buffer.String()
			if txt != "" {
				// print final buffer
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			return
		default:
		}

		count++
		if count > 1000 {
			count = 0
			txt := buffer.String()
			if txt != "" {
				// print current buffered output
				fmt.Fprintf(os.Stdout, "%s", txt)
			}
			buffer.Reset()
		}
	}
}

// JSONTokenizer sends sequential JSON tokens down a channel
func JSONTokenizer(inp io.Reader) <-chan string {

	if inp == nil {
		return nil
	}

	out := make(chan string, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create JSON tokenizer channel\n")
		os.Exit(1)
	}

	tokenizeJSON := func(inp io.Reader, out chan<- string) {

		// close channel when all tokens have been sent
		defer close(out)

		// use token decoder from encoding/json package
		dec := json.NewDecoder(inp)
		if dec == nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to create JSON Decoder\n")
			os.Exit(1)
		}
		dec.UseNumber()

		for {
			t, err := dec.Token()
			if err == io.EOF {
				return
			}
			if err != nil {
				fmt.Fprintf(os.Stderr, "\nERROR: Unable to read JSON token '%s'\n", err)
				os.Exit(1)
			}

			// type switch performs sequential type assertions until match is found
			switch v := t.(type) {
			case json.Delim:
				// opening or closing braces (for objects) or brackets (for arrays)
				out <- string(v)
			case string:
				str := v
				if HasAdjacentSpacesOrNewline(str) {
					str = CompressRunsOfSpaces(str)
				}
				out <- str
			case json.Number:
				out <- v.String()
			case float64:
				out <- strconv.FormatFloat(v, 'f', -1, 64)
			case bool:
				if v {
					out <- "true"
				} else {
					out <- "false"
				}
			case nil:
				out <- "null"
			default:
				out <- t.(string)
			}
		}
	}

	// launch single tokenizer goroutine
	go tokenizeJSON(inp, out)

	return out
}

// JSONConverter parses JSON token stream into XML object stream
func JSONConverter(inp <-chan string, set, rec, nest string) <-chan string {

	if inp == nil {
		return nil
	}

	out := make(chan string, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create JSON converter channel\n")
		os.Exit(1)
	}

	// opt is used for anonymous top-level objects, anon for anonymous top-level arrays
	opt := "opt"
	anon := "anon"
	if rec != "" {
		// override record delimiter
		opt = rec
		anon = rec
	}

	flat := false
	plural := false
	depth := false

	switch nest {
	case "flat":
		flat = true
	case "plural", "name":
		plural = true
	case "depth", "deep", "level":
		depth = true
	}

	// convertJSON sends XML records down a channel
	convertJSON := func(inp <-chan string, out chan<- string) {

		// close channel when all tokens have been processed
		defer close(out)

		// ensure that XML tags are legal (initial digit allowed by xtract for biological data in JSON)
		fixTag := func(tag string) string {

			if tag == "" {
				return tag
			}

			okay := true
			for _, ch := range tag {
				if !InElement[ch] {
					okay = false
				}
			}
			if okay {
				return tag
			}

			var temp strings.Builder

			// replace illegal characters with underscore
			for _, ch := range tag {
				if InElement[ch] {
					temp.WriteRune(ch)
				} else {
					temp.WriteRune('_')
				}
			}

			return temp.String()
		}

		// closure silently places local variable pointer onto inner function call stack
		var buffer strings.Builder

		// array to speed up indentation
		indentSpaces := []string{
			"",
			"  ",
			"    ",
			"      ",
			"        ",
			"          ",
			"            ",
			"              ",
			"                ",
			"                  ",
		}

		indent := 0
		if set != "" {
			indent = 1
		}

		// indent a specified number of spaces
		doIndent := func(indt int) {
			i := indt
			for i > 9 {
				buffer.WriteString("                    ")
				i -= 10
			}
			if i < 0 {
				return
			}
			buffer.WriteString(indentSpaces[i])
		}

		count := 0

		// recursive function definitions
		var parseObject func(tag string)
		var parseArray func(tag, pfx string, lvl int)

		// recursive descent parser uses mutual recursion
		parseValue := func(tag, pfx, tkn string, lvl int) {

			switch tkn {
			case "{":
				parseObject(tag)
				// no break needed, would use fallthrough to explicitly cause program control to flow to the next case
			case "[":
				if flat {
					parseArray(tag, pfx, lvl+1)
				} else if lvl > 0 {
					// nested JSON arrays create recursive XML objects
					doIndent(indent)
					indent++
					tg := tag
					if plural {
						tg = inflector.Pluralize(tag)
					}
					buffer.WriteString("<")
					buffer.WriteString(tg)
					buffer.WriteString(">\n")
					if depth {
						parseArray(pfx+"_"+strconv.Itoa(lvl), tag, lvl+1)
					} else {
						parseArray(tag, pfx, lvl+1)
					}
					indent--
					doIndent(indent)
					buffer.WriteString("</")
					buffer.WriteString(tg)
					buffer.WriteString(">\n")
				} else {
					parseArray(tag, pfx, lvl+1)
				}
			case "}", "]":
				// should not get here, decoder tracks nesting of braces and brackets
			case "":
				// empty value string generates self-closing object
				doIndent(indent)
				buffer.WriteString("<")
				buffer.WriteString(tag)
				buffer.WriteString("/>\n")
			default:
				// write object and contents to string builder
				doIndent(indent)
				tkn = strings.TrimSpace(tkn)
				tkn = html.EscapeString(tkn)
				buffer.WriteString("<")
				buffer.WriteString(tag)
				buffer.WriteString(">")
				buffer.WriteString(tkn)
				buffer.WriteString("</")
				buffer.WriteString(tag)
				buffer.WriteString(">\n")
			}

			count++
			if count > 1000 {
				count = 0
				txt := buffer.String()
				if txt != "" {
					// send current result through output channel
					out <- txt
				}
				buffer.Reset()
			}
		}

		parseObject = func(tag string) {

			doIndent(indent)
			indent++
			buffer.WriteString("<")
			buffer.WriteString(tag)
			buffer.WriteString(">\n")

			for {
				// shadowing tag variable inside for loop does not step on value of tag argument in outer scope
				tag, ok := <-inp
				if !ok {
					break
				}

				if tag == "}" || tag == "]" {
					break
				}

				tag = fixTag(tag)

				tkn, ok := <-inp
				if !ok {
					break
				}

				if tkn == "}" || tkn == "]" {
					break
				}

				parseValue(tag, tag, tkn, 0)
			}

			indent--
			doIndent(indent)
			buffer.WriteString("</")
			buffer.WriteString(tag)
			buffer.WriteString(">\n")
		}

		parseArray = func(tag, pfx string, lvl int) {

			for {
				tkn, ok := <-inp
				if !ok {
					break
				}

				if tkn == "}" || tkn == "]" {
					break
				}

				parseValue(tag, pfx, tkn, lvl)
			}
		}

		// process stream of catenated top-level JSON objects or arrays
		for {
			tkn, ok := <-inp
			if !ok {
				break
			}
			if tkn == "{" {
				parseObject(opt)
			} else if tkn == "[" {
				parseArray(anon, anon, 0)
			} else {
				break
			}

			txt := buffer.String()
			if txt != "" {
				// send remaining result through output channel
				out <- txt
			}

			buffer.Reset()

			runtime.Gosched()
		}
	}

	// launch single converter goroutine
	go convertJSON(inp, out)

	return out
}

// ASN1Tokenizer sends sequential ASN1 tokens down a channel
func ASN1Tokenizer(inp io.Reader) <-chan string {

	if inp == nil {
		return nil
	}

	out := make(chan string, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create ASN1 tokenizer channel\n")
		os.Exit(1)
	}

	var buf strings.Builder

	scanr := bufio.NewScanner(inp)

	tokenizeASN1 := func(inp io.Reader, out chan<- string) {

		// close channel when all tokens have been sent
		defer close(out)

		row := 0
		idx := 0
		line := ""

		sentinel := string(rune(0))

		nextLine := func() string {

			for scanr.Scan() {
				// read line
				line := scanr.Text()
				row++
				if line == "" {
					// ignore blank lines
					continue
				}
				// add sentinel
				line += sentinel
				return line
			}

			// end of data
			return sentinel
		}

		readRestOfAsnString := func() string {

			// continue reading additional lines of string
			for {

				line = nextLine()
				if line == "" || line == sentinel {
					break
				}
				idx = 0

				ch := line[idx]

				for InAsnString[ch] {
					idx++
					ch = line[idx]
				}
				str := line[:idx]
				line = line[idx:]
				idx = 0

				buf.WriteString(str)

				if strings.HasPrefix(line, "\"") {
					// "
					// skip past closing quote
					line = line[1:]
					idx = 0

					// break out of continuation loop
					return line
				}

				// string continues on additional lines
			}

			return line
		}

		readRestOfAsnBits := func() string {

			// continue reading additional lines of string
			for {

				line = nextLine()
				if line == "" || line == sentinel {
					break
				}
				idx = 0

				ch := line[idx]

				for InAsnBits[ch] {
					idx++
					ch = line[idx]
				}
				str := line[:idx]
				line = line[idx:]
				idx = 0

				buf.WriteString(str)

				if strings.HasPrefix(line, "'") {
					// skip past closing apostrophe
					line = line[1:]
					idx = 0

					// break out of continuation loop
					return line
				}

				// string continues on additional lines
			}

			return line
		}

		for {

			line = nextLine()
			if line == "" || line == sentinel {
				break
			}

			for {

				if line == "" || line == sentinel {
					break
				}
				idx = 0

				// trim leading blanks
				ch := line[idx]
				for InBlank[ch] {
					idx++
					ch = line[idx]
				}
				line = line[idx:]
				idx = 0

				if ch == ',' {
					out <- string(ch)
					line = line[1:]
					continue
				}

				if ch == '{' {
					// start structure
					out <- string(ch)
					line = line[1:]
					continue
				}

				if ch == '}' {
					// end structure
					out <- string(ch)
					line = line[1:]
					continue
				}

				if ch == '"' {
					// "
					// start of string
					buf.Reset()

					// skip past opening quote
					line = line[1:]
					idx = 0
					ch = line[idx]

					// read to closing quote or sentinel at end of line
					for InAsnString[ch] {
						idx++
						ch = line[idx]
					}
					str := line[:idx]
					line = line[idx:]
					idx = 0

					buf.WriteString(str)

					if strings.HasPrefix(line, "\"") {
						// "
						// skip past closing quote
						line = line[1:]
						idx = 0

					} else {

						// continue reading additional lines of string
						line = readRestOfAsnString()
						idx = 0
					}

					tmp := buf.String()
					if tmp == "" {
						// encode empty string
						tmp = "\"\""
					}

					out <- tmp
					buf.Reset()

					continue
				}

				if ch == '\'' {
					// start of bit string
					buf.Reset()

					// skip past opening apostrophe
					line = line[1:]
					idx = 0
					ch = line[idx]

					// read to closing apostrophe or sentinel at end of line
					for InAsnBits[ch] {
						idx++
						ch = line[idx]
					}
					str := line[:idx]
					line = line[idx:]
					idx = 0

					buf.WriteString(str)

					if strings.HasPrefix(line, "'") {
						// skip past closing apostrophe
						line = line[1:]
						idx = 0

					} else {

						// continue reading additional lines of bit string
						line = readRestOfAsnBits()
						idx = 0
					}

					// if apostrophe is at end of line, read next line
					if line == "" || line == sentinel {
						line = nextLine()
					}

					// then skip past trailing hex or binary indicator
					if strings.HasPrefix(line, "H") || strings.HasPrefix(line, "B") {
						line = line[1:]
						idx = 0
					}

					out <- buf.String()
					buf.Reset()

					continue
				}

				if ch == ':' && strings.HasPrefix(line, "::=") {
					// start of record contents
					out <- "::="
					line = line[3:]
					idx = 0
					continue
				}

				if ch == '-' && strings.HasPrefix(line, "--") {
					// skip comments
					break
				}
				if ch == ';' {
					// skip comments
					break
				}

				// read token or unquoted numeric value
				idx = 0
				for InAsnTag[ch] {
					idx++
					ch = line[idx]
				}
				tkn := line[:idx]
				line = line[idx:]
				idx = 0

				out <- tkn
			}
		}
	}

	// launch single tokenizer goroutine
	go tokenizeASN1(inp, out)

	return out
}

// ASN1Converter parses ASN1 token stream into XML object stream
func ASN1Converter(inp <-chan string, set, rec string) <-chan string {

	if inp == nil {
		return nil
	}

	out := make(chan string, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create ASN1 converter channel\n")
		os.Exit(1)
	}

	// convertASN1 sends XML records down a channel
	convertASN1 := func(inp <-chan string, out chan<- string) {

		// close channel when all tokens have been processed
		defer close(out)

		// ensure that XML tags are legal
		fixTag := func(tag string) string {

			if tag == "" {
				return tag
			}

			okay := true
			for _, ch := range tag {
				if !InElement[ch] {
					okay = false
				}
			}
			if okay {
				return tag
			}

			var temp strings.Builder

			// replace illegal characters with underscore
			for _, ch := range tag {
				if InElement[ch] {
					temp.WriteRune(ch)
				} else {
					temp.WriteRune('_')
				}
			}

			return temp.String()
		}

		nextToken := func() string {

			for {
				tkn, ok := <-inp
				if !ok {
					break
				}
				if tkn == "" {
					// ignore blank tokens
					continue
				}

				return tkn
			}

			// end of data
			return ""
		}

		// collects tags until next brace or comma
		var arry []string

		// builds XML output for current record
		var buffer strings.Builder

		// array to speed up indentation
		indentSpaces := []string{
			"",
			"  ",
			"    ",
			"      ",
			"        ",
			"          ",
			"            ",
			"              ",
			"                ",
			"                  ",
		}

		count := 0

		indent := 0
		if set != "" {
			indent = 1
		}

		// indent a specified number of spaces
		doIndent := func(indt int) {
			i := indt
			for i > 9 {
				buffer.WriteString("                    ")
				i -= 10
			}
			if i < 0 {
				return
			}
			buffer.WriteString(indentSpaces[i])
		}

		printOpeningTag := func(tag string) {

			tag = fixTag(tag)
			doIndent(indent)
			indent++
			buffer.WriteString("<")
			buffer.WriteString(tag)
			buffer.WriteString(">\n")
		}

		printClosingTag := func(tag string) {

			tag = fixTag(tag)
			indent--
			doIndent(indent)
			buffer.WriteString("</")
			buffer.WriteString(tag)
			buffer.WriteString(">\n")
		}

		printContent := func(tag, tkn string) {

			if tkn == "\"\"" {
				return
			}
			tag = fixTag(tag)
			doIndent(indent)
			buffer.WriteString("<")
			buffer.WriteString(tag)
			buffer.WriteString(">")
			tkn = strings.TrimSpace(tkn)
			tkn = html.EscapeString(tkn)
			buffer.WriteString(tkn)
			buffer.WriteString("</")
			buffer.WriteString(tag)
			buffer.WriteString(">\n")
		}

		popFromArry := func() (string, string, string) {

			fst, sec, trd := "", "", ""
			switch len(arry) {
			case 1:
				fst = arry[0]
			case 2:
				fst, sec = arry[0], arry[1]
			case 3:
				fst, sec, trd = arry[0], arry[1], arry[2]
			}
			arry = nil
			return fst, sec, trd
		}

		// recursive function definition
		var parseAsnObject func(prnt string, lvl int)

		parseAsnObject = func(prnt string, lvl int) {

			for {
				tkn := nextToken()
				if tkn == "" {
					return
				}

				switch tkn {
				case "{":
					fst, sec, trd := popFromArry()
					tag := fst
					if fst == "" {
						fst = prnt
					}
					if fst == "" {
						return
					}
					printOpeningTag(fst)
					tag = fst + "_E"
					if sec != "" {
						printOpeningTag(sec)
						tag = sec + "_E"
					}
					if trd != "" {
						printOpeningTag(trd)
						tag = trd
					}
					parseAsnObject(tag, lvl+1)
					if trd != "" {
						printClosingTag(trd)
					}
					if sec != "" {
						printClosingTag(sec)
					}
					printClosingTag(fst)
					if lvl == 0 {
						return
					}
				case ",":
					fst, sec, trd := popFromArry()
					if trd != "" {
						printOpeningTag(fst)
						printContent(sec, trd)
						printClosingTag(fst)
					} else if sec != "" {
						printContent(fst, sec)
					} else if fst != "" {
						printContent(prnt, fst)
					}
				case "}":
					fst, sec, trd := popFromArry()
					if fst == "" {
						return
					}
					if trd != "" {
						printOpeningTag(fst)
						printContent(sec, trd)
						printClosingTag(fst)
					} else if sec != "" {
						printContent(fst, sec)
					} else if fst != "" {
						printContent(prnt, fst)
					}
					return
				case "::=":
					fmt.Fprintf(os.Stderr, "\nERROR: Unexpected ::= token found\n")
					os.Exit(1)
				default:
					arry = append(arry, tkn)
				}

				count++
				if count > 1000 {
					count = 0
					txt := buffer.String()
					if txt != "" {
						// send current result through output channel
						out <- txt
					}
					buffer.Reset()
				}
			}
		}

		if set != "" {
			out <- "<" + set + ">"
		}

		// process stream of catenated top-level ASN1 records
		for {
			arry = nil

			top := nextToken()
			if top == "" {
				break
			}

			if rec != "" {
				top = rec
			}

			arry = append(arry, top)

			tkn := nextToken()
			if tkn == "" {
				fmt.Fprintf(os.Stderr, "\nERROR: Incomplete ASN1 starting with '%s'\n", top)
				os.Exit(1)
			}
			if tkn != "::=" {
				fmt.Fprintf(os.Stderr, "\nERROR: ASN1 message missing expected ::= token, found '%s'\n", tkn)
				os.Exit(1)
			}

			parseAsnObject(top, 0)

			txt := buffer.String()
			if txt != "" {
				// send remaining result through output channel
				out <- txt
			}

			buffer.Reset()

			runtime.Gosched()
		}

		if set != "" {
			out <- "</" + set + ">"
		}
	}

	// launch single converter goroutine
	go convertASN1(inp, out)

	return out
}

// READ TAB-DELIMITED FILE AND WRAP IN XML FIELDS

// TableConverter parses tab-delimited (-t2x) or comma-separated values (-c2x) files into XML
func TableConverter(inp io.Reader, delim string, args []string) int {

	// e.g., transmute -t2x -set Set -rec Rec -skip 0 Uid Name

	if inp == nil {
		return 0
	}

	cmmd := args[0]
	args = args[1:]

	head := ""
	tail := ""

	hd := ""
	tl := ""

	skip := 0
	header := false
	lower := false
	upper := false
	indent := true

	var fields []string
	numFlds := 0

	for len(args) > 0 {
		str := args[0]
		switch str {
		case "-set":
			args = args[1:]
			if len(args) < 1 {
				fmt.Fprintf(os.Stderr, "\nERROR: No argument after -set\n")
				os.Exit(1)
			}
			set := args[0]
			if set != "" && set != "-" {
				head = "<" + set + ">"
				tail = "</" + set + ">"
			}
			args = args[1:]
		case "-rec":
			args = args[1:]
			if len(args) < 1 {
				fmt.Fprintf(os.Stderr, "\nERROR: No argument after -rec\n")
				os.Exit(1)
			}
			rec := args[0]
			if rec != "" && rec != "-" {
				hd = "<" + rec + ">"
				tl = "</" + rec + ">"
			}
			args = args[1:]
		case "-skip":
			args = args[1:]
			if len(args) < 1 {
				fmt.Fprintf(os.Stderr, "\nERROR: No argument after -skip\n")
				os.Exit(1)
			}
			tmp := args[0]
			val, err := strconv.Atoi(tmp)
			if err != nil {
				fmt.Fprintf(os.Stderr, "\nERROR: -skip argument (%s) is not an integer\n", tmp)
				os.Exit(1)
			}
			skip = val
			args = args[1:]
		case "-header", "-headers", "-heading":
			header = true
			args = args[1:]
		case "-lower":
			lower = true
			args = args[1:]
		case "-upper":
			upper = true
			args = args[1:]
		case "-indent":
			indent = true
			args = args[1:]
		case "-flush":
			indent = false
			args = args[1:]
		default:
			// remaining arguments are names for columns
			if str != "" && str != "*" {
				fields = append(fields, str)
				numFlds++
			}
			args = args[1:]
		}
	}

	if numFlds < 1 && !header {
		fmt.Fprintf(os.Stderr, "\nERROR: Insufficient arguments for %s\n", cmmd)
		os.Exit(1)
	}

	var buffer strings.Builder
	count := 0
	okay := false
	row := 0
	recordCount := 0

	wrtr := bufio.NewWriter(os.Stdout)

	scanr := bufio.NewScanner(inp)

	if head != "" {
		buffer.WriteString(head)
		buffer.WriteString("\n")
	}

	if header {

		// -header uses fields from first row for column names
		for scanr.Scan() {

			line := scanr.Text()

			row++

			if skip > 0 {
				skip--
				continue
			}

			cols := strings.Split(line, delim)

			for _, str := range cols {
				fields = append(fields, str)
				numFlds++
			}
			break
		}

		if numFlds < 1 {
			fmt.Fprintf(os.Stderr, "\nERROR: Line with column names not found\n")
			os.Exit(1)
		}
	}

	for scanr.Scan() {

		line := scanr.Text()

		row++

		if skip > 0 {
			skip--
			continue
		}

		cols := strings.Split(line, delim)

		if len(cols) != numFlds {
			fmt.Fprintf(os.Stderr, "Mismatched columns in row %d - '%s'\n", row, line)
			continue
		}

		if hd != "" {
			if indent {
				buffer.WriteString("  ")
			}
			buffer.WriteString(hd)
			buffer.WriteString("\n")
		}

		for i, fld := range fields {
			val := cols[i]
			if lower {
				val = strings.ToLower(val)
			}
			if upper {
				val = strings.ToUpper(val)
			}
			if fld[0] == '*' {
				fld = fld[1:]
			} else {
				val = html.EscapeString(val)
			}
			val = strings.TrimSpace(val)
			if indent {
				buffer.WriteString("    ")
			}
			buffer.WriteString("<")
			buffer.WriteString(fld)
			buffer.WriteString(">")
			buffer.WriteString(val)
			buffer.WriteString("</")
			buffer.WriteString(fld)
			buffer.WriteString(">")
			buffer.WriteString("\n")
		}

		if tl != "" {
			if indent {
				buffer.WriteString("  ")
			}
			buffer.WriteString(tl)
			buffer.WriteString("\n")
		}

		recordCount++
		count++

		if count >= 1000 {
			count = 0
			txt := buffer.String()
			if txt != "" {
				// print current buffer
				wrtr.WriteString(txt[:])
			}
			buffer.Reset()
		}

		okay = true
	}

	if tail != "" {
		buffer.WriteString(tail)
		buffer.WriteString("\n")
	}

	if okay {
		txt := buffer.String()
		if txt != "" {
			// print current buffer
			wrtr.WriteString(txt[:])
		}
	}
	buffer.Reset()

	wrtr.Flush()

	return recordCount
}

// READ GENBANK FLATFILE AND TRANSLATE TO INSDSEQ XML

// GenBankConverter sends INSDSeq XML records down a channel
func GenBankConverter(inp io.Reader) <-chan string {

	if inp == nil {
		return nil
	}

	out := make(chan string, ChanDepth)
	if out == nil {
		fmt.Fprintf(os.Stderr, "Unable to create GenBank converter channel\n")
		os.Exit(1)
	}

	const twelvespaces = "            "
	const twentyonespaces = "                     "

	var rec strings.Builder
	var alt strings.Builder
	var con strings.Builder
	var seq strings.Builder

	scanr := bufio.NewScanner(inp)

	convertGenBank := func(inp io.Reader, out chan<- string) {

		// close channel when all records have been sent
		defer close(out)

		row := 0

		nextLine := func() string {

			for scanr.Scan() {
				line := scanr.Text()
				if line == "" {
					continue
				}
				return line
			}
			return ""

		}

		for {

			rec.Reset()

			// read first line of next record
			line := nextLine()
			if line == "" {
				break
			}

			row++

			for {
				if !strings.HasPrefix(line, "LOCUS") {
					// skip release file header information
					line = nextLine()
					row++
					continue
				}
				break
			}

			readContinuationLines := func(str string) string {

				for {
					// read next line
					line = nextLine()
					row++
					if !strings.HasPrefix(line, twelvespaces) {
						// if not continuation line, break out of loop
						break
					}
					// append subsequent line and continue with loop
					txt := strings.TrimPrefix(line, twelvespaces)
					str += " " + txt
				}

				str = CompressRunsOfSpaces(str)
				str = strings.TrimSpace(str)

				return str
			}

			writeOneElement := func(spaces, tag, value string) {

				rec.WriteString(spaces)
				rec.WriteString("<")
				rec.WriteString(tag)
				rec.WriteString(">")
				value = html.EscapeString(value)
				rec.WriteString(value)
				rec.WriteString("</")
				rec.WriteString(tag)
				rec.WriteString(">\n")
			}

			// each section will exit with the next line ready to process

			if strings.HasPrefix(line, "LOCUS") {

				// start of record
				rec.WriteString("  <INSDSeq>\n")

				// do not break if given artificial multi-line LOCUS
				str := readContinuationLines(line)

				cols := strings.Fields(str)
				ln := len(cols)
				if ln == 8 {
					moleculetype := cols[4]
					strandedness := ""
					if strings.HasPrefix(moleculetype, "ds-") {
						moleculetype = strings.TrimPrefix(moleculetype, "ds-")
						strandedness = "double"
					} else if strings.HasPrefix(moleculetype, "ss-") {
						moleculetype = strings.TrimPrefix(moleculetype, "ss-")
						strandedness = "single"
					} else if strings.HasPrefix(moleculetype, "ms-") {
						moleculetype = strings.TrimPrefix(moleculetype, "ms-")
						strandedness = "mixed"
					} else if strings.HasSuffix(moleculetype, "DNA") {
						strandedness = "double"
					} else if strings.HasSuffix(moleculetype, "RNA") {
						strandedness = "single"
					}

					writeOneElement("    ", "INSDSeq_locus", cols[1])

					writeOneElement("    ", "INSDSeq_length", cols[2])

					if strandedness != "" {
						writeOneElement("    ", "INSDSeq_strandedness", strandedness)
					}

					writeOneElement("    ", "INSDSeq_moltype", moleculetype)

					writeOneElement("    ", "INSDSeq_topology", cols[5])

					writeOneElement("    ", "INSDSeq_division", cols[6])

					writeOneElement("    ", "INSDSeq_update-date", cols[7])

				} else if ln == 7 {

					writeOneElement("    ", "INSDSeq_locus", cols[1])

					writeOneElement("    ", "INSDSeq_length", cols[2])

					writeOneElement("    ", "INSDSeq_moltype", "AA")

					writeOneElement("    ", "INSDSeq_topology", cols[4])

					writeOneElement("    ", "INSDSeq_division", cols[5])

					writeOneElement("    ", "INSDSeq_update-date", cols[6])

				} else {
					fmt.Fprintf(os.Stderr, "ERROR: "+str+"\n")
				}

				// read next line and continue - handled by readContinuationLines above
				// line = nextLine()
				// row++
			}

			if strings.HasPrefix(line, "DEFINITION") {

				txt := strings.TrimPrefix(line, "DEFINITION")
				def := readContinuationLines(txt)
				def = strings.TrimSuffix(def, ".")

				writeOneElement("    ", "INSDSeq_definition", def)
			}

			var secondaries []string

			if strings.HasPrefix(line, "ACCESSION") {

				txt := strings.TrimPrefix(line, "ACCESSION")
				str := readContinuationLines(txt)
				accessions := strings.Fields(str)
				ln := len(accessions)
				if ln > 1 {

					writeOneElement("    ", "INSDSeq_primary-accession", accessions[0])

					// skip past primary accession, collect secondaries
					secondaries = accessions[1:]

				} else if ln == 1 {

					writeOneElement("    ", "INSDSeq_primary-accession", accessions[0])

				} else {
					fmt.Fprintf(os.Stderr, "ERROR: ACCESSION "+str+"\n")
				}
			}

			accnver := ""
			gi := ""

			if strings.HasPrefix(line, "VERSION") {

				cols := strings.Fields(line)
				if len(cols) == 2 {

					accnver = cols[1]
					writeOneElement("    ", "INSDSeq_accession-version", accnver)

				} else if len(cols) == 3 {

					accnver = cols[1]
					writeOneElement("    ", "INSDSeq_accession-version", accnver)

					// collect gi for other-seqids
					if strings.HasPrefix(cols[2], "GI:") {
						gi = strings.TrimPrefix(cols[2], "GI:")
					}

				} else {
					fmt.Fprintf(os.Stderr, "ERROR: "+line+"\n")
				}

				// read next line and continue
				line = nextLine()
				row++

			}

			if gi != "" {

				rec.WriteString("    <INSDSeq_other-seqids>\n")

				writeOneElement("      ", "INSDSeqid", "gi|"+gi)

				rec.WriteString("    </INSDSeq_other-seqids>\n")
			}

			if len(secondaries) > 0 {

				rec.WriteString("    <INSDSeq_secondary-accessions>\n")

				for _, secndry := range secondaries {

					if strings.HasPrefix(secndry, "REGION") {
						break
					}
					writeOneElement("      ", "INSDSecondary-accn", secndry)
				}

				rec.WriteString("    </INSDSeq_secondary-accessions>\n")
			}

			if strings.HasPrefix(line, "DBLINK") {

				txt := strings.TrimPrefix(line, "DBLINK")
				readContinuationLines(txt)
				// collect for database-reference
				// out <- Token{DBLINK, dbl}
			}

			if strings.HasPrefix(line, "DBSOURCE") {

				txt := strings.TrimPrefix(line, "DBSOURCE")
				readContinuationLines(txt)
				// collect for database-source
				// out <- Token{DBSOURCE, dbl}
			}

			if strings.HasPrefix(line, "KEYWORDS") {

				txt := strings.TrimPrefix(line, "KEYWORDS")
				key := readContinuationLines(txt)
				key = strings.TrimSuffix(key, ".")

				if key != "" {
					rec.WriteString("    <INSDSeq_keywords>\n")
					kywds := strings.Split(key, ";")
					for _, kw := range kywds {
						kw = strings.TrimSpace(kw)
						if kw == "" || kw == "." {
							continue
						}

						writeOneElement("      ", "INSDKeyword", kw)
					}
					rec.WriteString("    </INSDSeq_keywords>\n")
				}
			}

			if strings.HasPrefix(line, "SOURCE") {

				txt := strings.TrimPrefix(line, "SOURCE")
				src := readContinuationLines(txt)

				writeOneElement("    ", "INSDSeq_source", src)
			}

			if strings.HasPrefix(line, "  ORGANISM") {

				org := strings.TrimPrefix(line, "  ORGANISM")
				org = CompressRunsOfSpaces(org)
				org = strings.TrimSpace(org)

				writeOneElement("    ", "INSDSeq_organism", org)

				line = nextLine()
				row++
				if strings.HasPrefix(line, twelvespaces) {
					txt := strings.TrimPrefix(line, twelvespaces)
					tax := readContinuationLines(txt)
					tax = strings.TrimSuffix(tax, ".")

					writeOneElement("    ", "INSDSeq_taxonomy", tax)
				}
			}

			rec.WriteString("    <INSDSeq_references>\n")
			for {
				if !strings.HasPrefix(line, "REFERENCE") {
					// exit out of reference section
					break
				}

				ref := "0"

				rec.WriteString("      <INSDReference>\n")

				txt := strings.TrimPrefix(line, "REFERENCE")
				str := readContinuationLines(txt)
				str = CompressRunsOfSpaces(str)
				str = strings.TrimSpace(str)
				idx := strings.Index(str, "(")
				if idx > 0 {
					ref = strings.TrimSpace(str[:idx])

					writeOneElement("        ", "INSDReference_reference", ref)

					posn := str[idx+1:]
					posn = strings.TrimSuffix(posn, ")")
					posn = strings.TrimSpace(posn)
					if posn == "sites" {

						writeOneElement("        ", "INSDReference_position", posn)

					} else {
						var arry []string
						cls := strings.Split(posn, ";")
						for _, item := range cls {
							item = strings.TrimPrefix(item, "bases ")
							item = strings.TrimPrefix(item, "residues ")
							item = strings.TrimSpace(item)
							cols := strings.Fields(item)
							if len(cols) == 3 && cols[1] == "to" {
								arry = append(arry, cols[0]+".."+cols[2])
							}
						}
						if len(arry) > 0 {
							posit := strings.Join(arry, ",")
							writeOneElement("        ", "INSDReference_position", posit)
						} else {
							fmt.Fprintf(os.Stderr, "ERROR: "+posn+"\n")
						}
					}
				} else {
					ref = strings.TrimSpace(str)

					writeOneElement("        ", "INSDReference_reference", ref)
				}
				row++

				if strings.HasPrefix(line, "  AUTHORS") {

					txt := strings.TrimPrefix(line, "  AUTHORS")
					auths := readContinuationLines(txt)

					rec.WriteString("        <INSDReference_authors>\n")
					authors := strings.Split(auths, ", ")
					for _, auth := range authors {
						auth = strings.TrimSpace(auth)
						if auth == "" {
							continue
						}
						pair := strings.Split(auth, " and ")
						for _, name := range pair {

							writeOneElement("          ", "INSDAuthor", name)
						}
					}
					rec.WriteString("        </INSDReference_authors>\n")
				}

				if strings.HasPrefix(line, "  CONSRTM") {

					txt := strings.TrimPrefix(line, "  CONSRTM")
					cons := readContinuationLines(txt)

					writeOneElement("        ", "INSDReference_consortium", cons)
				}

				if strings.HasPrefix(line, "  TITLE") {

					txt := strings.TrimPrefix(line, "  TITLE")
					titl := readContinuationLines(txt)

					writeOneElement("        ", "INSDReference_title", titl)
				}

				if strings.HasPrefix(line, "  JOURNAL") {

					txt := strings.TrimPrefix(line, "  JOURNAL")
					jour := readContinuationLines(txt)

					writeOneElement("        ", "INSDReference_journal", jour)
				}

				if strings.HasPrefix(line, "   PUBMED") {

					txt := strings.TrimPrefix(line, "   PUBMED")
					pmid := readContinuationLines(txt)

					writeOneElement("        ", "INSDReference_pubmed", pmid)
				}

				if strings.HasPrefix(line, "  MEDLINE") {

					txt := strings.TrimPrefix(line, "  MEDLINE")
					// old MEDLINE uid not supported
					readContinuationLines(txt)
				}

				if strings.HasPrefix(line, "  REMARK") {

					txt := strings.TrimPrefix(line, "  REMARK")
					rem := readContinuationLines(txt)

					writeOneElement("        ", "INSDReference_remark", rem)
				}

				// end of this reference
				rec.WriteString("      </INSDReference>\n")
				// continue to next reference
			}
			rec.WriteString("    </INSDSeq_references>\n")

			if strings.HasPrefix(line, "COMMENT") {

				txt := strings.TrimPrefix(line, "COMMENT")
				com := readContinuationLines(txt)

				writeOneElement("    ", "INSDSeq_comment", com)
			}

			if strings.HasPrefix(line, "PRIMARY") {

				txt := strings.TrimPrefix(line, "PRIMARY")
				pmy := readContinuationLines(txt)

				writeOneElement("    ", "INSDSeq_primary", pmy)
			}

			rec.WriteString("    <INSDSeq_feature-table>\n")
			if strings.HasPrefix(line, "FEATURES") {

				line = nextLine()
				row++

				for {
					if !strings.HasPrefix(line, "     ") {
						// exit out of features section
						break
					}
					if len(line) < 22 {
						fmt.Fprintf(os.Stderr, "ERROR: "+line+"\n")
						line = nextLine()
						row++
						continue
					}

					rec.WriteString("      <INSDFeature>\n")

					// read feature key and start of location
					fkey := line[5:21]
					fkey = strings.TrimSpace(fkey)

					writeOneElement("        ", "INSDFeature_key", fkey)

					loc := line[21:]
					loc = strings.TrimSpace(loc)
					for {
						line = nextLine()
						row++
						if !strings.HasPrefix(line, twentyonespaces) {
							break
						}
						txt := strings.TrimPrefix(line, twentyonespaces)
						if strings.HasPrefix(txt, "/") {
							// if not continuation of location, break out of loop
							break
						}
						// append subsequent line and continue with loop
						loc += strings.TrimSpace(txt)
					}

					writeOneElement("        ", "INSDFeature_location", loc)

					location_operator := ""
					is_comp := false
					prime5 := false
					prime3 := false

					// parseloc recursive definition
					var parseloc func(string) []string

					parseloc = func(str string) []string {

						var acc []string

						if strings.HasPrefix(str, "join(") && strings.HasSuffix(str, ")") {

							location_operator = "join"

							str = strings.TrimPrefix(str, "join(")
							str = strings.TrimSuffix(str, ")")
							items := strings.Split(str, ",")

							for _, thisloc := range items {
								inner := parseloc(thisloc)
								for _, sub := range inner {
									acc = append(acc, sub)
								}
							}

						} else if strings.HasPrefix(str, "order(") && strings.HasSuffix(str, ")") {

							location_operator = "order"

							str = strings.TrimPrefix(str, "order(")
							str = strings.TrimSuffix(str, ")")
							items := strings.Split(str, ",")

							for _, thisloc := range items {
								inner := parseloc(thisloc)
								for _, sub := range inner {
									acc = append(acc, sub)
								}
							}

						} else if strings.HasPrefix(str, "complement(") && strings.HasSuffix(str, ")") {

							is_comp = true

							str = strings.TrimPrefix(str, "complement(")
							str = strings.TrimSuffix(str, ")")
							items := parseloc(str)

							// reverse items
							for i, j := 0, len(items)-1; i < j; i, j = i+1, j-1 {
								items[i], items[j] = items[j], items[i]
							}

							// reverse from and to positions, flip direction of angle brackets (partial flags)
							for _, thisloc := range items {
								pts := strings.Split(thisloc, "..")
								ln := len(pts)
								if ln == 2 {
									fst := pts[0]
									scd := pts[1]
									lf := ""
									rt := ""
									if strings.HasPrefix(fst, "<") {
										fst = strings.TrimPrefix(fst, "<")
										rt = ">"
									}
									if strings.HasPrefix(scd, ">") {
										scd = strings.TrimPrefix(scd, ">")
										lf = "<"
									}
									acc = append(acc, lf+scd+".."+rt+fst)
								} else if ln > 0 {
									acc = append(acc, pts[0])
								}
							}

						} else {

							// save individual interval or point if no leading accession
							if strings.Index(str, ":") < 0 {
								acc = append(acc, str)
							}
						}

						return acc
					}

					items := parseloc(loc)

					rec.WriteString("        <INSDFeature_intervals>\n")

					num_ivals := 0

					// report individual intervals
					for _, thisloc := range items {
						if thisloc == "" {
							continue
						}

						num_ivals++

						rec.WriteString("          <INSDInterval>\n")
						pts := strings.Split(thisloc, "..")
						if len(pts) == 2 {

							// fr..to
							fr := pts[0]
							to := pts[1]
							if strings.HasPrefix(fr, "<") {
								fr = strings.TrimPrefix(fr, "<")
								prime5 = true
							}
							if strings.HasPrefix(to, ">") {
								to = strings.TrimPrefix(to, ">")
								prime3 = true
							}
							writeOneElement("            ", "INSDInterval_from", fr)
							writeOneElement("            ", "INSDInterval_to", to)
							if is_comp {
								rec.WriteString("            <INSDInterval_iscomp value=\"true\"/>\n")
							}
							writeOneElement("            ", "INSDInterval_accession", accnver)

						} else {

							crt := strings.Split(thisloc, "^")
							if len(crt) == 2 {

								// fr^to
								fr := crt[0]
								to := crt[1]
								writeOneElement("            ", "INSDInterval_from", fr)
								writeOneElement("            ", "INSDInterval_to", to)
								if is_comp {
									rec.WriteString("            <INSDInterval_iscomp value=\"true\"/>\n")
								}
								rec.WriteString("            <INSDInterval_interbp value=\"true\"/>\n")
								writeOneElement("            ", "INSDInterval_accession", accnver)

							} else {

								// pt
								pt := pts[0]
								if strings.HasPrefix(pt, "<") {
									pt = strings.TrimPrefix(pt, "<")
									prime5 = true
								}
								if strings.HasPrefix(pt, ">") {
									pt = strings.TrimPrefix(pt, ">")
									prime3 = true
								}
								writeOneElement("            ", "INSDInterval_point", pt)
								writeOneElement("            ", "INSDInterval_accession", accnver)
							}
						}
						rec.WriteString("          </INSDInterval>\n")
					}

					rec.WriteString("        </INSDFeature_intervals>\n")

					if num_ivals > 1 {
						writeOneElement("        ", "INSDFeature_operator", location_operator)
					}
					if prime5 {
						rec.WriteString("        <INSDFeature_partial5 value=\"true\"/>\n")
					}
					if prime3 {
						rec.WriteString("        <INSDFeature_partial3 value=\"true\"/>\n")
					}

					hasQual := false
					for {
						if !strings.HasPrefix(line, twentyonespaces) {
							// if not qualifier line, break out of loop
							break
						}
						txt := strings.TrimPrefix(line, twentyonespaces)
						qual := ""
						val := ""
						if strings.HasPrefix(txt, "/") {
							if !hasQual {
								hasQual = true
								rec.WriteString("        <INSDFeature_quals>\n")
							}
							// read new qualifier and start of value
							qual = strings.TrimPrefix(txt, "/")
							qual = strings.TrimSpace(qual)
							idx := strings.Index(qual, "=")
							if idx > 0 {
								val = qual[idx+1:]
								qual = qual[:idx]
							}

							for {
								line = nextLine()
								row++
								if !strings.HasPrefix(line, twentyonespaces) {
									break
								}
								txt := strings.TrimPrefix(line, twentyonespaces)
								if strings.HasPrefix(txt, "/") {
									// if not continuation of qualifier, break out of loop
									break
								}
								// append subsequent line to value and continue with loop
								if qual == "transcription" || qual == "translation" || qual == "peptide" || qual == "anticodon" {
									val += strings.TrimSpace(txt)
								} else {
									val += " " + strings.TrimSpace(txt)
								}
							}

							rec.WriteString("          <INSDQualifier>\n")

							writeOneElement("            ", "INSDQualifier_name", qual)

							val = strings.TrimPrefix(val, "\"")
							val = strings.TrimSuffix(val, "\"")
							val = strings.TrimSpace(val)
							if val != "" {

								writeOneElement("            ", "INSDQualifier_value", val)
							}

							rec.WriteString("          </INSDQualifier>\n")
						}
					}
					if hasQual {
						rec.WriteString("        </INSDFeature_quals>\n")
					}

					// end of this feature
					rec.WriteString("      </INSDFeature>\n")
					// continue to next feature
				}
			}
			rec.WriteString("    </INSDSeq_feature-table>\n")

			// TSA, TLS, WGS, or CONTIG lines may be next

			alt_name := ""

			if strings.HasPrefix(line, "TSA") ||
				strings.HasPrefix(line, "TLS") ||
				strings.HasPrefix(line, "WGS") {

				alt.Reset()

				alt_name = line[:3]
				line = line[3:]
			}

			if strings.HasPrefix(line, "WGS_CONTIG") ||
				strings.HasPrefix(line, "WGS_SCAFLD") {

				alt.Reset()

				alt_name = line[:3]
				line = line[10:]
			}

			if alt_name != "" {

				alt_name = strings.ToLower(alt_name)
				txt := strings.TrimSpace(line)
				alt.WriteString(txt)
				for {
					// read next line
					line = nextLine()
					row++
					if !strings.HasPrefix(line, twelvespaces) {
						// if not continuation of contig, break out of loop
						break
					}
					// append subsequent line and continue with loop
					txt = strings.TrimPrefix(line, twelvespaces)
					txt = strings.TrimSpace(txt)
					alt.WriteString(txt)
				}
			}

			if strings.HasPrefix(line, "CONTIG") {

				// pathological records can have over 90,000 components, use strings.Builder
				con.Reset()

				txt := strings.TrimPrefix(line, "CONTIG")
				txt = strings.TrimSpace(txt)
				con.WriteString(txt)
				for {
					// read next line
					line = nextLine()
					row++
					if !strings.HasPrefix(line, twelvespaces) {
						// if not continuation of contig, break out of loop
						break
					}
					// append subsequent line and continue with loop
					txt = strings.TrimPrefix(line, twelvespaces)
					txt = strings.TrimSpace(txt)
					con.WriteString(txt)
				}
			}

			if strings.HasPrefix(line, "BASE COUNT") {

				txt := strings.TrimPrefix(line, "BASE COUNT")
				readContinuationLines(txt)
				// not supported
			}

			if strings.HasPrefix(line, "ORIGIN") {

				line = nextLine()
				row++
			}

			// remainder should be sequence

			// sequence can be millions of bases, use strings.Builder
			seq.Reset()

			for line != "" {

				if strings.HasPrefix(line, "//") {

					// end of record, print collected sequence
					str := seq.String()
					if str != "" {

						writeOneElement("    ", "INSDSeq_sequence", str)
					}
					seq.Reset()

					// print contig section
					str = con.String()
					str = strings.TrimSpace(str)
					if str != "" {
						writeOneElement("    ", "INSDSeq_contig", str)
					}
					con.Reset()

					if alt_name != "" {
						rec.WriteString("    <INSDSeq_alt-seq>\n")
						rec.WriteString("      <INSDAltSeqData>\n")
						str = alt.String()
						str = strings.TrimSpace(str)
						if str != "" {
							writeOneElement("        ", "INSDAltSeqData_name", alt_name)
							rec.WriteString("        <INSDAltSeqData_items>\n")
							writeOneElement("          ", "INSDAltSeqItem_value", str)
							rec.WriteString("        </INSDAltSeqData_items>\n")
						}
						alt.Reset()
						rec.WriteString("      </INSDAltSeqData>\n")
						rec.WriteString("    </INSDSeq_alt-seq>\n")
					}

					// end of record
					rec.WriteString("  </INSDSeq>\n")

					// send formatted record down channel
					txt := rec.String()
					out <- txt
					rec.Reset()
					// go to top of loop for next record
					break
				}

				// read next sequence line

				cols := strings.Fields(line)
				if len(cols) > 0 && !IsAllDigits(cols[0]) {
					fmt.Fprintf(os.Stderr, "ERROR: Unrecognized section "+cols[0]+"\n")
				}

				for _, str := range cols {

					if IsAllDigits(str) {
						continue
					}

					// append letters to sequence
					seq.WriteString(str)
				}

				// read next line and continue
				line = nextLine()
				row++

			}

			// continue to next record
		}
	}

	// launch single converter goroutine
	go convertGenBank(inp, out)

	return out
}

// STRING CONVERTERS

func EncodeURL(inp io.Reader) {

	if inp == nil {
		return
	}

	data, _ := ioutil.ReadAll(inp)
	txt := string(data)
	txt = strings.TrimSuffix(txt, "\n")

	str := url.QueryEscape(txt)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func DecodeURL(inp io.Reader) {

	if inp == nil {
		return
	}

	byt, _ := ioutil.ReadAll(inp)
	txt := string(byt)
	txt = strings.TrimSuffix(txt, "\n")

	str, _ := url.QueryUnescape(txt)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func EncodeB64(inp io.Reader) {

	if inp == nil {
		return
	}

	data, _ := ioutil.ReadAll(inp)

	str := base64.StdEncoding.EncodeToString(data)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func DecodeB64(inp io.Reader) {

	if inp == nil {
		return
	}

	byt, _ := ioutil.ReadAll(inp)

	data, _ := base64.StdEncoding.DecodeString(string(byt))
	str := string(data)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func DecodeHGVS(inp io.Reader) {

	if inp == nil {
		return
	}

	byt, _ := ioutil.ReadAll(inp)
	txt := string(byt)

	str := ParseHGVS(txt)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

// COLUMN ALIGNMENT FORMATTER

// ProcessAlign aligns a tab-delimited table by individual column widths
func ProcessAlign(inp io.Reader, args []string) {

	// tab-delimited-table to padded-by-spaces alignment inspired by
	// Steve Kinzler's align script - see http://kinzler.com/me/align/

	if inp == nil {
		return
	}

	spcs := "                              "

	mrg := ""
	pad := "  "

	lettrs := make(map[int]rune)
	lst := 'l'

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-g":
			val := GetNumericArg(args, "-g spacing between columns", 0, 1, 30)
			pad = spcs[0:val]
			args = args[2:]
		case "-h":
			val := GetNumericArg(args, "-i indent before columns", 0, 1, 30)
			mrg = spcs[0:val]
			args = args[2:]
		case "-a":
			val := GetStringArg(args, "-a column alignment code string")
			for i, ch := range val {
				lettrs[i] = ch
				lst = ch
			}
			args = args[2:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -align command\n")
			os.Exit(1)
		}
	}

	var arry []string

	width := make(map[int]int)
	whole := make(map[int]int)
	fract := make(map[int]int)

	scanr := bufio.NewScanner(inp)

	row := 0
	numCols := 0

	// allows leading plus or minus, digits interspersed with optional commas, decimal point, and digits
	isNumeric := func(str string) bool {

		has_num := false
		has_period := false

		for i, ch := range str {
			switch ch {
			case '0', '1', '2', '3', '4', '5', '6', '7', '8', '9':
				has_num = true
			case '+', '-':
				if i > 0 {
					return false
				}
			case '.':
				has_period = true
			case ',':
				if has_period {
					return false
				}
			default:
				return false
			}
		}

		return has_num
	}

	processLine := func(line string) string {

		var flds []string

		cols := strings.Split(line, "\t")
		if numCols == 0 {
			numCols = len(cols)
		} else if numCols != len(cols) {
			fmt.Fprintf(os.Stderr, "ERROR: Mismatched number of columns in row ")
			fmt.Fprintf(os.Stderr, strconv.Itoa(row))
			fmt.Fprintf(os.Stderr, ": actual ")
			fmt.Fprintf(os.Stderr, strconv.Itoa(len(cols)))
			fmt.Fprintf(os.Stderr, ", expected ")
			fmt.Fprintf(os.Stderr, strconv.Itoa(numCols))
			fmt.Fprintf(os.Stderr, "\n")
			// os.Exit(1)
		}

		for i, str := range cols {

			str = CompressRunsOfSpaces(str)
			str = strings.TrimSpace(str)

			flds = append(flds, str)

			// determine maximum length in each column
			ln := utf8.RuneCountInString(str)
			if ln > width[i] {
				width[i] = ln
			}

			code, ok := lettrs[i]
			if !ok {
				code = lst
			}

			switch code {
			case 'n', 'N', 'z', 'Z':
				if isNumeric(str) {
					// determine maximum length of decimal number parts
					wh, fr := SplitInTwoAt(str, ".", LEFT)
					if fr != "" {
						fr = "." + fr
					}

					lf := utf8.RuneCountInString(wh)
					if lf > whole[i] {
						whole[i] = lf
					}
					rt := utf8.RuneCountInString(fr)
					if rt > fract[i] {
						fract[i] = rt
					}
					ln = whole[i] + fract[i]
					if ln > width[i] {
						width[i] = ln
					}
				}
			}
		}

		return strings.Join(flds, "\t")
	}

	for i := 0; i < numCols; i++ {

		code, ok := lettrs[i]
		if !ok {
			code = lst
		}

		switch code {
		case 'n', 'N', 'z', 'Z':
			// adjust maximum widths with aligned decimal points
			ln := whole[i] + fract[i]
			if ln > width[i] {
				width[i] = ln
			}
		}
	}

	// clean up spaces, calculate column widths
	for scanr.Scan() {

		row++
		line := scanr.Text()
		if line == "" {
			continue
		}

		line = processLine(line)
		arry = append(arry, line)
	}

	var buffer strings.Builder

	for _, line := range arry {

		buffer.Reset()

		cols := strings.Split(line, "\t")

		btwn := mrg
		for i, str := range cols {

			buffer.WriteString(btwn)

			code, ok := lettrs[i]
			if !ok {
				code = lst
			}

			ln := utf8.RuneCountInString(str)
			mx := width[i]
			diff := mx - ln
			lft := 0
			rgt := 0
			lft_pad := " "
			rgt_pad := " "

			if diff > 0 {
				switch code {
				case 'l':
					rgt = diff
				case 'c':
					lft = diff / 2
					rgt = diff - lft
				case 'r':
					lft = diff
				case 'n', 'N', 'z', 'Z':
					lft = diff
					if isNumeric(str) {
						switch code {
						case 'N':
							rgt_pad = "0"
						case 'z':
							lft_pad = "0"
						case 'Z':
							lft_pad = "0"
							rgt_pad = "0"
						}
						sn := whole[i]
						rc := fract[i]
						wh, fr := SplitInTwoAt(str, ".", LEFT)
						if fract[i] > 0 {
							if fr == "" {
								fr = "."
							} else {
								fr = "." + fr
							}
							lf := utf8.RuneCountInString(wh)
							lft = sn - lf
							rt := utf8.RuneCountInString(fr)
							rgt = rc - rt
							str = wh + fr
						}
					}
				default:
					rgt = diff
				}
			}

			for lft > 0 {
				lft--
				buffer.WriteString(lft_pad)
			}

			buffer.WriteString(str)
			btwn = pad

			for rgt > 0 {
				rgt--
				buffer.WriteString(rgt_pad)
			}
		}

		txt := buffer.String()
		txt = strings.TrimRight(txt, " ")

		os.Stdout.WriteString(txt)
		os.Stdout.WriteString("\n")
	}
}

// SEQUENCE EDITING

func ReadAllIntoSequence(inp io.Reader) string {

	if inp == nil {
		return ""
	}

	var buffer strings.Builder

	scanr := bufio.NewScanner(inp)

	for scanr.Scan() {

		str := scanr.Text()

		// skip FASTA definition line
		if strings.HasPrefix(str, ">") {
			continue
		}
		// convert everything to upper case
		str = strings.ToUpper(str)

		// leave only upper case letters or asterisk
		str = strings.Map(func(c rune) rune {
			if (c < 'A' || c > 'Z') && c != '*' && c != '-' {
				return -1
			}
			return c
		}, str)

		buffer.WriteString(str)
	}

	txt := buffer.String()

	return txt
}

func ReadFromFileIntoSequence(fname string) string {

	f, err := os.Open(fname)
	if err != nil {
		fmt.Fprintf(os.Stderr, "Unable to open file %s - %s\n", fname, err.Error())
		os.Exit(1)
	}

	defer f.Close()

	seq := ReadAllIntoSequence(f)

	return seq
}

func SequenceRemove(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	first := ""
	last := ""

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-first":
			first = GetStringArg(args, "Bases to delete at beginning")
			first = strings.ToUpper(first)
			args = args[2:]
		case "-last":
			last = GetStringArg(args, "Bases to delete at end")
			last = strings.ToUpper(last)
			args = args[2:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -remove command\n")
			os.Exit(1)
		}
	}

	str := ReadAllIntoSequence(inp)

	ln := len(str)

	if IsAllDigits(first) {
		val, err := strconv.Atoi(first)
		if err == nil && val > 0 {
			if val <= ln {
				str = str[val:]
				ln = len(str)
			} else {
				fmt.Fprintf(os.Stderr, "\nERROR: -first argument %d is greater than sequence length %d\n", val, ln)
				str = ""
			}
		}
	} else {
		val := len(first)
		if val > 0 {
			if val <= ln {
				// warn if existing sequence does not match deletion argument
				ext := str[:val]
				if first != ext {
					fmt.Fprintf(os.Stderr, "\nWARNING: -first argument %s does not match existing sequence %s\n", first, ext)
				}
				// delete characters
				str = str[val:]
				ln = len(str)
			} else {
				fmt.Fprintf(os.Stderr, "\nERROR: -first argument %d is greater than sequence length %d\n", val, ln)
				str = ""
			}
		}
	}

	if IsAllDigits(last) {
		val, err := strconv.Atoi(last)
		if err == nil && val > 0 {
			if val <= ln {
				str = str[:ln-val]
			} else {
				fmt.Fprintf(os.Stderr, "\nERROR: -last argument %d is greater than remaining sequence length %d\n", val, ln)
				str = ""
			}
		}
	} else {
		val := len(last)
		if val > 0 {
			if val <= ln {
				// warn if existing sequence does not match deletion argument
				ext := str[ln-val:]
				if last != ext {
					fmt.Fprintf(os.Stderr, "\nWARNING: -last argument %s does not match existing sequence %s\n", last, ext)
				}
				// delete characters
				str = str[:ln-val]
				ln = len(str)
			} else {
				fmt.Fprintf(os.Stderr, "\nERROR: -last argument %d is greater than sequence length %d\n", val, ln)
				str = ""
			}
		}
	}

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func SequenceRetain(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	lead := 0
	trail := 0

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-leading":
			lead = GetNumericArg(args, "Bases to keep at beginning", 0, -1, -1)
			args = args[2:]
		case "-trailing":
			trail = GetNumericArg(args, "Bases to keep at end", 0, -1, -1)
			args = args[2:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -retain command\n")
			os.Exit(1)
		}
	}

	str := ReadAllIntoSequence(inp)

	ln := len(str)
	if lead > 0 && trail > 0 {
		fmt.Fprintf(os.Stderr, "\nERROR: Cannot have both -leading and -trailing arguments\n")
		str = ""
	} else if lead > 0 {
		if lead <= ln {
			str = str[:lead]
		} else {
			fmt.Fprintf(os.Stderr, "\nERROR: -leading argument %d is greater than sequence length %d\n", lead, ln)
			str = ""
		}
	} else if trail > 0 {
		if trail <= ln {
			str = str[ln-trail:]
		} else {
			fmt.Fprintf(os.Stderr, "\nERROR: -trailing argument %d is greater than sequence length %d\n", trail, ln)
			str = ""
		}
	}

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func SequenceReplace(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	pos := 0
	del := ""
	ins := ""

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-offset":
			pos = GetNumericArg(args, "0-based position", 0, -1, -1)
			args = args[2:]
		case "-column":
			val := GetNumericArg(args, "1-based position", 1, -1, -1)
			pos = val - 1
			args = args[2:]
		case "-delete":
			del = GetStringArg(args, "Number to delete")
			del = strings.ToUpper(del)
			args = args[2:]
		case "-insert":
			ins = GetStringArg(args, "Bases to insert")
			ins = strings.ToUpper(ins)
			args = args[2:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -replace command\n")
			os.Exit(1)
		}
	}

	str := ReadAllIntoSequence(inp)

	if del == "" && ins == "" {
		fmt.Fprintf(os.Stderr, "\nERROR: -replace command requires either -delete or -insert\n")
		return
	}

	ln := len(str)
	if pos > ln {

		if pos == ln+1 && del == "" && ins != "" {

			// append to end of sequence
			str = str[:] + ins

		} else {
			fmt.Fprintf(os.Stderr, "\nERROR: -replace position %d is greater than sequence length %d\n", pos, ln)
			return
		}

	} else {

		if IsAllDigits(del) {
			val, err := strconv.Atoi(del)
			if err == nil && val > 0 {
				if val <= ln-pos {
					str = str[:pos] + str[pos+val:]
				} else {
					fmt.Fprintf(os.Stderr, "\nERROR: -replace deletion %d is greater than remaining sequence length %d\n", val, ln-pos)
					return
				}
			}
		} else {
			val := len(del)
			if val > 0 {
				if val <= ln-pos {
					// warn if existing sequence does not match deletion argument
					ext := str[pos : pos+val]
					if del != ext {
						fmt.Fprintf(os.Stderr, "\nWARNING: -replace deletion %s does not match existing sequence %s\n", del, ext)
					}
					// delete characters
					str = str[:pos] + str[pos+val:]
				} else {
					fmt.Fprintf(os.Stderr, "\nERROR: -replace deletion %d is greater than remaining sequence length %d\n", val, ln-pos)
					return
				}
			}
		}

		ln = len(str)
		if ins != "" {
			if pos <= ln {
				str = str[:pos] + ins + str[pos:]
			} else {
				fmt.Fprintf(os.Stderr, "\nERROR: -replace position %d is greater than remaining sequence length %d\n", pos, ln-pos)
				return
			}
		}
	}

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

func SequenceExtract(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	// skip past command name
	args = args[1:]

	if len(args) < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: Missing argument after -extract command\n")
		os.Exit(1)
	}

	// read output of xtract -insd feat_location qualifier
	feat_loc := args[0]

	str := ReadAllIntoSequence(inp)

	ln := len(str)

	// split intervals, e.g., "201..224,1550..1920,1986..2085,2317..2404,2466..2629"
	comma := strings.Split(feat_loc, ",")

	for _, item := range comma {

		// also allow dash separator, e.g., "201-224,1550-1920,1986-2085,2317-2404,2466-2629"
		item = strings.Replace(item, "-", "..", -1)

		fr, to := SplitInTwoAt(item, "..", LEFT)

		fr = strings.TrimSpace(fr)
		to = strings.TrimSpace(to)

		min, err := strconv.Atoi(fr)
		if err != nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized number '%s'\n", fr)
			os.Exit(1)
		}
		if min < 1 || min > ln {
			fmt.Fprintf(os.Stderr, "\nERROR: Starting point '%s' out of range\n", fr)
			os.Exit(1)
		}

		max, err := strconv.Atoi(to)
		if err != nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized number '%s'\n", to)
			os.Exit(1)
		}
		if max < 1 || max > ln {
			fmt.Fprintf(os.Stderr, "\nERROR: Ending point '%s' out of range\n", to)
			os.Exit(1)
		}

		if min < max {
			min--
			sub := str[min:max]
			os.Stdout.WriteString(sub)
		} else if min > max {
			max--
			sub := str[max:min]
			sub = ReverseComplement(sub)
			os.Stdout.WriteString(sub)
		} else {
			// need more information to know strand if single point
		}
	}

	os.Stdout.WriteString("\n")
}

// REVERSE SEQUENCE

// SeqFlip reverses without complementing - e.g., minus strand proteins translated in reverse order
func SeqFlip(inp io.Reader) {

	if inp == nil {
		return
	}

	str := ReadAllIntoSequence(inp)

	runes := []rune(str)
	// reverse sequence letters - middle base in odd-length sequence is not touched
	for i, j := 0, len(runes)-1; i < j; i, j = i+1, j-1 {
		runes[i], runes[j] = runes[j], runes[i]
	}
	str = string(runes)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

// REVERSE COMPLEMENT

func NucRevComp(inp io.Reader) {

	if inp == nil {
		return
	}

	str := ReadAllIntoSequence(inp)

	str = ReverseComplement(str)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

// FASTA DIFFERENCES

func PrintFastaPairs(frst, scnd string) {

	frst = strings.ToLower(frst)
	scnd = strings.ToLower(scnd)

	fst := frst[:]
	scd := scnd[:]

	// next functions return spaces after end of sequence
	nextF := func() rune {

		if len(fst) < 1 {
			return ' '
		}
		ch := fst[0]
		fst = fst[1:]

		return rune(ch)
	}

	nextS := func() rune {

		if len(scd) < 1 {
			return ' '
		}
		ch := scd[0]
		scd = scd[1:]

		return rune(ch)
	}

	var fs []rune
	var sc []rune
	mx := 0

	// populate output arrays
	for {

		f, s := nextF(), nextS()
		// if both spaces, end of both sequences
		if f == ' ' && s == ' ' {
			break
		}
		if f == s {
			fs = append(fs, f)
			sc = append(sc, ' ')
		} else {
			// show mismatches in upper case
			fs = append(fs, unicode.ToUpper(f))
			sc = append(sc, unicode.ToUpper(s))
		}
		mx++
	}

	// pad output to multiple of 50
	j := mx % 50
	if j > 0 {
		for j < 50 {
			fs = append(fs, ' ')
			sc = append(sc, ' ')
			j++
			mx++
		}
	}

	// print in blocks of 50 bases or residues
	for i := 0; i < mx; i += 50 {
		dl := 50
		if mx-i < 50 {
			dl = mx - i
		}
		lf := fs[:dl]
		rt := sc[:dl]
		fs = fs[dl:]
		sc = sc[dl:]
		tm := strings.TrimRight(string(lf), " ")
		fmt.Fprintf(os.Stdout, "%s %6d\n%s\n", string(lf), i+len(tm), string(rt))
	}
}

func FastaDiff(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	// skip past command name
	args = args[1:]

	if len(args) != 2 {
		fmt.Fprintf(os.Stderr, "\nERROR: Two files required by -diff command\n")
		os.Exit(1)
	}

	frst := args[0]
	scnd := args[1]

	frstFasta := ReadFromFileIntoSequence(frst)
	scndFasta := ReadFromFileIntoSequence(scnd)

	if frstFasta == scndFasta {
		return
	}

	// sequences are assumed to be aligned, this code highlight mismatches
	PrintFastaPairs(frstFasta, scndFasta)
}

// PROTEIN WEIGHT

func ProtWeight(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	trim_leading_met := true

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-met":
			trim_leading_met = false
			args = args[1:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -molwt command\n")
			os.Exit(1)
		}
	}

	str := ReadAllIntoSequence(inp)

	str = ProteinWeight(str, trim_leading_met)

	os.Stdout.WriteString(str)
	if !strings.HasSuffix(str, "\n") {
		os.Stdout.WriteString("\n")
	}
}

// PROTEIN TRANSLATION

const (
	Base_gap = iota
	Base_A   // A
	Base_C   // C
	Base_M   // AC
	Base_G   // G
	Base_R   // AG
	Base_S   // CG
	Base_V   // ACG
	Base_T   // T
	Base_W   // AT
	Base_Y   // CT
	Base_H   // ACT
	Base_K   // GT
	Base_D   // AGT
	Base_B   // CGT
	Base_N   // ACGT
)

/*
func init() {

	// generate baseToIdx for source code

	var baseToIdx [256]int

	upperToBase := "-ACMGRSVTWYHKDBN"
	lowerToBase := "-acmgrsvtwyhkdbn"

	// illegal characters map to 0
	for i := 0; i < 256; i++ {
		baseToIdx[i] = Base_gap
	}

	// map iupacna alphabet to int
	for i := Base_gap; i <= Base_N; i++ {
		ch := upperToBase[i]
		baseToIdx[int(ch)] = i
		ch = lowerToBase[i]
		baseToIdx[int(ch)] = i
	}
	baseToIdx['U'] = Base_T
	baseToIdx['u'] = Base_T
	baseToIdx['X'] = Base_N
	baseToIdx['x'] = Base_N

	// also map ncbi4na alphabet to int
	for i := Base_gap; i <= Base_N; i++ {
		baseToIdx[i] = i
	}

	fmt.Fprintf(os.Stdout, "var baseToIdx = map[int]int{\n")
	for i := 0; i < 256; i++ {
		if baseToIdx[i] != 0 {
			fmt.Fprintf(os.Stdout, "\t%d: %d,\n", i, baseToIdx[i])
		}
	}
	fmt.Fprintf(os.Stdout, "}\n\n")
}
*/

var baseToIdx = map[int]int{
	1:   1,
	2:   2,
	3:   3,
	4:   4,
	5:   5,
	6:   6,
	7:   7,
	8:   8,
	9:   9,
	10:  10,
	11:  11,
	12:  12,
	13:  13,
	14:  14,
	15:  15,
	65:  1,
	66:  14,
	67:  2,
	68:  13,
	71:  4,
	72:  11,
	75:  12,
	77:  3,
	78:  15,
	82:  5,
	83:  6,
	84:  8,
	85:  8,
	86:  7,
	87:  9,
	88:  15,
	89:  10,
	97:  1,
	98:  14,
	99:  2,
	100: 13,
	103: 4,
	104: 11,
	107: 12,
	109: 3,
	110: 15,
	114: 5,
	115: 6,
	116: 8,
	117: 8,
	118: 7,
	119: 9,
	120: 15,
	121: 10,
}

/*
func init() {

	// populate nextState and rvCpState arrays

	var nextState [4096]int
	var rvCpState [4096]int

	baseToComp := "-TGKCYSBAWRDMHVN"

	// states 0 through 4095 are triple letter states (---, --A, ..., NNT, NNN)
	for i, st := Base_gap, 0; i <= Base_N; i++ {
		for j, nx := Base_gap, 0; j <= Base_N; j++ {
			for k := Base_gap; k <= Base_N; k++ {
				nextState[st] = nx
				p := baseToIdx[int(baseToComp[k])]
				q := baseToIdx[int(baseToComp[j])]
				r := baseToIdx[int(baseToComp[i])]
				rvCpState[st] = int(256*p + 16*q + r)
				st++
				nx += 16
			}
		}
	}

	// NextCodonState indexes through nextState array and adds Base_* value
	fmt.Fprintf(os.Stdout, "var nextState = [4096]int{\n")
	for i := 0; i < 4096; i += 16 {
		fmt.Fprintf(os.Stdout, "\t")
		for j := 0; j < 16; j++ {
			fmt.Fprintf(os.Stdout, " %d,", nextState[i+j])
		}
		fmt.Fprintf(os.Stdout, "\n")
	}
	fmt.Fprintf(os.Stdout, "}\n\n")

	// RevCompState is a direct index to the reverse complement of a state
	fmt.Fprintf(os.Stdout, "var rvCpState = [4096]int{\n")
	for i := 0; i < 4096; i += 16 {
		fmt.Fprintf(os.Stdout, "\t")
		for j := 0; j < 16; j++ {
			fmt.Fprintf(os.Stdout, " %d,", rvCpState[i+j])
		}
		fmt.Fprintf(os.Stdout, "\n")
	}
	fmt.Fprintf(os.Stdout, "}\n\n")
}
*/

var nextState = [4096]int{
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
	   0,   16,   32,   48,   64,   80,   96,  112,  128,  144,  160,  176,  192,  208,  224,  240,
	 256,  272,  288,  304,  320,  336,  352,  368,  384,  400,  416,  432,  448,  464,  480,  496,
	 512,  528,  544,  560,  576,  592,  608,  624,  640,  656,  672,  688,  704,  720,  736,  752,
	 768,  784,  800,  816,  832,  848,  864,  880,  896,  912,  928,  944,  960,  976,  992, 1008,
	1024, 1040, 1056, 1072, 1088, 1104, 1120, 1136, 1152, 1168, 1184, 1200, 1216, 1232, 1248, 1264,
	1280, 1296, 1312, 1328, 1344, 1360, 1376, 1392, 1408, 1424, 1440, 1456, 1472, 1488, 1504, 1520,
	1536, 1552, 1568, 1584, 1600, 1616, 1632, 1648, 1664, 1680, 1696, 1712, 1728, 1744, 1760, 1776,
	1792, 1808, 1824, 1840, 1856, 1872, 1888, 1904, 1920, 1936, 1952, 1968, 1984, 2000, 2016, 2032,
	2048, 2064, 2080, 2096, 2112, 2128, 2144, 2160, 2176, 2192, 2208, 2224, 2240, 2256, 2272, 2288,
	2304, 2320, 2336, 2352, 2368, 2384, 2400, 2416, 2432, 2448, 2464, 2480, 2496, 2512, 2528, 2544,
	2560, 2576, 2592, 2608, 2624, 2640, 2656, 2672, 2688, 2704, 2720, 2736, 2752, 2768, 2784, 2800,
	2816, 2832, 2848, 2864, 2880, 2896, 2912, 2928, 2944, 2960, 2976, 2992, 3008, 3024, 3040, 3056,
	3072, 3088, 3104, 3120, 3136, 3152, 3168, 3184, 3200, 3216, 3232, 3248, 3264, 3280, 3296, 3312,
	3328, 3344, 3360, 3376, 3392, 3408, 3424, 3440, 3456, 3472, 3488, 3504, 3520, 3536, 3552, 3568,
	3584, 3600, 3616, 3632, 3648, 3664, 3680, 3696, 3712, 3728, 3744, 3760, 3776, 3792, 3808, 3824,
	3840, 3856, 3872, 3888, 3904, 3920, 3936, 3952, 3968, 3984, 4000, 4016, 4032, 4048, 4064, 4080,
}

var rvCpState = [4096]int{
	  0, 2048, 1024, 3072, 512, 2560, 1536, 3584, 256, 2304, 1280, 3328,  768, 2816, 1792, 3840,
	128, 2176, 1152, 3200, 640, 2688, 1664, 3712, 384, 2432, 1408, 3456,  896, 2944, 1920, 3968,
	 64, 2112, 1088, 3136, 576, 2624, 1600, 3648, 320, 2368, 1344, 3392,  832, 2880, 1856, 3904,
	192, 2240, 1216, 3264, 704, 2752, 1728, 3776, 448, 2496, 1472, 3520,  960, 3008, 1984, 4032,
	 32, 2080, 1056, 3104, 544, 2592, 1568, 3616, 288, 2336, 1312, 3360,  800, 2848, 1824, 3872,
	160, 2208, 1184, 3232, 672, 2720, 1696, 3744, 416, 2464, 1440, 3488,  928, 2976, 1952, 4000,
	 96, 2144, 1120, 3168, 608, 2656, 1632, 3680, 352, 2400, 1376, 3424,  864, 2912, 1888, 3936,
	224, 2272, 1248, 3296, 736, 2784, 1760, 3808, 480, 2528, 1504, 3552,  992, 3040, 2016, 4064,
	 16, 2064, 1040, 3088, 528, 2576, 1552, 3600, 272, 2320, 1296, 3344,  784, 2832, 1808, 3856,
	144, 2192, 1168, 3216, 656, 2704, 1680, 3728, 400, 2448, 1424, 3472,  912, 2960, 1936, 3984,
	 80, 2128, 1104, 3152, 592, 2640, 1616, 3664, 336, 2384, 1360, 3408,  848, 2896, 1872, 3920,
	208, 2256, 1232, 3280, 720, 2768, 1744, 3792, 464, 2512, 1488, 3536,  976, 3024, 2000, 4048,
	 48, 2096, 1072, 3120, 560, 2608, 1584, 3632, 304, 2352, 1328, 3376,  816, 2864, 1840, 3888,
	176, 2224, 1200, 3248, 688, 2736, 1712, 3760, 432, 2480, 1456, 3504,  944, 2992, 1968, 4016,
	112, 2160, 1136, 3184, 624, 2672, 1648, 3696, 368, 2416, 1392, 3440,  880, 2928, 1904, 3952,
	240, 2288, 1264, 3312, 752, 2800, 1776, 3824, 496, 2544, 1520, 3568, 1008, 3056, 2032, 4080,
	  8, 2056, 1032, 3080, 520, 2568, 1544, 3592, 264, 2312, 1288, 3336,  776, 2824, 1800, 3848,
	136, 2184, 1160, 3208, 648, 2696, 1672, 3720, 392, 2440, 1416, 3464,  904, 2952, 1928, 3976,
	 72, 2120, 1096, 3144, 584, 2632, 1608, 3656, 328, 2376, 1352, 3400,  840, 2888, 1864, 3912,
	200, 2248, 1224, 3272, 712, 2760, 1736, 3784, 456, 2504, 1480, 3528,  968, 3016, 1992, 4040,
	 40, 2088, 1064, 3112, 552, 2600, 1576, 3624, 296, 2344, 1320, 3368,  808, 2856, 1832, 3880,
	168, 2216, 1192, 3240, 680, 2728, 1704, 3752, 424, 2472, 1448, 3496,  936, 2984, 1960, 4008,
	104, 2152, 1128, 3176, 616, 2664, 1640, 3688, 360, 2408, 1384, 3432,  872, 2920, 1896, 3944,
	232, 2280, 1256, 3304, 744, 2792, 1768, 3816, 488, 2536, 1512, 3560, 1000, 3048, 2024, 4072,
	 24, 2072, 1048, 3096, 536, 2584, 1560, 3608, 280, 2328, 1304, 3352,  792, 2840, 1816, 3864,
	152, 2200, 1176, 3224, 664, 2712, 1688, 3736, 408, 2456, 1432, 3480,  920, 2968, 1944, 3992,
	 88, 2136, 1112, 3160, 600, 2648, 1624, 3672, 344, 2392, 1368, 3416,  856, 2904, 1880, 3928,
	216, 2264, 1240, 3288, 728, 2776, 1752, 3800, 472, 2520, 1496, 3544,  984, 3032, 2008, 4056,
	 56, 2104, 1080, 3128, 568, 2616, 1592, 3640, 312, 2360, 1336, 3384,  824, 2872, 1848, 3896,
	184, 2232, 1208, 3256, 696, 2744, 1720, 3768, 440, 2488, 1464, 3512,  952, 3000, 1976, 4024,
	120, 2168, 1144, 3192, 632, 2680, 1656, 3704, 376, 2424, 1400, 3448,  888, 2936, 1912, 3960,
	248, 2296, 1272, 3320, 760, 2808, 1784, 3832, 504, 2552, 1528, 3576, 1016, 3064, 2040, 4088,
	  4, 2052, 1028, 3076, 516, 2564, 1540, 3588, 260, 2308, 1284, 3332,  772, 2820, 1796, 3844,
	132, 2180, 1156, 3204, 644, 2692, 1668, 3716, 388, 2436, 1412, 3460,  900, 2948, 1924, 3972,
	 68, 2116, 1092, 3140, 580, 2628, 1604, 3652, 324, 2372, 1348, 3396,  836, 2884, 1860, 3908,
	196, 2244, 1220, 3268, 708, 2756, 1732, 3780, 452, 2500, 1476, 3524,  964, 3012, 1988, 4036,
	 36, 2084, 1060, 3108, 548, 2596, 1572, 3620, 292, 2340, 1316, 3364,  804, 2852, 1828, 3876,
	164, 2212, 1188, 3236, 676, 2724, 1700, 3748, 420, 2468, 1444, 3492,  932, 2980, 1956, 4004,
	100, 2148, 1124, 3172, 612, 2660, 1636, 3684, 356, 2404, 1380, 3428,  868, 2916, 1892, 3940,
	228, 2276, 1252, 3300, 740, 2788, 1764, 3812, 484, 2532, 1508, 3556,  996, 3044, 2020, 4068,
	 20, 2068, 1044, 3092, 532, 2580, 1556, 3604, 276, 2324, 1300, 3348,  788, 2836, 1812, 3860,
	148, 2196, 1172, 3220, 660, 2708, 1684, 3732, 404, 2452, 1428, 3476,  916, 2964, 1940, 3988,
	 84, 2132, 1108, 3156, 596, 2644, 1620, 3668, 340, 2388, 1364, 3412,  852, 2900, 1876, 3924,
	212, 2260, 1236, 3284, 724, 2772, 1748, 3796, 468, 2516, 1492, 3540,  980, 3028, 2004, 4052,
	 52, 2100, 1076, 3124, 564, 2612, 1588, 3636, 308, 2356, 1332, 3380,  820, 2868, 1844, 3892,
	180, 2228, 1204, 3252, 692, 2740, 1716, 3764, 436, 2484, 1460, 3508,  948, 2996, 1972, 4020,
	116, 2164, 1140, 3188, 628, 2676, 1652, 3700, 372, 2420, 1396, 3444,  884, 2932, 1908, 3956,
	244, 2292, 1268, 3316, 756, 2804, 1780, 3828, 500, 2548, 1524, 3572, 1012, 3060, 2036, 4084,
	 12, 2060, 1036, 3084, 524, 2572, 1548, 3596, 268, 2316, 1292, 3340,  780, 2828, 1804, 3852,
	140, 2188, 1164, 3212, 652, 2700, 1676, 3724, 396, 2444, 1420, 3468,  908, 2956, 1932, 3980,
	 76, 2124, 1100, 3148, 588, 2636, 1612, 3660, 332, 2380, 1356, 3404,  844, 2892, 1868, 3916,
	204, 2252, 1228, 3276, 716, 2764, 1740, 3788, 460, 2508, 1484, 3532,  972, 3020, 1996, 4044,
	 44, 2092, 1068, 3116, 556, 2604, 1580, 3628, 300, 2348, 1324, 3372,  812, 2860, 1836, 3884,
	172, 2220, 1196, 3244, 684, 2732, 1708, 3756, 428, 2476, 1452, 3500,  940, 2988, 1964, 4012,
	108, 2156, 1132, 3180, 620, 2668, 1644, 3692, 364, 2412, 1388, 3436,  876, 2924, 1900, 3948,
	236, 2284, 1260, 3308, 748, 2796, 1772, 3820, 492, 2540, 1516, 3564, 1004, 3052, 2028, 4076,
	 28, 2076, 1052, 3100, 540, 2588, 1564, 3612, 284, 2332, 1308, 3356,  796, 2844, 1820, 3868,
	156, 2204, 1180, 3228, 668, 2716, 1692, 3740, 412, 2460, 1436, 3484,  924, 2972, 1948, 3996,
	 92, 2140, 1116, 3164, 604, 2652, 1628, 3676, 348, 2396, 1372, 3420,  860, 2908, 1884, 3932,
	220, 2268, 1244, 3292, 732, 2780, 1756, 3804, 476, 2524, 1500, 3548,  988, 3036, 2012, 4060,
	 60, 2108, 1084, 3132, 572, 2620, 1596, 3644, 316, 2364, 1340, 3388,  828, 2876, 1852, 3900,
	188, 2236, 1212, 3260, 700, 2748, 1724, 3772, 444, 2492, 1468, 3516,  956, 3004, 1980, 4028,
	124, 2172, 1148, 3196, 636, 2684, 1660, 3708, 380, 2428, 1404, 3452,  892, 2940, 1916, 3964,
	252, 2300, 1276, 3324, 764, 2812, 1788, 3836, 508, 2556, 1532, 3580, 1020, 3068, 2044, 4092,
	  2, 2050, 1026, 3074, 514, 2562, 1538, 3586, 258, 2306, 1282, 3330,  770, 2818, 1794, 3842,
	130, 2178, 1154, 3202, 642, 2690, 1666, 3714, 386, 2434, 1410, 3458,  898, 2946, 1922, 3970,
	 66, 2114, 1090, 3138, 578, 2626, 1602, 3650, 322, 2370, 1346, 3394,  834, 2882, 1858, 3906,
	194, 2242, 1218, 3266, 706, 2754, 1730, 3778, 450, 2498, 1474, 3522,  962, 3010, 1986, 4034,
	 34, 2082, 1058, 3106, 546, 2594, 1570, 3618, 290, 2338, 1314, 3362,  802, 2850, 1826, 3874,
	162, 2210, 1186, 3234, 674, 2722, 1698, 3746, 418, 2466, 1442, 3490,  930, 2978, 1954, 4002,
	 98, 2146, 1122, 3170, 610, 2658, 1634, 3682, 354, 2402, 1378, 3426,  866, 2914, 1890, 3938,
	226, 2274, 1250, 3298, 738, 2786, 1762, 3810, 482, 2530, 1506, 3554,  994, 3042, 2018, 4066,
	 18, 2066, 1042, 3090, 530, 2578, 1554, 3602, 274, 2322, 1298, 3346,  786, 2834, 1810, 3858,
	146, 2194, 1170, 3218, 658, 2706, 1682, 3730, 402, 2450, 1426, 3474,  914, 2962, 1938, 3986,
	 82, 2130, 1106, 3154, 594, 2642, 1618, 3666, 338, 2386, 1362, 3410,  850, 2898, 1874, 3922,
	210, 2258, 1234, 3282, 722, 2770, 1746, 3794, 466, 2514, 1490, 3538,  978, 3026, 2002, 4050,
	 50, 2098, 1074, 3122, 562, 2610, 1586, 3634, 306, 2354, 1330, 3378,  818, 2866, 1842, 3890,
	178, 2226, 1202, 3250, 690, 2738, 1714, 3762, 434, 2482, 1458, 3506,  946, 2994, 1970, 4018,
	114, 2162, 1138, 3186, 626, 2674, 1650, 3698, 370, 2418, 1394, 3442,  882, 2930, 1906, 3954,
	242, 2290, 1266, 3314, 754, 2802, 1778, 3826, 498, 2546, 1522, 3570, 1010, 3058, 2034, 4082,
	 10, 2058, 1034, 3082, 522, 2570, 1546, 3594, 266, 2314, 1290, 3338,  778, 2826, 1802, 3850,
	138, 2186, 1162, 3210, 650, 2698, 1674, 3722, 394, 2442, 1418, 3466,  906, 2954, 1930, 3978,
	 74, 2122, 1098, 3146, 586, 2634, 1610, 3658, 330, 2378, 1354, 3402,  842, 2890, 1866, 3914,
	202, 2250, 1226, 3274, 714, 2762, 1738, 3786, 458, 2506, 1482, 3530,  970, 3018, 1994, 4042,
	 42, 2090, 1066, 3114, 554, 2602, 1578, 3626, 298, 2346, 1322, 3370,  810, 2858, 1834, 3882,
	170, 2218, 1194, 3242, 682, 2730, 1706, 3754, 426, 2474, 1450, 3498,  938, 2986, 1962, 4010,
	106, 2154, 1130, 3178, 618, 2666, 1642, 3690, 362, 2410, 1386, 3434,  874, 2922, 1898, 3946,
	234, 2282, 1258, 3306, 746, 2794, 1770, 3818, 490, 2538, 1514, 3562, 1002, 3050, 2026, 4074,
	 26, 2074, 1050, 3098, 538, 2586, 1562, 3610, 282, 2330, 1306, 3354,  794, 2842, 1818, 3866,
	154, 2202, 1178, 3226, 666, 2714, 1690, 3738, 410, 2458, 1434, 3482,  922, 2970, 1946, 3994,
	 90, 2138, 1114, 3162, 602, 2650, 1626, 3674, 346, 2394, 1370, 3418,  858, 2906, 1882, 3930,
	218, 2266, 1242, 3290, 730, 2778, 1754, 3802, 474, 2522, 1498, 3546,  986, 3034, 2010, 4058,
	 58, 2106, 1082, 3130, 570, 2618, 1594, 3642, 314, 2362, 1338, 3386,  826, 2874, 1850, 3898,
	186, 2234, 1210, 3258, 698, 2746, 1722, 3770, 442, 2490, 1466, 3514,  954, 3002, 1978, 4026,
	122, 2170, 1146, 3194, 634, 2682, 1658, 3706, 378, 2426, 1402, 3450,  890, 2938, 1914, 3962,
	250, 2298, 1274, 3322, 762, 2810, 1786, 3834, 506, 2554, 1530, 3578, 1018, 3066, 2042, 4090,
	  6, 2054, 1030, 3078, 518, 2566, 1542, 3590, 262, 2310, 1286, 3334,  774, 2822, 1798, 3846,
	134, 2182, 1158, 3206, 646, 2694, 1670, 3718, 390, 2438, 1414, 3462,  902, 2950, 1926, 3974,
	 70, 2118, 1094, 3142, 582, 2630, 1606, 3654, 326, 2374, 1350, 3398,  838, 2886, 1862, 3910,
	198, 2246, 1222, 3270, 710, 2758, 1734, 3782, 454, 2502, 1478, 3526,  966, 3014, 1990, 4038,
	 38, 2086, 1062, 3110, 550, 2598, 1574, 3622, 294, 2342, 1318, 3366,  806, 2854, 1830, 3878,
	166, 2214, 1190, 3238, 678, 2726, 1702, 3750, 422, 2470, 1446, 3494,  934, 2982, 1958, 4006,
	102, 2150, 1126, 3174, 614, 2662, 1638, 3686, 358, 2406, 1382, 3430,  870, 2918, 1894, 3942,
	230, 2278, 1254, 3302, 742, 2790, 1766, 3814, 486, 2534, 1510, 3558,  998, 3046, 2022, 4070,
	 22, 2070, 1046, 3094, 534, 2582, 1558, 3606, 278, 2326, 1302, 3350,  790, 2838, 1814, 3862,
	150, 2198, 1174, 3222, 662, 2710, 1686, 3734, 406, 2454, 1430, 3478,  918, 2966, 1942, 3990,
	 86, 2134, 1110, 3158, 598, 2646, 1622, 3670, 342, 2390, 1366, 3414,  854, 2902, 1878, 3926,
	214, 2262, 1238, 3286, 726, 2774, 1750, 3798, 470, 2518, 1494, 3542,  982, 3030, 2006, 4054,
	 54, 2102, 1078, 3126, 566, 2614, 1590, 3638, 310, 2358, 1334, 3382,  822, 2870, 1846, 3894,
	182, 2230, 1206, 3254, 694, 2742, 1718, 3766, 438, 2486, 1462, 3510,  950, 2998, 1974, 4022,
	118, 2166, 1142, 3190, 630, 2678, 1654, 3702, 374, 2422, 1398, 3446,  886, 2934, 1910, 3958,
	246, 2294, 1270, 3318, 758, 2806, 1782, 3830, 502, 2550, 1526, 3574, 1014, 3062, 2038, 4086,
	 14, 2062, 1038, 3086, 526, 2574, 1550, 3598, 270, 2318, 1294, 3342,  782, 2830, 1806, 3854,
	142, 2190, 1166, 3214, 654, 2702, 1678, 3726, 398, 2446, 1422, 3470,  910, 2958, 1934, 3982,
	 78, 2126, 1102, 3150, 590, 2638, 1614, 3662, 334, 2382, 1358, 3406,  846, 2894, 1870, 3918,
	206, 2254, 1230, 3278, 718, 2766, 1742, 3790, 462, 2510, 1486, 3534,  974, 3022, 1998, 4046,
	 46, 2094, 1070, 3118, 558, 2606, 1582, 3630, 302, 2350, 1326, 3374,  814, 2862, 1838, 3886,
	174, 2222, 1198, 3246, 686, 2734, 1710, 3758, 430, 2478, 1454, 3502,  942, 2990, 1966, 4014,
	110, 2158, 1134, 3182, 622, 2670, 1646, 3694, 366, 2414, 1390, 3438,  878, 2926, 1902, 3950,
	238, 2286, 1262, 3310, 750, 2798, 1774, 3822, 494, 2542, 1518, 3566, 1006, 3054, 2030, 4078,
	 30, 2078, 1054, 3102, 542, 2590, 1566, 3614, 286, 2334, 1310, 3358,  798, 2846, 1822, 3870,
	158, 2206, 1182, 3230, 670, 2718, 1694, 3742, 414, 2462, 1438, 3486,  926, 2974, 1950, 3998,
	 94, 2142, 1118, 3166, 606, 2654, 1630, 3678, 350, 2398, 1374, 3422,  862, 2910, 1886, 3934,
	222, 2270, 1246, 3294, 734, 2782, 1758, 3806, 478, 2526, 1502, 3550,  990, 3038, 2014, 4062,
	 62, 2110, 1086, 3134, 574, 2622, 1598, 3646, 318, 2366, 1342, 3390,  830, 2878, 1854, 3902,
	190, 2238, 1214, 3262, 702, 2750, 1726, 3774, 446, 2494, 1470, 3518,  958, 3006, 1982, 4030,
	126, 2174, 1150, 3198, 638, 2686, 1662, 3710, 382, 2430, 1406, 3454,  894, 2942, 1918, 3966,
	254, 2302, 1278, 3326, 766, 2814, 1790, 3838, 510, 2558, 1534, 3582, 1022, 3070, 2046, 4094,
	  1, 2049, 1025, 3073, 513, 2561, 1537, 3585, 257, 2305, 1281, 3329,  769, 2817, 1793, 3841,
	129, 2177, 1153, 3201, 641, 2689, 1665, 3713, 385, 2433, 1409, 3457,  897, 2945, 1921, 3969,
	 65, 2113, 1089, 3137, 577, 2625, 1601, 3649, 321, 2369, 1345, 3393,  833, 2881, 1857, 3905,
	193, 2241, 1217, 3265, 705, 2753, 1729, 3777, 449, 2497, 1473, 3521,  961, 3009, 1985, 4033,
	 33, 2081, 1057, 3105, 545, 2593, 1569, 3617, 289, 2337, 1313, 3361,  801, 2849, 1825, 3873,
	161, 2209, 1185, 3233, 673, 2721, 1697, 3745, 417, 2465, 1441, 3489,  929, 2977, 1953, 4001,
	 97, 2145, 1121, 3169, 609, 2657, 1633, 3681, 353, 2401, 1377, 3425,  865, 2913, 1889, 3937,
	225, 2273, 1249, 3297, 737, 2785, 1761, 3809, 481, 2529, 1505, 3553,  993, 3041, 2017, 4065,
	 17, 2065, 1041, 3089, 529, 2577, 1553, 3601, 273, 2321, 1297, 3345,  785, 2833, 1809, 3857,
	145, 2193, 1169, 3217, 657, 2705, 1681, 3729, 401, 2449, 1425, 3473,  913, 2961, 1937, 3985,
	 81, 2129, 1105, 3153, 593, 2641, 1617, 3665, 337, 2385, 1361, 3409,  849, 2897, 1873, 3921,
	209, 2257, 1233, 3281, 721, 2769, 1745, 3793, 465, 2513, 1489, 3537,  977, 3025, 2001, 4049,
	 49, 2097, 1073, 3121, 561, 2609, 1585, 3633, 305, 2353, 1329, 3377,  817, 2865, 1841, 3889,
	177, 2225, 1201, 3249, 689, 2737, 1713, 3761, 433, 2481, 1457, 3505,  945, 2993, 1969, 4017,
	113, 2161, 1137, 3185, 625, 2673, 1649, 3697, 369, 2417, 1393, 3441,  881, 2929, 1905, 3953,
	241, 2289, 1265, 3313, 753, 2801, 1777, 3825, 497, 2545, 1521, 3569, 1009, 3057, 2033, 4081,
	  9, 2057, 1033, 3081, 521, 2569, 1545, 3593, 265, 2313, 1289, 3337,  777, 2825, 1801, 3849,
	137, 2185, 1161, 3209, 649, 2697, 1673, 3721, 393, 2441, 1417, 3465,  905, 2953, 1929, 3977,
	 73, 2121, 1097, 3145, 585, 2633, 1609, 3657, 329, 2377, 1353, 3401,  841, 2889, 1865, 3913,
	201, 2249, 1225, 3273, 713, 2761, 1737, 3785, 457, 2505, 1481, 3529,  969, 3017, 1993, 4041,
	 41, 2089, 1065, 3113, 553, 2601, 1577, 3625, 297, 2345, 1321, 3369,  809, 2857, 1833, 3881,
	169, 2217, 1193, 3241, 681, 2729, 1705, 3753, 425, 2473, 1449, 3497,  937, 2985, 1961, 4009,
	105, 2153, 1129, 3177, 617, 2665, 1641, 3689, 361, 2409, 1385, 3433,  873, 2921, 1897, 3945,
	233, 2281, 1257, 3305, 745, 2793, 1769, 3817, 489, 2537, 1513, 3561, 1001, 3049, 2025, 4073,
	 25, 2073, 1049, 3097, 537, 2585, 1561, 3609, 281, 2329, 1305, 3353,  793, 2841, 1817, 3865,
	153, 2201, 1177, 3225, 665, 2713, 1689, 3737, 409, 2457, 1433, 3481,  921, 2969, 1945, 3993,
	 89, 2137, 1113, 3161, 601, 2649, 1625, 3673, 345, 2393, 1369, 3417,  857, 2905, 1881, 3929,
	217, 2265, 1241, 3289, 729, 2777, 1753, 3801, 473, 2521, 1497, 3545,  985, 3033, 2009, 4057,
	 57, 2105, 1081, 3129, 569, 2617, 1593, 3641, 313, 2361, 1337, 3385,  825, 2873, 1849, 3897,
	185, 2233, 1209, 3257, 697, 2745, 1721, 3769, 441, 2489, 1465, 3513,  953, 3001, 1977, 4025,
	121, 2169, 1145, 3193, 633, 2681, 1657, 3705, 377, 2425, 1401, 3449,  889, 2937, 1913, 3961,
	249, 2297, 1273, 3321, 761, 2809, 1785, 3833, 505, 2553, 1529, 3577, 1017, 3065, 2041, 4089,
	  5, 2053, 1029, 3077, 517, 2565, 1541, 3589, 261, 2309, 1285, 3333,  773, 2821, 1797, 3845,
	133, 2181, 1157, 3205, 645, 2693, 1669, 3717, 389, 2437, 1413, 3461,  901, 2949, 1925, 3973,
	 69, 2117, 1093, 3141, 581, 2629, 1605, 3653, 325, 2373, 1349, 3397,  837, 2885, 1861, 3909,
	197, 2245, 1221, 3269, 709, 2757, 1733, 3781, 453, 2501, 1477, 3525,  965, 3013, 1989, 4037,
	 37, 2085, 1061, 3109, 549, 2597, 1573, 3621, 293, 2341, 1317, 3365,  805, 2853, 1829, 3877,
	165, 2213, 1189, 3237, 677, 2725, 1701, 3749, 421, 2469, 1445, 3493,  933, 2981, 1957, 4005,
	101, 2149, 1125, 3173, 613, 2661, 1637, 3685, 357, 2405, 1381, 3429,  869, 2917, 1893, 3941,
	229, 2277, 1253, 3301, 741, 2789, 1765, 3813, 485, 2533, 1509, 3557,  997, 3045, 2021, 4069,
	 21, 2069, 1045, 3093, 533, 2581, 1557, 3605, 277, 2325, 1301, 3349,  789, 2837, 1813, 3861,
	149, 2197, 1173, 3221, 661, 2709, 1685, 3733, 405, 2453, 1429, 3477,  917, 2965, 1941, 3989,
	 85, 2133, 1109, 3157, 597, 2645, 1621, 3669, 341, 2389, 1365, 3413,  853, 2901, 1877, 3925,
	213, 2261, 1237, 3285, 725, 2773, 1749, 3797, 469, 2517, 1493, 3541,  981, 3029, 2005, 4053,
	 53, 2101, 1077, 3125, 565, 2613, 1589, 3637, 309, 2357, 1333, 3381,  821, 2869, 1845, 3893,
	181, 2229, 1205, 3253, 693, 2741, 1717, 3765, 437, 2485, 1461, 3509,  949, 2997, 1973, 4021,
	117, 2165, 1141, 3189, 629, 2677, 1653, 3701, 373, 2421, 1397, 3445,  885, 2933, 1909, 3957,
	245, 2293, 1269, 3317, 757, 2805, 1781, 3829, 501, 2549, 1525, 3573, 1013, 3061, 2037, 4085,
	 13, 2061, 1037, 3085, 525, 2573, 1549, 3597, 269, 2317, 1293, 3341,  781, 2829, 1805, 3853,
	141, 2189, 1165, 3213, 653, 2701, 1677, 3725, 397, 2445, 1421, 3469,  909, 2957, 1933, 3981,
	 77, 2125, 1101, 3149, 589, 2637, 1613, 3661, 333, 2381, 1357, 3405,  845, 2893, 1869, 3917,
	205, 2253, 1229, 3277, 717, 2765, 1741, 3789, 461, 2509, 1485, 3533,  973, 3021, 1997, 4045,
	 45, 2093, 1069, 3117, 557, 2605, 1581, 3629, 301, 2349, 1325, 3373,  813, 2861, 1837, 3885,
	173, 2221, 1197, 3245, 685, 2733, 1709, 3757, 429, 2477, 1453, 3501,  941, 2989, 1965, 4013,
	109, 2157, 1133, 3181, 621, 2669, 1645, 3693, 365, 2413, 1389, 3437,  877, 2925, 1901, 3949,
	237, 2285, 1261, 3309, 749, 2797, 1773, 3821, 493, 2541, 1517, 3565, 1005, 3053, 2029, 4077,
	 29, 2077, 1053, 3101, 541, 2589, 1565, 3613, 285, 2333, 1309, 3357,  797, 2845, 1821, 3869,
	157, 2205, 1181, 3229, 669, 2717, 1693, 3741, 413, 2461, 1437, 3485,  925, 2973, 1949, 3997,
	 93, 2141, 1117, 3165, 605, 2653, 1629, 3677, 349, 2397, 1373, 3421,  861, 2909, 1885, 3933,
	221, 2269, 1245, 3293, 733, 2781, 1757, 3805, 477, 2525, 1501, 3549,  989, 3037, 2013, 4061,
	 61, 2109, 1085, 3133, 573, 2621, 1597, 3645, 317, 2365, 1341, 3389,  829, 2877, 1853, 3901,
	189, 2237, 1213, 3261, 701, 2749, 1725, 3773, 445, 2493, 1469, 3517,  957, 3005, 1981, 4029,
	125, 2173, 1149, 3197, 637, 2685, 1661, 3709, 381, 2429, 1405, 3453,  893, 2941, 1917, 3965,
	253, 2301, 1277, 3325, 765, 2813, 1789, 3837, 509, 2557, 1533, 3581, 1021, 3069, 2045, 4093,
	  3, 2051, 1027, 3075, 515, 2563, 1539, 3587, 259, 2307, 1283, 3331,  771, 2819, 1795, 3843,
	131, 2179, 1155, 3203, 643, 2691, 1667, 3715, 387, 2435, 1411, 3459,  899, 2947, 1923, 3971,
	 67, 2115, 1091, 3139, 579, 2627, 1603, 3651, 323, 2371, 1347, 3395,  835, 2883, 1859, 3907,
	195, 2243, 1219, 3267, 707, 2755, 1731, 3779, 451, 2499, 1475, 3523,  963, 3011, 1987, 4035,
	 35, 2083, 1059, 3107, 547, 2595, 1571, 3619, 291, 2339, 1315, 3363,  803, 2851, 1827, 3875,
	163, 2211, 1187, 3235, 675, 2723, 1699, 3747, 419, 2467, 1443, 3491,  931, 2979, 1955, 4003,
	 99, 2147, 1123, 3171, 611, 2659, 1635, 3683, 355, 2403, 1379, 3427,  867, 2915, 1891, 3939,
	227, 2275, 1251, 3299, 739, 2787, 1763, 3811, 483, 2531, 1507, 3555,  995, 3043, 2019, 4067,
	 19, 2067, 1043, 3091, 531, 2579, 1555, 3603, 275, 2323, 1299, 3347,  787, 2835, 1811, 3859,
	147, 2195, 1171, 3219, 659, 2707, 1683, 3731, 403, 2451, 1427, 3475,  915, 2963, 1939, 3987,
	 83, 2131, 1107, 3155, 595, 2643, 1619, 3667, 339, 2387, 1363, 3411,  851, 2899, 1875, 3923,
	211, 2259, 1235, 3283, 723, 2771, 1747, 3795, 467, 2515, 1491, 3539,  979, 3027, 2003, 4051,
	 51, 2099, 1075, 3123, 563, 2611, 1587, 3635, 307, 2355, 1331, 3379,  819, 2867, 1843, 3891,
	179, 2227, 1203, 3251, 691, 2739, 1715, 3763, 435, 2483, 1459, 3507,  947, 2995, 1971, 4019,
	115, 2163, 1139, 3187, 627, 2675, 1651, 3699, 371, 2419, 1395, 3443,  883, 2931, 1907, 3955,
	243, 2291, 1267, 3315, 755, 2803, 1779, 3827, 499, 2547, 1523, 3571, 1011, 3059, 2035, 4083,
	 11, 2059, 1035, 3083, 523, 2571, 1547, 3595, 267, 2315, 1291, 3339,  779, 2827, 1803, 3851,
	139, 2187, 1163, 3211, 651, 2699, 1675, 3723, 395, 2443, 1419, 3467,  907, 2955, 1931, 3979,
	 75, 2123, 1099, 3147, 587, 2635, 1611, 3659, 331, 2379, 1355, 3403,  843, 2891, 1867, 3915,
	203, 2251, 1227, 3275, 715, 2763, 1739, 3787, 459, 2507, 1483, 3531,  971, 3019, 1995, 4043,
	 43, 2091, 1067, 3115, 555, 2603, 1579, 3627, 299, 2347, 1323, 3371,  811, 2859, 1835, 3883,
	171, 2219, 1195, 3243, 683, 2731, 1707, 3755, 427, 2475, 1451, 3499,  939, 2987, 1963, 4011,
	107, 2155, 1131, 3179, 619, 2667, 1643, 3691, 363, 2411, 1387, 3435,  875, 2923, 1899, 3947,
	235, 2283, 1259, 3307, 747, 2795, 1771, 3819, 491, 2539, 1515, 3563, 1003, 3051, 2027, 4075,
	 27, 2075, 1051, 3099, 539, 2587, 1563, 3611, 283, 2331, 1307, 3355,  795, 2843, 1819, 3867,
	155, 2203, 1179, 3227, 667, 2715, 1691, 3739, 411, 2459, 1435, 3483,  923, 2971, 1947, 3995,
	 91, 2139, 1115, 3163, 603, 2651, 1627, 3675, 347, 2395, 1371, 3419,  859, 2907, 1883, 3931,
	219, 2267, 1243, 3291, 731, 2779, 1755, 3803, 475, 2523, 1499, 3547,  987, 3035, 2011, 4059,
	 59, 2107, 1083, 3131, 571, 2619, 1595, 3643, 315, 2363, 1339, 3387,  827, 2875, 1851, 3899,
	187, 2235, 1211, 3259, 699, 2747, 1723, 3771, 443, 2491, 1467, 3515,  955, 3003, 1979, 4027,
	123, 2171, 1147, 3195, 635, 2683, 1659, 3707, 379, 2427, 1403, 3451,  891, 2939, 1915, 3963,
	251, 2299, 1275, 3323, 763, 2811, 1787, 3835, 507, 2555, 1531, 3579, 1019, 3067, 2043, 4091,
	  7, 2055, 1031, 3079, 519, 2567, 1543, 3591, 263, 2311, 1287, 3335,  775, 2823, 1799, 3847,
	135, 2183, 1159, 3207, 647, 2695, 1671, 3719, 391, 2439, 1415, 3463,  903, 2951, 1927, 3975,
	 71, 2119, 1095, 3143, 583, 2631, 1607, 3655, 327, 2375, 1351, 3399,  839, 2887, 1863, 3911,
	199, 2247, 1223, 3271, 711, 2759, 1735, 3783, 455, 2503, 1479, 3527,  967, 3015, 1991, 4039,
	 39, 2087, 1063, 3111, 551, 2599, 1575, 3623, 295, 2343, 1319, 3367,  807, 2855, 1831, 3879,
	167, 2215, 1191, 3239, 679, 2727, 1703, 3751, 423, 2471, 1447, 3495,  935, 2983, 1959, 4007,
	103, 2151, 1127, 3175, 615, 2663, 1639, 3687, 359, 2407, 1383, 3431,  871, 2919, 1895, 3943,
	231, 2279, 1255, 3303, 743, 2791, 1767, 3815, 487, 2535, 1511, 3559,  999, 3047, 2023, 4071,
	 23, 2071, 1047, 3095, 535, 2583, 1559, 3607, 279, 2327, 1303, 3351,  791, 2839, 1815, 3863,
	151, 2199, 1175, 3223, 663, 2711, 1687, 3735, 407, 2455, 1431, 3479,  919, 2967, 1943, 3991,
	 87, 2135, 1111, 3159, 599, 2647, 1623, 3671, 343, 2391, 1367, 3415,  855, 2903, 1879, 3927,
	215, 2263, 1239, 3287, 727, 2775, 1751, 3799, 471, 2519, 1495, 3543,  983, 3031, 2007, 4055,
	 55, 2103, 1079, 3127, 567, 2615, 1591, 3639, 311, 2359, 1335, 3383,  823, 2871, 1847, 3895,
	183, 2231, 1207, 3255, 695, 2743, 1719, 3767, 439, 2487, 1463, 3511,  951, 2999, 1975, 4023,
	119, 2167, 1143, 3191, 631, 2679, 1655, 3703, 375, 2423, 1399, 3447,  887, 2935, 1911, 3959,
	247, 2295, 1271, 3319, 759, 2807, 1783, 3831, 503, 2551, 1527, 3575, 1015, 3063, 2039, 4087,
	 15, 2063, 1039, 3087, 527, 2575, 1551, 3599, 271, 2319, 1295, 3343,  783, 2831, 1807, 3855,
	143, 2191, 1167, 3215, 655, 2703, 1679, 3727, 399, 2447, 1423, 3471,  911, 2959, 1935, 3983,
	 79, 2127, 1103, 3151, 591, 2639, 1615, 3663, 335, 2383, 1359, 3407,  847, 2895, 1871, 3919,
	207, 2255, 1231, 3279, 719, 2767, 1743, 3791, 463, 2511, 1487, 3535,  975, 3023, 1999, 4047,
	 47, 2095, 1071, 3119, 559, 2607, 1583, 3631, 303, 2351, 1327, 3375,  815, 2863, 1839, 3887,
	175, 2223, 1199, 3247, 687, 2735, 1711, 3759, 431, 2479, 1455, 3503,  943, 2991, 1967, 4015,
	111, 2159, 1135, 3183, 623, 2671, 1647, 3695, 367, 2415, 1391, 3439,  879, 2927, 1903, 3951,
	239, 2287, 1263, 3311, 751, 2799, 1775, 3823, 495, 2543, 1519, 3567, 1007, 3055, 2031, 4079,
	 31, 2079, 1055, 3103, 543, 2591, 1567, 3615, 287, 2335, 1311, 3359,  799, 2847, 1823, 3871,
	159, 2207, 1183, 3231, 671, 2719, 1695, 3743, 415, 2463, 1439, 3487,  927, 2975, 1951, 3999,
	 95, 2143, 1119, 3167, 607, 2655, 1631, 3679, 351, 2399, 1375, 3423,  863, 2911, 1887, 3935,
	223, 2271, 1247, 3295, 735, 2783, 1759, 3807, 479, 2527, 1503, 3551,  991, 3039, 2015, 4063,
	 63, 2111, 1087, 3135, 575, 2623, 1599, 3647, 319, 2367, 1343, 3391,  831, 2879, 1855, 3903,
	191, 2239, 1215, 3263, 703, 2751, 1727, 3775, 447, 2495, 1471, 3519,  959, 3007, 1983, 4031,
	127, 2175, 1151, 3199, 639, 2687, 1663, 3711, 383, 2431, 1407, 3455,  895, 2943, 1919, 3967,
	255, 2303, 1279, 3327, 767, 2815, 1791, 3839, 511, 2559, 1535, 3583, 1023, 3071, 2047, 4095,
}

/*

// Standard Genetic Code Table

                           Second Position
  First      T             C             A             G      Third
  -----------------------------------------------------------------
    T   TTT Phe [F]   TCT Ser [S]   TAT Tyr [Y]   TGT Cys [C]   T
        TTC Phe [F]   TCC Ser [S]   TAC Tyr [Y]   TGC Cys [C]   C
        TTA Leu [L]   TCA Ser [S]   TAA Och [*]   TGA Opa [*]   A
        TTG Leu [L]   TCG Ser [S]   TAG Amb [*]   TGG Trp [W]   G
  -----------------------------------------------------------------
    C   CTT Leu [L]   CCT Pro [P]   CAT His [H]   CGT Arg [R]   T
        CTC Leu [L]   CCC Pro [P]   CAC His [H]   CGC Arg [R]   C
        CTA Leu [L]   CCA Pro [P]   CAA Gln [Q]   CGA Arg [R]   A
        CTG Leu [L]   CCG Pro [P]   CAG Gln [Q]   CGG Arg [R]   G
  -----------------------------------------------------------------
    A   ATT Ile [I]   ACT Thr [T]   AAT Asn [N]   AGT Ser [S]   T
        ATC Ile [I]   ACC Thr [T]   AAC Asn [N]   AGC Ser [S]   C
        ATA Ile [I]   ACA Thr [T]   AAA Lys [K]   AGA Arg [R]   A
        ATG Met [M]   ACG Thr [T]   AAG Lys [K]   AGG Arg [R]   G
  -----------------------------------------------------------------
    G   GTT Val [V]   GCT Ala [A]   GAT Asp [D]   GGT Gly [G]   T
        GTC Val [V]   GCC Ala [A]   GAC Asp [D]   GGC Gly [G]   C
        GTA Val [V]   GCA Ala [A]   GAA Glu [E]   GGA Gly [G]   A
        GTG Val [V]   GCG Ala [A]   GAG Glu [E]   GGG Gly [G]   G
  -----------------------------------------------------------------

// Amino Acid Abbreviations

  Alanine          Ala   A                  |   A   Ala   Alanine
  Arginine         Arg   R   R-ginine       |   B   Asx   Asp or Asn
  Asparagine       Asn   N   asparagiNe     |   C   Cys   Cysteine
  Aspartic Acid    Asp   D   asparDic       |   D   Asp   Aspartic Acid
  Cysteine         Cys   C                  |   E   Glu   Glutamic Acid
  Glutamic Acid    Glu   E   gluE           |   F   Phe   Phenylalanine
  Glutamine        Gln   Q   Q-tamine       |   G   Gly   Glycine
  Glycine          Gly   G                  |   H   His   Histidine
  Histidine        His   H                  |   I   Ile   Isoleucine
  Isoleucine       Ile   I                  |   J   Xle   Leu or Ile
  Leucine          Leu   L                  |   K   Lys   Lysine
  Lysine           Lys   K   next to L      |   L   Leu   Leucine
  Methionine       Met   M                  |   M   Met   Methionine
  Phenylalanine    Phe   F   Fenylalanine   |   N   Asn   Asparagine
  Proline          Pro   P                  |   O   Pyl   Pyrrolysine
  Pyrrolysine      Pyl   O   pyrrOlysine    |   P   Pro   Proline
  Selenocysteine   Sec   U                  |   Q   Gln   Glutamine
  Serine           Ser   S                  |   R   Arg   Arginine
  Threonine        Thr   T                  |   S   Ser   Serine
  Tryptophan       Trp   W   tWyptophan     |   T   Thr   Threonine
  Tyrosine         Tyr   Y   tYrosine       |   U   Sec   Selenocysteine
  Valine           Val   V                  |   V   Val   Valine
  Asp or Asn       Asx   B                  |   W   Trp   Tryptophan
  Glu or Gln       Glx   Z                  |   X   Xxx   Undetermined
  Leu or Ile       Xle   J                  |   Y   Tyr   Tyrosine
  Undetermined     Xxx   X                  |   Z   Glx   Glu or Gln
  Termination      Ter   *                  |   *   Ter   Termination

  -----------------------------------------------------------------

// Nucleotide Abbreviations

            R   AG   puRine       |   H   ACT    not G
            Y   CT   pYrimidine   |   B   CGT    not A
            M   AC   aMino        |   V   ACG    not T
            K   GT   Keto         |   D   AGT    not C
            S   CG   Strong       |   N   ACGT   unkNown
            W   AT   Weak         |   X   ACGT   unknown

// Genetic Code Names

  1:  "Standard",
  2:  "Vertebrate Mitochondrial",
  3:  "Yeast Mitochondrial",
  4:  "Mold Mitochondrial; Protozoan Mitochondrial; Coelenterate Mitochondrial; Mycoplasma; Spiroplasma",
  5:  "Invertebrate Mitochondrial",
  6:  "Ciliate Nuclear; Dasycladacean Nuclear; Hexamita Nuclear",
  9:  "Echinoderm Mitochondrial; Flatworm Mitochondrial",
  10: "Euplotid Nuclear",
  11: "Bacterial, Archaeal and Plant Plastid",
  12: "Alternative Yeast Nuclear",
  13: "Ascidian Mitochondrial",
  14: "Alternative Flatworm Mitochondrial",
  15: "Blepharisma Macronuclear",
  16: "Chlorophycean Mitochondrial",
  21: "Trematode Mitochondrial",
  22: "Scenedesmus obliquus Mitochondrial",
  23: "Thraustochytrium Mitochondrial",
  24: "Rhabdopleuridae Mitochondrial",
  25: "Candidate Division SR1 and Gracilibacteria",
  26: "Pachysolen tannophilus Nuclear",
  27: "Karyorelict Nuclear",
  28: "Condylostoma Nuclear",
  29: "Mesodinium Nuclear",
  30: "Peritrich Nuclear",
  31: "Blastocrithidia Nuclear",
  32: "Balanophoraceae Plastid",
  33: "Cephalodiscidae Mitochondrial",

*/

/*
func init() {

	// raw genetic code tables - update these and rerun when new codes are added

	// Base
	//    1  TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG
	//    2  TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG
	//    3  TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG

	ncbieaaCode := map[int]string{
		1:  "FFLLSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		2:  "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSS**VVVVAAAADDEEGGGG",
		3:  "FFLLSSSSYY**CCWWTTTTPPPPHHQQRRRRIIMMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		4:  "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		5:  "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSSSSVVVVAAAADDEEGGGG",
		6:  "FFLLSSSSYYQQCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		9:  "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
		10: "FFLLSSSSYY**CCCWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		11: "FFLLSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		12: "FFLLSSSSYY**CC*WLLLSPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		13: "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNKKSSGGVVVVAAAADDEEGGGG",
		14: "FFLLSSSSYYY*CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
		15: "FFLLSSSSYY*QCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		16: "FFLLSSSSYY*LCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		21: "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIMMTTTTNNNKSSSSVVVVAAAADDEEGGGG",
		22: "FFLLSS*SYY*LCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		23: "FF*LSSSSYY**CC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		24: "FFLLSSSSYY**CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSSKVVVVAAAADDEEGGGG",
		25: "FFLLSSSSYY**CCGWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		26: "FFLLSSSSYY**CC*WLLLAPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		27: "FFLLSSSSYYQQCCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		28: "FFLLSSSSYYQQCCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		29: "FFLLSSSSYYYYCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		30: "FFLLSSSSYYEECC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		31: "FFLLSSSSYYEECCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		32: "FFLLSSSSYY*WCC*WLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSRRVVVVAAAADDEEGGGG",
		33: "FFLLSSSSYYY*CCWWLLLLPPPPHHQQRRRRIIIMTTTTNNKKSSSKVVVVAAAADDEEGGGG",
	}

	// Base
	//    1  TTTTTTTTTTTTTTTTCCCCCCCCCCCCCCCCAAAAAAAAAAAAAAAAGGGGGGGGGGGGGGGG
	//    2  TTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGGTTTTCCCCAAAAGGGG
	//    3  TCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAGTCAG

	sncbieaaCode := map[int]string{
		1:  "---M------**--*----M---------------M----------------------------",
		2:  "----------**--------------------MMMM----------**---M------------",
		3:  "----------**----------------------MM---------------M------------",
		4:  "--MM------**-------M------------MMMM---------------M------------",
		5:  "---M------**--------------------MMMM---------------M------------",
		6:  "--------------*--------------------M----------------------------",
		9:  "----------**-----------------------M---------------M------------",
		10: "----------**-----------------------M----------------------------",
		11: "---M------**--*----M------------MMMM---------------M------------",
		12: "----------**--*----M---------------M----------------------------",
		13: "---M------**----------------------MM---------------M------------",
		14: "-----------*-----------------------M----------------------------",
		15: "----------*---*--------------------M----------------------------",
		16: "----------*---*--------------------M----------------------------",
		21: "----------**-----------------------M---------------M------------",
		22: "------*---*---*--------------------M----------------------------",
		23: "--*-------**--*-----------------M--M---------------M------------",
		24: "---M------**-------M---------------M---------------M------------",
		25: "---M------**-----------------------M---------------M------------",
		26: "----------**--*----M---------------M----------------------------",
		27: "--------------*--------------------M----------------------------",
		28: "----------**--*--------------------M----------------------------",
		29: "--------------*--------------------M----------------------------",
		30: "--------------*--------------------M----------------------------",
		31: "----------**-----------------------M----------------------------",
		32: "---M------*---*----M------------MMMM---------------M------------",
		33: "---M-------*-------M---------------M---------------M------------",
	}

	// translation table structure
	type TransTable struct {
		aminoAcid [4096]int
		orfStart  [4096]int
		orfStop   [4096]int
	}

	// create translation table for given genetic code
	newTransTable := func(genCode int) *TransTable {

		tbl := new(TransTable)
		if tbl == nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to allocate translation table for genetic code %d\n", genCode)
			os.Exit(1)
		}

		// genetic code number corrections
		if genCode == 7 {
			genCode = 4
		} else if genCode == 8 {
			genCode = 1
		} else if genCode == 0 {
			genCode = 1
		}

		// return if unable to find ncbieaa and sncbieaa strings
		ncbieaa, ok := ncbieaaCode[genCode]
		if !ok {
			fmt.Fprintf(os.Stderr, "\nERROR: Genetic code %d does not exist\n", genCode)
			os.Exit(1)
		}
		sncbieaa, ok := sncbieaaCode[genCode]
		if !ok {
			fmt.Fprintf(os.Stderr, "\nERROR: Genetic code %d does not exist\n", genCode)
			os.Exit(1)
		}

		// also check length of ncbieaa and sncbieaa strings
		if len(ncbieaa) != 64 || len(sncbieaa) != 64 {
			fmt.Fprintf(os.Stderr, "\nERROR: Genetic code %d length mismatch\n", genCode)
			os.Exit(1)
		}

		// ambiguous codons map to unknown amino acid or not start
		for i := 0; i < 4096; i++ {
			tbl.aminoAcid[i] = int('X')
			tbl.orfStart[i] = int('-')
			tbl.orfStop[i] = int('-')
		}

		var expansions = [4]int{Base_A, Base_C, Base_G, Base_T}
		// T = 0, C = 1, A = 2, G = 3
		var codonIdx = [9]int{0, 2, 1, 0, 3, 0, 0, 0, 0}

		// lookup amino acid for each codon in genetic code table
		for i, st := Base_gap, 0; i <= Base_N; i++ {
			for j := Base_gap; j <= Base_N; j++ {
				for k := Base_gap; k <= Base_N; k++ {

					aa := 0
					orf := 0
					go_on := true

					// expand ambiguous IJK nucleotide symbols into component bases XYZ
					for p := 0; p < 4 && go_on; p++ {
						x := expansions[p]
						if (x & i) != 0 {
							for q := 0; q < 4 && go_on; q++ {
								y := expansions[q]
								if (y & j) != 0 {
									for r := 0; r < 4 && go_on; r++ {
										z := expansions[r]
										if (z & k) != 0 {

											// calculate offset in genetic code string

											// the T = 0, C = 1, A = 2, G = 3 order is
											// necessary because the genetic code
											// strings are presented in TCAG order
											cd := 16*codonIdx[x] + 4*codonIdx[y] + codonIdx[z]

											// lookup amino acid for codon XYZ
											ch := int(ncbieaa[cd])
											if aa == 0 {
												aa = ch
											} else if aa != ch {
												// allow Asx (Asp or Asn) and Glx (Glu or Gln)
												if (aa == 'B' || aa == 'D' || aa == 'N') && (ch == 'D' || ch == 'N') {
													aa = 'B'
												} else if (aa == 'Z' || aa == 'E' || aa == 'Q') && (ch == 'E' || ch == 'Q') {
													aa = 'Z'
												} else if (aa == 'J' || aa == 'I' || aa == 'L') && (ch == 'I' || ch == 'L') {
													aa = 'J'
												} else {
													aa = 'X'
												}
											}

											// lookup translation start flag
											ch = int(sncbieaa[cd])
											if orf == 0 {
												orf = ch
											} else if orf != ch {
												orf = 'X'
											}

											// drop out of loop as soon as answer is known
											if aa == 'X' && orf == 'X' {
												go_on = false
											}
										}
									}
								}
							}
						}
					}

					// assign amino acid
					if aa != 0 {
						tbl.aminoAcid[st] = aa
					}
					// assign orf start/stop
					if orf == '*' {
						tbl.orfStop[st] = orf
					} else if orf != 0 && orf != '-' && orf != 'X' {
						tbl.orfStart[st] = orf
					}

					st++
				}
			}
		}

		return tbl
	}

	// generate aminoAcidMaps, orfStartMaps, and orfStopMaps for source code

	// sort genetic code keys in numerical order
	var keys []int
	for ky := range sncbieaaCode {
		keys = append(keys, ky)
	}
	sort.Slice(keys, func(i, j int) bool { return keys[i] < keys[j] })

	tbl := newTransTable(1)
	if tbl == nil {
		fmt.Fprintf(os.Stderr, "newTransTable failed\n")
		return
	}

	fmt.Fprintf(os.Stdout, "// standard genetic code [1] shows only non-X residues,\n")
	fmt.Fprintf(os.Stdout, "// remaining tables show differences from the standard\n\n")
	fmt.Fprintf(os.Stdout, "var aminoAcidMaps = map[int]map[int]int{\n")

	fmt.Fprintf(os.Stdout, "\t1: {\n")
	for i := 0; i < 4096; i++ {
		aa := tbl.aminoAcid[i]
		if aa != 0 && aa != 'X' {
			fmt.Fprintf(os.Stdout, "\t\t%d: '%c', // %s\n", i, aa, GetCodonFromState(i))
		}
	}
	fmt.Fprintf(os.Stdout, "\t},\n")

	for _, id := range keys {

		// skip standard code
		if id == 1 {
			continue
		}

		tb := newTransTable(id)
		if tb == nil {
			fmt.Fprintf(os.Stderr, "newTransTable failed for code %d, skipping\n", id)
			continue
		}

		fmt.Fprintf(os.Stdout, "\t%d: {\n", id)
		for i := 0; i < 4096; i++ {
			aa := tbl.aminoAcid[i]
			nw := tb.aminoAcid[i]
			if aa == nw {
				continue
			}
			// allow 'X' to override specific amino acid in standard code
			if aa != 0 {
				fmt.Fprintf(os.Stdout, "\t\t%d: '%c', // %s\n", i, nw, GetCodonFromState(i))
			}
		}
		fmt.Fprintf(os.Stdout, "\t},\n")
	}

	fmt.Fprintf(os.Stdout, "}\n\n")

	// continue with start and stop tables, but all are instantiated

	fmt.Fprintf(os.Stdout, "// start and stop tables show all relevant entries for each genetic code\n\n")
	fmt.Fprintf(os.Stdout, "var orfStartMaps = map[int]map[int]int{\n")

	for _, id := range keys {

		tb := newTransTable(id)
		if tb == nil {
			fmt.Fprintf(os.Stderr, "newTransTable failed for code %d, skipping\n", id)
			continue
		}

		fmt.Fprintf(os.Stdout, "\t%d: {\n", id)
		for i := 0; i < 4096; i++ {
			aa := tb.orfStart[i]
			if aa != 0 && aa != '*' && aa != '-' && aa != 'X' {
				fmt.Fprintf(os.Stdout, "\t\t%d: '%c', // %s\n", i, aa, GetCodonFromState(i))
			}
		}
		fmt.Fprintf(os.Stdout, "\t},\n")
	}

	fmt.Fprintf(os.Stdout, "}\n\n")

	fmt.Fprintf(os.Stdout, "var orfStopMaps = map[int]map[int]int{\n")

	for _, id := range keys {

		tb := newTransTable(id)
		if tb == nil {
			fmt.Fprintf(os.Stderr, "newTransTable failed for code %d, skipping\n", id)
			continue
		}

		fmt.Fprintf(os.Stdout, "\t%d: {\n", id)
		for i := 0; i < 4096; i++ {
			aa := tb.orfStop[i]
			if aa == '*' {
				fmt.Fprintf(os.Stdout, "\t\t%d: '%c', // %s\n", i, aa, GetCodonFromState(i))
			}
		}
		fmt.Fprintf(os.Stdout, "\t},\n")
	}

	fmt.Fprintf(os.Stdout, "}\n\n")

	// generate static string of standard genetic code array for performance

	fmt.Fprintf(os.Stdout, "// standard genetic code array for performance\n\n")
	fmt.Fprintf(os.Stdout, "const stdGenCode = \"\" +\n")

	for i := 0; i < 4096; i += 64 {
		fmt.Fprintf(os.Stdout, "\t\"")
		for j := 0; j < 64; j++ {
			aa := tbl.aminoAcid[i+j]
			fmt.Fprintf(os.Stdout, "%c", aa)
		}
		fmt.Fprintf(os.Stdout, "\"")
		if i < 4032 {
			fmt.Fprintf(os.Stdout, " +")
		}
		fmt.Fprintf(os.Stdout, "\n")
	}

	fmt.Fprintf(os.Stdout, "\n")
}
*/

/*
func init() {

	// print codon to state table

	upperToBase := "-ACMGRSVTWYHKDBN"

	for _, ch1 := range upperToBase {
		for  _, ch2 := range upperToBase {
			for  _, ch3 := range upperToBase {
				st := SetCodonState(int(ch1), int(ch2), int(ch3))
				fmt.Fprintf(os.Stdout, "%c%c%c\t%d\n", ch1, ch2, ch3, st)
			}
		}
	}
}
*/

// standard genetic code [1] shows only non-X residues,
// remaining tables show differences from the standard

var aminoAcidMaps = map[int]map[int]int{
	1: {
		273:  'K', // AAA
		274:  'N', // AAC
		276:  'K', // AAG
		277:  'K', // AAR
		280:  'N', // AAT
		282:  'N', // AAY
		289:  'T', // ACA
		290:  'T', // ACC
		291:  'T', // ACM
		292:  'T', // ACG
		293:  'T', // ACR
		294:  'T', // ACS
		295:  'T', // ACV
		296:  'T', // ACT
		297:  'T', // ACW
		298:  'T', // ACY
		299:  'T', // ACH
		300:  'T', // ACK
		301:  'T', // ACD
		302:  'T', // ACB
		303:  'T', // ACN
		321:  'R', // AGA
		322:  'S', // AGC
		324:  'R', // AGG
		325:  'R', // AGR
		328:  'S', // AGT
		330:  'S', // AGY
		385:  'I', // ATA
		386:  'I', // ATC
		387:  'I', // ATM
		388:  'M', // ATG
		392:  'I', // ATT
		393:  'I', // ATW
		394:  'I', // ATY
		395:  'I', // ATH
		529:  'Q', // CAA
		530:  'H', // CAC
		532:  'Q', // CAG
		533:  'Q', // CAR
		536:  'H', // CAT
		538:  'H', // CAY
		545:  'P', // CCA
		546:  'P', // CCC
		547:  'P', // CCM
		548:  'P', // CCG
		549:  'P', // CCR
		550:  'P', // CCS
		551:  'P', // CCV
		552:  'P', // CCT
		553:  'P', // CCW
		554:  'P', // CCY
		555:  'P', // CCH
		556:  'P', // CCK
		557:  'P', // CCD
		558:  'P', // CCB
		559:  'P', // CCN
		577:  'R', // CGA
		578:  'R', // CGC
		579:  'R', // CGM
		580:  'R', // CGG
		581:  'R', // CGR
		582:  'R', // CGS
		583:  'R', // CGV
		584:  'R', // CGT
		585:  'R', // CGW
		586:  'R', // CGY
		587:  'R', // CGH
		588:  'R', // CGK
		589:  'R', // CGD
		590:  'R', // CGB
		591:  'R', // CGN
		641:  'L', // CTA
		642:  'L', // CTC
		643:  'L', // CTM
		644:  'L', // CTG
		645:  'L', // CTR
		646:  'L', // CTS
		647:  'L', // CTV
		648:  'L', // CTT
		649:  'L', // CTW
		650:  'L', // CTY
		651:  'L', // CTH
		652:  'L', // CTK
		653:  'L', // CTD
		654:  'L', // CTB
		655:  'L', // CTN
		833:  'R', // MGA
		836:  'R', // MGG
		837:  'R', // MGR
		897:  'J', // MTA
		898:  'J', // MTC
		899:  'J', // MTM
		904:  'J', // MTT
		905:  'J', // MTW
		906:  'J', // MTY
		907:  'J', // MTH
		1041: 'E', // GAA
		1042: 'D', // GAC
		1044: 'E', // GAG
		1045: 'E', // GAR
		1048: 'D', // GAT
		1050: 'D', // GAY
		1057: 'A', // GCA
		1058: 'A', // GCC
		1059: 'A', // GCM
		1060: 'A', // GCG
		1061: 'A', // GCR
		1062: 'A', // GCS
		1063: 'A', // GCV
		1064: 'A', // GCT
		1065: 'A', // GCW
		1066: 'A', // GCY
		1067: 'A', // GCH
		1068: 'A', // GCK
		1069: 'A', // GCD
		1070: 'A', // GCB
		1071: 'A', // GCN
		1089: 'G', // GGA
		1090: 'G', // GGC
		1091: 'G', // GGM
		1092: 'G', // GGG
		1093: 'G', // GGR
		1094: 'G', // GGS
		1095: 'G', // GGV
		1096: 'G', // GGT
		1097: 'G', // GGW
		1098: 'G', // GGY
		1099: 'G', // GGH
		1100: 'G', // GGK
		1101: 'G', // GGD
		1102: 'G', // GGB
		1103: 'G', // GGN
		1153: 'V', // GTA
		1154: 'V', // GTC
		1155: 'V', // GTM
		1156: 'V', // GTG
		1157: 'V', // GTR
		1158: 'V', // GTS
		1159: 'V', // GTV
		1160: 'V', // GTT
		1161: 'V', // GTW
		1162: 'V', // GTY
		1163: 'V', // GTH
		1164: 'V', // GTK
		1165: 'V', // GTD
		1166: 'V', // GTB
		1167: 'V', // GTN
		1298: 'B', // RAC
		1304: 'B', // RAT
		1306: 'B', // RAY
		1553: 'Z', // SAA
		1556: 'Z', // SAG
		1557: 'Z', // SAR
		2065: '*', // TAA
		2066: 'Y', // TAC
		2068: '*', // TAG
		2069: '*', // TAR
		2072: 'Y', // TAT
		2074: 'Y', // TAY
		2081: 'S', // TCA
		2082: 'S', // TCC
		2083: 'S', // TCM
		2084: 'S', // TCG
		2085: 'S', // TCR
		2086: 'S', // TCS
		2087: 'S', // TCV
		2088: 'S', // TCT
		2089: 'S', // TCW
		2090: 'S', // TCY
		2091: 'S', // TCH
		2092: 'S', // TCK
		2093: 'S', // TCD
		2094: 'S', // TCB
		2095: 'S', // TCN
		2113: '*', // TGA
		2114: 'C', // TGC
		2116: 'W', // TGG
		2120: 'C', // TGT
		2122: 'C', // TGY
		2129: '*', // TRA
		2177: 'L', // TTA
		2178: 'F', // TTC
		2180: 'L', // TTG
		2181: 'L', // TTR
		2184: 'F', // TTT
		2186: 'F', // TTY
		2433: 'J', // WTA
		2689: 'L', // YTA
		2692: 'L', // YTG
		2693: 'L', // YTR
		2945: 'J', // HTA
	},
	2: {
		321:  '*', // AGA
		324:  '*', // AGG
		325:  '*', // AGR
		385:  'M', // ATA
		387:  'X', // ATM
		389:  'M', // ATR
		393:  'X', // ATW
		395:  'X', // ATH
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		897:  'X', // MTA
		899:  'X', // MTM
		905:  'X', // MTW
		907:  'X', // MTH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2433: 'X', // WTA
		2945: 'X', // HTA
	},
	3: {
		385:  'M', // ATA
		387:  'X', // ATM
		389:  'M', // ATR
		393:  'X', // ATW
		395:  'X', // ATH
		641:  'T', // CTA
		642:  'T', // CTC
		643:  'T', // CTM
		644:  'T', // CTG
		645:  'T', // CTR
		646:  'T', // CTS
		647:  'T', // CTV
		648:  'T', // CTT
		649:  'T', // CTW
		650:  'T', // CTY
		651:  'T', // CTH
		652:  'T', // CTK
		653:  'T', // CTD
		654:  'T', // CTB
		655:  'T', // CTN
		897:  'X', // MTA
		898:  'X', // MTC
		899:  'X', // MTM
		904:  'X', // MTT
		905:  'X', // MTW
		906:  'X', // MTY
		907:  'X', // MTH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2433: 'X', // WTA
		2689: 'X', // YTA
		2692: 'X', // YTG
		2693: 'X', // YTR
		2945: 'X', // HTA
	},
	4: {
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
	},
	5: {
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'S', // AGG
		325:  'S', // AGR
		326:  'S', // AGS
		327:  'S', // AGV
		329:  'S', // AGW
		331:  'S', // AGH
		332:  'S', // AGK
		333:  'S', // AGD
		334:  'S', // AGB
		335:  'S', // AGN
		385:  'M', // ATA
		387:  'X', // ATM
		389:  'M', // ATR
		393:  'X', // ATW
		395:  'X', // ATH
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		897:  'X', // MTA
		899:  'X', // MTM
		905:  'X', // MTW
		907:  'X', // MTH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2433: 'X', // WTA
		2945: 'X', // HTA
	},
	6: {
		2065: 'Q', // TAA
		2068: 'Q', // TAG
		2069: 'Q', // TAR
		2129: 'X', // TRA
		2577: 'Q', // YAA
		2580: 'Q', // YAG
		2581: 'Q', // YAR
		3089: 'Z', // KAA
		3092: 'Z', // KAG
		3093: 'Z', // KAR
		3601: 'Z', // BAA
		3604: 'Z', // BAG
		3605: 'Z', // BAR
	},
	9: {
		273:  'N', // AAA
		275:  'N', // AAM
		277:  'X', // AAR
		281:  'N', // AAW
		283:  'N', // AAH
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'S', // AGG
		325:  'S', // AGR
		326:  'S', // AGS
		327:  'S', // AGV
		329:  'S', // AGW
		331:  'S', // AGH
		332:  'S', // AGK
		333:  'S', // AGD
		334:  'S', // AGB
		335:  'S', // AGN
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
	},
	10: {
		2113: 'C', // TGA
		2115: 'C', // TGM
		2121: 'C', // TGW
		2123: 'C', // TGH
		2129: 'X', // TRA
	},
	11: {},
	12: {
		644:  'S', // CTG
		645:  'X', // CTR
		646:  'X', // CTS
		647:  'X', // CTV
		652:  'X', // CTK
		653:  'X', // CTD
		654:  'X', // CTB
		655:  'X', // CTN
		2692: 'X', // YTG
		2693: 'X', // YTR
	},
	13: {
		321:  'G', // AGA
		324:  'G', // AGG
		325:  'G', // AGR
		385:  'M', // ATA
		387:  'X', // ATM
		389:  'M', // ATR
		393:  'X', // ATW
		395:  'X', // ATH
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		897:  'X', // MTA
		899:  'X', // MTM
		905:  'X', // MTW
		907:  'X', // MTH
		1345: 'G', // RGA
		1348: 'G', // RGG
		1349: 'G', // RGR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2433: 'X', // WTA
		2945: 'X', // HTA
	},
	14: {
		273:  'N', // AAA
		275:  'N', // AAM
		277:  'X', // AAR
		281:  'N', // AAW
		283:  'N', // AAH
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'S', // AGG
		325:  'S', // AGR
		326:  'S', // AGS
		327:  'S', // AGV
		329:  'S', // AGW
		331:  'S', // AGH
		332:  'S', // AGK
		333:  'S', // AGD
		334:  'S', // AGB
		335:  'S', // AGN
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		2065: 'Y', // TAA
		2067: 'Y', // TAM
		2069: 'X', // TAR
		2073: 'Y', // TAW
		2075: 'Y', // TAH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
	},
	15: {
		2068: 'Q', // TAG
		2069: 'X', // TAR
		2580: 'Q', // YAG
		3092: 'Z', // KAG
		3604: 'Z', // BAG
	},
	16: {
		2068: 'L', // TAG
		2069: 'X', // TAR
		2196: 'L', // TWG
	},
	21: {
		273:  'N', // AAA
		275:  'N', // AAM
		277:  'X', // AAR
		281:  'N', // AAW
		283:  'N', // AAH
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'S', // AGG
		325:  'S', // AGR
		326:  'S', // AGS
		327:  'S', // AGV
		329:  'S', // AGW
		331:  'S', // AGH
		332:  'S', // AGK
		333:  'S', // AGD
		334:  'S', // AGB
		335:  'S', // AGN
		385:  'M', // ATA
		387:  'X', // ATM
		389:  'M', // ATR
		393:  'X', // ATW
		395:  'X', // ATH
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		897:  'X', // MTA
		899:  'X', // MTM
		905:  'X', // MTW
		907:  'X', // MTH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2433: 'X', // WTA
		2945: 'X', // HTA
	},
	22: {
		2068: 'L', // TAG
		2069: 'X', // TAR
		2081: '*', // TCA
		2083: 'X', // TCM
		2085: 'X', // TCR
		2087: 'X', // TCV
		2089: 'X', // TCW
		2091: 'X', // TCH
		2093: 'X', // TCD
		2095: 'X', // TCN
		2097: '*', // TMA
		2145: '*', // TSA
		2161: '*', // TVA
		2196: 'L', // TWG
	},
	23: {
		2177: '*', // TTA
		2181: 'X', // TTR
		2193: '*', // TWA
		2241: '*', // TKA
		2257: '*', // TDA
		2433: 'X', // WTA
		2689: 'X', // YTA
		2693: 'X', // YTR
		2945: 'X', // HTA
	},
	24: {
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'K', // AGG
		325:  'X', // AGR
		329:  'S', // AGW
		331:  'S', // AGH
		340:  'K', // ARG
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
	},
	25: {
		2113: 'G', // TGA
		2129: 'X', // TRA
		3137: 'G', // KGA
	},
	26: {
		644:  'A', // CTG
		645:  'X', // CTR
		646:  'X', // CTS
		647:  'X', // CTV
		652:  'X', // CTK
		653:  'X', // CTD
		654:  'X', // CTB
		655:  'X', // CTN
		2692: 'X', // YTG
		2693: 'X', // YTR
	},
	27: {
		2065: 'Q', // TAA
		2068: 'Q', // TAG
		2069: 'Q', // TAR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2577: 'Q', // YAA
		2580: 'Q', // YAG
		2581: 'Q', // YAR
		3089: 'Z', // KAA
		3092: 'Z', // KAG
		3093: 'Z', // KAR
		3601: 'Z', // BAA
		3604: 'Z', // BAG
		3605: 'Z', // BAR
	},
	28: {
		2065: 'Q', // TAA
		2068: 'Q', // TAG
		2069: 'Q', // TAR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2577: 'Q', // YAA
		2580: 'Q', // YAG
		2581: 'Q', // YAR
		3089: 'Z', // KAA
		3092: 'Z', // KAG
		3093: 'Z', // KAR
		3601: 'Z', // BAA
		3604: 'Z', // BAG
		3605: 'Z', // BAR
	},
	29: {
		2065: 'Y', // TAA
		2067: 'Y', // TAM
		2068: 'Y', // TAG
		2069: 'Y', // TAR
		2070: 'Y', // TAS
		2071: 'Y', // TAV
		2073: 'Y', // TAW
		2075: 'Y', // TAH
		2076: 'Y', // TAK
		2077: 'Y', // TAD
		2078: 'Y', // TAB
		2079: 'Y', // TAN
		2129: 'X', // TRA
	},
	30: {
		2065: 'E', // TAA
		2068: 'E', // TAG
		2069: 'E', // TAR
		2129: 'X', // TRA
		2577: 'Z', // YAA
		2580: 'Z', // YAG
		2581: 'Z', // YAR
		3089: 'E', // KAA
		3092: 'E', // KAG
		3093: 'E', // KAR
		3601: 'Z', // BAA
		3604: 'Z', // BAG
		3605: 'Z', // BAR
	},
	31: {
		2065: 'E', // TAA
		2068: 'E', // TAG
		2069: 'E', // TAR
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
		2577: 'Z', // YAA
		2580: 'Z', // YAG
		2581: 'Z', // YAR
		3089: 'E', // KAA
		3092: 'E', // KAG
		3093: 'E', // KAR
		3601: 'Z', // BAA
		3604: 'Z', // BAG
		3605: 'Z', // BAR
	},
	32: {
		2068: 'W', // TAG
		2069: 'X', // TAR
		2132: 'W', // TRG
	},
	33: {
		321:  'S', // AGA
		323:  'S', // AGM
		324:  'K', // AGG
		325:  'X', // AGR
		329:  'S', // AGW
		331:  'S', // AGH
		340:  'K', // ARG
		833:  'X', // MGA
		836:  'X', // MGG
		837:  'X', // MGR
		2065: 'Y', // TAA
		2067: 'Y', // TAM
		2069: 'X', // TAR
		2073: 'Y', // TAW
		2075: 'Y', // TAH
		2113: 'W', // TGA
		2117: 'W', // TGR
		2129: 'X', // TRA
	},
}

// start and stop tables show all relevant entries for each genetic code

var orfStartMaps = map[int]map[int]int{
	1: {
		388:  'M', // ATG
		644:  'M', // CTG
		900:  'M', // MTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		2692: 'M', // YTG
		2948: 'M', // HTG
	},
	2: {
		385:  'M', // ATA
		386:  'M', // ATC
		387:  'M', // ATM
		388:  'M', // ATG
		389:  'M', // ATR
		390:  'M', // ATS
		391:  'M', // ATV
		392:  'M', // ATT
		393:  'M', // ATW
		394:  'M', // ATY
		395:  'M', // ATH
		396:  'M', // ATK
		397:  'M', // ATD
		398:  'M', // ATB
		399:  'M', // ATN
		1156: 'M', // GTG
		1412: 'M', // RTG
	},
	3: {
		385:  'M', // ATA
		388:  'M', // ATG
		389:  'M', // ATR
		1156: 'M', // GTG
		1412: 'M', // RTG
	},
	4: {
		385:  'M', // ATA
		386:  'M', // ATC
		387:  'M', // ATM
		388:  'M', // ATG
		389:  'M', // ATR
		390:  'M', // ATS
		391:  'M', // ATV
		392:  'M', // ATT
		393:  'M', // ATW
		394:  'M', // ATY
		395:  'M', // ATH
		396:  'M', // ATK
		397:  'M', // ATD
		398:  'M', // ATB
		399:  'M', // ATN
		644:  'M', // CTG
		900:  'M', // MTG
		1156: 'M', // GTG
		1412: 'M', // RTG
		1668: 'M', // STG
		1924: 'M', // VTG
		2177: 'M', // TTA
		2180: 'M', // TTG
		2181: 'M', // TTR
		2433: 'M', // WTA
		2436: 'M', // WTG
		2437: 'M', // WTR
		2692: 'M', // YTG
		2948: 'M', // HTG
		3204: 'M', // KTG
		3460: 'M', // DTG
		3716: 'M', // BTG
		3972: 'M', // NTG
	},
	5: {
		385:  'M', // ATA
		386:  'M', // ATC
		387:  'M', // ATM
		388:  'M', // ATG
		389:  'M', // ATR
		390:  'M', // ATS
		391:  'M', // ATV
		392:  'M', // ATT
		393:  'M', // ATW
		394:  'M', // ATY
		395:  'M', // ATH
		396:  'M', // ATK
		397:  'M', // ATD
		398:  'M', // ATB
		399:  'M', // ATN
		1156: 'M', // GTG
		1412: 'M', // RTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		3204: 'M', // KTG
		3460: 'M', // DTG
	},
	6: {
		388: 'M', // ATG
	},
	9: {
		388:  'M', // ATG
		1156: 'M', // GTG
		1412: 'M', // RTG
	},
	10: {
		388: 'M', // ATG
	},
	11: {
		385:  'M', // ATA
		386:  'M', // ATC
		387:  'M', // ATM
		388:  'M', // ATG
		389:  'M', // ATR
		390:  'M', // ATS
		391:  'M', // ATV
		392:  'M', // ATT
		393:  'M', // ATW
		394:  'M', // ATY
		395:  'M', // ATH
		396:  'M', // ATK
		397:  'M', // ATD
		398:  'M', // ATB
		399:  'M', // ATN
		644:  'M', // CTG
		900:  'M', // MTG
		1156: 'M', // GTG
		1412: 'M', // RTG
		1668: 'M', // STG
		1924: 'M', // VTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		2692: 'M', // YTG
		2948: 'M', // HTG
		3204: 'M', // KTG
		3460: 'M', // DTG
		3716: 'M', // BTG
		3972: 'M', // NTG
	},
	12: {
		388: 'M', // ATG
		644: 'M', // CTG
		900: 'M', // MTG
	},
	13: {
		385:  'M', // ATA
		388:  'M', // ATG
		389:  'M', // ATR
		1156: 'M', // GTG
		1412: 'M', // RTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		3204: 'M', // KTG
		3460: 'M', // DTG
	},
	14: {
		388: 'M', // ATG
	},
	15: {
		388: 'M', // ATG
	},
	16: {
		388: 'M', // ATG
	},
	21: {
		388:  'M', // ATG
		1156: 'M', // GTG
		1412: 'M', // RTG
	},
	22: {
		388: 'M', // ATG
	},
	23: {
		388:  'M', // ATG
		392:  'M', // ATT
		396:  'M', // ATK
		1156: 'M', // GTG
		1412: 'M', // RTG
	},
	24: {
		388:  'M', // ATG
		644:  'M', // CTG
		900:  'M', // MTG
		1156: 'M', // GTG
		1412: 'M', // RTG
		1668: 'M', // STG
		1924: 'M', // VTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		2692: 'M', // YTG
		2948: 'M', // HTG
		3204: 'M', // KTG
		3460: 'M', // DTG
		3716: 'M', // BTG
		3972: 'M', // NTG
	},
	25: {
		388:  'M', // ATG
		1156: 'M', // GTG
		1412: 'M', // RTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		3204: 'M', // KTG
		3460: 'M', // DTG
	},
	26: {
		388: 'M', // ATG
		644: 'M', // CTG
		900: 'M', // MTG
	},
	27: {
		388: 'M', // ATG
	},
	28: {
		388: 'M', // ATG
	},
	29: {
		388: 'M', // ATG
	},
	30: {
		388: 'M', // ATG
	},
	31: {
		388: 'M', // ATG
	},
	32: {
		385:  'M', // ATA
		386:  'M', // ATC
		387:  'M', // ATM
		388:  'M', // ATG
		389:  'M', // ATR
		390:  'M', // ATS
		391:  'M', // ATV
		392:  'M', // ATT
		393:  'M', // ATW
		394:  'M', // ATY
		395:  'M', // ATH
		396:  'M', // ATK
		397:  'M', // ATD
		398:  'M', // ATB
		399:  'M', // ATN
		644:  'M', // CTG
		900:  'M', // MTG
		1156: 'M', // GTG
		1412: 'M', // RTG
		1668: 'M', // STG
		1924: 'M', // VTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		2692: 'M', // YTG
		2948: 'M', // HTG
		3204: 'M', // KTG
		3460: 'M', // DTG
		3716: 'M', // BTG
		3972: 'M', // NTG
	},
	33: {
		388:  'M', // ATG
		644:  'M', // CTG
		900:  'M', // MTG
		1156: 'M', // GTG
		1412: 'M', // RTG
		1668: 'M', // STG
		1924: 'M', // VTG
		2180: 'M', // TTG
		2436: 'M', // WTG
		2692: 'M', // YTG
		2948: 'M', // HTG
		3204: 'M', // KTG
		3460: 'M', // DTG
		3716: 'M', // BTG
		3972: 'M', // NTG
	},
}

var orfStopMaps = map[int]map[int]int{
	1: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
	},
	2: {
		321:  '*', // AGA
		324:  '*', // AGG
		325:  '*', // AGR
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	3: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	4: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	5: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	6: {
		2113: '*', // TGA
	},
	9: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	10: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	11: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
	},
	12: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
	},
	13: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	14: {
		2068: '*', // TAG
	},
	15: {
		2065: '*', // TAA
		2113: '*', // TGA
		2129: '*', // TRA
	},
	16: {
		2065: '*', // TAA
		2113: '*', // TGA
		2129: '*', // TRA
	},
	21: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	22: {
		2065: '*', // TAA
		2081: '*', // TCA
		2097: '*', // TMA
		2113: '*', // TGA
		2129: '*', // TRA
		2145: '*', // TSA
		2161: '*', // TVA
	},
	23: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
		2177: '*', // TTA
		2193: '*', // TWA
		2241: '*', // TKA
		2257: '*', // TDA
	},
	24: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	25: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	26: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
	},
	27: {
		2113: '*', // TGA
	},
	28: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
		2113: '*', // TGA
		2129: '*', // TRA
	},
	29: {
		2113: '*', // TGA
	},
	30: {
		2113: '*', // TGA
	},
	31: {
		2065: '*', // TAA
		2068: '*', // TAG
		2069: '*', // TAR
	},
	32: {
		2065: '*', // TAA
		2113: '*', // TGA
		2129: '*', // TRA
	},
	33: {
		2068: '*', // TAG
	},
}

// standard genetic code array for performance

const stdGenCode = "" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXKNXKKXXNXNXXXXXXTTTTTTTTTTTTTTTXXXXXXXXXXXXXXXX" +
	"XRSXRRXXSXSXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XIIIMXXXIIIIXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXQHXQQXXHXHXXXXXXPPPPPPPPPPPPPPPXXXXXXXXXXXXXXXX" +
	"XRRRRRRRRRRRRRRRXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XLLLLLLLLLLLLLLLXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XRXXRRXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XJJJXXXXJJJJXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXEDXEEXXDXDXXXXXXAAAAAAAAAAAAAAAXXXXXXXXXXXXXXXX" +
	"XGGGGGGGGGGGGGGGXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XVVVVVVVVVVVVVVVXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXBXXXXXBXBXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXZXXZZXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXX*YX**XXYXYXXXXXXSSSSSSSSSSSSSSSXXXXXXXXXXXXXXXX" +
	"X*CXWXXXCXCXXXXXX*XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XLFXLLXXFXFXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XJXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XLXXLLXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XJXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" +
	"XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"

// translation finite state machine functions

func SetCodonState(ch1, ch2, ch3 int) int {

	if ch1 < 0 || ch1 > 255 || ch2 < 0 || ch2 > 255 || ch3 < 0 || ch3 > 255 {
		return 0
	}

	return 256*baseToIdx[ch1] + 16*baseToIdx[ch2] + baseToIdx[ch3]
}

func GetCodonFromState(state int) string {

	if state < 0 || state > 4095 {
		return ""
	}

	upperToBase := "-ACMGRSVTWYHKDBN"

	ch3 := upperToBase[state&15]
	state = state >> 4
	ch2 := upperToBase[state&15]
	state = state >> 4
	ch1 := upperToBase[state&15]

	str := string(ch1) + string(ch2) + string(ch3)

	return str
}

func NextCodonState(state, ch int) int {

	if state < 0 || state > 4095 {
		return 0
	}

	return nextState[state] + baseToIdx[ch]
}

func RevCompState(state int) int {

	if state < 0 || state > 4095 {
		return 0
	}

	return rvCpState[state]
}

// translation codon lookup functions

func GetCodonResidue(genCode, state int) int {

	if state < 0 || state > 4095 {
		return 'X'
	}

	// look for override in alternative genetic code
	gc, ok := aminoAcidMaps[genCode]
	if ok {
		aa, ok := gc[state]
		if ok {
			return aa
		}
	}

	if genCode != 1 {
		// now fetch residue from standard genetic code
		gc, ok = aminoAcidMaps[1]
		if ok {
			aa, ok := gc[state]
			if ok {
				return aa
			}
		}
	}

	// no entry in either map defaults to unknown residue X
	return 'X'
}

func GetStartResidue(genCode, state int) int {

	if state < 0 || state > 4095 {
		return '-'
	}

	gc, ok := orfStartMaps[genCode]
	if ok {
		aa, ok := gc[state]
		if ok {
			return aa
		}
	}

	return '-'
}

func GetStopResidue(genCode, state int) int {

	if state < 0 || state > 4095 {
		return '-'
	}

	gc, ok := orfStopMaps[genCode]
	if ok {
		aa, ok := gc[state]
		if ok {
			return aa
		}
	}

	return '-'
}

func IsOrfStart(genCode, state int) bool {

	if state < 0 || state > 4095 {
		return false
	}

	return GetStartResidue(genCode, state) == 'M'
}

func IsOrfStop(genCode, state int) bool {

	if state < 0 || state > 4095 {
		return false
	}

	return GetStopResidue(genCode, state) == '*'
}

func IsATGStart(genCode, state int) bool {

	const ATG_state = 388

	if state < 0 || state > 4095 {
		return false
	}

	return IsOrfStart(genCode, state) && state == ATG_state
}

func IsAltStart(genCode, state int) bool {

	const ATG_state = 388

	if state < 0 || state > 4095 {
		return false
	}

	return IsOrfStart(genCode, state) && state != ATG_state
}

// TranslateCdRegion converts a single coding region to a protein sequence
func TranslateCdRegion(seq string, genCode, frame int, include_stop, do_every_codon, remove_trailing_X, is_5prime_complete, is_3prime_complete bool) string {

	var buffer strings.Builder

	usable_size := len(seq) - frame
	mod := usable_size % 3
	length := usable_size / 3
	check_start := is_5prime_complete && frame == 0

	aa := 0
	state := 0
	// start_state := 0
	first_time := true

	// standard and bacterial code use built-in array for fast access
	fast := genCode == 1 || genCode == 11

	pos := frame

	// first codon has extra logic, process separately
	if length > 0 {

		// loop through first codon to initialize state
		for k := 0; k < 3; k++ {

			// internal version of NextCodonState without value reality check
			ch := int(seq[pos])
			state = nextState[state] + baseToIdx[ch]
			// state = NextCodonState(state, int(seq[pos]))
			pos++
		}

		// start_state = state

		// save first translated amino acid
		if check_start {

			aa = GetStartResidue(genCode, state)
			buffer.WriteByte(byte(aa))

		} else {

			aa = GetCodonResidue(genCode, state)
			buffer.WriteByte(byte(aa))
		}

		first_time = false
	}

	// start at offset 1, continue with rest of sequence
	for i := 1; i < length; i++ {

		// loop through next triplet codon
		for k := 0; k < 3; k++ {

			// internal version of NextCodonState without value reality check
			ch := int(seq[pos])
			state = nextState[state] + baseToIdx[ch]
			// state = NextCodonState(state, int(seq[pos]))
			pos++
		}

		// save translated amino acid
		if fast {

			// use high-efficiency built-in lookup table for standard code
			aa = int(stdGenCode[state])
			buffer.WriteByte(byte(aa))

		} else {

			aa = GetCodonResidue(genCode, state)
			buffer.WriteByte(byte(aa))
		}

		first_time = false
	}

	if mod > 0 {

		k := 0
		for ; k < mod; k++ {
			state = NextCodonState(state, int(seq[pos]))
			pos++
		}

		for ; k < 3; k++ {
			state = NextCodonState(state, int('N'))
		}

		if first_time {
			// start_state = state
		}

		// save translated amino acid
		ch := GetCodonResidue(genCode, state)

		if first_time && check_start {

			aa = GetStartResidue(genCode, state)
			buffer.WriteByte(byte(aa))

		} else if ch != 'X' {

			aa = GetCodonResidue(genCode, state)
			buffer.WriteByte(byte(aa))
		}
	}

	txt := buffer.String()

	// check for stop codon that normally encodes an amino acid
	if aa != '*' && include_stop && mod > 0 && len(txt) > 0 && is_3prime_complete {

		aa = GetStopResidue(genCode, state)
		if aa == '*' {
			rgt := len(txt) - 1
			txt = txt[:rgt] + "*"
		}
	}

	if do_every_codon {

		// read through all stop codons

	} else if include_stop {

		idx := strings.Index(txt, "*")
		if idx >= 0 && idx+1 < len(txt) {
			txt = txt[:idx+1]
		}

	} else {

		idx := strings.Index(txt, "*")
		if idx >= 0 {
			txt = txt[:idx]
		}
	}

	if remove_trailing_X {

		txt = strings.TrimRight(txt, "X")
	}

	return txt
}

// CdRegionToProtein reads all of stdin as sequence data
func CdRegionToProtein(inp io.Reader, args []string) {

	if inp == nil {
		return
	}

	genCode := 1
	frame := 0
	include_stop := false
	do_every_codon := false
	remove_trailing_X := false
	is_5prime_complete := true
	is_3prime_complete := true

	repeat := 1

	// skip past command name
	args = args[1:]

	for len(args) > 0 {

		switch args[0] {
		case "-code", "-gencode":
			genCode = GetNumericArg(args, "genetic code number", 0, 1, 30)
			args = args[2:]
		case "-frame":
			frame = GetNumericArg(args, "offset into coding sequence", 0, 1, 30)
			args = args[2:]
		case "-stop", "-stops":
			include_stop = true
			args = args[1:]
		case "-every", "-all":
			do_every_codon = true
			args = args[1:]
		case "-trim", "-trailing":
			remove_trailing_X = true
			args = args[1:]
		case "-part5", "-partial5", "-lt5":
			is_5prime_complete = false
			args = args[1:]
		case "-part3", "-partial3", "-gt3":
			is_3prime_complete = false
			args = args[1:]
		case "-repeat":
			repeat = GetNumericArg(args, "number of repetitions for testing", 1, 1, 100)
			args = args[2:]
		default:
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized option after -cds2prot command\n")
			os.Exit(1)
		}
	}

	txt := ReadAllIntoSequence(inp)

	for i := 0; i < repeat; i++ {

		// repeat multiple times for performance testing (undocumented)
		str := TranslateCdRegion(txt, genCode, frame, include_stop, do_every_codon, remove_trailing_X, is_5prime_complete, is_3prime_complete)

		os.Stdout.WriteString(str)
		if !strings.HasSuffix(str, "\n") {
			os.Stdout.WriteString("\n")
		}
	}
}

// MAIN FUNCTION

func main() {

	// skip past executable name
	args := os.Args[1:]

	if len(args) < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: No command-line arguments supplied to transmute\n")
		os.Exit(1)
	}

	// CONCURRENCY, CLEANUP, AND DEBUGGING FLAGS

	// do these first because -defcpu and -maxcpu can be sent from wrapper before other arguments

	ncpu := runtime.NumCPU()
	if ncpu < 1 {
		ncpu = 1
	}

	// wrapper can limit maximum number of processors to use (undocumented)
	maxProcs := ncpu
	defProcs := 0

	// concurrent performance tuning parameters, can be overridden by -proc and -cons
	numProcs := 0
	serverRatio := 4

	// number of servers usually calculated by -cons server ratio, but can be overridden by -serv
	NumServe = 0

	// number of channels usually equals number of servers, but can be overridden by -chan
	ChanDepth = 0

	// miscellaneous tuning parameters
	HeapSize = 16
	FarmSize = 64

	// garbage collector control can be set by environment variable or default value with -gogc 0
	goGc := 600

	// -flag sets -strict or -mixed cleanup flags from argument
	flgs := ""

	DeStop = true

	unicodePolicy := ""
	scriptPolicy := ""
	mathmlPolicy := ""

	// read data from file instead of stdin
	fileName := ""

	// debugging
	stts := false
	timr := false

	// profiling
	prfl := false

	inSwitch := true

	// get concurrency, cleanup, and debugging flags in any order
	for {

		inSwitch = true

		switch args[0] {
		// concurrency override arguments can be passed in by local wrapper script (undocumented)
		case "-maxcpu":
			maxProcs = GetNumericArg(args, "Maximum number of processors", 1, 1, ncpu)
			args = args[1:]
		case "-defcpu":
			defProcs = GetNumericArg(args, "Default number of processors", ncpu, 1, ncpu)
			args = args[1:]
		// performance tuning flags
		case "-proc":
			numProcs = GetNumericArg(args, "Number of processors", ncpu, 1, ncpu)
			args = args[1:]
		case "-cons":
			serverRatio = GetNumericArg(args, "Parser to processor ratio", 4, 1, 32)
			args = args[1:]
		case "-serv":
			NumServe = GetNumericArg(args, "Concurrent parser count", 0, 1, 128)
			args = args[1:]
		case "-chan":
			ChanDepth = GetNumericArg(args, "Communication channel depth", 0, ncpu, 128)
			args = args[1:]
		case "-heap":
			HeapSize = GetNumericArg(args, "Unshuffler heap size", 8, 8, 64)
			args = args[1:]
		case "-farm":
			FarmSize = GetNumericArg(args, "Node buffer length", 4, 4, 2048)
			args = args[1:]
		case "-gogc":
			goGc = GetNumericArg(args, "Garbage collection percentage", 0, 50, 1000)
			args = args[1:]

		// read data from file
		case "-input":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Input file name is missing\n")
				os.Exit(1)
			}
			fileName = args[1]
			// skip past first of two arguments
			args = args[1:]

		// data cleanup flags
		case "-compress", "-compressed":
			DoCompress = true
		case "-spaces", "-cleanup":
			DoCleanup = true
		case "-strict":
			DoStrict = true
			AllowEmbed = true
			ContentMods = true
		case "-mixed":
			DoMixed = true
			AllowEmbed = true
			ContentMods = true
		case "-accent":
			DeAccent = true
		case "-ascii":
			DoASCII = true

		// previously visible processing flags (undocumented)
		case "-stems", "-stem":
			DoStem = true
		case "-stops", "-stop":
			DeStop = false

		// allow setting of unicode, script, and mathml flags (undocumented)
		case "-unicode":
			unicodePolicy = GetStringArg(args, "Unicode argument")
			args = args[1:]
		case "-script":
			scriptPolicy = GetStringArg(args, "Script argument")
			args = args[1:]
		case "-mathml":
			mathmlPolicy = GetStringArg(args, "MathML argument")
			args = args[1:]

		case "-flag", "-flags":
			flgs = GetStringArg(args, "Flags argument")
			args = args[1:]

		// debugging flags
		case "-stats", "-stat":
			stts = true
		case "-timer":
			timr = true
		case "-profile":
			prfl = true

		default:
			// if not any of the controls, set flag to break out of for loop
			inSwitch = false
		}

		if !inSwitch {
			break
		}

		// skip past argument
		args = args[1:]

		if len(args) < 1 {
			break
		}
	}

	// -flag allows script to set -strict or -mixed (or -stems, or -stops) from argument
	switch flgs {
	case "strict":
		DoStrict = true
	case "mixed":
		DoMixed = true
	case "stems", "stem":
		DoStem = true
	case "stops", "stop":
		DeStop = false
	case "none", "default":
	default:
		if flgs != "" {
			fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized -flag value '%s'\n", flgs)
			os.Exit(1)
		}
	}

	UnicodeFix = ParseMarkup(unicodePolicy, "-unicode")
	ScriptFix = ParseMarkup(scriptPolicy, "-script")
	MathMLFix = ParseMarkup(mathmlPolicy, "-mathml")

	if UnicodeFix != NOMARKUP {
		DoUnicode = true
	}

	if ScriptFix != NOMARKUP {
		DoScript = true
	}

	if MathMLFix != NOMARKUP {
		DoMathML = true
	}

	// set dependent flags
	CountLines = DoMixed
	AllowEmbed = DoStrict || DoMixed
	ContentMods = AllowEmbed || DoCompress || DoUnicode || DoScript || DoMathML || DeAccent || DoASCII

	// reality checks on number of processors to use
	// performance degrades if capacity is above maximum number of partitions per second (context switching?)
	if numProcs == 0 {
		if defProcs > 0 {
			numProcs = defProcs
		} else {
			// best performance measurement with current code is obtained when 6 to 8 processors are assigned,
			// varying slightly among queries on PubmedArticle, gene DocumentSummary, and INSDSeq sequence records
			numProcs = 8
			if cpuid.CPU.ThreadsPerCore > 1 {
				cores := ncpu / cpuid.CPU.ThreadsPerCore
				if cores > 4 && cores < 8 {
					numProcs = cores
				}
			}
		}
	}
	if numProcs > ncpu {
		numProcs = ncpu
	}
	if numProcs > maxProcs {
		numProcs = maxProcs
	}

	// allow simultaneous threads for multiplexed goroutines
	runtime.GOMAXPROCS(numProcs)

	// adjust garbage collection target percentage
	if goGc >= 50 && goGc <= 1000 {
		debug.SetGCPercent(goGc)
	}

	// explicit -serv argument overrides -cons ratio
	if NumServe > 0 {
		serverRatio = NumServe / numProcs
		// if numServers / numProcs is not a whole number, do not print serverRatio in -stats
		if NumServe != numProcs*serverRatio {
			serverRatio = 0
		}
	} else {
		NumServe = numProcs * serverRatio
	}
	// server limits
	if NumServe > 128 {
		NumServe = 128
	} else if NumServe < 1 {
		NumServe = numProcs
	}

	// explicit -chan argument overrides default to number of servers
	if ChanDepth == 0 {
		ChanDepth = NumServe
	}

	// -stats prints number of CPUs and performance tuning values if no other arguments (undocumented)
	if stts && len(args) < 1 {

		fmt.Fprintf(os.Stderr, "Core %d\n", ncpu/cpuid.CPU.ThreadsPerCore)
		fmt.Fprintf(os.Stderr, "Thrd %d\n", ncpu)
		fmt.Fprintf(os.Stderr, "Sock %d\n", ncpu/cpuid.CPU.LogicalCores)
		fmt.Fprintf(os.Stderr, "Mmry %d\n", memory.TotalMemory()/(1024*1024*1024))

		fmt.Fprintf(os.Stderr, "Proc %d\n", numProcs)
		if serverRatio > 0 {
			fmt.Fprintf(os.Stderr, "Cons %d\n", serverRatio)
		}
		fmt.Fprintf(os.Stderr, "Serv %d\n", NumServe)
		fmt.Fprintf(os.Stderr, "Chan %d\n", ChanDepth)
		fmt.Fprintf(os.Stderr, "Heap %d\n", HeapSize)
		fmt.Fprintf(os.Stderr, "Farm %d\n", FarmSize)
		fmt.Fprintf(os.Stderr, "Gogc %d\n", goGc)

		fi, err := os.Stdin.Stat()
		if err == nil {
			mode := fi.Mode().String()
			fmt.Fprintf(os.Stderr, "Mode %s\n", mode)
		}

		fmt.Fprintf(os.Stderr, "\n")

		return
	}

	if len(args) < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: Insufficient command-line arguments supplied to transmute\n")
		os.Exit(1)
	}

	// DOCUMENTATION COMMANDS

	inSwitch = true

	switch args[0] {
	case "-version":
		fmt.Printf("%s\n", EDirectVersion)
	case "-help":
		fmt.Printf("transmute %s\n%s\n", EDirectVersion, transmuteHelp)
	case "-extra", "-extras":
		fmt.Printf("transmute %s\n%s\n", EDirectVersion, transmuteExtra)
	default:
		// if not any of the documentation commands, keep going
		inSwitch = false
	}

	if inSwitch {
		return
	}

	// FILE NAME CAN BE SUPPLIED WITH -input COMMAND

	in := os.Stdin

	// check for data being piped into stdin
	isPipe := false
	fi, err := os.Stdin.Stat()
	if err == nil {
		isPipe = bool((fi.Mode() & os.ModeNamedPipe) != 0)
	}

	usingFile := false

	if fileName != "" {

		inFile, err := os.Open(fileName)
		if err != nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to open input file '%s'\n", fileName)
			os.Exit(1)
		}

		defer inFile.Close()

		// use indicated file instead of stdin
		in = inFile
		usingFile = true

		if isPipe && runtime.GOOS != "windows" {
			mode := fi.Mode().String()
			fmt.Fprintf(os.Stderr, "\nERROR: Input data from both stdin and file '%s', mode is '%s'\n", fileName, mode)
			os.Exit(1)
		}
	}

	// check for -input command after extraction arguments
	for _, str := range args {
		if str == "-input" {
			fmt.Fprintf(os.Stderr, "\nERROR: Misplaced -input command\n")
			os.Exit(1)
		}
	}

	// START PROFILING IF REQUESTED

	if prfl {

		f, err := os.Create("cpu.pprof")
		if err != nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to create profile output file\n")
			os.Exit(1)
		}

		pprof.StartCPUProfile(f)

		defer pprof.StopCPUProfile()
	}

	// INITIALIZE RECORD COUNT

	recordCount := 0
	byteCount := 0

	// print processing rate and program duration
	printDuration := func(name string) {

		PrintDuration(name, recordCount, byteCount)
	}

	// JSON TO XML CONVERTER

	if args[0] == "-j2x" || args[0] == "-json2xml" {

		set := "root"
		rec := ""
		nest := ""

		nextArg := func() (string, bool) {

			if len(args) < 1 {
				return "", false
			}

			// remove next token from slice
			nxt := args[0]
			args = args[1:]

			return nxt, true
		}

		// look for optional arguments
		args = args[1:]
		for {
			arg, ok := nextArg()
			if !ok {
				break
			}

			switch arg {
			case "-set":
				// override set wrapper
				set, ok = nextArg()
				if ok && set == "-" {
					set = ""
				}
			case "-rec":
				// override record wrapper
				rec, ok = nextArg()
				if ok && rec == "-" {
					rec = ""
				}
			case "-nest":
				// specify nested array naming policy
				nest, ok = nextArg()
				if !ok {
					fmt.Fprintf(os.Stderr, "Nested array naming policy is missing\n")
					os.Exit(1)
				}
				if ok && nest == "-" {
					nest = "flat"
				}
				switch nest {
				case "flat", "plural", "name", "recurse", "recursive", "same", "depth", "deep", "level":
				default:
					fmt.Fprintf(os.Stderr, "Unrecognized nested array naming policy\n")
					os.Exit(1)
				}
			default:
				// alternative form uses positional arguments to override set and rec
				set = arg
				if set == "-" {
					set = ""
				}
				rec, ok = nextArg()
				if ok && rec == "-" {
					rec = ""
				}
			}
		}

		// use output channel of tokenizer as input channel of converter
		jtkn := JSONTokenizer(in)
		jcnv := JSONConverter(jtkn, set, rec, nest)

		if jtkn == nil || jcnv == nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to create JSON to XML converter\n")
			os.Exit(1)
		}

		if set != "" {
			fmt.Fprintf(os.Stdout, "<%s>\n", set)
		}

		// drain output of last channel in service chain
		for str := range jcnv {

			if str == "" {
				continue
			}

			// send result to output
			os.Stdout.WriteString(str)
			if !strings.HasSuffix(str, "\n") {
				os.Stdout.WriteString("\n")
			}

			recordCount++
			runtime.Gosched()
		}

		if set != "" {
			fmt.Fprintf(os.Stdout, "</%s>\n", set)
		}

		debug.FreeOSMemory()

		if timr {
			printDuration("blocks")
		}

		return
	}

	// ASN.1 TO XML CONVERTER

	if args[0] == "-a2x" || args[0] == "-asn2xml" {

		set := ""
		rec := ""

		nextArg := func() (string, bool) {

			if len(args) < 1 {
				return "", false
			}

			// remove next token from slice
			nxt := args[0]
			args = args[1:]

			return nxt, true
		}

		// look for optional arguments
		args = args[1:]
		for {
			arg, ok := nextArg()
			if !ok {
				break
			}

			switch arg {
			case "-set":
				// override set wrapper
				set, ok = nextArg()
				if ok && set == "-" {
					set = ""
				}
			case "-rec":
				// override record wrapper
				rec, ok = nextArg()
				if ok && rec == "-" {
					rec = ""
				}
			}
		}

		atkn := ASN1Tokenizer(in)
		acnv := atkn
		acnv = ASN1Converter(atkn, set, rec)

		if atkn == nil || acnv == nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to create ASN.1 to XML converter\n")
			os.Exit(1)
		}

		// drain output of last channel in service chain
		for str := range acnv {

			if str == "" {
				continue
			}

			// send result to output
			os.Stdout.WriteString(str)
			if !strings.HasSuffix(str, "\n") {
				os.Stdout.WriteString("\n")
			}

			recordCount++
			runtime.Gosched()
		}

		debug.FreeOSMemory()

		if timr {
			printDuration("blocks")
		}

		return
	}

	// READ TAB-DELIMITED FILE AND WRAP IN XML FIELDS

	// must be called before CreateReader starts draining stdin
	if len(args) > 1 && args[0] == "-t2x" {

		recordCount = TableConverter(in, "\t", args)

		debug.FreeOSMemory()

		if timr {
			printDuration("lines")
		}

		return
	}

	if len(args) > 1 && args[0] == "-c2x" {

		recordCount = TableConverter(in, ",", args)

		debug.FreeOSMemory()

		if timr {
			printDuration("lines")
		}

		return
	}

	// READ GENBANK FLATFILE AND TRANSLATE TO INSDSEQ XML

	// must be called before CreateReader starts draining stdin
	if len(args) > 0 && args[0] == "-g2x" {

		gbk := GenBankConverter(in)

		if gbk == nil {
			fmt.Fprintf(os.Stderr, "Unable to create GenBank to XML converter\n")
			os.Exit(1)
		}

		head := `<?xml version="1.0" encoding="UTF-8" ?>
<!DOCTYPE INSDSet PUBLIC "-//NCBI//INSD INSDSeq/EN" "https://www.ncbi.nlm.nih.gov/dtd/INSD_INSDSeq.dtd">
<INSDSet>
`
		tail := ""

		// drain output of last channel in service chain
		for str := range gbk {

			if str == "" {
				continue
			}

			if head != "" {
				os.Stdout.WriteString(head)
				head = ""
				tail = `</INSDSet>
`
			}

			// send result to stdout
			os.Stdout.WriteString(str)
			if !strings.HasSuffix(str, "\n") {
				os.Stdout.WriteString("\n")
			}

			recordCount++

			runtime.Gosched()
		}

		if tail != "" {
			os.Stdout.WriteString(tail)
		}

		debug.FreeOSMemory()

		if timr {
			printDuration("records")
		}

		return
	}

	// STRING CONVERSION COMMANDS

	inSwitch = true

	switch args[0] {
	case "-encodeURL":
		EncodeURL(in)
	case "-decodeURL":
		DecodeURL(in)
	case "-encode64", "-encodeB64", "-encodeBase64":
		EncodeB64(in)
	case "-decode64", "-decodeB64", "-decodeBase64":
		DecodeB64(in)
	case "-hgvs":
		DecodeHGVS(in)
	case "-align":
		ProcessAlign(in, args)
	case "-remove":
		SequenceRemove(in, args)
	case "-retain":
		SequenceRetain(in, args)
	case "-replace":
		SequenceReplace(in, args)
	case "-extract":
		SequenceExtract(in, args)
	case "-revcomp":
		NucRevComp(in)
	case "-reverse":
		SeqFlip(in)
	case "-molwt":
		ProtWeight(in, args)
	case "-cds2prot":
		CdRegionToProtein(in, args)
	case "-diff":
		FastaDiff(in, args)
	default:
		// if not any of the conversion commands, keep going
		inSwitch = false
	}

	if inSwitch {

		debug.FreeOSMemory()

		return
	}

	// CREATE XML BLOCK READER FROM STDIN OR FILE

	rdr := CreateReader(in)
	if rdr == nil {
		fmt.Fprintf(os.Stderr, "\nERROR: Unable to create XML Block Reader\n")
		os.Exit(1)
	}

	// CONFIRM INPUT DATA AVAILABILITY AFTER RUNNING COMMAND GENERATORS

	if fileName == "" && runtime.GOOS != "windows" {

		fromStdin := bool((fi.Mode() & os.ModeCharDevice) == 0)
		if !isPipe || !fromStdin {
			mode := fi.Mode().String()
			fmt.Fprintf(os.Stderr, "\nERROR: No data supplied to transmute from stdin or file, mode is '%s'\n", mode)
			os.Exit(1)
		}
	}

	if !usingFile && !isPipe {

		fmt.Fprintf(os.Stderr, "\nERROR: No XML input data supplied to transmute\n")
		os.Exit(1)
	}

	// SPECIAL FORMATTING COMMANDS

	inSwitch = true
	leaf := false

	switch args[0] {
	case "-format":
		recordCount = ProcessFormat(rdr, args, timr)
	case "-filter":
		ProcessFilter(rdr, args)
	case "-normalize", "-normal":
		DoMixed = true
		AllowEmbed = true
		ContentMods = true
		ProcessNormalize(rdr, args)
	case "-outline":
		ProcessOutline(rdr)
	case "-contour":
		leaf = true
		fallthrough
	case "-synopsis":
		args = args[1:]
		delim := "/"
		if len(args) > 0 {
			delim = args[0]
			if len(delim) > 3 {
				delim = "/"
			}
		}
		ProcessSynopsis(rdr, leaf, delim)
	case "-tokens":
		ProcessTokens(rdr)
	default:
		// if not any of the formatting commands, keep going
		inSwitch = false
	}

	if inSwitch {

		debug.FreeOSMemory()

		// suppress printing of lines if not properly counted
		if recordCount == 1 {
			recordCount = 0
		}

		if timr {
			printDuration("lines")
		}

		return
	}

	// SPECIFY STRINGS TO GO BEFORE AND AFTER ENTIRE OUTPUT OR EACH RECORD

	head := ""
	tail := ""

	hd := ""
	tl := ""

	for {

		inSwitch = true

		switch args[0] {
		case "-head":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -head command\n")
				os.Exit(1)
			}
			head = ConvertSlash(args[1])
		case "-tail":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -tail command\n")
				os.Exit(1)
			}
			tail = ConvertSlash(args[1])
		case "-hd":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -hd command\n")
				os.Exit(1)
			}
			hd = ConvertSlash(args[1])
		case "-tl":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -tl command\n")
				os.Exit(1)
			}
			tl = ConvertSlash(args[1])
		case "-wrp":
			// shortcut to wrap records in XML tags
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -wrp command\n")
				os.Exit(1)
			}
			tmp := ConvertSlash(args[1])
			lft, rgt := SplitInTwoAt(tmp, ",", LEFT)
			if lft != "" {
				head = "<" + lft + ">"
				tail = "</" + lft + ">"
			}
			if rgt != "" {
				hd = "<" + rgt + ">"
				tl = "</" + rgt + ">"
			}
		case "-set":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -set command\n")
				os.Exit(1)
			}
			tmp := ConvertSlash(args[1])
			if tmp != "" {
				head = "<" + tmp + ">"
				tail = "</" + tmp + ">"
			}
		case "-rec":
			if len(args) < 2 {
				fmt.Fprintf(os.Stderr, "\nERROR: Pattern missing after -rec command\n")
				os.Exit(1)
			}
			tmp := ConvertSlash(args[1])
			if tmp != "" {
				hd = "<" + tmp + ">"
				tl = "</" + tmp + ">"
			}
		default:
			// if not any of the controls, set flag to break out of for loop
			inSwitch = false
		}

		if !inSwitch {
			break
		}

		// skip past arguments
		args = args[2:]

		if len(args) < 1 {
			fmt.Fprintf(os.Stderr, "\nERROR: Insufficient command-line arguments supplied to transmute\n")
			os.Exit(1)
		}
	}

	// ENSURE PRESENCE OF PATTERN ARGUMENT

	if len(args) < 1 {
		fmt.Fprintf(os.Stderr, "\nERROR: Insufficient command-line arguments supplied to transmute\n")
		os.Exit(1)
	}

	// allow -record as synonym of -pattern (undocumented)
	if args[0] == "-record" || args[0] == "-Record" {
		args[0] = "-pattern"
	}

	// make sure top-level -pattern command is next
	if args[0] != "-pattern" && args[0] != "-Pattern" {
		fmt.Fprintf(os.Stderr, "\nERROR: No -pattern in command-line arguments\n")
		os.Exit(1)
	}
	if len(args) < 2 {
		fmt.Fprintf(os.Stderr, "\nERROR: Item missing after -pattern command\n")
		os.Exit(1)
	}

	topPat := args[1]
	if topPat == "" {
		fmt.Fprintf(os.Stderr, "\nERROR: Item missing after -pattern command\n")
		os.Exit(1)
	}
	if strings.HasPrefix(topPat, "-") {
		fmt.Fprintf(os.Stderr, "\nERROR: Misplaced %s command\n", topPat)
		os.Exit(1)
	}

	// look for -pattern Parent/* construct for heterogeneous data, e.g., -pattern PubmedArticleSet/*
	topPattern, star := SplitInTwoAt(topPat, "/", LEFT)
	if topPattern == "" {
		return
	}

	// CONCURRENT REFORMATTING OF PARSED XML RECORDS

	// -pattern plus -format does concurrent flush-left reformatting
	if len(args) > 2 && args[2] == "-format" {

		xmlq := CreateProducer(topPattern, star, rdr)
		fchq := CreateFormatters(topPattern, xmlq)
		unsq := CreateUnshuffler(fchq)

		if xmlq == nil || fchq == nil || unsq == nil {
			fmt.Fprintf(os.Stderr, "\nERROR: Unable to create formatter\n")
			os.Exit(1)
		}

		if head != "" {
			os.Stdout.WriteString(head)
			os.Stdout.WriteString("\n")
		}

		// drain output channel
		for curr := range unsq {

			str := curr.Text

			if str == "" {
				continue
			}

			if hd != "" {
				os.Stdout.WriteString(hd)
				os.Stdout.WriteString("\n")
			}

			// send result to output
			os.Stdout.WriteString(str)
			if !strings.HasSuffix(str, "\n") {
				os.Stdout.WriteString("\n")
			}

			if tl != "" {
				os.Stdout.WriteString(tl)
				os.Stdout.WriteString("\n")
			}

			recordCount++
			runtime.Gosched()
		}

		if tail != "" {
			os.Stdout.WriteString(tail)
			os.Stdout.WriteString("\n")
		}

		debug.FreeOSMemory()

		if timr {
			printDuration("records")
		}

		return
	}

	// REPORT UNRECOGNIZED COMMAND

	fmt.Fprintf(os.Stderr, "\nERROR: Unrecognized transmute command\n")
	os.Exit(1)
}
