/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Utility 
{
  /**
   * Helper functions, absent from System.Array.
   */
  public module NArray
  {
    /**
     * Iterates a function over an array.
     */
    public Iter ['a] (arr : array ['a], f : 'a -> void) : void
    {
      def loop (i)
      {
        when (i < arr.Length) {
          f (arr [i]);
          loop (i + 1)
        }
      }
      
      loop (0)
    }

    
    /**
     * Iterates a function over an array, passing both the array index
     * and value as the iterated function parameters.
     */
    public IterI ['a] (arr : array ['a], f : int * 'a -> void) : void
    {
      def loop (i)
      {
        when (i < arr.Length) {
          f (i, arr [i]);
          loop (i + 1)
        }
      }
      
      loop (0)
    }
    
    public Map ['a, 'b] (res_type : System.Type, ar : array ['a], f : 'a -> 'b) : array ['b]
    {
      def result = System.Array.CreateInstance (res_type, ar.Length) :> array['b];

      for (mutable i = 0; i < ar.Length; ++i)
        result [i] = f (ar [i]);

      result
    }



    /**
     * Folds a function over an array.
     */
    public Fold ['a, 'b] (arr : array ['b], ini : 'a, f : 'b * 'a -> 'a) : 'a
    {
      def loop (acc, i)
      {
        if (i >= arr.Length) acc
        else loop (f (arr [i], acc), i + 1)
      }
      
      loop (ini, 0)
    }


    /**
     * Folds a function over an array, passing the array index
     * as an additional parameter to the folded function parameters.
     */
    public FoldI ['a, 'b] (arr : array ['b], ini : 'a, f : int * 'b * 'a -> 'a) : 'a
    {
      def loop (acc, i)
      {
        if (i >= arr.Length) acc
        else loop (f (i, arr [i], acc), i + 1)
      }
      
      loop (ini, 0)
    }

    
    /**
     * Returns 'true' if at least one of the 'l' arrays's elements
     * satisfies the condition 'f'.
     *
     * Example of use:
     *
     *   NArray.Exists (array ["a", "b", "abc", "d", "e"], fun (x) { x.Length > 2 })
     *
     * evaluates to 'true' as there is one string of length 3 on the list.
     */
    public Exists ['a] (a : array ['a], f : 'a -> bool) : bool
    {
      def walk_array (i : int) : bool
      {
        i < a.Length && (f (a [i]) || walk_array (i + 1))
      }

      walk_array (0)
    }

    
    /**
     * Returns 'true' if exactly all of the 'l' arrays's elements
     * satisfy the condition 'f'.
     *
     * Example of use:
     *
     *   NArray.ForAll (array [2, 4, 6, 8, 10], fun (x) { x % 2 == 0 })
     *
     * evaluates to 'true' as all the array's elements are even integers.
     */
    public ForAll ['a] (a : array ['a], f : 'a -> bool) : bool
    {
      def walk_array (i : int) : bool
      {
        i >= a.Length || (f (a [i]) && walk_array (i + 1))
      }

      walk_array (0)
    }

    
    /**
     * Converts a list into an array. (it doesn't work too well in current type system)
     */  
    public FromList ['a] (t : System.Type, x : list ['a]) : array ['a]
    {
      def list_length = Nemerle.Collections.List.Length (x);
      def result = (System.Array.CreateInstance (t, list_length) :> array['a]);
      
      mutable i = 0;
      
      Nemerle.Collections.List.Iter (x, fun (x) {
        result [i] = x; ++i
      });
      
      result
    }

    public Iter2['a, 'b] (a : list ['a], b : array ['b], f : 'a  * 'b -> void) : void {
      def loop (l, acc) : void {
        match (l) {
          | [] => () 
          | x :: xs => f (x, b [acc]); loop (xs, acc + 1)
        }
      }

      loop (a, 0)
    }

    public Iter2['a, 'b] (a : array ['a], b : list ['b], f : 'a  * 'b -> void) : void {
      def loop (l, acc) : void {
        match (l) {
          | [] => ()
          | x :: xs  => f (a [acc], x); loop (xs, acc + 1)
        }
      }

      loop (b, 0)
    }

    public Map2['a, 'b, 'c] (a : list ['a], b : array ['b], f : 'a * 'b -> 'c) : list ['c] {
      def loop (l, acc) : list ['c] {
        match (l) {
          | [] => []
          | x :: xs => f (x, b [acc]) :: loop (xs, acc + 1)
        }
      }

      loop (a, 0)
    }

    public Map2['a, 'b, 'c] (a : array ['a], b : list ['b], f : 'a * 'b -> 'c) : list ['c] {
      def loop (l, acc) : list ['c] {
        match (l) {
          | [] => []
          | x :: xs => f (a [acc], x) :: loop (xs, acc + 1)
        }
      }

      loop (b, 0)
    }

    public RevMap2['a,'b,'c] (a : list ['a], b : array ['b], f : 'a * 'b -> 'c) : list ['c] {
      def loop (x, i, acc) : list ['c] {
        match (x) {
          | [] => acc
          | h :: t => loop (t, i + 1, f (h, b [i]) :: acc)
        }
      }

      loop(a, 0, [])
    }

    public RevMap2['a,'b,'c] (a : array ['a], b : list ['b], f : 'a * 'b -> 'c) : list ['c] {
      def loop (x, i, acc) : list ['c] {
        match (x) {
          | [] => acc
          | h :: t => loop (t, i + 1, f (a [i], h) :: acc)
        }
      }

      loop(b, 0, [])
    }

    public FoldLeft2['a, 'b, 'c] (a : list ['a], b : array ['b], acc : 'c, f : 'a  * 'b * 'c -> 'c) : 'c {
      def loop (x, i, ac) : 'c {
        match (x) {
          | [] => ac
          | h :: t => loop (t, i + 1, f (h, b [i], ac))
        }
      }

      loop (a, 0, acc)
    }

    public FoldLeft2['a, 'b, 'c] (a : array ['a], b : list ['b], acc : 'c, f : 'a  * 'b * 'c -> 'c) : 'c {
      def loop (x, i, ac) : 'c {
        match (x) {
          | [] => ac
          | h :: t => loop (t, i + 1, f (a [i], h, ac))
        }
      }

      loop (b, 0, acc)
    }

    public FoldRight2['a, 'b, 'c] (a : list ['a], b : array ['b], c : 'c, f : 'a * 'b  * 'c -> 'c) : 'c {
      def loop (x, i, acc) : 'c {
        match (x) {
          | [] => acc
          | h :: t => f (h, b [i], loop (t, i + 1, acc))
        }
      }

      loop (a, 0, c)
    }

    public FoldRight2['a, 'b, 'c] (a : array ['a], b : list ['b], c : 'c, f : 'a * 'b  * 'c -> 'c) : 'c {
      def loop (x, i, acc) : 'c {
        match (x) {
          | [] => acc
          | h :: t => f (a [i], h, loop (t, i + 1, acc))
        }
      }

      loop (b, 0, c)
    }

    public ForAll2['a, 'b] (a : list ['a], b : array ['b], f : 'a * 'b -> bool) : bool {
      def loop (x, i) : bool {
        match (x) {
          | [] => true
          | h :: t => f (h, b[i]) && loop (t, i + 1)
        }
      }

      loop (a, 0)
    }

    public ForAll2['a, 'b] (a : array ['a], b : list ['b], f : 'a * 'b -> bool) : bool {
      def loop (x, i) : bool {
        match (x) {
          | [] => true
          | h :: t => f (a [i], h) && loop (t, i + 1)
        }
      }

      loop (b, 0)
    }

    public Exists2['a, 'b] (a : array ['a], b : list ['b], f : 'a * 'b -> bool) : bool {
      def loop (x, i) : bool {
        match (x) {
          | [] => false
          | h :: t => f (a [i], h) || loop (t, i + 1)
        }
      }

      loop (b, 0)
    }

    public Exists2['a, 'b] (a : list ['a], b : array ['b], f : 'a * 'b -> bool) : bool {
      def loop (x, i) : bool {
        match (x) {
          | [] => false
          | h :: t => f (h, b [i]) || loop (t, i + 1)
        }
      }

      loop (a, 0)
    }
  }
} /* end of namespace */
