/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Collections 
{ 
  /**
   * Exception throws by queue service function where queue is empty.
   */
  public class EmptyQueue : EmptyCollection 
  {
    public this () { base ("Nemerle.Collections.Queue") }
  }


  /**
   * Class representing first-in-first-out queue.
   */
  public class Queue ['a] : ICollection['a]
  {
    /* -- PUBLIC CONSTRUCTORS ----------------------------------------------- */
    
    /** Create a new empty queue. */
    public this ()
    {
      m_first = [];
      m_last  = [];
    }


    /* -- PUBLIC PROPERTIES ------------------------------------------------- */

    /**
     * Return `true` iff the queue is empty.
     */
    public IsEmpty : bool
    {
      get
      {
        move_cond ();

        match (m_first) {
          | [] => true
          | _  => false
        }
      }
    }

    
    /**
     * Returns the number of elements in the queue.
     */
    public Count : int
    {
      get
      {
        List.Length (m_first) + List.Length (m_last)
      }
    }


    /**
     * Alias for Count.
     */
    public Length : int
    { 
      get { Count }
    }

    
    /* -- PUBLIC METHODS ---------------------------------------------------- */

    /**
     * Append element to the end of the queue.
     */
    public Add (x : 'a) : void 
    {
      m_last = x :: m_last;
    }
     

    /**
     * Alias for Add.
     */
    public Push (x : 'a) : void
    {
      Add (x)
    }

     
    /**
     * Return the first element of the queue and remove it.  Throw
     * EmptyQueue exception if the queue is empty.
     */
    public Take () : 'a
    {
      move_cond ();

      match (m_first) {
        | x :: xs => 
          m_first = xs;
          x
          
        | [] =>
          throw EmptyQueue ()
      }
    }


    /**
     * Alias for Take.
     */
    public Pop () : 'a
    {
      Take ()
    }


    /**
     * Return the first element of the queue.
     *
     * Throw EmptyQueue exception if the queue is empty.
     */
    public Peek () : 'a
    {
      move_cond ();

      match (m_first) {
        | x :: _ => x
        | [] => throw EmptyQueue ()
      }
    }


    /**
     * Alias for Peek.
     */
    public Top () : 'a
    {
      Peek ()
    }


    /** 
     * Return some element from the queue, implements ICollection.First.
     */
    public First () : option ['a]
    {
      move_cond ();

      match (m_first) {
        | x :: _ => Some (x)
        | []     => None ()
      }
    }


    /**
     * Empty the queue.
     */
    public Clear () : void
    {
      m_first = [];
      m_last  = [];
    }


    /**
     * Create a shallow copy of the queue.
     */
    public Clone () : Queue ['a]
    {
      def q     = Queue ();
      q.m_first = List.Copy (m_first);
      q.m_last  = List.Copy (m_last);
      q
    }

    
    /**
     * Call supplied function for every element of the queue.
     */
    public Iter (f : 'a -> void) : void
    {
      move ();
      List.Iter (m_first, f);
    }
        
    
    /**
     * Fold elements of the queue with supplied function and initial
     * value.
     */
    public Fold['b] (f : 'a * 'b -> 'b, x : 'b) : 'b
    {
      move ();
      List.FoldLeft (m_first, x, f)
    }


    /**
     * Transfer all elements of the queue q to the end of this queue.
     */
    public Transfer (q : Queue['a]) : void
    {
      move ();
            
      m_last = q.m_last + List.Rev (q.m_first);
      q.m_first   = [];
      q.m_last    = [];
    }


    /**
     * Returns `true' iff the element x is a member of the queue.
     */
    public Contains (x : 'a) : bool
    {
      List.Member (m_first, x) || List.Member (m_last, x)
    }


    /**
     * Removes element x from the queue.
     */
    public Remove (x : 'a) : void
    {
      m_first = List.Remove (m_first, x);
      m_last  = List.Remove (m_last, x);
    }


    /**
     * Return `true' iff every element of the queue satisfy predicate f.
     */
    public ForAll (f : 'a -> bool) : bool
    {
      List.ForAll (m_first, f) && List.ForAll (m_last, f)
    }

          
    /**
     * Return true iff the queue contains an element that
     * satisfies predicate f.
     */
    public Exists (f : 'a -> bool) : bool
    {
      List.Exists (m_first, f) || List.Exists (m_last, f)
    }


    /**
     * Remove from queue every element that does not satisfy
     * predicate f.
     */
    public Filter (f : 'a -> bool) : void
    {
      m_first = List.Filter (m_first, f);
      m_last  = List.Filter (m_last, f);
    }


    /**
     * Map queue to a new queue using mapping f.
     */
    public Map ['b] (f : 'a -> 'b) : Queue ['b]
    {
      def q     = Queue ();
      q.m_first = List.Map (m_first, f);
      q.m_last  = List.Map (m_last, f);
      q
    }


    /**
     * Partition the queue into two queues: first with elements
     * that satisfy predicate f, second with the rest.
     */
    public Partition (f : 'a -> bool) : Queue ['a] * Queue ['a]
    {
      def sat        = Queue ();
      def nonsat     = Queue ();

      def (s, n)     = List.Partition (m_first, f);
      sat.m_first    = s; 
      nonsat.m_first = n;

      def (s, n)     = List.Partition (m_last, f);
      sat.m_last     = s;
      nonsat.m_last  = n;

      (sat, nonsat)
    }

    /**
     * Return enumerator for elements of the queue.
     */
    public GetEnumerator () : IEnumerator ['a]
    {
      move ();

      ListEnumerator (m_first)
    }

           
    /* -- PRIVATE METHODS --------------------------------------------------- */
    
    /**
     * Ensure that if there are any elements in the queue, then
     * the first list is not empty.
     */
    private move_cond () : void
    {
      match (m_first) {
        | [] =>
          m_first = List.Rev (m_last);
          m_last  = []
        | _  => ()
      }
    }


    /**
     * Unconditionally move all the elements of the second list
     * to the end of the first list.
     */
    private move () : void
    {
      m_first = m_first + List.Rev (m_last)
    }

    
    /* -- PRIVATE FIELDS ---------------------------------------------------- */
    
    /**
     * First part of the queue.
     */
    private mutable m_first : list ['a];
    
    /**
     * Second part of the queue
     */
    private mutable m_last  : list ['a];
    
  } /* end of class Queue */
  
} /* end of namespace */
