/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Collections
{
  using System;
  
  public class Vector['a] : IEnumerable['a]
  {
    mutable storage : array['a];
    mutable current_size : int;

    public this(n : int) 
    {
      storage = array (if (n > 0) n else 16);
      current_size = 0;
    }

    public Length : int
    {
      get { storage.Length }
    }

    public Count : int
    {
      get { current_size }
    }

    public Capacity : int
    {
      get { storage.Length }

      //[Nemerle.Assertions.Requires (value <= Count, throw System.ArgumentOutOfRangeException ())]    
      set {
        def newstorage = array (value);
        Array.Copy (storage, newstorage, storage.Length);
        storage = newstorage;
      }
    }
    
    public IsEmpty : bool
    {
      get { current_size == 0 }
    }

    public Add (x : 'a) : void
    {
      when (current_size >= storage.Length)
        Capacity = storage.Length * 2;
        
      storage [current_size] = x;
      ++current_size;
    }

    public RemoveLast () : void
    {
      if (current_size < 1) throw System.InvalidOperationException ("vector is empty")
      else
        --current_size;
    }
    
    public Item [index : int] : 'a
    {
      get { storage [index] }
      set { storage [index] = value }
    }

    public Clear () : void
    {
      current_size = 0;
    }

    public GetEnumerator () : IEnumerator['a]
    {
      VectorEnumerator (this) : IEnumerator['a];
    }

    internal class VectorEnumerator['a] : IEnumerator['a], System.ICloneable
    {
      enumeratee : Vector['a];
      mutable position : int;
      length : int;

      public this (vectorToEnumerate : Vector['a])
      {
        enumeratee = vectorToEnumerate;
        position = -1;
        length = enumeratee.Count;
      }

      public Current : 'a
      {
        get {
          when (position < 0)
            throw InvalidOperationException ("Enumeraion has not yet started.");
          when (position >= length)
            throw InvalidOperationException ("Enumeration has already ended.");
          enumeratee[position] : 'a;
        }
      }

      public MoveNext () : bool
      {
        when (position < length)
          ++position;
        if (position < length)
          true;
        else
          false;
      }

      public Reset () : void
      {
        position = -1;
      }

      public Clone () : object
      {
        MemberwiseClone ();
      }
    }
  }
}
