/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


using Nemerle.Collections;
using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;

using Nemerle.Logging.Helper;

namespace Nemerle.Logging
{
  internal module Helper
  {
    public Flags : Hashtable [string, bool] = Hashtable ();
    public mutable PrintExpression : PExpr;
    public mutable SetPrintExpression : bool;

    public GetName (expr : PExpr) : string
    {
      | PExpr.Ref (name) =>
        name.Id
      | _ =>
        Message.FatalError ($ "expected a plain identifier, got $expr");
    }

    Init () : void
    {
      Flags.Clear ();
    }

    this ()
    {
      Passes.OnInit += Init;
      PrintExpression = <[ System.Console.WriteLine ]>;
      SetPrintExpression = false;
      Init ();
    }
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class)]
  macro LogFlag (_ : TypeBuilder, id, is_on : bool)
  {
    def name = GetName (id);
    when (Flags.Contains (name))
      Message.Error ($ "redefinition of the flag `$(name)'");
    Flags [name] = is_on;
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class)]
  macro LogFunction (_ : TypeBuilder, fn)
  {
    when (SetPrintExpression)
      Message.Error ("the logging expression already set");
    PrintExpression = fn;
  }

  macro log (flag, str)
  {
    def name = GetName (flag);
    if (Flags.Contains (name))
      if (Flags [name])
        <[ $PrintExpression ($str) ]>
      else <[ {} ]>
    else {
      Message.Error ($ "there is no debug flag named `$(name)'");
      <[ {} ]>
    }
  }

  macro @whenlogging (flag, body)
    syntax ("whenlogging", "(", flag, ")", body)
  {
    def name = GetName (flag);
    if (Flags.Contains (name))
      if (Flags [name]) body
      else <[ {} ]>
    else {
      Message.Error ($ "there is no debug flag named `$(name)'");
      <[ {} ]>
    }
  }
}
