/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Text;

using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;

namespace Nemerle 
{
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class)]
  macro MarkOptions (t : TypeBuilder, attribute)
  {
    // iterate through members of this type and select only variant options
    foreach (ClassMember.TypeDeclaration
              (TopDeclaration.VariantOption as vo) in t.GetParsedMembers ())
    {
      // add given custom attribute to this variant option
      vo.AddCustomAttribute (attribute)
    }
  }
}

namespace Nemerle.Utility
{
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Field,
                       Inherited = false, AllowMultiple = true)]
  macro Accessor (current_type : TypeBuilder, storage_field : ParsedField, params args : list [PExpr])
  {
    def (name, flags) =
      match (args) {
        | [ <[ $(n : dyn) ]>, <[ $("flags" : dyn) = $opts ]> ] =>
          (Some (n), Some (opts))
        | [ <[ $("flags" : dyn) = $opts ]> ] =>
          (None (), Some (opts))
        | [ <[ $(n : dyn) ]> ] =>
          (Some (n), None ())
        | [] =>
          (None (), None ())
        | _ =>
          Message.FatalError ("usage: Accessor (name, flags = SOMEFLAGS), "
                              "where both name and flags are optional");
      }

    // __some_foo__bar ==> SomeFooBar
    def transformed_name = 
      {
        def sb = StringBuilder ();
        mutable next_upper = true;
        foreach (ch in storage_field.Name)
          if (ch == '_') {
            next_upper = true;
          } else if (next_upper) {
            _ = sb.Append (char.ToUpper (ch));
            next_upper = false;
          } else
            _ = sb.Append (ch);
        sb.ToString ()
      }

    def name = name.WithDefault (transformed_name);

    mutable want_setter = false;
    mutable want_internal = false;
    mutable want_protected = false;

    def parse_opts (expr) {
      | <[ $("WantSetter" : dyn) ]> =>
        want_setter = true
      | <[ $("Setter" : dyn) ]> =>
        want_setter = true
      | <[ $("Internal" : dyn) ]> =>
        want_internal = true;
      | <[ $("Protected" : dyn) ]> =>
        want_protected = true;
      | <[ $e1 | $e2 ]> =>
        parse_opts (e1);
        parse_opts (e2);
      | e =>
        Message.FatalError ($ "bad accessor option, $e");
    }
    match (flags) {
      | Some (e) =>
        parse_opts (e)
      | None => {}
    }

    def fieldref = <[ $(storage_field.ParsedName : name) ]>;

    def prop = 
      if (want_setter)
        <[ decl:
          public $(name : dyn) : $(storage_field.ty)
          {
            set { $fieldref = value; }
            get { $fieldref }
          }
        ]>
      else
        <[ decl:
          public $(name : dyn) : $(storage_field.ty)
          {
            get { $fieldref }
          }
        ]>;

    when (want_internal) {
      prop.Attributes &= ~NemerleAttributes.Public;
      prop.Attributes |= NemerleAttributes.Internal;
    }

    when (want_protected) {
      prop.Attributes &= ~NemerleAttributes.Public;
      prop.Attributes |= NemerleAttributes.Protected;
    }

    when (storage_field.Attributes %&& NemerleAttributes.Static)
      prop.Attributes |= NemerleAttributes.Static;

    current_type.Define (prop);
  }


  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Field,
                       Inherited = false, AllowMultiple = true)]
  macro FlagAccessor (current_type : TypeBuilder, storage_field : ParsedField, params args : list [PExpr])
  {
    def usage =
        "usage: FlagAccessor (name1, name2, ..., flags = SOMEFLAGS), "
        "where flags are optional";

    mutable opts = None ();
    
    def names =
      args.Filter (fun (_) {
        | <[ $("flags" : dyn) = $o ]> =>
          if (opts.IsNone) opts = Some (o);
          else Message.FatalError (usage);
          false
        | <[ $(_ : name) ]> => true
        | _ => 
          Message.FatalError (usage)
      });

    mutable want_setter = false;
    mutable want_internal = false;
    mutable want_protected = false;

    def parse_opts (expr) {
      | <[ $("WantSetter" : dyn) ]> =>
        want_setter = true
      | <[ $("Setter" : dyn) ]> =>
        want_setter = true
      | <[ $("Internal" : dyn) ]> =>
        want_internal = true;
      | <[ $("Protected" : dyn) ]> =>
        want_protected = true;
      | <[ $e1 | $e2 ]> =>
        parse_opts (e1);
        parse_opts (e2);
      | e =>
        Message.FatalError ($ "bad accessor option, $e");
    }
    
    match (opts) {
      | Some (e) => parse_opts (e)
      | None => {}
    }

    def fieldref = <[ $(storage_field.ParsedName : name) ]>;
    def enumref = storage_field.ty;

    foreach (<[ $(name : dyn) ]> in names) {
      def flag = <[ $enumref . $(name : dyn) ]>;
      def prop = 
        if (want_setter)
          <[ decl:
            public $(name : dyn) : bool
            {
              get { $fieldref %&& $flag }
              set {
                if (value)
                  $fieldref |= $flag;
                else
                  $fieldref &= ~ $flag
              }
            }
          ]>
        else
          <[ decl:
            public $(name : dyn) : bool
            {
              get { $fieldref %&& $flag }
            }
          ]>;

      when (want_internal) {
        prop.Attributes &= ~NemerleAttributes.Public;
        prop.Attributes |= NemerleAttributes.Internal;
      }

      when (want_protected) {
        prop.Attributes &= ~NemerleAttributes.Public;
        prop.Attributes |= NemerleAttributes.Protected;
      }

      when (storage_field.Attributes %&& NemerleAttributes.Static)
        prop.Attributes |= NemerleAttributes.Static;
        
      current_type.Define (prop);
    }
  }
}
