/*
 * Copyright (c) 2004, 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Macros used only inside the compiler.  */

using Nemerle.Compiler;

namespace Nemerle.Compiler.Util
{
  macro locate (l, body)
  {
    <[
      def pushpop = $l != Location.Default;
      when (pushpop) Location_stack.push ($l);
      try { $body } finally { when (pushpop) Location_stack.pop () }
    ]>
  }

  macro ice (msg = <[ "(see backtrace)" ]>)
  {
    <[ Message.Debug ("Internal compiler error, please report a bug to bugs.nemerle.org. "
                      "You can try modifying program near this location.");
       assert (false, $msg) ]>
  }

  macro cassert (cond, message = <[ "" ]>)
  {
    <[ unless ($cond) {
         Message.Debug ("Internal compiler error, please report a bug to bugs.nemerle.org. "
                        "You can try modifying program near this location.");
         throw AssertionException ($(cond.loc.File : string), 
                                   $(cond.loc.Line : int),
                                   $cond.ToString (), $message)
       }
    ]>
  }
}

namespace Nemerle.Compiler.SolverMacros
{
  macro ReportError (messenger, msg)
  {
    <[
      if ($messenger.NeedMessage)
        $messenger.Error ($msg)
      else
        $messenger.MarkError ()
    ]>
  }
  
  macro ReportFatal (messenger, msg)
  {
    <[
      if ($messenger.NeedMessage) {
        $messenger.Error ($msg);
        when (IsTopLevel)
          throw Recovery ();
      } else
        $messenger.MarkError ();

      Typedtree.TExpr.Error ()
    ]>
  }
  
  macro SaveError (messenger, msg)
  {
    <[
      if ($messenger.NeedMessage)
        $messenger.SaveError ($msg);
      else
        $messenger.MarkError ();
    ]>
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Method)]
  macro PossiblyLooping (_ : TypeBuilder, m : ParsedMethod, solver)
  {
    m.Body = <[
      if ($solver.CanEnterPossiblyLooping ())
        try {
          $(m.Body)
        } finally {
          $solver.LeavePossiblyLooping ()
        }
      else false
    ]>
  }
}
