//
// NemerleCodeCompiler.n
//
// Author:
//  Atsushi Enomoto <atsushi@ximian.com>
//  Kamil Skalski <nazgul@nemerle.org>
//
// Original: CSharpCodeCompiler.cs
// Original Authors:
//  Sean Kasun (seank@users.sf.net)
//  Gonzalo Paniagua Javier (gonzalo@ximian.com)
//
// Copyright (c) Novell, Inc. (http://www.novell.com)
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
using System;
using System.CodeDom;
using System.CodeDom.Compiler;
using System.Configuration;
using System.IO;
using System.Text;
using System.Reflection;
using System.Collections;
using System.Collections.Specialized;
using System.Diagnostics;

using Nemerle.Assertions;
using Nemerle.Collections;

namespace Nemerle.Compiler
{

  public class NemerleCodeCompiler : NemerleCodeGenerator, ICodeCompiler
  {
    public this ()  {  }

    public CompileAssemblyFromDom (options : CompilerParameters,
                                   e : CodeCompileUnit) : CompilerResults
    {
      CompileAssemblyFromDomBatch (options, array [e]);
    }
    
    public CompileAssemblyFromDomBatch (options : CompilerParameters,
                                        ea : array [CodeCompileUnit]) : CompilerResults
    {
      def fileNames = array (ea.Length);
      mutable i = 0;
      def options = if (options == null) CompilerParameters () else options;
      
      def assemblies = options.ReferencedAssemblies;

      foreach (e : CodeCompileUnit in ea) {
        fileNames [i] = GetTempFileNameWithExtension (options.TempFiles,
                                                      i.ToString () + ".n");
        using (f = FileStream (fileNames [i], FileMode.OpenOrCreate))
        {
          def s = StreamWriter (f, Encoding.UTF8);
          when (e.ReferencedAssemblies != null) {
            foreach (str in e.ReferencedAssemblies) {
              when (!assemblies.Contains (str))
                ignore (assemblies.Add (str));
            }
          }

          (this : ICodeGenerator).
            GenerateCodeFromCompileUnit (e, s, CodeGeneratorOptions());
            
          s.Close ();
        }
        ++i;
      }
      CompileAssemblyFromFileBatch (options, fileNames);
    }
    
    public CompileAssemblyFromFile (options : CompilerParameters,
                                    fileName : string) : CompilerResults
    {
      CompileAssemblyFromFileBatch (options, array [fileName]);
    }

    [Requires (options != null && fileNames != null)]
    public CompileAssemblyFromFileBatch (options : CompilerParameters,
                                         fileNames : array [string]) : CompilerResults
    {
      def results = CompilerResults (options.TempFiles);
      mutable succeeded = false;
      mutable failed = false;
      
      def err_event (is_warning, loc : Location, msg) {
        def loc = if (loc == Location.Default) Location_stack.top () else loc;
        def error = CompilerError ();
        error.FileName = loc.File;
        error.Line = loc.Line;
        error.Column = loc.Column;
        error.IsWarning = is_warning;
        error.ErrorText = msg;
        ignore (results.Errors.Add (error));
      }

      Message.ErrorOccured += fun (loc : Location, msg) {
        err_event (false, loc, msg);
        failed = true;
      }
      Message.WarningOccured += fun (loc : Location, msg) {
        err_event (true, loc, msg);
      }

      def fullOutput = System.IO.StringWriter ();
      Message.InitOutput (fullOutput);
      Options.ProgressBar = false;
      Options.IgnoreConfusion = true;
      Options.ReferencedLibraries = [];
      Options.TargetIsLibrary = !options.GenerateExecutable;

      when (options.OutputAssembly == null)
        options.OutputAssembly = GetTempFileNameWithExtension (options.TempFiles, "dll");
      Options.OutputFileName = options.OutputAssembly;

      when (null != options.ReferencedAssemblies)
        foreach (import in options.ReferencedAssemblies)
          Options.ReferencedLibraries = import :: Options.ReferencedLibraries;

      try {
        if (fileNames.Length < 1)
          Message.Error ("need at least one file to compile");
        else {
          Options.Sources = List.FromArray (fileNames);
          Passes.Run ();
          succeeded = !failed;
        }
      }
      catch {
        | _e is System.IO.FileNotFoundException => Message.Error (_e.Message)
        | _e is Recovery => succeeded = false;
//        | _e : System.ArgumentException =>  Message.Error (_e.Message);
        | _e is MatchFailureException =>   Message.Error (_e.Message);
        | _e is ICE =>  Message.Error (_e.Message);
        | _e is AssertionException =>  Message.Error (_e.Message);
        | _e is AssemblyFindException =>  Message.Error (_e.Message);
//        | _e : System.Exception =>  Message.Error (_e.Message);
      };

      if (succeeded) {
        results.NativeCompilerReturnValue = 0;
        results.CompiledAssembly = Assembly.LoadFrom (options.OutputAssembly);
      }
      else {
        results.NativeCompilerReturnValue = 1;
        results.CompiledAssembly = null;
      }

      results;
    }

    public CompileAssemblyFromSource (options : CompilerParameters,
                                      source : string) : CompilerResults
    {
      CompileAssemblyFromSourceBatch (options, array [source]);
    }

    public CompileAssemblyFromSourceBatch (options : CompilerParameters,
                                           sources : array [string]) : CompilerResults
    {
      def fileNames = array (sources.Length);
      mutable i = 0;
      foreach (source in sources) {
        fileNames [i] = GetTempFileNameWithExtension (
          options.TempFiles, i.ToString () + ".n");
        using (f = FileStream (fileNames [i], FileMode.OpenOrCreate)) {
          def s = StreamWriter (f);
          s.Write (source);
          s.Close ();
        }
        ++i;
      }
      CompileAssemblyFromFileBatch (options, fileNames);
    }

    static GetTempFileNameWithExtension (temp_files : TempFileCollection,
                                         extension : string) : string
    {
      temp_files.AddExtension (extension);
    }
  }
}
