/*
 * Copyright (c) 2004, 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;

using Nemerle.Utility;
using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;

using SRE = System.Reflection.Emit;

namespace Nemerle.Compiler {

/** This module is used to decode and encode Nemerle specific information about types,
    methods, etc. which are not directly expressible in .NET metadata.

    We use custom attributes to save / read this data in emitted / loaded assemblies.
 */
module TyCodec 
{
  variant Term
  {
    | App { name : string; args : list [Term]; }
  }

  // --------------- DECODING -------------------------  

  ParseTerm (s : string) : Term
  {
    def get_name (pos) {
      def idx = s.IndexOf ('(', pos);
      assert (idx != -1);
      (idx + 1, s.Substring (pos, idx - pos))
    };
    
    def maybe_get (pos) {
      if (s[pos] == ')') (pos + 1, None ())
      else {
        def (pos, name) = get_name (pos);
        def (pos, args) = get_list ([], pos);
        (pos, Some (Term.App (name, args)))
      }
    } 
    and get_list (acc, pos) {
      match (maybe_get (pos)) {
        | (pos, None) => (pos, List.Rev (acc))
        | (pos, Some (x)) => get_list (x :: acc, pos)
      }
    };
    
    match (maybe_get (0)) {
      | (pos, Some (ret)) =>
        assert (pos == s.Length);
        ret
      | (_, None) =>
        assert (false);
    }
  }
  

  decode (lib : LibraryReference, tenv : SystemMap [int, StaticTyVar], t : Term) : MType
  {
    def self (t) { decode (lib, tenv, t) : TyVar };
    match (t) {
      | Term.App (name, args) when name[0] != '.' =>
        match (NamespaceTree.LookupExactType (NString.Split (name, array ['.', '+']))) {
          | Some (tc) =>
            if (tc.FullName == "System.Void") InternalType.Void
            else
              MType.Class (tc, List.Map (args, self)).Expand ()
          | None =>
            match (lib.LookupInternalType (name)) {
              | Some (tc) =>
                MType.Class (tc, List.Map (args, self)).Expand ()
              | None => Util.ice ("unbound encoded type " + name)
            }
        }
      | Term.App (".a", [Term.App (srank, []), t]) =>
        MType.Array (decode (lib, tenv, t), System.Int32.Parse (srank))
      | Term.App (".r", [t]) =>
        MType.Ref (decode (lib, tenv, t))
      | Term.App (".o", [t]) =>
        MType.Out (decode (lib, tenv, t))
      | Term.App (".f", [t1, t2]) =>
        MType.Fun (decode (lib, tenv, t1), decode (lib, tenv, t2))
      | Term.App (".v", [Term.App (no, [])]) =>
        match (tenv.Find (System.Int32.Parse (no))) {
          | Some (tv) => MType.TyVarRef (tv)
          | None => Util.ice ("unboud type variable in encoded type " + no)
        }
      | Term.App (".p", types) => MType.Tuple (List.Map (types, self))
      | Term.App (name, _) => Util.ice ("invalid encoded type opcode " + name)
    }
  }

  decode_typarms (lib : LibraryReference, 
                  tenv : SystemMap [int, StaticTyVar], 
                  t : Term) : list [StaticTyVar] * SystemMap [int, StaticTyVar]
  {
    mutable tyvars = [];
    def decode_tyvar (t, acc : SystemMap [int, StaticTyVar]) {
      match (t) {
        | Term.App (_, [Term.App (no, []), Term.App (name, []), _]) =>
          def tv = StaticTyVar (name);
          tyvars = tv :: tyvars;
          acc.Replace (System.Int32.Parse (no), tv)
        | _ => Util.ice ("evil encoded tyvar")
      }
    };

    def vars =
      match (t) {
        | Term.App (".tp", vars) => vars
        | Term.App (name, _) => Util.ice ("invalid typarms in encoded type " + name)
      };
      
    def tenv = List.FoldLeft (vars, tenv, decode_tyvar);
    
    def set_constraints (t, tv : StaticTyVar) {
      match (t) {
        | Term.App (_, [_, _, Term.App (_, constraints)]) =>
          tv.Constraints = 
            List.Map (constraints, fun (t) { decode (lib, tenv, t) });
        | _ => Util.ice ("evil encoded tyvar")
      }
    };

    tyvars = List.Rev (tyvars);
    List.Iter2 (vars, tyvars, set_constraints);
    
    (tyvars, tenv)
  }


  reflect_typarms (_lib : LibraryReference, 
                  mutable tenv : SystemMap [int, StaticTyVar], 
                  _t : System.Type) : list [StaticTyVar] * SystemMap [int, StaticTyVar]
  {
    mutable tyvars = [];
#if _GENERICS    
    def vars = _t.GetGenericArguments ();

    foreach (gparm in vars) {
      def tv = StaticTyVar (gparm.Name, gparm);
      tyvars = tv :: tyvars;
      tenv = tenv.Replace (gparm.MetadataToken, tv)
    }
    
    def set_constraints (t, tv : StaticTyVar) {
      def constraints = t.GetGenericParameterConstraints ();
      tv.Constraints = 
        List.MapFromArray (constraints, fun (t) { _lib.TypeOfType (tenv, t) });
    };
#endif
    tyvars = List.Rev (tyvars);
#if _GENERICS    
    NArray.Iter2 (vars, tyvars, set_constraints);
#endif
    
    (tyvars, tenv)
  }

  
  /**
   * Used to decode Nemerle types extracted from assembly metadata
   */
  public DecodeType (lib : LibraryReference, tenv : SystemMap [int, StaticTyVar], tokens : string) : MType
  {
    decode (lib, tenv, ParseTerm (tokens))
  }

  public DecodeTypeBuilder (lib : LibraryReference, 
                       tenv : SystemMap [int, StaticTyVar], 
                       s : string) 
    : list [StaticTyVar] * SystemMap [int, StaticTyVar] * list [MType.Class] * list [MType.Class]
  {
    match (ParseTerm (s)) {
      | Term.App (".ti", [tp, Term.App (_, dst), Term.App (_, st)]) =>
          def (tyvars, tenv) = decode_typarms (lib, tenv, tp);
          def decode_ty (t) { decode (lib, tenv, t) :> MType.Class };
          (tyvars, tenv, List.Map (dst, decode_ty), List.Map (st, decode_ty))
      | _ => Util.ice ("evil encoded tyinfo " + s)
    }
  }

  public ReflectTypeBuilder (lib : LibraryReference, 
                             tenv : SystemMap [int, StaticTyVar], 
                             original : System.Type) 
    : list [StaticTyVar] * SystemMap [int, StaticTyVar] * list [MType.Class] * list [MType.Class]
  {
    def (tyvars, tenv) = reflect_typarms (lib, tenv, original);

    // compute direct supertypes
    def ifaces = List.MapFromArray (original.GetInterfaces (), fun (ty) {
      lib.TypeOfType (tenv, ty) :> MType.Class
    });
    def dst = 
      match (original.BaseType) {
        | null => ifaces
        | t => (lib.TypeOfType (tenv, t) :> MType.Class) :: ifaces
      }

    // compute all supertypes
    def supertypes = Hashtable (30);
    def add_bt (t : MType.Class) {
      def tc = t.tycon;
      unless (supertypes.Contains (tc.GetId ()))
      {
        supertypes.Add (tc.GetId (), t);
        match (tc.BaseType) {
          | null => ()          
          | tc => add_bt (MType.Class (tc, []))
        }
      }
    };
    List.Iter (dst, add_bt);
    def st = supertypes.Fold ([], fun (_, tc, acc) { tc :: acc });
      
//    Message.Debug ($"reflecting $original, created dts $dst");
    (tyvars, tenv, dst, st)
  }

  public FixupFunctionHeader (lib : LibraryReference,
                              tenv : SystemMap [int, StaticTyVar], 
                              fh : Fun_header, 
                              [Nemerle.Assertions.NotNull] tokens : string) : void
  {
    match (ParseTerm (tokens)) {
      | Term.App (".m", [tp, ty]) =>
        def (tyvars, tenv) = decode_typarms (lib, tenv, tp);
        fh.typarms = tyvars;
        match (decode (lib, tenv, ty)) {
          | MType.Fun (arg, rt) =>
            fh.ret_type = rt;
            def arg_types = arg.Fix ().GetFunctionArguments ();
            def set_type (p : Fun_parm, t) {
              p.ty = t;
            };
            if (fh.parms.Length == 1 && arg_types.Length > 1)
              set_type (List.Hd (fh.parms), arg)
            else
              List.Iter2 (fh.parms, arg_types, set_type);
          | _ => Util.ice ("encoded method type aint ->")
        }
      | _ => Util.ice ("evil encoded method header " + tokens)
    }
  }


  // -------------- ENCODING ----------------------

  FlattenTerm (t : Term) : string
  {
    def ret = System.Text.StringBuilder ();
    def walk (t) {
      | Term.App (name, args) =>
        ignore (ret.Append (name));
        ignore (ret.Append ('('));
        List.Iter (args, walk);
        ignore (ret.Append (')'));
    };
    walk (t);
    ret.ToString ()
  }

  needs_encoding (t : TyVar) : bool
  {
    match (t.Fix ()) {
      | MType.Ref (t)
      | MType.Out (t)
      | MType.Array (t, _) => needs_encoding (t)
      | MType.Void 
      | MType.Class (_, []) => false
#if _GENERICS
      | MType.Class 
      | MType.TyVarRef => false
#else
      | MType.TyVarRef
      | MType.Class         
#endif        
      | MType.Fun | MType.Tuple => true
      | MType.Intersection => assert (false)
    }
  } 


  encode_tvs (t : list [TyVar]) : list [Term]
  {
    List.Map (t, encode_tv)
  }
  
  encode_tv (t : TyVar) : Term
  {
    encode (t.Fix ())
  }
  
  encode (t : MType) : Term
  {
    | MType.Array (t, rank) =>
      Term.App (".a", [Term.App (rank.ToString (), []), encode_tv (t)])
    | MType.TyVarRef (tv) =>
      Term.App (".v", [Term.App (tv.Id.ToString (), [])])
    | MType.Void =>
      Term.App ("System.Void", [])
    | MType.Fun (arg, rt) =>
      Term.App (".f", [encode_tv (arg), encode_tv (rt)])
    | MType.Tuple (args) =>
      Term.App (".p", encode_tvs (args))
    | MType.Class (ti, args) =>
      Term.App (ti.FrameworkTypeName, encode_tvs (args))
    | MType.Ref (t) =>
      Term.App (".r", [encode_tv (t)])
    | MType.Out (t) =>
      Term.App (".o", [encode_tv (t)])
    | MType.Intersection => assert (false)
  }
    
  /**
   * Used to emit Nemerle types in assembly metadata
   *
   * <remarks>
   *   The type tree is converted to prefix, term-like notation.
   * </remarks>
   */
  public EncodeType (t : MType) : string
  {
    FlattenTerm (encode (t))
  }

  encode_typarms (tyvars : list [StaticTyVar]) : Term
  {
    def encode_tyvar (tv : StaticTyVar) {
      Term.App ("", [Term.App (tv.Id.ToString (), []),
                Term.App (tv.Name, []),
                Term.App ("", List.Map (tv.Constraints, encode))])
    };
    
    Term.App (".tp", List.Map (tyvars, encode_tyvar))
  }


  encode_tyinfo (ti : TypeBuilder) : Term
  {
    def generic_parms = ti.GetTyparms ();
    def direct_super = ti.GetDirectSuperTypes ();
    def super = ti.GetSuperTypes ();

#if _GENERICS    
    if (ti.IsNemerleSpecific
#else        
    if (ti.IsNemerleSpecific || !generic_parms.IsEmpty
#endif        
        || List.Exists (direct_super, needs_encoding)
        || List.Exists (super, needs_encoding))
      Term.App (".ti", [
        encode_typarms (generic_parms),
        Term.App ("", List.Map (direct_super, encode)),
        Term.App ("", List.Map (super, encode))
      ])
    else null
  }
 
  
  public EncodeMemberType (tb : TypeBuilder) : SRE.CustomAttributeBuilder
  {
    def coded = encode_tyinfo (tb);
    if (coded != null) {
      when (tb.DeclaringType != null)
        tb.DeclaringType.IsNemerleSpecific = true;
      HierarchyEmitter.make_nemerle_type_attr (FlattenTerm (coded))
    }
    else 
      null
  }

  public EncodeMemberType (m : NemerleMember) : SRE.CustomAttributeBuilder
  {
    def t = m.GetMemType ();
    if (needs_encoding (t)) {
      m.DeclaringType.IsNemerleSpecific = true;
      HierarchyEmitter.make_nemerle_type_attr (EncodeType (t)) 
    }
    else
      null
  }

  public EncodeMemberType (m : NemerleMethod) : SRE.CustomAttributeBuilder
  { 
    def generic_params = m.GetHeader ().typarms;
    def mem_type = m.GetMemType ();
    def need_enc = 
      match (mem_type) {
        | MType.Fun (MType.Tuple (parms), to) =>
          List.Exists (to :: parms, needs_encoding)
        | MType.Fun (from, to) =>
          needs_encoding (from) || needs_encoding (to)
        | _ => Util.ice ("method of non -> type")
      }
#if _GENERICS
    if (need_enc) {
#else
    if (!generic_params.IsEmpty || need_enc) {
#endif      
      m.DeclaringType.IsNemerleSpecific = true;
      def tp = encode_typarms (generic_params);
      def mt = encode (mem_type);
      def flat = FlattenTerm (Term.App (".m", [tp, mt]));
      HierarchyEmitter.make_nemerle_type_attr (flat)
    }
    else null
  }
}

} // ns
