/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.IO;
using Nemerle.Collections;

using Nemerle.Compiler.Typedtree;

using SR = System.Reflection;

namespace Nemerle.Compiler {

public module SystemType 
{
  public mutable Array : System.Type;
  public mutable Boolean : System.Type;
  public mutable Byte : System.Type;
  public mutable Char : System.Type;
  public mutable Decimal : System.Type;
  public mutable Double : System.Type;
  public mutable Enum : System.Type;
  public mutable FlagsAttribute : System.Type;
  public mutable Int16 : System.Type;
  public mutable Int32 : System.Type;
  public mutable Int64 : System.Type;
  public mutable IntPtr : System.Type;
  public mutable Delegate : System.Type;
  public mutable MulticastDelegate : System.Type;
  public mutable Object : System.Type;
  public mutable Reflection_AssemblyConfigurationAttribute : System.Type;
  public mutable Reflection_FieldInfo : System.Type;
  public mutable Reflection_PropertyInfo : System.Type;
  public mutable Reflection_DefaultMemberAttribute : System.Type;
  public mutable Runtime_CompilerServices_IsVolatile : System.Type;
  public mutable SByte : System.Type;
  public mutable Single : System.Type;
  public mutable String : System.Type;
  public mutable Type : System.Type;
  public mutable UInt16 : System.Type;
  public mutable UInt32 : System.Type;
  public mutable UInt64 : System.Type;
  public mutable Void : System.Type;
  public mutable ParamArrayAttribute : System.Type;

  public mutable Decimal_ctors : Hashtable [string, SR.ConstructorInfo];
  public mutable Type_GetTypeFromHandle : SR.MethodInfo;
  public mutable AssemblyBuilder_EmbedResourceFile : SR.MethodInfo; 
  public mutable String_opEquality : SR.MethodInfo;
  public mutable String_opInequality : SR.MethodInfo;
  public mutable Decimal_opEquality : SR.MethodInfo;
  public mutable String_Concat : SR.MethodInfo;
  public mutable ObjectCtor : SR.ConstructorInfo;
  public mutable Delegate_Combine : SR.MethodInfo;
  public mutable Delegate_Remove : SR.MethodInfo;
  
  public NullMatchException : System.Type
  {
    get { InternalType.NullMatchException_tc.GetSystemType () }
  }
  
  public ContainsMacroAttribute : System.Type
  {
    get { InternalType.ContainsMacroAttribute_tc.GetSystemType () }
  }
  
  public TypeAttribute : System.Type
  {
    get { InternalType.TypeAttribute_tc.GetSystemType () }
  }
  
  public VariantAttribute : System.Type
  {
    get { InternalType.VariantAttribute_tc.GetSystemType () }
  }
  
  public VariantOptionAttribute : System.Type
  {
    get { InternalType.VariantOptionAttribute_tc.GetSystemType () }
  }

  public VolatileModifier : System.Type
  {
    get { InternalType.VolatileModifier_tc.GetSystemType () }
  }
  
  public ConstantVariantOptionAttribute : System.Type
  {
    get { InternalType.ConstantVariantOptionAttribute_tc.GetSystemType () }
  }
  
  public TypeAliasAttribute : System.Type
  {
    get { InternalType.TypeAliasAttribute_tc.GetSystemType () }
  }

  /**
   * Reflects a type using GlobalEnv
   */
  public Reflect (type_name : string) : System.Type
  {
    match (NamespaceTree.LookupSystemType (type_name)) {
      | Some (t) => t
      | _ =>
        Util.ice ("cannot reflect `" + type_name + "'")
    }
  }

  internal Init () : void
  {
    Array = Reflect ("System.Array");
    Boolean = Reflect ("System.Boolean");
    Byte = Reflect ("System.Byte");
    Char = Reflect ("System.Char");
    Decimal = Reflect ("System.Decimal");
    Double = Reflect ("System.Double");
    Enum = Reflect ("System.Enum");
    FlagsAttribute = Reflect ("System.FlagsAttribute");
    Int16 = Reflect ("System.Int16");
    Int32 = Reflect ("System.Int32");
    Int64 = Reflect ("System.Int64");
    IntPtr = Reflect ("System.IntPtr");
    Delegate = Reflect ("System.Delegate");
    MulticastDelegate = Reflect ("System.MulticastDelegate");
    Object = Reflect ("System.Object");
    Reflection_FieldInfo = Reflect ("System.Reflection.FieldInfo");
    Reflection_PropertyInfo = Reflect ("System.Reflection.PropertyInfo");
    Reflection_AssemblyConfigurationAttribute = Reflect ("System.Reflection.AssemblyConfigurationAttribute");
    Runtime_CompilerServices_IsVolatile = Reflect ("System.Runtime.CompilerServices.IsVolatile");
    SByte = Reflect ("System.SByte");
    Single = Reflect ("System.Single");
    String = Reflect ("System.String");
    SystemType.Type = Reflect ("System.Type");
    UInt16 = Reflect ("System.UInt16");
    UInt32 = Reflect ("System.UInt32");
    UInt64 = Reflect ("System.UInt64");
    Void = Reflect ("System.Void");
    ParamArrayAttribute = Reflect ("System.ParamArrayAttribute");
    Reflection_DefaultMemberAttribute = Reflect ("System.Reflection.DefaultMemberAttribute");

    {
      Decimal_ctors = Hashtable ();
      def decimal_ctors = SystemType.Decimal.GetConstructors ();

      foreach (decimal_ctor : SR.ConstructorInfo in decimal_ctors) {
        def parameters = decimal_ctor.GetParameters ();

        when (parameters.Length == 1) {
          def parameter_ty = parameters [0].ParameterType;
          Decimal_ctors [parameter_ty.Name] = decimal_ctor
        }
      }

      assert (Decimal_ctors.Count >=  7)
    }
    
    Type_GetTypeFromHandle = SystemType.Type.GetMethod ("GetTypeFromHandle");
    AssemblyBuilder_EmbedResourceFile = Reflect ("System.Reflection.Emit.AssemblyBuilder")
      .GetMethod ("EmbedResourceFile", SR.BindingFlags.Instance %| SR.BindingFlags.Public
                   %| BindingFlags.NonPublic, null, SR.CallingConventions.Any,
                   array [String, String], null);
    String_opEquality = SystemType.String.GetMethod ("op_Equality");
    String_opInequality = SystemType.String.GetMethod ("op_Inequality");                     
    String_Concat = String.GetMethod ("Concat", array [String, String]);
    Decimal_opEquality = SystemType.Decimal.GetMethod ("op_Equality");
    ObjectCtor = Object.GetConstructor (System.Type.EmptyTypes);
    assert (ObjectCtor != null);

    Delegate_Combine = Delegate.GetMethod ("Combine", array [Delegate, Delegate]);
    Delegate_Remove = Delegate.GetMethod ("Remove", array [Delegate, Delegate]);
  }
}

public class TupleType 
{
  internal tycon : TypeInfo;
  internal fields : array [IField];
  internal ctor : IMethod;

  public SystemType : System.Type
  {
    get { tycon.GetSystemType () }
  }

  public GetField (pos : int) : IField
  {
    fields [pos]
  }

  public Ctor : IMethod
  {
    get { ctor }
  }

  get_field (pos : int) : IField
  {
    match (tycon.LookupMember (sprintf ("field%d", pos))) {
      | [x] =>
        match (x.GetKind ()) {
          | MemberKind.Field (f) => f
          | _ => assert (false)
        }
      | _ => assert (false)
    }
  }

  internal this (size : int)
  {
    def name = ["Nemerle", "Internal", sprintf ("Tuple%d", size)];
    tycon = NamespaceTree.LookupInternalType (name);
    fields = array (size + 1);
    for (mutable i = 1; i <= size; ++i)
      fields [i] = get_field (i);
    match (tycon.LookupMember (".ctor")) {
      | [x] =>
        match (x.GetKind ()) {
          | MemberKind.Method (m) => ctor = m
          | _ => assert (false)
        }
      | xs => assert (false, xs.ToString ())
    }
  }
}

public class FunctionType {
  internal tycon : TypeInfo;
  internal apply : IMethod;

  public ApplyMethod : IMethod
  {
    get { apply }
  }

  public SystemType : System.Type
  {
    get { tycon.GetSystemType () }
  }

  public InternalType : MType
  {
    get { tycon.GetMemType () }
  }

  internal this (size : int)
  {
     def name = ["Nemerle", "Internal", sprintf ("Func%d", size)];
     tycon = NamespaceTree.LookupInternalType (name);
     foreach (meth :> IMethod in tycon.LookupMember ("apply")) {
       when (List.Length (meth.GetParameters ()) == size) {
         assert (apply == null);
         apply = meth
       }
     };
     assert (apply != null);
  }
}
  

public module InternalType
{
  public mutable Array_tc : TypeInfo;
  public mutable Attribute_tc : TypeInfo;
  public mutable Boolean_tc : TypeInfo;
  public mutable Byte_tc : TypeInfo;
  public mutable Char_tc : TypeInfo;
  public mutable Decimal_tc : TypeInfo;
  public mutable Delegate_tc : TypeInfo;
  public mutable Double_tc : TypeInfo;
  public mutable Enum_tc : TypeInfo;
  public mutable Exception_tc : TypeInfo;
  public mutable Int16_tc : TypeInfo;
  public mutable Int32_tc : TypeInfo;
  public mutable Int64_tc : TypeInfo;
  public mutable Object_tc : TypeInfo;
  public mutable SByte_tc : TypeInfo;
  public mutable Single_tc : TypeInfo;
  public mutable String_tc : TypeInfo;
  public mutable Type_tc : TypeInfo;
  public mutable UInt16_tc : TypeInfo;
  public mutable UInt32_tc : TypeInfo;
  public mutable UInt64_tc : TypeInfo;
  public mutable ValueType_tc : TypeInfo;
  public mutable MatchFailureException_tc : TypeInfo;
  public mutable NullMatchException_tc : TypeInfo;
  public mutable ContainsMacroAttribute_tc : TypeInfo;
  public mutable TypeAttribute_tc : TypeInfo;
  public mutable VariantAttribute_tc : TypeInfo;
  public mutable TypeAliasAttribute_tc : TypeInfo;
  public mutable VariantOptionAttribute_tc : TypeInfo;
  public mutable VolatileModifier_tc : TypeInfo;
  public mutable ConstantVariantOptionAttribute_tc : TypeInfo;
  public mutable FlagsAttribute_tc : TypeInfo;
  public mutable ParamArrayAttribute_tc : TypeInfo;
  public mutable AssemblyVersionAttribute_tc : TypeInfo;
  public mutable AssemblyKeyFileAttribute_tc : TypeInfo;
  public mutable AssemblyCultureAttribute_tc : TypeInfo;  
  public mutable Nemerle_list_tc : TypeInfo;
  public mutable IEnumerable_tc : TypeInfo;
  public mutable DllImport_tc : TypeInfo;
  public mutable Serializable_tc : TypeInfo;    

  mutable function_types : array [FunctionType];
  mutable tuple_types : array [TupleType];

  public mutable Void : MType.Void;
  public mutable Array : MType;
  public mutable Attribute : MType;
  public mutable Boolean : MType;
  public mutable Byte : MType.Class;
  public mutable Char : MType;
  public mutable Decimal : MType;
  public mutable Delegate : MType;
  public mutable Double : MType;
  public mutable Enum : MType.Class;
  public mutable Exception : MType;
  public mutable Int16 : MType.Class;
  public mutable Int32 : MType.Class;
  public mutable Int64 : MType.Class;
  public mutable Object : MType.Class;
  public mutable SByte : MType.Class;
  public mutable Single : MType;
  public mutable String : MType;
  public mutable Type : MType;
  public mutable UInt16 : MType.Class;
  public mutable UInt32 : MType.Class;
  public mutable UInt64 : MType.Class;
  public mutable ValueType : MType.Class;
  public mutable MatchFailureException : MType;
  public mutable IObjectReference : MType.Class;  

  public mutable Delegate_Combine : IMethod;
  public mutable Delegate_Remove : IMethod;
  public mutable String_Concat : IMethod;

  mutable f_MatchFailureException_ctor : IMethod;
  public NewMatchFailureException_ctor : IMethod
  {
    get {
      when (f_MatchFailureException_ctor == null)
        f_MatchFailureException_ctor = get_single_method (MatchFailureException_tc, ".ctor");
      f_MatchFailureException_ctor
    }
  }

  public MatchFailureException_ctor : SR.ConstructorInfo
  {
    get { NewMatchFailureException_ctor.GetConstructorInfo () }
  }
  
  mutable f_String_opEquality : IMethod;
  public String_opEquality : IMethod
  {
    get {
      when (f_String_opEquality == null)
        f_String_opEquality = get_single_method (String_tc, "op_Equality");
      f_String_opEquality
    }
  }

  mutable f_Decimal_opEquality : IMethod;
  public Decimal_opEquality : IMethod
  {
    get {
      when (f_Decimal_opEquality == null)
        f_Decimal_opEquality = get_single_method (Decimal_tc, "op_Equality");
      f_Decimal_opEquality
    }
  }

  get_single_method (tc : TypeInfo, name : string) : IMethod
  {
    match (tc.LookupMember (name)) {
      | [x is IMethod] => x
      | _ => Util.ice ()
    }
  }

  public GetFunctionType (len : int) : FunctionType
  {
    when (function_types [len] == null)
      function_types [len] = FunctionType (len);

    function_types [len]
  }

  public GetTupleType (len : int) : TupleType
  {
    when (tuple_types [len] == null)
      tuple_types [len] = TupleType (len);

    tuple_types [len]
  }


  lookup (type_name : string) : TypeInfo
  {
    match (NamespaceTree.LookupExactType (type_name)) {
      | Some (t) => t
      | None =>
        Util.ice ("internal type " + type_name + " not found")
    }
  }

    
  internal InitSystemTypes () : void
  {
    // ordering is important here
    Boolean_tc = lookup ("System.Boolean"); Boolean = MType.Class (Boolean_tc, []);
    Int32_tc = lookup ("System.Int32"); Int32 = MType.Class (Int32_tc, []);

    LibraryReference.add_buildins = true;
    (Boolean_tc :> LibraryReference.NetType).AddBuiltins ();
    (Int32_tc :> LibraryReference.NetType).AddBuiltins ();
    
    // and here not
    Void = MType.Void ();
    Array_tc = lookup ("System.Array"); Array = MType.Class (Array_tc, []);
    Attribute_tc = lookup ("System.Attribute"); Attribute = MType.Class (Attribute_tc, []);
    Byte_tc = lookup ("System.Byte"); Byte = MType.Class (Byte_tc, []);
    Char_tc = lookup ("System.Char"); Char = MType.Class (Char_tc, []);
    Decimal_tc = lookup ("System.Decimal"); Decimal = MType.Class (Decimal_tc, []);
    Delegate_tc = lookup ("System.Delegate"); Delegate = MType.Class (Delegate_tc, []);
    Double_tc = lookup ("System.Double"); Double = MType.Class (Double_tc, []);
    Enum_tc = lookup ("System.Enum"); Enum = MType.Class (Enum_tc, []);
    Exception_tc = lookup ("System.Exception"); Exception = MType.Class (Exception_tc, []);
    Int16_tc = lookup ("System.Int16"); Int16 = MType.Class (Int16_tc, []);
    Int64_tc = lookup ("System.Int64"); Int64 = MType.Class (Int64_tc, []);
    Object_tc = lookup ("System.Object"); Object = MType.Class (Object_tc, []);
    (Object_tc :> LibraryReference.NetType).AddBuiltins ();
    SByte_tc = lookup ("System.SByte"); SByte = MType.Class (SByte_tc, []);
    Single_tc = lookup ("System.Single"); Single = MType.Class (Single_tc, []);
    String_tc = lookup ("System.String"); String = MType.Class (String_tc, []);
    Type_tc = lookup ("System.Type"); InternalType.Type = MType.Class (Type_tc, []);
    UInt16_tc = lookup ("System.UInt16"); UInt16 = MType.Class (UInt16_tc, []);
    UInt32_tc = lookup ("System.UInt32"); UInt32 = MType.Class (UInt32_tc, []);
    UInt64_tc = lookup ("System.UInt64"); UInt64 = MType.Class (UInt64_tc, []);
    ValueType_tc = lookup ("System.ValueType"); ValueType = MType.Class (ValueType_tc, []);
    IEnumerable_tc = lookup ("System.Collections.IEnumerable");
    DllImport_tc = lookup ("System.Runtime.InteropServices.DllImportAttribute");
    Serializable_tc = lookup ("System.SerializableAttribute");    
    IObjectReference = MType.Class (lookup ("System.Runtime.Serialization.IObjectReference"), []);
    
    ParamArrayAttribute_tc = lookup ("System.ParamArrayAttribute");
    FlagsAttribute_tc = lookup ("System.FlagsAttribute");
    AssemblyVersionAttribute_tc = lookup ("System.Reflection.AssemblyVersionAttribute");
    AssemblyKeyFileAttribute_tc = lookup ("System.Reflection.AssemblyKeyFileAttribute");
    AssemblyCultureAttribute_tc = lookup ("System.Reflection.AssemblyCultureAttribute");

    def is_right (mem : IMember) {
      match (mem) {
        | meth is IMethod =>
          def parms = meth.GetParameters ();
          meth.IsStatic &&
          parms.Length == 2 &&
          ! List.Hd (parms).ty.Equals (Object)
        | _ => false
      }
    }

    def single (tc : TypeInfo, name) {
      match (List.Filter (tc.LookupMember (name), is_right)) {
        | [s] => s :> IMethod
        | _ => assert (false)
      }
    }

    Delegate_Combine = single (Delegate_tc, "Combine");
    Delegate_Remove = single (Delegate_tc, "Remove");
    String_Concat = single (String_tc, "Concat");
    
    function_types = array (20);
    tuple_types = array (20);
  }

  // to be called after scan_globals (think about compiling nemerle.dll)
  internal InitNemerleTypes () : void
 {
    InternalType.MatchFailureException_tc = lookup ("Nemerle.Core.MatchFailureException"); 
    InternalType.MatchFailureException = MType.Class (InternalType.MatchFailureException_tc, []);
    
    InternalType.NullMatchException_tc = lookup ("Nemerle.Core.NullMatchException"); 
    InternalType.ContainsMacroAttribute_tc = lookup ("Nemerle.Internal.ContainsMacroAttribute"); 
    InternalType.TypeAttribute_tc = lookup ("Nemerle.Internal.TypeAttribute"); 
    InternalType.VariantAttribute_tc = lookup ("Nemerle.Internal.VariantAttribute"); 
    InternalType.TypeAliasAttribute_tc = lookup ("Nemerle.Internal.TypeAliasAttribute"); 
    InternalType.VariantOptionAttribute_tc = lookup ("Nemerle.Internal.VariantOptionAttribute");
    InternalType.VolatileModifier_tc = lookup ("Nemerle.Internal.VolatileModifier");
    InternalType.ConstantVariantOptionAttribute_tc = lookup ("Nemerle.Internal.ConstantVariantOptionAttribute"); 
    InternalType.Nemerle_list_tc = lookup ("Nemerle.Core.list");
  }
}

} // end ns
