/*
 * Copyright (c) 2003-2005 University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Compiler
{
  using Nemerle.Compiler.Typedtree;
  using Nemerle.Collections;
  using Nemerle.Utility;

  using System.Reflection;
  using System.Reflection.Emit;
  using System.IO;
  using System.Diagnostics.SymbolStore;

  using NC = Nemerle.Compiler;
  using SR = System.Reflection;

  /**
   * Generator of System.Reflection.Emit structures composing program hierarchy.
   */
  class HierarchyEmitter
  {
    /* -- PUBLIC CONSTRUCTORS ---------------------------------------------- */

    public this (assem_name : AssemblyName)
    {
      _OutputFileName = Options.OutputFileName;

      _assembly_name = assem_name;

      /* define a dynamic assembly */
      _assembly_name.Name = Path.GetFileNameWithoutExtension (_OutputFileName);

      def assembly_requirements =
        if (Options.CompileToMemory)
          AssemblyBuilderAccess.Run
        else
          AssemblyBuilderAccess.RunAndSave;
      
      this._assembly_builder =
        System.AppDomain.CurrentDomain.DefineDynamicAssembly
          (this._assembly_name,
           assembly_requirements,
           Path.GetDirectoryName(Path.GetFullPath (_OutputFileName)));

      when (_assembly_name.Name == "") Message.FatalError ("name of output assembly cannot be empty");

      /* create a dynamic module */
      this._module_builder =
        if (Options.CompileToMemory)
          // we cannot give output filename if we are compiling only to Run
          this._assembly_builder.DefineDynamicModule (_assembly_name.Name, Options.EmitDebug); 
        else
          this._assembly_builder.DefineDynamicModule (_assembly_name.Name,
                                                      Path.GetFileName (_OutputFileName),
                                                      Options.EmitDebug); 

      when (Options.EmitDebug) _debug_emit = _module_builder.GetSymWriter ();

      unless (Options.TargetIsLibrary) {
        _need_entry_point = true;
        _entry_point = None ();
      };

      _aux_types = [];
    }


    add_resources_to_assembly () : void 
    {
      def escape_resource (x : string) {
        def cp = x.IndexOf (',');
        if (cp != -1)
          (x.Substring (0, cp), x.Substring (cp + 1))
        else
          // change name from /bar/bar/file to bar.bar.file namespace  
          (x, x.Replace ('/','.').Replace ('\\', '.'));
      }

      def escape_linked_resource (x : string) {
        def cp = x.IndexOf (',');
        if (cp != -1)
          // change name from /bar/bar/file to bar.bar.file namespace          
          (x.Substring (0, cp), x.Substring (cp + 1))
        else
          (x, x);
      }

      /* we can embed resources only on mono or .NET 2.0 */
#if _NET_2_0
      foreach (element in Options.EmbeddedResources) {
        def (file, name) = escape_resource (element);
        try {
          def resource_builder = _module_builder.DefineResource (file, name);        
          if (File.Exists (file)) {
            using (fs = File.OpenRead (file)) {
              def res = array (fs.Length :> int);
              _ = fs.Read(res, 0, file.Length);
              resource_builder.AddResource (name, res);
            }
          }
          else
            Message.Error ("Could not find resource " + file);
        }
        catch {
          | e => Message.Error ("Could not embed resource: " + e.Message);
        }
      }
#else      
      if (SystemType.AssemblyBuilder_EmbedResourceFile != null)
        foreach (element in Options.EmbeddedResources) {
          def (file, name) = escape_resource (element);
          try {
            def margs = array [name : object, file : object];
            if (File.Exists (file))
                _ = SystemType.AssemblyBuilder_EmbedResourceFile
                        .Invoke (_assembly_builder, margs);
            else
              Message.Error ("Could not find resource " + file);
          }
          catch {
            | e => Message.Error ("Could not embed resource: " + e.Message);
          }
        }
      else
        unless (Options.EmbeddedResources.IsEmpty)
          Message.Error ("Cannot embed resource on this runtime... use mono or .NET 2.0");
#endif

      foreach (element in Options.LinkedResources) {
        def (file, name) = escape_linked_resource (element);
        try {
          _assembly_builder.AddResourceFile (name, file);
        }
        catch {
          | _ is FileNotFoundException =>
            Message.Error ("Could not find resource " + file);
          | e is System.ArgumentException =>
            Message.Error ("Could not link resource: " + e.Message);
        }
      }
    }

    /**
     *
     */
    public EmitAuxDecls () : void
    {
      compile_all_tyinfos (true)
    }


    /**
     *
     */
    public EmitDecls () : void
    {
      Passes.Solver.Enqueue (fun () {
        compile_all_tyinfos (false);
        foreach (x in AttributeCompiler.GetCompiledAssemblyAttributes ())
          _assembly_builder.SetCustomAttribute (x);
      })
    }

    /**
     * Returns generated assembly for runtime instantations of its types
     */
    public GeneratedAssembly : Assembly
    {
      get { _assembly_builder }
    }
    
    
    /**
     * Saves the constructed assembly to a file
     */
    public SaveAssembly () : void
    {
      add_resources_to_assembly ();

      // if there are some nemerle specific metadata encoded in attributes
      when (contains_nemerle_specifics) {
        def attr = make_string_attribute (SystemType.Reflection_AssemblyConfigurationAttribute,
                                          "ContainsNemerleTypes");
        this._assembly_builder.SetCustomAttribute (attr);
      }
          
      // set the entry point
      match ((_need_entry_point, _entry_point)) {
        | (true, Some (entry_point_method_info)) =>
          _assembly_builder.SetEntryPoint (entry_point_method_info,
                                           if (Options.TargetIsWinexe)
                                             PEFileKinds.WindowApplication
                                           else
                                             PEFileKinds.ConsoleApplication)
        | (true, None) =>
          Message.Error ("no suitable entry point (Main function) found")
        | _ => ()
      };

      // save the assembly
      try {
        _assembly_builder.Save (Path.GetFileName (_OutputFileName));
        //when (_debug_emit != null) _debug_emit.Close ();
      }
      catch {
        | e is System.UnauthorizedAccessException =>
          Message.Error ($"could not write to output file `$(this._OutputFileName)'"
                         " -- `$(e.Message)'")  
          
        | e is IOException =>
          Message.Error ($"could not write to output file `$(this._OutputFileName)'"
                         " -- `$(e.Message)'")  
      }
    }

    /* -- PRIVATE METHODS -------------------------------------------------- */

    /**
     *
     */
    internal static emit_implementation (ti : NC.TypeBuilder) : void
    {
      ti.IsFinalized = true;
      mutable skip_types = false;
      def emit_member (m : IMember) {
        def mods = m.GetModifiers ();
        unless (mods.IsEmpty) {
          def handle = m.GetHandle ();
          //Message.Debug (handle.ToString () + ": " + handle.GetType ().ToString ());
          mutable context = m.DeclaringType;
          def adder =
            match (m.GetKind ()) {
              | MemberKind.Method =>
                // FIXME: we should allow [return:] here, but lame S.R.E. doesn't allow
                // this, mono has an extension though
                match (handle) {
                  | handle is ConstructorBuilder =>
                    fun (target, attribute) {
                      if (target %&& System.AttributeTargets.Method) {
                        handle.SetCustomAttribute (attribute);
                        null
                      }
                      else "constructor " + m.ToString ()
                    }
                  | handle is MethodBuilder =>
                    fun (target, a) {
                      if (target %&& System.AttributeTargets.Method) {
                        handle.SetCustomAttribute (a);
                        null
                      }
                      else "method " + m.ToString ()
                    }

                  | _ => Util.ice ("not a constructor or method handler in MemberKind.Method")
                }
                
              | MemberKind.Property =>
                fun (target, a) {
                  if (target %&& System.AttributeTargets.Property) {
                    (handle :> PropertyBuilder).SetCustomAttribute (a);
                    null
                  }
                  else "property " + m.ToString ()
                }
              | MemberKind.Event =>
                fun (target, a) {
                  if (target %&& System.AttributeTargets.Event) {
                    (m :> NemerleEvent).event_builder.SetCustomAttribute (a);
                    null
                  }
                  else "event " + m.ToString ()
                }
              | MemberKind.Field  =>
                fun (target, a) {
                  if (target %&& System.AttributeTargets.Field) {
                    (handle :> FieldBuilder).SetCustomAttribute (a);
                    null
                  }
                  else "field " + m.ToString ()
                }
              | MemberKind.Type (tc) =>
                if (skip_types)
                  null
                else {
                  context = tc;
                  def handle = handle :> SR.Emit.TypeBuilder;
                  fun (target, a) {
                    def valid = target %&& System.AttributeTargets.Class ||
                                handle.IsEnum && target %&& System.AttributeTargets.Enum;
                    if (valid) {
                      handle.SetCustomAttribute (a);
                      null
                    }
                    else "type " + tc.ToString ()
                  }
                }
            };

          when (adder != null)
            mods.SaveCustomAttributes (context :> NC.TypeBuilder, adder)
        }

        // Message.Debug ($"emit: $m");

        match (m.GetKind ()) {
          | MemberKind.Field =>
            when ((m :> NemerleField).IsVolatile)
            {
              def volatile_attr =
                make_void_attribute (SystemType.VolatileModifier);

              (m.GetHandle () :> FieldBuilder).SetCustomAttribute (volatile_attr)
            }
          | MemberKind.Method (m) => CompileTypedMethod.CompileMethodImmediately ((m :> NemerleMethod))
          | MemberKind.Property (p) => compile_property (p)
          | MemberKind.Event (prop) =>
            def builder = (prop :> NemerleEvent).event_builder;
            builder.SetAddOnMethod (prop.GetAdder ().GetMethodInfo () :> MethodBuilder);
            builder.SetRemoveOnMethod (prop.GetRemover ().GetMethodInfo () :> MethodBuilder);

          | _ => ()
        }
      };

      emit_member (ti);
      skip_types = true;
      List.Iter (ti.GetDirectMembers (), emit_member);
      //Message.Debug ("finalizing " + ti.FullName);
      unless (Message.SeenError)
        ti.FinalizeType ();
      //Message.Debug ("finalized " + ti.FullName);
    }


    static compile_property (prop : IProperty) : void
    {
      def builder = prop.GetPropertyInfo () :> PropertyBuilder;
      def declaring = prop.DeclaringType : object;
      match (prop.GetGetter ()) {
        | null => ()
        | getter when getter.DeclaringType == declaring =>
          builder.SetGetMethod (getter.GetMethodInfo () :> MethodBuilder);
        | _ => ()    // emit getter only if it was defined in this type, not derived  
      }
      match (prop.GetSetter ()) {
        | null => ()
        | setter when setter.DeclaringType == declaring =>
          builder.SetSetMethod (setter.GetMethodInfo () :> MethodBuilder);
        | _ => ()    // emit setter only if it was defined in this type, not derived  
      }
    }


    finalize_aux_types () : void
    {
      unless (Message.SeenError)
        foreach (x : Emit.TypeBuilder in _aux_types)
          _ = x.CreateType ();
      _aux_types = [];
    }


    /**
     * - create S.R.E.TypeBuilders for entire hierarchy of program 
     * - add members to those TypeBuilders (only stubs for methods)
     * - emit bodies of methods
     * - finalize value types
     * - finalize all types
     */  
    compile_all_tyinfos (aux_phase : bool) : void
    {
      def allow_it (ti : NC.TypeBuilder) {
        !ti.IsFinalized &&
        is_aux_decl (ti) == aux_phase
      };
      def set_type_builder (ti : NC.TypeBuilder) {
        when (allow_it (ti)) {
          //Message.Debug ("make type builder for " + ti.FullName);
          ti.ConstructTypeBuilder ();
          when (ti.Attributes %&& NemerleAttributes.Macro) {
            def attr = make_string_attribute (SystemType.ContainsMacroAttribute,
                                              ti.GetTypeBuilder ().FullName);
            _assembly_builder.SetCustomAttribute (attr);
          }
        }
      };
      // create members' declarations in SRE.TypeBuilders
      def emit_decls (ti : NC.TypeBuilder) {
        when (allow_it (ti)) {
          //Message.Debug ("generating declarations " + ti.FullName);
          add_declarations (ti)
        }
      };
      def emit_impl (ti : NC.TypeBuilder) {
        when (allow_it (ti)) {
          //Message.Debug ("generating code for " + ti.FullName);
          Passes.MarkTypeBuilderCompiled ();
          emit_implementation (ti)
        }
      };

      this._emit_nemerle_attributes = !aux_phase;

      when (!aux_phase) _cgil_phase = 1;
      NC.TypeBuilder.Iter (set_type_builder);

      when (!aux_phase) _cgil_phase = 2;
      NC.TypeBuilder.Iter (emit_decls);

      when (!aux_phase) _cgil_phase = 3;

      // we first finalize value types, because MS.NET runtime requires so
      NC.TypeBuilder.IterConditionally (emit_impl, fun (x : NC.TypeBuilder) { x.IsValueType });

      // now we can finalize everything else
      NC.TypeBuilder.Iter (emit_impl);

      Message.MaybeBailout ();

      when (!aux_phase) _cgil_phase = 4;
      finalize_aux_types ();
    }

    internal MemberAdded (ti : NC.TypeBuilder, _mem : IMember) : void
    {
      // Message.Debug ($"ma: $ti -> $mem ");
      when (_cgil_phase >= 1)
        ti.ConstructTypeBuilder ();

      // when (_cgil_phase >= 2)
      //  add_declaration (ti, mem);
    }


    internal MaybeCompile (ti : NC.TypeBuilder, mem : IMember) : void
    {
      add_declaration (ti, mem);
      
      when (_cgil_phase >= 3)
        match (mem) { 
          | meth is NemerleMethod =>
            ti.DoBeforeFinalization (fun () {
              CompileTypedMethod.CompileMethodImmediately (meth)
            })
          | _ => {}
        }
    }


    /**
     * Check if declaration is auxiliary, used internally etc.
     */
    private static is_aux_decl (ti : NC.TypeBuilder) : bool
    {
      ti.FullName.StartsWith ("Nemerle.Internal.")
    }


    private static make_string_attribute (attr_type : System.Type, value : string) : CustomAttributeBuilder
    {
      def constructor_param_types = array [SystemType.String];
      def constructor_info = attr_type.GetConstructor (constructor_param_types);
      assert (constructor_info != null);
      def constructor_params = array [(value : object)];
      CustomAttributeBuilder (constructor_info, constructor_params)
    }


    private static make_void_attribute (attr_type : System.Type) : CustomAttributeBuilder
    {
      def constructor_info = attr_type.GetConstructor (System.Type.EmptyTypes);
      CustomAttributeBuilder (constructor_info, array [])
    }


    /**
     * Builds a Nemerle type attribute
     */
    internal static make_nemerle_type_attr (encoded_type : string) : CustomAttributeBuilder
    {
      make_string_attribute (SystemType.TypeAttribute, encoded_type)
    }


    /**
     * Builds a Nemerle variant attribute
     */
    private static make_nemerle_variant_attribute (decls : list [TypeInfo]) : CustomAttributeBuilder
    {
      def names = List.Map (decls, fun (decl : TypeInfo) { decl.FrameworkTypeName });
      make_string_attribute (SystemType.VariantAttribute, NString.Concat (",", names))
    }


    /**
     * Builds a Nemerle type alias
     */
    private static make_nemerle_type_alias_attribute (t : MType) : CustomAttributeBuilder
    {
      make_string_attribute (SystemType.TypeAliasAttribute, TyCodec.EncodeType (t))
    }


    /**
     * Builds a Nemerle variant option attribute
     */
    private static make_nemerle_variant_option_attribute (is_const : bool) : CustomAttributeBuilder
    {
      make_void_attribute (if (is_const)
                             SystemType.ConstantVariantOptionAttribute
                           else
                             SystemType.VariantOptionAttribute)
    }


    /**
     * Converts Nemerle property modifiers to the CLI property attributes.
     */
    private static make_property_attributes (attrs : NemerleAttributes) : PropertyAttributes
    {
      mutable result = PropertyAttributes.None;

      when (attrs %&& NemerleAttributes.SpecialName) 
        result |= PropertyAttributes.SpecialName %| PropertyAttributes.RTSpecialName;

      result
    }


    /**
     * Converts Nemerle field modifiers to the CLI field attributes.
     */
    private static make_field_attributes (attrs : NemerleAttributes) : FieldAttributes
    {
      mutable result = FieldAttributes.PrivateScope;

      when (attrs %&& NemerleAttributes.Public) result |= FieldAttributes.Public;
      when (attrs %&& NemerleAttributes.Private) result |= FieldAttributes.Private;
      when (attrs %&& NemerleAttributes.Protected) result |= FieldAttributes.Family;
      when (attrs %&& NemerleAttributes.Internal) result |= FieldAttributes.FamORAssem;
      when (attrs %&& NemerleAttributes.Static) result |= FieldAttributes.Static;
      when (attrs %&& NemerleAttributes.SpecialName) result |= FieldAttributes.SpecialName %|
          FieldAttributes.RTSpecialName;

      result
    }


    /**
     * Converts Nemerle method modifiers to the CLI method attributes.
     */
    private static make_method_attributes (attrs : NemerleAttributes) : MethodAttributes
    {
      mutable result = MethodAttributes.HideBySig;

      when (attrs %&& NemerleAttributes.Public) result |= MethodAttributes.Public;
      when (attrs %&& NemerleAttributes.Private) result |= MethodAttributes.Private;
      if (attrs %&& NemerleAttributes.Protected)
        if (attrs %&& NemerleAttributes.Internal)
          result |= MethodAttributes.FamORAssem;
        else
          result |= MethodAttributes.Family;
      else
        when (attrs %&& NemerleAttributes.Internal)
          result |= MethodAttributes.Assembly;
      when (attrs %&& NemerleAttributes.Static) result |= MethodAttributes.Static;
      when (attrs %&& NemerleAttributes.Override)
        result |= MethodAttributes.Virtual %| MethodAttributes.ReuseSlot;
      // when method is static, then new is only for compile-time information
      when (!(attrs %&& NemerleAttributes.Static) && attrs %&& NemerleAttributes.New)
        result |= MethodAttributes.NewSlot | MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.Abstract)
        result |= MethodAttributes.Virtual %| MethodAttributes.Abstract;
      when (attrs %&& NemerleAttributes.Virtual) result |= MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.Sealed)
        result |= MethodAttributes.Final %| MethodAttributes.Virtual;
      when (attrs %&& NemerleAttributes.SpecialName) result |= MethodAttributes.SpecialName;

      result
    }


    /**
     * Converts Nemerle method modifiers to the Framework type attributes.
     */
    private static make_type_attributes (attrs : NemerleAttributes, is_nested : bool) : TypeAttributes
    {
      mutable result = TypeAttributes.AutoLayout;
      when (attrs %&& NemerleAttributes.Public)
        if (is_nested) result |= TypeAttributes.NestedPublic
        else result |= TypeAttributes.Public;
      when (attrs %&& NemerleAttributes.Private)
        if (is_nested) result |= TypeAttributes.NestedPrivate
        else Message.Error ("Private is not allowed for top level types");
      when (attrs %&& NemerleAttributes.Protected)
        if (is_nested) result |= TypeAttributes.NestedFamily
        else Message.Error ("Protected is not allowed for top level types");
      when (attrs %&& NemerleAttributes.Internal && is_nested)
        result |= TypeAttributes.NestedFamORAssem;
      when (attrs %&& NemerleAttributes.Abstract) result |= TypeAttributes.Abstract;
      when (attrs %&& NemerleAttributes.Sealed) result |= TypeAttributes.Sealed;
      when (attrs %&& NemerleAttributes.SpecialName) result |= TypeAttributes.SpecialName;
      // we cannot do Sealed a the same time on MS.NET 1.1
      when (attrs %&& NemerleAttributes.Static) result |= TypeAttributes.Abstract;
        
      result
    }


    /**
     * Converts a System.Type list to a System.Type array.
     * FIXME: array [x] won't work here as the list's length is not know.
     */
    internal static type_list_to_type_array (x : list [System.Type]) : array [System.Type]
    {
      Nemerle.Utility.NArray.FromList (typeof (System.Type), x)
    }


    /**
     * Reads the inheritance list of a class and retrieves the base
     * type and builds the array of interfaces implemented by the class.
     */
    private static determine_inheritance (ti : NC.TypeBuilder) : (System.Type * array [System.Type])
    {
      match (ti.GetTydecl ()) {
        | TypeDeclaration.Enum => (SystemType.Enum, array [])
        | _ =>
#if _GENERICS
          def supertypes = ti.GetDirectSuperTypes ();
          match (ti.SuperClass ()) {
            | Some =>
              (List.Head (supertypes).GetSystemType (),
               type_list_to_type_array (List.RevMap (List.Tail (supertypes),
                                                     fun (tc) { tc.GetSystemType () })))
            | None =>
              (null, type_list_to_type_array (List.RevMap (supertypes, fun (tc) { tc.GetSystemType () })))
          }
#else
          def base_class_type =
            match (ti.SuperClass ()) {
              | Some (x) => x.GetSystemType ()
              | _ => null
            };

          def ifaces_types = List.RevMap (ti.InterfacesToImplement (),
                                          fun (tc : TypeInfo) { tc.GetSystemType () });

          (base_class_type, type_list_to_type_array (ifaces_types))
#endif          
      }
    }


    /**
     * This method makes a skeleton of a class -- the class partial type and
     * partial types for the nested classes are created. The parent class and
     * the interfaces being implemented are assumed to have been processed
     * before constructing a dependant class. All the declarations and method
     * implementations will be created successively in subsequent passes.
     */
    public ConstructTypeBuilder (tyinfo : NC.TypeBuilder) : Emit.TypeBuilder
    {
      /* create optional custom attributes for this type */
      def custom_attribute =
        match (tyinfo.GetTydecl ()) {
          | TypeDeclaration.Variant (decls) => make_nemerle_variant_attribute (decls)
          | TypeDeclaration.VariantOption =>
            def has_co_ctor = tyinfo.GetConstantObject () != null;
            make_nemerle_variant_option_attribute (has_co_ctor)

          | TypeDeclaration.Alias (t) => make_nemerle_type_alias_attribute (t)
          | _ => null
        };

      /* decide the new type attributes */
      def is_nested = tyinfo.DeclaringType != null;

      def type_kind_attrs =
        match (tyinfo.GetTydecl ()) {
          | TypeDeclaration.Variant 
          | TypeDeclaration.VariantOption 
          | TypeDeclaration.Class          => TypeAttributes.Class
          | TypeDeclaration.Alias // we pretend type alias is an interface
          | TypeDeclaration.Interface      => TypeAttributes.Interface %| TypeAttributes.Abstract
          | TypeDeclaration.Enum           => TypeAttributes.Sealed %| TypeAttributes.Class
        };

      def attrs = make_type_attributes (tyinfo.Attributes, is_nested) %| type_kind_attrs;

#if _GENERICS
      /* create the type builder for a top-level or nested class declaration */
      def new_type_builder =
        if (!is_nested)
          this._module_builder.DefineType (tyinfo.FullName, attrs)
        else {
          def type_builder = (tyinfo.DeclaringType :> NC.TypeBuilder).GetTypeBuilder ();
          type_builder.DefineNestedType (tyinfo.Name, attrs)
        };

      // creates and store generic parameters in our StaticTyVars
      def typarms = tyinfo.GetTyparms ();
      unless (typarms.IsEmpty) {
        def names = array (typarms.Length);
        _ = List.FoldLeft (typarms, 0, fun (x, idx) { 
          names [idx] = x.Name; 
          idx + 1 
        });
        def generic_parms = new_type_builder.DefineGenericParameters (names);
        _ = List.FoldLeft (typarms, 0, fun (x, idx) { 
          x.SetGenericBuilder (generic_parms [idx]); 
          idx + 1 
        });
      }
        
      /* determine the inheritance relation for this type
         generic builders are now properly stored */
      def (extends_class, implements_interfaces) = determine_inheritance (tyinfo);

      // save inheritance information in our type builder
      new_type_builder.SetParent (extends_class);
      foreach (inter in implements_interfaces)
        new_type_builder.AddInterfaceImplementation (inter);

#else
      /* determine the inheritance relation for this type */
      def (extends_class, implements_interfaces) = determine_inheritance (tyinfo);

      /* create the type builder for a top-level or nested class declaration */
      def new_type_builder =
        if (!is_nested)
          this._module_builder.DefineType (tyinfo.FullName, attrs,
                                           extends_class, implements_interfaces)
        else {
          def type_builder = (tyinfo.DeclaringType :> NC.TypeBuilder).GetTypeBuilder ();
          type_builder.DefineNestedType (tyinfo.Name, attrs, extends_class,
                                         implements_interfaces)
        };

#endif

#if _GENERICS
#endif

      when (custom_attribute != null && _emit_nemerle_attributes) {
        tyinfo.IsNemerleSpecific = true;
        contains_nemerle_specifics = true;
        new_type_builder.SetCustomAttribute (custom_attribute);
      };

      // Structs with no fields need to have at least one byte.
      // The right thing would be to set the PackingSize in a DefineType
      // but there are no functions that allow interfaces *and* the size to
      // be specified.
      // maybe in 2.0 there is a better API
      when (tyinfo.IsStruct && tyinfo.GetFields (BindingFlags.Instance %|
                                                 BindingFlags.Public %| 
                                                 BindingFlags.NonPublic).IsEmpty)
        _ = new_type_builder.DefineField ("$PLACE_HOLDER$", SystemType.Byte,
                                          FieldAttributes.Private %| FieldAttributes.SpecialName);

      new_type_builder
    }


    /**
     * Adds a constructor builder to a type builder
     */
    internal static AddConstructorBuilder (meth : NemerleMethod, tb : Emit.TypeBuilder,
                                    is_delegate_ctor : bool) : void
    {
      // Message.Debug ("-- AddConstructorBuilder: $meth");

      def hd = meth.GetHeader ();

      /* build the parameter types array */
      def parm_types_list = hd.parms.Map (fun (parm) { parm.SystemType });
      def parm_types_array = type_list_to_type_array (parm_types_list);

      mutable skip_it_altogether = false;

      when (is_delegate_ctor) {
        if (meth.Attributes %&& NemerleAttributes.Public)
          skip_it_altogether = true;
        else
          meth.Attributes = NemerleAttributes.Public;
      }

      unless (skip_it_altogether) {
        /* create the constructor builder */
        def constructor_builder =
          tb.DefineConstructor (make_method_attributes (meth.Attributes) %|
                                MethodAttributes.RTSpecialName %|
                                MethodAttributes.SpecialName,
                                CallingConventions.Standard, parm_types_array);

        /* add the runtime modifiers for delegate constructors */
        when (is_delegate_ctor) {
          constructor_builder.SetImplementationFlags (
            MethodImplAttributes.Runtime %| MethodImplAttributes.Managed
          )
        }

        // CompileTypedMethod.compile_parms (hd);
      
        def name_parms (pos, parms) {
          match (parms) {
            | [] => ()
            | (p : Fun_parm) :: ps =>
              p.builder =
                 constructor_builder.DefineParameter
                       (pos, parameter_attributes (p), p.name);
              match (p.default_value) {
                | Some (lit) =>
                  p.builder.SetConstant (lit.AsObject ());
                | None => {}
              }
              name_parms (pos + 1, ps)
          }
        };
        name_parms (1, hd.parms);

        meth.ctor_builder = constructor_builder;
      }
    }


    /**
     * Emits a constructor's body
     */
    internal EmitConstructor (context : NC.TypeBuilder, meth : NemerleMethod) : void
    {
      unless (context.IsDelegate) {
        def cb = meth.ctor_builder;
        def hd = meth.GetHeader ();
        assign_parameter_indexes (cb.IsStatic, hd);

        emit_parameter_attributes (context, hd);

        //CGpost.Postprocess (hd);

        // Message.Debug ($"il builder for : $(hd.name) from $context");

        /* emit the constructor's body */
        def gen = ILEmitter (cb, _module_builder, hd);
        when (Options.EmitDebug) gen.SetDocument (context.Location);
        gen.Run ();
      }
    }

    static parameter_attributes (fp : Fun_parm) : ParameterAttributes
    {
      (match (fp.kind) {
        | ParmKind.Out => ParameterAttributes.Out
        | ParmKind.Ref
        | ParmKind.Normal => ParameterAttributes.None
      }) 
      | 
      (if (fp.default_value.IsSome)
         ParameterAttributes.HasDefault | ParameterAttributes.Optional
       else
         ParameterAttributes.None)
    }

    /**
     * Adds a method builder to a type builder
     */
    internal static AddMethodBuilder (meth : NemerleMethod, tb : Emit.TypeBuilder, is_delegate_method : bool) : void
    {
      assert (meth != null);
      // Message.Debug ($"-- AddMethodBuilder: $meth $(meth.Attributes) ");

      def hd = meth.GetHeader ();

      /* build the parameter types array */
      def parm_types_list = hd.parms.Map (fun (parm) { parm.SystemType });
      def parm_types_array = type_list_to_type_array (parm_types_list);

      /* update the modifiers list */
      match (meth.GetFunKind ()) {
        | FunKind.BoundMethod (_ :: _) =>
          meth.Attributes |= NemerleAttributes.Virtual;
        | _ => {}
      }
      
      def attrs = make_method_attributes (meth.Attributes);
      def attrs =
        if (meth.Name == ".cctor")
          attrs %| MethodAttributes.RTSpecialName %| MethodAttributes.SpecialName
        else
          attrs;

      def pinvoke_meth =
        if (meth != null)
          AttributeCompiler.CheckPInvoking (meth, tb, attrs, parm_types_array);
        else
          null;
          
      /* add the method to the type builder */
#if _GENERICS
      def method_builder =
        tb.DefineMethod (meth.Name, attrs,
                         meth.ReturnType.Fix ().GetSystemType (),
                         parm_types_array);
#else
      def method_builder =
        if (pinvoke_meth != null) pinvoke_meth
        else
          tb.DefineMethod (meth.Name, attrs,
                           meth.ReturnType.Fix ().GetSystemType (),
                           parm_types_array);
#endif

      /* add the runtime modifiers for delegate methods */
      when (is_delegate_method) {
        assert (pinvoke_meth == null);
        method_builder.SetImplementationFlags (
          MethodImplAttributes.Runtime | MethodImplAttributes.Managed
        )
      }

      when (pinvoke_meth == null && (meth.Attributes & NemerleAttributes.Extern != 0))
        Message.Error (meth.Location, "method is marked `extern' but do not have DllImport attribute");

      // CompileTypedMethod.compile_parms (hd);
      
      def name_parms (pos, parms) {
        match (parms) {
          | [] => ()
          | (p : Fun_parm) :: ps =>
            p.builder =
               method_builder.DefineParameter
                     (pos, parameter_attributes (p), p.name);
            match (p.default_value) {
              | Some (lit) =>
                p.builder.SetConstant (lit.AsObject ());
              | None => {}
            }
            name_parms (pos + 1, ps)
        }
      };
      name_parms (1, hd.parms);

      meth.method_builder = method_builder;
    }


    static emit_parameter_attributes (context : NC.TypeBuilder, hd : Fun_header) : void
    {
      foreach (parm in hd.parms) {
        parm.modifiers.SaveCustomAttributes (context, fun (target, attr) {
          if (target %&& System.AttributeTargets.Parameter) {
            parm.builder.SetCustomAttribute (attr);
            null
          }
          else "parameter " + parm.name
        })
      }
    }


    /**
     * Emits a method's body
     */
    internal EmitMethod (context : NC.TypeBuilder, meth : NemerleMethod) : void
    {
      // make sure type is not finalized
      assert (context.GetTypeBuilder () != null);
      
      unless (context.IsDelegate) {
        def mb = meth.method_builder;
        def hd = meth.GetHeader ();
        Util.cassert (mb != null, $"method builder is null for $meth");
        assign_parameter_indexes (mb.IsStatic, hd);

        emit_parameter_attributes (context, hd);

        /* update the entry point settings, if necessary */
        when (_need_entry_point && meth.Name == "Main") {
          def has_correct_signature () {
            mb.IsStatic &&
            (meth.ReturnType.Equals (InternalType.Int32) || 
             meth.ReturnType.Equals (InternalType.Void)) &&
            match (hd.parms) {
              // workaround bug in MS PEVerify
              | [fp] =>
                fp.ty.Fix ().Equals (MType.Array (InternalType.String, 1))
              | [] => true
              | _ => false
            }
          }
          
          if (has_correct_signature ()) {
            match (_entry_point) {
              | Some =>
                Message.Error ($ "more then one entry point (Main function) "
                                 "found in $(mb.DeclaringType)")
              | None =>
                when (_debug_emit != null) _debug_emit.SetUserEntryPoint (SymbolToken (1));
                _entry_point = Some ((mb : MethodInfo))
            }
          }
          else
            Message.Warning (28, $"$meth has the wrong signature to be an entry point");
        };

        /* update the overriding relationships */
        def type_builder = mb.DeclaringType :> Emit.TypeBuilder;

        match (meth.GetFunKind ()) {
          | FunKind.BoundMethod (impl) =>
            foreach (overridden : IMethod in impl) {
              def overridden_info = overridden.GetMethodInfo ();

              // Message.Debug ("override " + method_info.Name + " " + overridden_info.Name + " in " +
              //                type_builder.FullName);
              type_builder.DefineMethodOverride (mb, overridden_info)
            }
          | _ => {}
        }

        /* emit the method body */
        when (_debug_emit != null) {
          //_debug_emit.SetMethodSourceRange (_debug_doc, 2, 3, _debug_doc, 7, 8);
          _debug_emit.OpenMethod (SymbolToken (1));
        }

        // Message.Debug ($"making il generator for $meth");
        def gen = ILEmitter (mb, _module_builder, hd);
        when (Options.EmitDebug) gen.SetDocument (context.Location);
        gen.Run ();

        when (_debug_emit != null) _debug_emit.CloseMethod ();        
      }
    }


    /**
     * Adds a field builder
     */
    internal AddFieldBuilder (field : NemerleField, tb : Emit.TypeBuilder) : FieldBuilder
    {
      mutable attrs = make_field_attributes (field.Attributes);
      when (field.IsLiteral)
        attrs |= FieldAttributes.Literal;

      // prevent verification failure
      when (field.Name != "value__")
        attrs &= ~FieldAttributes.RTSpecialName;

      def field_builder =
        tb.DefineField (field.Name, field.GetMemType ().GetSystemType (), attrs);

      when (_emit_nemerle_attributes) {
        def attr = TyCodec.EncodeMemberType (field);
        when (attr != null) field_builder.SetCustomAttribute (attr);
      };

      when (field.IsLiteral) {
        field_builder.SetConstant (field.const_value.AsObject ());
      };

      field.field_builder = field_builder;
      field_builder
    }

    /**
     * Adds a property builder
     */
    internal AddPropertyBuilder (prop : NemerleProperty, tb : Emit.TypeBuilder) : PropertyBuilder
    {
      def parms = List.Map (prop.parms, fun (t : MType) { t.GetSystemType () });
      def parms = type_list_to_type_array (parms);
      def attrs = make_property_attributes (prop.Attributes);
      def ty = prop.GetMemType ().GetSystemType ();

      def prop_builder = tb.DefineProperty (prop.Name, attrs, ty, parms);

      when (_emit_nemerle_attributes) {
        def attr = TyCodec.EncodeMemberType (prop);
        when (attr != null)
          prop_builder.SetCustomAttribute (attr);
      };

      prop.property_builder = prop_builder;
      prop_builder
    }

    /**
     * Adds an event builder
     */
    internal AddEventBuilder (the_event : NemerleEvent, tb : Emit.TypeBuilder) : EventBuilder
    {
      def ty = the_event.GetMemType ().GetSystemType ();
      def event_builder =
        tb.DefineEvent (the_event.Name, EventAttributes.None, ty);

      when (_emit_nemerle_attributes) {
        def attr = TyCodec.EncodeMemberType (the_event);
        when (attr != null)
          event_builder.SetCustomAttribute (attr);
      };

      the_event.event_builder = event_builder;
      event_builder
    }

    add_declaration (ti : NC.TypeBuilder, m : IMember) : void
    {
      def type_builder = ti.GetTypeBuilder ();
      match (m.GetKind ()) {
        | MemberKind.Field (field) =>
          ignore (AddFieldBuilder (field :> NemerleField, type_builder))

        | MemberKind.Event (ev) =>
          ignore (AddEventBuilder (ev :> NemerleEvent, type_builder))

        | MemberKind.Property (p) =>
          ignore (AddPropertyBuilder (p :> NemerleProperty, type_builder))

        | MemberKind.Method (m) =>
          def m = m :> NemerleMethod;
          def attr =
            if (_emit_nemerle_attributes)
              TyCodec.EncodeMemberType (m)
            else
              null;
          match (m.GetFunKind ()) {
            | FunKind.Constructor =>
              AddConstructorBuilder (m, type_builder, ti.IsDelegate);
              def builder = m.ctor_builder;
              // we don't need our atributes on delegate ctors
              when (!ti.IsDelegate && attr != null)
                builder.SetCustomAttribute (attr);

            | _ =>
              AddMethodBuilder (m, type_builder, ti.IsDelegate);
              def builder = m.method_builder;
              when (attr != null) builder.SetCustomAttribute (attr);
          }

        | MemberKind.Type => () // skip
      }
    }


    /**
     * This methods walks the class tree and adds field and method declarations.
     * The method implementation will be filled in a separate pass.
     */
    internal add_declarations (ti : NC.TypeBuilder) : void
    {
      foreach (m in ti.GetDirectMembers ()) add_declaration (ti, m);

      when (_emit_nemerle_attributes) {
        def tyattr = TyCodec.EncodeMemberType (ti);
        when (tyattr != null) {
          contains_nemerle_specifics = true;
          ti.GetTypeBuilder ().SetCustomAttribute (tyattr);
        }
      };

      when (ti.IsDelegate) {
        // a lil' fixup
        match (ti.LookupMember (".ctor")) {
          | [c1 is NemerleMethod, c2 is NemerleMethod] =>
            if (c1.ctor_builder == null)
              c1.ctor_builder = c2.ctor_builder
            else
              c2.ctor_builder = c1.ctor_builder
          | _ => assert (false)
        }
      }
    }


    /**
     * Assigns an ordinal index to each of a method's parameter
     */
    private static assign_parameter_indexes (is_static : bool, hd : Fun_header) : void
    {
      mutable index = if (is_static) 0 else 1;
      foreach (parm in hd.parms) {
        parm.decl.SetParmIndex (index,
                                is_by_ref = parm.kind != ParmKind.Normal);
        index++;
      }
    }


    /* -- PRIVATE FIELDS --------------------------------------------------- */

    private _assembly_name : System.Reflection.AssemblyName;
    private _assembly_builder : AssemblyBuilder;
    internal _module_builder : ModuleBuilder;
    public static mutable _debug_emit : ISymbolWriter;
    private mutable _emit_nemerle_attributes : bool;
    private mutable contains_nemerle_specifics : bool = false;

    private _need_entry_point : bool;
    /** updated when method with static Main signature is met */
    private mutable _entry_point : option [MethodInfo];

    private _OutputFileName : string;
    private mutable _aux_types : list [Emit.TypeBuilder];
    private mutable _cgil_phase : int;
  }
}
