/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.IO;

using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;

using System.Xml;
using System.Text;

namespace Nemerle.Compiler 
{
  public module XmlDoc {
    mutable document : XmlDocument;
    mutable docNode : XmlNode;
    mutable last_loc : Location;

    public mutable OutputFileName : string;

    public Init () : void {
      document = XmlDocument ();
      _ = document.AppendChild (document.CreateXmlDeclaration ("1.0", null, null));
      def d = document.AppendChild (document.CreateElement ("doc"));
      def name = d.AppendChild (document.CreateElement ("assembly")).
        AppendChild (document.CreateElement ("name"));
      _ = name.AppendChild (document.CreateTextNode (Options.OutputFileName));
      docNode = d.AppendChild (d.AppendChild (document.CreateElement ("members")));
    }

    public Save () : void {
      document.Save (OutputFileName);
    }

    CreateMember (name : string) : XmlNode {
      def mem = document.CreateElement ("member");
      def attr = document.CreateAttribute("name");
      attr.Value = name;
      _ = mem.SetAttributeNode(attr);
      mem
    }

    DumpMember (m : IMember) : void {
      def full_name = m.DeclaringType.FullName + "." + m.Name;
      def (prefix, suffix) = 
        match (m.GetKind ()) {
          | MemberKind.Field => ("F:", "")
          | MemberKind.Method (method) =>
            def parms = List.Map (method.GetParameters (),
                                  fun (p : Typedtree.Fun_parm) {
                                    p.ty.Fix ().GetSystemType ().FullName
                                  });
            if (!(parms is []))                                  
              ("M:", "(" + Nemerle.Utility.NString.Concat (",", parms) + ")")
            else 
              ("M:", "")

          | MemberKind.Property => ("P:", "")
          | MemberKind.Type (tycon) => DumpType (tycon); ("", "")
          | MemberKind.Event => ("E:", "")
        }
      unless (prefix == "" && suffix == "") {
        def member = docNode.AppendChild (CreateMember (prefix + full_name + suffix));
        add_comments (member, m);
      }
    }

    parse_comment (comment : string) : XmlNode
    {
      def lines = comment.Split (array ['\n']);
      mutable all_stars = true;
      def white_space = array [' ', '\t', '\r', '\n'];
      
      for (mutable i = 0; i < lines.Length; ++i) {
        lines [i] = lines [i].Trim (white_space);
        when (i > 0 && lines[i] != "" && !lines [i].StartsWith ("*"))
          all_stars = false;
      }

      when (all_stars)
        for (mutable i = 1; i < lines.Length; ++i)
          when (lines [i] != "")
            lines [i] = lines [i].Substring (1).Trim (white_space);

      def sb = StringBuilder ("<summary>");
      mutable summary = true;
      mutable remarks = false;
      mutable para = false;

      mutable i = 0;
      while (i < lines.Length && lines [i] == "")
        ++i;

      for (; i < lines.Length; ++i) {
        def line = lines [i];
        if (line == "") {
          if (summary) {
            _ = sb.Append ("</summary>");
            summary = false;
          } else if (para) {
            _ = sb.Append ("</para>");
            para = false;
          } else {}
        } else {
          when (!summary && !remarks) {
            _ = sb.Append ("<remarks>");
            remarks = true;
          }
/*          
          when (!summary && !para) {
            _ = sb.Append ("<para>");
            para = true;
          }
*/          
          _ = sb.Append (line).Append ('\n');
        }
      }

      when (para)
        _ = sb.Append ("</para>");
      when (summary)
        _ = sb.Append ("</summary>");
      when (remarks)
        _ = sb.Append ("</remarks>");

      def frag = document.CreateDocumentFragment ();
      frag.InnerXml = sb.ToString ();
      frag
    }

    add_comments (x : XmlNode, mem : IMember) : void
    {
      def curr = mem.Location;

      def choose (loc : Location, com, acc) {
        if (loc.File == curr.File && 
            loc.CompareTo (curr) <= 0 && (curr.CompareTo (last_loc) < 0 || loc.CompareTo (last_loc) > 0))
          com;
        else
          acc
      }

      def comment = PreParser.doc_comments.Fold (null, choose);
      when (comment != null && comment != "") {
        try {
          _ = x.AppendChild (parse_comment (comment));
        }
        catch {
          | e is XmlException => Message.Warning (curr, e.Message);       
        }
      }

      last_loc = curr;
    }

    public DumpType (t : TypeInfo) : void {
      def node = docNode.AppendChild (CreateMember ("T:" + t.FullName));
      add_comments (node, t);
      def mems = t.GetMembers (BindingFlags.Static %| BindingFlags.Instance %|
                               BindingFlags.Public %| BindingFlags.NonPublic %| 
                               BindingFlags.DeclaredOnly);
      List.Iter (mems, DumpMember);
    }
  }
} // end ns
