/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;
using Nemerle.Utility;

namespace Nemerle.Compiler.Parsetree
{
  [Record]
  public variant TopDeclaration : DeclarationBase
  {
    | Class {
      typarms : Typarms;
      mutable t_extends : list [PExpr];
      decls : list [ClassMember];
    }
    | Alias {
      typarms : Typarms;
      ty : PExpr;
    }
    | Interface {
      typarms : Typarms;
      mutable t_extends : list [PExpr];
      methods : list [ClassMember]; // only iface_member
    } 
    | Variant {
      typarms : Typarms;
      mutable t_extends : list [PExpr];
      mutable decls : list [ClassMember];
    }
    | VariantOption { decls : list [ClassMember]; }

    | Macro {
      header : Fun_header;
      synt : list [PExpr];
      expr : PExpr;
    }
    | Delegate { header : Fun_header; }
    | Enum {
      t_extends : list [PExpr];
      decls : list[ClassMember];
    }

    public this () { base (Splicable.Name (Name ("")), Modifiers.Empty) }
  }

  [Record]
  public variant ClassMember : DeclarationBase
  {
    | TypeDeclaration { td : TopDeclaration; }
    | Field { mutable ty : PExpr; }
    | Function { 
        header : Fun_header;
        kind : FunKind;
        mutable body : FunBody;
      }
    | Property {
        ty : PExpr;
        prop_ty : PExpr;
        dims : list [Fun_parm]; // parameters of indexer property
        get : option [ClassMember];
        set : option [ClassMember];
      }
    | Event {
        ty : PExpr;
        field : ClassMember.Field;
        add : ClassMember.Function;
        remove : ClassMember.Function;
      }
    | EnumOption { value : option [PExpr]; }

    /** accessible only for ClassMember.Function, when its FunBody is not typed or compiled */
    public Body : PExpr {
      get {
        match (this) {
          | ClassMember.Function (body = bd) =>
            match (bd) {
              | FunBody.Parsed (expr) => expr
              | FunBody.Typed =>
                Message.Error ("Body of typed method is not accessible");
                null
              | FunBody.ILed =>
                Message.Error ("Body of compiled method is not accessible");
                null
              | FunBody.Abstract => <[ () ]>
            }
          | _ =>
            Message.Error ("Body is accessible only for ClassMember.Function variant option");
            null
        }
      }
      set {
        match (this) {
          | ClassMember.Function as fn =>
            fn.body = FunBody.Parsed (value);
          | _ =>
            Message.Error ("Body is accessible only for ClassMember.Function variant option")
        }
      }
    }
  }

  [Record]
  public class DeclarationBase : Located
  {
    public mutable name : Splicable;
    public mutable modifiers : Modifiers;

    public ParsedName : Name {
      get { name.GetName () }
    }
    
    public Name : string {
      get { name.GetName ().Id }
    }

    public Attributes : NemerleAttributes {
      get { modifiers.mods }
      set { modifiers.mods = value }
    }

    public GetCustomAttributes () : list [PExpr] {
      modifiers.custom_attrs
    }

    public AddCustomAttribute (e : PExpr) : void {
      modifiers.custom_attrs = e :: modifiers.custom_attrs
    }
  }

  [Record]
  public class Constraint
  {
    public tyvar : string * int;
    public ty : PExpr;
  }

  [Record]
  public class Typarms
  {
    public tyvars : list [string * int];
    public constraints : list [Constraint];

    public static Empty : Typarms = Typarms ([], []);
  }

  /** class encapsulating name of variable for purpose of
      quotation and renaming
   */    
  public class Name : Nemerle.IComparable [Name]
  {
    public idl : string;
    public color : int;
    public context : GlobalEnv;

    public this (id : string)
    {
      this.color = MacroColorizer.Color;
      this.context = null; // no global context
      idl = id;
    }

    public this (id : string, color : int, context : GlobalEnv)
    {
      this.color = color;
      this.context = context;
      idl = id;
    }

    static public NameInCurrentColor (id : string, context : GlobalEnv) : Name
    {
      Name (id, MacroColorizer.Color, context)
    }

    public NewName (id : string) : Name {
      Name (id, color, context);
    }
    
    /** Returns plain identifier string of this name.
     */
    public Id : string
    {
      get { idl }
    }

    public override ToString () : string
    {
      Id
    }

    [Nemerle.OverrideObjectEquals]
    public Equals (other : Name) : bool
    {
      this.CompareTo (other) == 0
    }

    public CompareTo (other : Name) : int
    {
      if (color == other.color || color < 0 || other.color < 0)
        if (idl == other.idl) 0
        else string.CompareOrdinal (idl, other.idl)
      else
        color - other.color
    }

    public GetEnv (default : GlobalEnv) : GlobalEnv
    {
//      when (context == null) Message.Debug ("null: " + Id);
      if (context != null)
        context
      else
        default
    }
  }

  [Record]
  public variant PExpr : Located 
  {
    | Wildcard        // `_' used mainly in patterns, but also in `_ = ignored'
    | Void            // `void' used only in types
    | As              { pat : PExpr; name : Splicable; }
    | Is              { pat : PExpr; ty : PExpr; }
    | Where           { name : PExpr; fields : PExpr; }
    | Match           { expr : PExpr; cases : list [MatchCase]; }

    | Ref             { name : Name; }
    | Member          { obj : PExpr; member : Splicable; }
    | Call            { func : PExpr; parms : list [PExpr]; }
    | GenericSpecifier { func : PExpr; generic_parms : list [PExpr]; }
    | ListLiteral     { elements : list [PExpr]; }
    | Assign          { target : PExpr; source : PExpr; }
    | DefMutable      { name : PExpr; val : PExpr; }
    | Define          { pattern : PExpr; val : PExpr; }            
    | DefFunctions    { funs : list [Function_decl]; }
    | Lambda          { decl : Function_decl; }
    | Throw           { exn : PExpr; }
    | TryWith         { body : PExpr; exn : Splicable; exn_ty : PExpr; 
                        handler : PExpr; }
    | TryFinally      { body : PExpr; handler : PExpr; }
    | Literal         { val : Nemerle.Compiler.Literal; }
    | This
    | Base
    | Typeof          { ty : PExpr; }
    | TypeConversion  { expr : PExpr; ty : PExpr; }  // (expr :> ty)
    | TypeEnforcement { expr : PExpr; ty : PExpr; } // (expr : ty)
    | Sequence        { body : list [PExpr]; }
    | Tuple           { args : list [PExpr]; }
    | Array           { rank : PExpr; args : PExpr; }
    | EmptyArray      { sizes : list [PExpr]; }
    | Indexer         { obj : PExpr; args : list [PExpr]; }
    | ParmByRef       { parm : PExpr; }
    | ParmOut         { parm : PExpr; }

    | Error // placeholder of missing tree (where some errors occured)
      
    // macros stuff    
    | MacroCall       { name : Name; ns : NamespaceTree.Node;
                        parms : list [SyntaxElement]; }
    | Quoted          { body : SyntaxElement; }
    | Spliced         { body : PExpr; }
    | Ellipsis        { body : PExpr; }
    | Typed           { body : Typedtree.TExpr; }
    | TypedPattern    { body : Typedtree.Pattern; }  
    | TypedType       { body : TyVar; }

    public override ToString () : string {
      PrettyPrint.SprintExpr (None (), this);
    }
  }

  [Record]
  public variant Splicable : Located 
  {
    | Name { body : Parsetree.Name; }
    | Expression { expr : PExpr; }

    public GetName () : Parsetree.Name {
      match (this) {
        | Splicable.Name (x) => x
        | _ => throw System.ArgumentException ("GetName () called for option " + this.ToString ())
      }
    }

    public GetIdentifier () : string {
      match (this) {
        | Splicable.Name (x) => x.Id
        | _ => throw System.ArgumentException ("GetIdList () called for option " + this.ToString ())
      }
    }
  }

  public class Fun_parm : DeclarationBase
  {
    public mutable ty : PExpr;
    
    public this (name : Splicable, ty : PExpr, modifiers : Modifiers) 
    {
      base (Location.Default, name, modifiers);
      this.ty = ty;
    }
    public this (loc : Location, name : Splicable, ty : PExpr,
                 modifiers : Modifiers) 
    {
      base (loc, name, modifiers);      
      this.ty = ty;
    }

    public this (from : PExpr) 
    {
      base (from.loc, null, Modifiers (NemerleAttributes.None, []));
      match (from) {
        | PExpr.Ref (n) => name = Splicable.Name (n); ty = PExpr.Wildcard (from.loc);
        | PExpr.Spliced (s) => name = Splicable.Expression (s); ty = PExpr.Wildcard (from.loc);          
        | PExpr.TypeEnforcement (PExpr.Ref (n), t) =>
          name = Splicable.Name (n); ty = t;
          
        | PExpr.TypeEnforcement (PExpr.Spliced (s), t) =>
          name = Splicable.Expression (s); ty = t;

        | _ => throw System.ArgumentException ($"incorrect expression supplied for parameter creation: $from")
      }
    }
  }

  public class Fun_header : Located
  {
    public typarms : Typarms;
    public mutable name : Splicable; // is changed when typing lambda
    public ret_type : PExpr;
    public parms : list [Fun_parm];

    public ParsedName : Name {
      get { name.GetName () }
    }
    
    public this (loc : Location, name : Splicable, ret_type : PExpr, 
                 parms : list [Fun_parm]) 
    {
      base (loc);
      this.typarms = Typarms.Empty;
      this.name = name;
      this.ret_type = ret_type;
      this.parms = parms;
    }

    public this (typarms : Typarms, name : Splicable, ret_type : PExpr,
                 parms : list [Fun_parm]) 
    {
      this (Location.Default, typarms, name, ret_type, parms)
    }

    public this (loc : Location, typarms : Typarms, 
                 name : Splicable, ret_type : PExpr, parms : list [Fun_parm]) 
    {
      this (loc, name, ret_type, parms);
      this.typarms = typarms;
    }
  }

  [Record]
  public class Function_decl
  {
    public header : Fun_header;
    public mutable body : PExpr;
  }

  [Record]
  public class MatchCase
  {
    public this (patterns : list [PExpr], body : PExpr)
    {
      this (patterns, body, false)
    }

    public patterns : list [PExpr];
    public body : PExpr;
    public disable_warnings : bool;
  }

 
  public variant SyntaxElement {
    | Expression       { body : PExpr; }
    | MatchCase        { body : Parsetree.MatchCase; }
    | Function         { body : Function_decl; }
    | Parameter        { body : Fun_parm; }
    | TType            { body : PExpr; }

    | RawToken         { body : Token; }

    | ClassMember      { body : Parsetree.ClassMember; }    
    | TypeBuilder      { body : Nemerle.Compiler.TypeBuilder; }
    | FieldBuilder     { body : NemerleField; }
    | MethodBuilder    { body : NemerleMethod; }
    | PropertyBuilder  { body : NemerleProperty; }
    | EventBuilder     { body : NemerleEvent; }
    | ParameterBuilder { body : Typedtree.Fun_parm; }

    public override ToString () : string {
      match (this) {
        | SyntaxElement.Expression (null) => ""
        | SyntaxElement.Expression (body) => body.ToString ()
        | SyntaxElement.TType (body)  => "ttype: " + body.ToString ()
        | SyntaxElement.RawToken (t)  => t.ToString ()
        | SyntaxElement.MatchCase     => "match case"
        | SyntaxElement.Function      => "function"
        | SyntaxElement.Parameter     => "parameter"

        | SyntaxElement.ClassMember   => "ClassMember"
        | SyntaxElement.TypeBuilder   => "TypeBuilder"
        | SyntaxElement.FieldBuilder  => "FieldBuilder"
        | SyntaxElement.MethodBuilder => "MethodBuilder"
        | SyntaxElement.PropertyBuilder => "PropertyBuilder"
        | SyntaxElement.EventBuilder    => "EventBuilder"
        | SyntaxElement.ParameterBuilder => "ParameterBuilder"
      }
    }
  }
} // Nemerle.Compiler

