/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler
{
  class PreParserException : System.Exception {
    public Location : Location;

    public this (loc : Location, msg : string) {
      base (msg);
      this.Location = loc;
    }
  }

  /** Transforms stream of tokens from given LexerBase to token tree
      with matched brackets.
   */
  public class PreParser
  {
    lexer : LexerBase;
    mutable last_token : Token = null;
    mutable Env : GlobalEnv;

    mutable finished : bool = false;
    
    /** Parent stream is the stack of processed token nodes,
        which are already assigned to be in currently build sequence.
        For example:
          a; b; c (); d e _we_are_here_
        'a, b, c()', are alredy known to be in parent sequence,
        while 'd e' are in current temporary sequence, which might
        get added to parent_stream if separator (e.g. ';') occurs
     */
    mutable parent_stream : array [Token] = array (4000);
    mutable parent_pos : int;

    /** Currently builded stream of token nodes is an array of
        loose tokens, which have occured after last separator.
        It will probably form LooseGroup as an element of parent
        sequence or all elements will constitue parent
     */
    mutable current_stream : array [Token] = array (2000);
    mutable current_pos : int;

    internal static mutable doc_comments : NemerleMap [Location, string];

    public static Init () : void {
      if (Options.XmlDoc)
        doc_comments = NemerleMap ();
      else
        doc_comments = null;
    }
    
    public this (lex : LexerBase) {
      Env = GlobalEnv.Core;
      lex.Keywords = Env.Keywords;
      lexer = lex;
    }

    static reset_comment (tok : Token) : void {
      when (Options.XmlDoc) doc_comments = doc_comments.Replace (tok.Location, "");
    }
    static reset_comment (loc : Location) : void {
      when (Options.XmlDoc) doc_comments = doc_comments.Replace (loc, "");
    }

    /** Fetch next token (from one token buffer or lexer if it's empty */
    get_token () : Token {
      if (last_token != null) {
        def result = last_token;
        last_token = null;
        result;
      }
      else {
        try {
          lexer.GetToken ()
        }
        catch {
          | e is LexerBase.Error =>
            Message.Error (lexer.Location, e.name);
            get_token ()
        }
      }
    }

    /** Store token in our mini one token buffer */
    push_back (tok : Token) : void {
      assert (last_token == null);
      last_token = tok;
    }

    static store (tokens : ref array [Token], idx : ref int, tok : Token) : void
    {
      tokens [idx] = tok;
      ++idx;
    }

    /** links Tokens from specified subarray to form a list and return its head */
    static make_list (tokens : array [Token], start : int, end : int) : Token
    {
      for (mutable i = end - 2; i >= start; --i) 
        tokens [i].Next = tokens [i + 1];
      tokens [start]
    }

    
    public static Dump (tok : Token, ident : string) : string {
      def (open, close, sepstr, elements) =
        match (tok) {
          | Token.RoundGroup => ("(", ")", ", ", tok)
          | Token.BracesGroup => ("{\n" + ident, "}", ";\n" + ident, tok)
          | Token.SquareGroup => ("[", "]", ", ", tok)
          | Token.QuoteGroup  => ("<[\n", "]>", "; ", tok)
          | Token.LooseGroup  => ("", "", " ", tok)

          | _ => ("", tok.ToString (false), "", null)
        }
      def builder = System.Text.StringBuilder (open);
      when (elements != null)
        foreach (e is Token in elements) 
          _ = builder.Append (Dump (e, ident + "  ")).Append (sepstr);
      builder.Append (close).ToString ()
    }

    /** Closes both currently created LooseGroup and parent group.
        Returns list of tokens composing parent group */
    finish_parent (parent_begin : int, current_begin : int) : Token {
      finish_current (current_begin);
      def parent_group = 
        if (parent_begin == parent_pos)
          null // case of `(` `)`
        else
          make_list (parent_stream, parent_begin, parent_pos);
      parent_pos = parent_begin;
      parent_group
    }

    /** Closes currently created LooseGroup and adds it at the end of the
        parent group. After that we are ready to make another LooseGroup.

        It is called mainly when separator token occurs.
     */   
    finish_current (current_begin : int) : void {
      unless (current_begin == current_pos) {
        def loose_group = make_list (current_stream, current_begin, current_pos);
        def loose = Token.LooseGroup (current_stream [current_begin].Location, loose_group);
        store (ref parent_stream, ref parent_pos, loose);
        current_pos = current_begin;
      }
    }

    /** Handle standard situations when new bracket group is beginning
        or there is erronous situation. Any non bracket token is
        appended to current LooseGroup.

        Throws PreParserException when there is unmatched end bracket.
     */
    handle_default_token (current_begin : int, tok : Token) : void {
      match (tok) {
        | Token.BeginBrace =>
          def brace_group = parse_brace_group (tok.Location, parent_pos, current_pos);
          store (ref current_stream, ref current_pos, brace_group);
          finish_current (current_begin);

        | Token.BeginRound =>
          def round_group = parse_round_group (tok.Location, parent_pos, current_pos);
          store (ref current_stream, ref current_pos, round_group);

        | Token.BeginSquare =>
          def square_group = parse_square_group (tok.Location, parent_pos, current_pos);
          store (ref current_stream, ref current_pos, square_group);

        | Token.BeginQuote =>
          def quote_group = parse_quote_group (tok.Location, parent_pos, current_pos);
          store (ref current_stream, ref current_pos, quote_group);

        | Token.EndRound | Token.EndSquare | Token.EndQuote | Token.EndBrace =>
          push_back (tok);
          throw PreParserException (tok.Location, $"unexpected closing bracket `$(tok)'");

        | Token.EndOfFile =>
          throw PreParserException (tok.Location, "unexpected end of file");

        | Token.Comment (comment) when Options.XmlDoc =>
          doc_comments = doc_comments.Replace (tok.Location, comment);

        | Token.Comment => ()
          
        | _ => store (ref current_stream, ref current_pos, tok);
      }
    }
    
    parse_brace_group (loc : Location, parent_begin : int, current_begin : int) : Token.BracesGroup
    {
      reset_comment (loc);

      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndBrace =>
            reset_comment (tok);
            def brace_group = finish_parent (parent_begin, current_begin);
            Token.BracesGroup (loc + tok.Location, brace_group);

          // finish current loose group
          | Token.Semicolon => 
            reset_comment (tok);
            finish_current (current_begin); loop ()

          | Token.EndOfFile when parent_begin == 0 =>
            def brace_group = finish_parent (parent_begin, current_begin);
            finished = true;
            Token.BracesGroup (loc + tok.Location, brace_group);
            
          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `{' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.BracesGroup (loc, group);
      }
    }

    parse_round_group (loc : Location, parent_begin : int, current_begin : int) : Token.RoundGroup
    {
      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire round group
          | Token.EndRound =>
            def round_group = finish_parent (parent_begin, current_begin);
            Token.RoundGroup (loc + tok.Location, round_group);

          // finish current loose group
          | Token.Comma =>
            finish_current (current_begin);
            loop ()

          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `(' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.RoundGroup (loc, group);
      }
    }

    parse_square_group (loc : Location, parent_begin : int, current_begin : int) : Token.SquareGroup
    {
      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndSquare =>
            def group = finish_parent (parent_begin, current_begin);
            Token.SquareGroup (loc + tok.Location, group);

          // finish current loose group
          | Token.Comma => finish_current (current_begin); loop ()

          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `[' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.SquareGroup (loc, group);
      }
    }

    parse_quote_group (loc : Location, parent_begin : int, current_begin : int) : Token.QuoteGroup
    {
      def loop () {
        def tok = get_token ();
        match (tok) {
          // finish entire brace group
          | Token.EndQuote =>
            def group = finish_parent (parent_begin, current_begin);
            Token.QuoteGroup (loc + tok.Location, group);

          // finish current loose group
          | Token.Semicolon => finish_current (current_begin); loop ()

          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (loc, "when parsing this `<[' brace group");
        Message.Error (e.Location, e.Message);
        def group = finish_parent (parent_begin, current_begin);
        Token.QuoteGroup (loc, group);
      }
    }

    ParseTopLevel (parent_begin : int, current_begin : int) : Token
    {
      def get_qid () {
        match (get_token ()) {
          | Token.Identifier (x) =>
            match (get_token ()) {
              | Token.Operator (".") => x :: get_qid ()
              | t => push_back (t); [x]
            }
          | t => Message.Error (t.Location, "expected qualified identifier"); []
        }
      }
      
      def loop () {
        def tok = get_token ();
        match (tok) {
          | Token.Keyword ("using") =>
            finish_current (current_begin);
            
            def id = get_qid ();
            match (get_token ()) {
              | Token.Semicolon as st =>
                def loc = tok.Location + st.Location;
                Env = Env.AddOpenNamespace (id, loc);
                lexer.Keywords = Env.Keywords;

                def using_tok = Token.Using (loc, Env);
                store (ref current_stream, ref current_pos, using_tok);
                
              | Token.Operator ("=") =>
                def id' = get_qid ();

                def st = get_token ();
                match (st) {
                  | Token.Semicolon => ()
                  | _ => Message.Error (st.Location, "expecting `;' after using alias")
                }

                match (id) {
                  | [name] => Env = Env.AddNamespaceAlias (name, id', tok.Location);
                  | _ => Message.Error (tok.Location, "using alias must be simple name without dots")
                }
             
                def using_tok = Token.Using (tok.Location + st.Location, Env);
                store (ref current_stream, ref current_pos, using_tok);
                
              | x => Message.Error (x.Location, "expecting `;' or `='")
            }
            finish_current (current_begin);
            loop ()

          | Token.Keyword ("namespace") =>
            finish_current (current_begin);
            
            def id = get_qid ();
            match (get_token ()) {
              | Token.BeginBrace as br =>
                def loc = tok.Location + br.Location;
                def oldenv = Env;
                Env = Env.EnterIntoNamespace (id);
                lexer.Keywords = Env.Keywords;
                
                def decls = ParseTopLevel (parent_pos, current_pos);
                def namespace_tok = Token.Namespace (loc, Env, decls);                

                Env = oldenv;
                lexer.Keywords = Env.Keywords;

                store (ref current_stream, ref current_pos, namespace_tok);                  

              | x => Message.Error (x.Location, "expecting `{' opening namespace scope")
            }
            finish_current (current_begin);            
            loop ()
          
          // finish entire brace group
          | Token.EndBrace =>
            reset_comment (tok);
            finish_parent (parent_begin, current_begin);

          // finish current loose group
          | Token.Semicolon => finish_current (current_begin); loop ()

          | Token.EndOfFile when parent_begin == 0 =>
            def brace_group = finish_parent (parent_begin, current_begin);
            finished = true;
            brace_group;
            
          | _ => handle_default_token (current_begin, tok); loop ()
        }
      }
      try { loop () }
      catch { e is PreParserException =>
        Message.Error (e.Location, e.Message);
        finish_parent (parent_begin, current_begin);
      }
    }

    public ParseTopLevel () : Token.BracesGroup {
      def stream = ParseTopLevel (0, 0);
      lexer.Dispose (); // make lexer release file handle
      Token.BracesGroup (stream)
    }
    
    public PreParse () : Token {
      def top = parse_brace_group (Location.Default, 0, 0);
      unless (finished) Message.Error (lexer.Location, "expected end of file, encountered closing brace");
      top
    }
  }
}
