
using Nemerle.Collections;
using Nemerle.IO;

module M1
{
  method_a (b : bool) : int
  {
    if (b) 123 else 321;
  }

  method_b (b : bool) : int
  {
    match (b) {
      | true => 123
      | _ => 321
    }
  }

  method_c (b : bool) : int
  {
    match (b) {
      | false => 321
      | _ => 123
    }
  }

  method_d (b : bool) : int
  {
    match (b) {
      | false => 321
      | true => 123
    }
  }

  method_e (b : bool) : int
  {
    /* should emit the match failed exception */
    match (b) { // W: matching is not exhaustive, example unmatched value:
      | false when b == false => 321
      | true => 123
    }
  }

  method_f (b : bool) : int
  {
    /* should not emit the the match failed exception */
    match (b) {
      | false when b == false => 321
      | false => 321
      | true => 123
    }
  }


  internal test_1 () : void 
  {
    printf ("--------------------\nTest 1...\n");
    
    def run_method (method : bool -> int) : void
    {
      printf ("%d\n", method (true));
      printf ("%d\n", method (false));
    };

    def methods = [method_a, method_b, method_c, method_d, method_e, method_f];

    List.Iter (methods, run_method)  
  }
}


module M2
{
  method_lex (ch : char) : void {
    | '-' when (true || false) => printf ("9\n")

    | '+' | '-' =>  printf ("10\n")

    | '?' | '#' =>  printf ("11\n")

    | '=' | '<' => printf ("12\n")

    | _ => printf ("17\n")      
  }      

  // test case for pattern overriding //
  method_a (parm : char, phaser : int) : int
  {
    match (parm) {
      | 'A' when phaser % 4 == 0 => 1000 + (parm :> int)
      | 'A' when phaser % 4 == 1 => 1
      | 'A' => 2 
      | 'A' => 3  // W: unused
      | 'A' when phaser % 4 == 2 => 4  // W: unused
      | 'A' => 5 // W: unused
      | _ when parm == 'x' => 13
      | 'A' => 6  // W: unused
      | 'A' => 7  // W: unused
      | 'A' when parm == 'y' => 8  // W: unused
      | 'A' => 9  // W: unused
      | 'B' => 10
      | 'C' => 11
      | x when x >= '0' && x <= '9' => 255
      | _ => -1
    }
  }

  // test case for no literal patterns //
  method_b (parm : char, _phaser : int) : int {
    match (parm) {
      | x when x == 'A' => 0
      | _ => 1
    }
  }

  // test case for non-value result types //
  method_c (parm : char) : option [int] {
    match (parm) {
      | 'A' => Some (42)
      | _ => None ()
    }
  }

  // test case for As //
  method_d (parm : char, _phaser : int) : int {
    match (parm) {
      | 'A' => 0
      | ch => (ch :> int)
    }
  }

  internal test_2 () : void 
  {
    printf ("--------------------\nTest 2...\n");
    
    def run_method (method : char * int -> int) : void
    {
      printf ("%d\n", method ('A', 0));
      printf ("%d\n", method ('A', 1));
      printf ("%d\n", method ('A', 2));
      printf ("%d\n", method ('A', 3));
      printf ("%d\n", method ('A', 0));
      printf ("%d\n", method ('B', 0));
      printf ("%d\n", method ('0', 0));
      printf ("%d\n", method ('x', 0));
      printf ("---\n")
    };

    def methods = [method_a, method_b, method_d];

    List.Iter (methods, run_method);

    if (Option.IsSome (method_c ('A'))) printf ("method_c OK\n")
    else printf ("method_c FAILED\n");

    printf ("---\n");

    method_lex ('-');
  }
}

variant RedBlackTree
{
  | Red { left : RedBlackTree; value : int; right : RedBlackTree; }
  | Black { left : RedBlackTree; value : int; right : RedBlackTree; }
  | Leaf { value : int; }
}

variant SomeLargeVariant
{
  | Zero
  | One
  | Two { value : int; }
  | Three { value : int; }
  | Four { value : int; }
  | Five { value : int; }
  | Six { value : int; }
  | Seven { value : int; }
  | Eight { value : int; }
  | Nine { value : int; }
  | Ten { value : int; }
  | Eleven { value : int; }
}


module M3
{
  TestA (rbt : RedBlackTree) : void 
  {
    match (rbt) {
      | RedBlackTree.Red => printf ("Red\n")
      | RedBlackTree.Black => printf ("Black\n")
      | RedBlackTree.Leaf => printf ("RedBlackTree.Leaf\n")
    }
  }

  TestB (rbt : RedBlackTree) : void 
  {
    match (rbt) {
      | RedBlackTree.Red (RedBlackTree.Leaf (i), _, _) => printf ("Red / RedBlackTree.Leaf: %i\n", i)
      | RedBlackTree.Black (RedBlackTree.Leaf, _, _) => printf ("Black / RedBlackTree.Leaf\n")
      | RedBlackTree.Black (RedBlackTree.Red, _, _) => printf ("Black / Red\n")
      | RedBlackTree.Black (RedBlackTree.Black, _, _) => printf ("Black / Black\n")
      | RedBlackTree.Leaf (i) => printf ("RedBlackTree.Leaf: %i\n", i)
      | _ => printf ("[FAILED]\n")
    }
  }
  
  TestC (some_list : list [int]) : void
  {
    def msg =
      match (some_list) {
        | [] => "empty list"
        | [_] => "one element"
        | [_, _] => "two elements"
        | x => List.Length (x).ToString () + " elements"    
      };
    
    printf ("%s\n", msg)
  }

  TestD (rbt : RedBlackTree) : void
  {
    def result =
      match (rbt) {
        | RedBlackTree.Leaf (7) => "7"
        | RedBlackTree.Leaf (i) => (-i).ToString ()
        | _ => "[FAILED]"
      };
      
    printf ("TestD: %s\n", result)
  }

  public variant mylist ['a] 
  {
    | MyCons { hd : 'a; tl : mylist ['a]; }
    | MyNil {} 
  }
  
  public MyNth['a] (l : mylist ['a], n : int) : 'a
  {
    match (l) {
      | mylist.MyCons (h, t) => if (n == 0) h else MyNth (t, n - 1)
      | mylist.MyNil => throw System.ArgumentOutOfRangeException ("List.Nth")
    }
  }

  TestE (slv : SomeLargeVariant) : void
  {
    def result =
      match (slv) {
        | SomeLargeVariant.Three => "three"
        | _ => "something else"
      };
      
    printf ("TestE: %s\n", result)
  }

  TestF (slv : SomeLargeVariant) : void
  {
    def result =
      match (slv) {
        | SomeLargeVariant.Three => "three"
        | SomeLargeVariant.Four => "four"
        | _ => "something else"
      };
      
    printf ("TestF: %s\n", result)
  }


  TestG (slv : SomeLargeVariant) : void
  {
    def result =
      match (slv) {
        | SomeLargeVariant.Zero => "zero"
        | SomeLargeVariant.One => "one"
        | SomeLargeVariant.Three => "three"
        | _ => "something else"
      };
      
    printf ("TestG: %s\n", result)
  }

  internal test_3 () : void
  {
    printf ("--------------------\nTest 3...\n");
    
    TestA (RedBlackTree.Red (RedBlackTree.Leaf (0), 1, RedBlackTree.Leaf (2)));
    TestA (RedBlackTree.Black (RedBlackTree.Leaf (3), 4, RedBlackTree.Leaf (5)));
    TestA (RedBlackTree.Leaf (6));
    
    TestB (RedBlackTree.Red (RedBlackTree.Leaf (0), 1, RedBlackTree.Leaf (2)));
    TestB (RedBlackTree.Black (RedBlackTree.Leaf (0), 1, RedBlackTree.Leaf (2)));
    TestB (RedBlackTree.Black (RedBlackTree.Red (RedBlackTree.Leaf (3), 4,
                                                 RedBlackTree.Leaf (5)), 6, RedBlackTree.Leaf (7)));
    TestB (RedBlackTree.Black (RedBlackTree.Black (RedBlackTree.Leaf (3), 4,
                                                   RedBlackTree.Leaf (5)), 6, RedBlackTree.Leaf (7)));
    TestB (RedBlackTree.Leaf (8));
    
    TestC ([3, 4]);
    TestC ([1]);
    TestC ([]);
    TestC ([6, 7, 8, 9, 10]);
    
    TestD (RedBlackTree.Leaf (8));
    TestD (RedBlackTree.Leaf (7));
    TestD (RedBlackTree.Leaf (6));
    
    TestE (SomeLargeVariant.Three (10));
    TestE (SomeLargeVariant.Four (7));
    TestF (SomeLargeVariant.Three (10));
    TestF (SomeLargeVariant.Four (7));
    TestF (SomeLargeVariant.Five (13));
    TestG (SomeLargeVariant.Zero ());
    TestG (SomeLargeVariant.One ());
    TestG (SomeLargeVariant.Three (7));
    TestG (SomeLargeVariant.Five (13));
  }
}

module M4
{
  method_a (parm : int) : int {
    match (parm) {
      | 5281 => 1   
      | 5297 => 7     
      | 5303 => 3   
      | 5309 => 9            
      | 5323 => 3           
      | 5333 => 3
      | 5347 => 7
      | 5351 => 1
      | 5381 => 1
      | 5387 => 7   
      | 5393 => 3
      | 5399 => 9
      | 5407 => 7
      | 5413 => 3
      | 5417 => 7
      | 5419 => 9
      | 5431 => 1
      | 5437 => 7
      | 5441 => 1
      | 5443 => 3   
      | 5449 => 9
      | 5471 => 1
      | 5477 => 7
      | 5479 => 9
      | 5483 => 3
      | 5501 => 1
      | 5503 => 3
      | 5507 => 7
      | 5519 => 9
      | 5521 => 1   
      | 5527 => 7
      | 5531 => 1
      | 5557 => 7
      | 5563 => 3
      | 5569 => 9
      | 5573 => 3
      | 5581 => 1
      | 5591 => 1
      | 5623 => 3
      | 5639 => 9
      | 5641 => 1
      | 5647 => 7
      | 5651 => 1
      | 5653 => 3
      | 5657 => 7
      | 5659 => 9
      | 5669 => 9
      | 5683 => 3
      | 5689 => 9
      | 5693 => 3   
      | 5701 => 1
      | 5711 => 1
      | 5717 => 7
      | 5737 => 7
      | 5741 => 1
      | 5743 => 3
      | 5749 => 9
      | 5779 => 9
      | 5783 => 3
      | 5791 => 1   
      | 5801 => 1
      | 5807 => 7
      | 5813 => 3
      | 5821 => 1
      | 5827 => 7
      | 5839 => 9
      | 5843 => 3
      | 5849 => 9
      | 5851 => 1
      | 5857 => 7   
      | 5861 => 1
      | 5867 => 7
      | 5869 => 9
      | 5879 => 9
      | 5881 => 1
      | 5897 => 7
      | 5903 => 3
      | 5923 => 3
      | 5927 => 7
      | 5939 => 9   
      | 5953 => 3
      | 5981 => 1
      | 5987 => 7
      | 6007 => 7
      | 6011 => 1
      | 6029 => 9
      | 6037 => 7
      | 6043 => 3
      | 6047 => 7
      | 6053 => 3   
      | 6067 => 7
      | 6073 => 3
      | 6079 => 9
      | 6089 => 9
      | 6091 => 1
      | 6101 => 1
      | 6113 => 3
      | 6121 => 1
      | 6131 => 1
      | 6133 => 3   
      | 6143 => 3
      | 6151 => 1
      | 6163 => 3
      | 6173 => 3
      | 6197 => 7
      | 6199 => 9
      | 6203 => 3
      | 6211 => 1
      | 6217 => 7
      | 6221 => 1   
      | 6229 => 9
      | 6247 => 7
      | 6257 => 7
      | 6263 => 3
      | 6269 => 9
      | 6271 => 1
      | 6277 => 7
      | 6287 => 7
      | 6299 => 9
      | 6301 => 1   
      | 6311 => 1
      | 6317 => 7
      | 6323 => 3
      | 6329 => 9
      | 6337 => 7
      | 6343 => 3
      | 6353 => 3
      | 6359 => 9
      | 6361 => 1
      | 6367 => 7   
      | 6373 => 3
      | 6379 => 9
      | 6389 => 9
      | 6397 => 7
      | 6421 => 1
      | 6427 => 7
      | 6449 => 9
      | 6451 => 1
      | 6469 => 9
      | 6473 => 3   
      | 6481 => 1
      | 6491 => 1
      | 6521 => 1
      | 6529 => 9
      | 6547 => 7
      | 6551 => 1
      | 6553 => 3
      | 6563 => 3
      | 6569 => 9
      | 6571 => 1
      | 6577 => 7
      | 6581 => 1
      | 6599 => 9
      | 6607 => 7
      | 6619 => 9
      | 6637 => 7
      | 6653 => 3
      | 6659 => 9
      | 6661 => 1
      | 6673 => 3   
      | 6679 => 9
      | 6689 => 9
      | 6691 => 1
      | 6701 => 1
      | 6703 => 3
      | 6709 => 9
      | 6719 => 9
      | 6733 => 3
      | 6737 => 7
      | 6761 => 1
      | 6763 => 3
      | 6779 => 9
      | 6781 => 1
      | 6791 => 1
      | 6793 => 3
      | 6803 => 3
      | 6823 => 3
      | 6827 => 7
      | 6829 => 9
      | 6833 => 3   
      | 6841 => 1
      | 6857 => 7
      | 6863 => 3
      | 6869 => 9
      | 6871 => 1
      | 6883 => 3
      | 6899 => 9
      | 6907 => 7
      | 6911 => 1
      | 6917 => 7   
      | 6947 => 7
      | 6949 => 9
      | 6959 => 9
      | 6961 => 1
      | 6967 => 7
      | 6971 => 1
      | 6977 => 7
      | 6983 => 3
      | 6991 => 1
      | 6997 => 7   
      | 7001 => 1
      | 7013 => 3
      | 7019 => 9
      | 7027 => 7
      | 7039 => 9
      | 7043 => 3
      | 7057 => 7
      | 7069 => 9
      | 7079 => 9
      | 7103 => 3   
      | 7109 => 9
      | 7121 => 1
      | 7127 => 7
      | 7129 => 9
      | 7151 => 1
      | 7159 => 9
      | 7177 => 7
      | 7187 => 7
      | 7193 => 3
      | 7207 => 7
      | 7211 => 1
      | 7213 => 3
      | 7219 => 9
      | 7229 => 9
      | 7237 => 7
      | 7243 => 3
      | 7247 => 7
      | 7253 => 3
      | 7283 => 3
      | 7297 => 7   
      | 7307 => 7
      | 7309 => 9
      | 7321 => 1
      | 7331 => 1
      | 7333 => 3
      | 7349 => 9
      | 7351 => 1
      | 7369 => 9
      | 7393 => 3
      | 7411 => 1   
      | 7417 => 7
      | 7433 => 3
      | 7451 => 1
      | 7457 => 7
      | 7459 => 9
      | 7477 => 7
      | 7481 => 1
      | 7487 => 7
      | 7489 => 9
      | 7499 => 9   
      | 7507 => 7
      | 7517 => 7
      | 7523 => 3
      | 7529 => 9
      | 7537 => 7
      | 7541 => 1
      | 7547 => 7
      | 7549 => 9
      | 7559 => 9
      | 7561 => 1   
      | 7573 => 3
      | 7577 => 7
      | 7583 => 3
      | 7589 => 9
      | 7591 => 1
      | 7603 => 3
      | 7607 => 7
      | 7621 => 1
      | 7639 => 9
      | 7643 => 3   
      | 7649 => 9
      | 7669 => 9
      | 7673 => 3
      | 7681 => 1
      | 7687 => 7
      | 7691 => 1
      | 7699 => 9
      | 7703 => 3
      | 7717 => 7
      | 7723 => 3   
      | 7727 => 7
      | 7741 => 1
      | 7753 => 3
      | 7757 => 7
      | 7759 => 9
      | 7789 => 9
      | 7793 => 3
      | 7817 => 7
      | 7823 => 3
      | 7829 => 9   
      | 7841 => 1
      | 7853 => 3
      | 7867 => 7
      | 7873 => 3
      | 7877 => 7
      | 7879 => 9
      | 7883 => 3
      | 7901 => 1
      | 7907 => 7
      | 7919 => 9
      | 0 => 0
      | 1 => 1
      | 2 => 2
      | 3 => 3
      | 4 => 4
      | 5 => 5
      | 6 => 6
      | 7 => 7
      | 8 => 8
      | 9 => 9
      | 10 => 0
      | 11 => 1
      | 12 => 2
      | 7813 => 3
      | 14 => 4
      | 15 => 5
      | 24216 => 6
      | 17 => 7
      | 18 => 8
      | 4219 => 9
      | 120 => 0
      | 21 => 1
      | 22 => 2
      | 2423443 => 3
      | 24 => 4
      | 25 => 5
      | 26 => 6
      | 427 => 7
      | 28 => 8
      | 29 => 9
      | 30 => 0
      | 32131 => 1
      | 32 => 2
      | 33 => 3
      | 4234 => 4
      | 35 => 5
      | 316 => 6
      | 37 => 7
      | 1238 => 8
      | 39 => 9
      | 43240 => 0
      | 41 => 1
      | 42 => 2
      | 43 => 3
      | 244 => 4
      | 45 => 5
      | 92346 => 6
      | 47 => 7
      | 48 => 8
      | 2349 => 9
      | 50 => 0
      | 93451 => 1
      | 52 => 2
      | 89453 => 3
      | 54 => 4
      | 55 => 5
      | 56 => 6
      | 557 => 7
      | 58 => 8
      | 24259 => 9
      | 60 => 0
      | 461 => 1
      | 62 => 2
      | 643 => 3
      | 234264 => 4
      | 65 => 5
      | 44266 => 6
      | 67 => 7
      | 68 => 8
      | 243869 => 9
      | 70 => 0
      | 9271 => 1
      | 72 => 2
      | 673 => 3
      | 74 => 4
      | 75 => 5
      | 9276 => 6
      | 77 => 7
      | 924378 => 8
      | 79 => 9
      | 80 => 0
      | 481 => 1
      | 82 => 2
      | 93483 => 3
      | 84 => 4
      | 85 => 5
      | 786 => 6
      | 87 => 7
      | 88 => 8
      | 42389 => 9
      | 90 => 0
      | 4291 => 1
      | 92 => 2
      | 932393 => 3
      | 94 => 4
      | 32495 => 5
      | 96 => 6
      | 9297 => 7
      | 898 => 8
      | 99 => 9
      | _ => parm % 10
    }
  }

  run_method_a (counter : int) : void
  {
    when (counter > 0) {
      def x = method_a (counter % 10000);
      if (x != counter % 10)
        printf ("failed on %i (had %i, expected %i)\n", counter, x, counter % 10)
      else
        run_method_a (counter - 1)
    }  
  }

  /* test case for pattern overriding */
  method_b (parm : int, phaser : int) : int
  {
    match (parm) {
      | 42 when phaser % 4 == 0 => 0
      | 42 when phaser % 4 == 1 => 1
      | 42 => 2 
      | 42 => 3 // W: unused
      | 42 when phaser % 4 == 2 => 4  // W: unused
      | 42 => 5  // W: unused
      | _ when parm == 123 => 13
      | 42 => 6 // W: unused
      | 42 => 7 // W: unused
      | 42 when parm == 44 => 8 // W: unused
      | 42 => 9 // W: unused
      | 69 => 10
      | 123 => 11
      | x when x > 300 && x < 310 => 12
      | _ => -1
    }
  }

  /* test case for no literal patterns */
  method_c (parm : int, _phaser : int) : int
  {
    match (parm) {
      | x when x == 42 => 0
      | _ => 1
    }
  }

  /* test case for non-value result types */
  method_d (parm : int) : option [int]
  {
    match (parm) {
      | 42 => Some (42)
      | _ => None ()
    }
  }

  /* test case for the switch instruction generator */
  method_e (parm : int) : option [int]
  {
    match (parm) {
      | 40 => None ()
      | 41 => None ()
      | 42 => Some (42)
      | 43 => None ()
      | 44 => None ()
      | _ => None ()
    }
  }

  /* test case for the normalizer of negatives in switch instruction generator */
  method_f (parm : int) : option [int]
  {
    match (parm) {
      | -12 => None ()
      | -11 => None ()
      | -10 => Some (-10)
      | -4 => None ()
      | -3 => Some (-3)
      | -2 => None ()
      | -1 => None ()
      | 0 => None ()
      | 1 => Some (1)
      | 2 => None ()
      | 3 => None ()
      | 5 => None ()
      | _ => None ()
    }
  }

  internal test_4 () : void 
  {
    printf ("--------------------\nTest 4...\n");
    
    run_method_a (2000);

    def run_method (method : int * int -> int) : void
    {
      printf ("%d\n", method ( 42, 0));
      printf ("%d\n", method ( 42, 1));
      printf ("%d\n", method ( 42, 2));
      printf ("%d\n", method ( 42, 3));
      printf ("%d\n", method ( 69, 0));
      printf ("%d\n", method (123, 0));
      printf ("%d\n", method (303, 0));
      printf ("%d\n", method (321, 0));
      printf ("---\n")
    };

    def methods = [method_b, method_c];

    List.Iter (methods, run_method);

    if (Option.IsSome (method_d (42))) printf ("method_d OK\n")
    else printf ("method_d FAILED\n");

    if (Option.IsSome (method_e (42))) printf ("method_e OK\n")
    else printf ("method_e FAILED\n");

    if (Option.IsSome (method_f (-10))) printf ("method_f OK\n")
    else printf ("method_f FAILED\n");
    if (Option.IsSome (method_f (1))) printf ("method_f OK\n")
    else printf ("method_f FAILED\n");

    printf ("---\n")
  }
}


module M5
{
  intize (s : string) : int
  {
    match (s) {
      | "eins" => 1
      | "zwei" => 2
      | "drei" => 3
      | _ => 0
    }
  }

  stringize (i : int) : string
  {
    match (i) {
      | -1 => "minus-odin"
      | -2 => "minus-dwa"
      | 1 => "odin"
      | 2 => "dwa"
      | 3 => "tri"
      | 4 => "czjetyrie"
      | _ => "?"
    }
  }

  alfabetize (c : char) : string
  {
    match (c) {
      | 'a' => "aj"
      | 'b' => "bi"
      | 'z' => "zet"
      | _ => "??"
    }
  }

  string_of_bool (x : bool) : string
  {
    match (x) {
      | true => "true"
      | false => "false"
    }
  }

  internal test_5 () : void
  {
    printf ("--------------------\nTest 5...\n");
    
    printf ("%d\n", intize ("eins"));
    printf ("%d\n", intize ("zwei"));
    printf ("%d\n", intize ("vier"));
    printf ("%s\n", stringize (-2));
    printf ("%s\n", stringize (-1));
    printf ("%s\n", stringize (1));
    printf ("%s\n", stringize (2));
    printf ("%s\n", stringize (4));
    printf ("%s\n", stringize (0));
    printf ("%s\n", stringize (42));
    printf ("%s\n", stringize (2147483647));
    printf ("%s\n", alfabetize ('a'));
    printf ("%s\n", alfabetize ('b'));
    printf ("%s\n", alfabetize ('c'));
    printf ("%s\n", string_of_bool (true));
    printf ("%s\n", string_of_bool (false));
  }
}


module M6
{
  internal test_6 () : void
  {
    printf ("--------------------\nTest 6...\n");
    
    def x = "foo3";
    def y = "foo" + 3.ToString ();

    match (Some (x)) {
      | Some ("foo3") => printf ("OK/A\n")
      | _ => ()
    };

    match (Some (y)) {
      | Some ("foo3") => printf ("OK/B\n")
      | _ => ()
    };

    when (x == y) {
      printf ("OK/C\n")
    };
  }
}


module M7
{
  f (x : int * int) : string {
    match (x) {
      | (1, 2) => "1,2"
      | (2, 3)
      | (3, 2) => "2,3"
      | (x, 12) when x < 50 => x.ToString ()
      | (x, _) when x > 10 => "+10"
      | (x, 3) when x > 3 => "+3,3"
      | (x, y) when x > y 
      | (1, 55) => "x>y"
      | _ => "?"
    }
  }

  test (x : int, y : int) : void {
    printf ("%s\n", f (x, y))
  }

  internal test_7 () : void
  {
    printf ("--------------------\nTest 7...\n");
    
    test (1, 2);
    test (2, 3);
    test (3, 2);
    test (11, 7);
    test (9, 7);
    test (4, 3);
    test (0, 3);
    test (14, 12);
    test (33, 12);
    test (50, 12);
    
    test (1, -1);
    test (1, 55);
  }
}

enum A {
  | Ene 
  | Due 
  | Rabe
}

[System.Flags]
enum B {
  | Ene = 0x01
  | Due = 0x02
  | Rabe = 0x04
}


module Enums {
  f (a : A) : string
  {
    match (a) {
      | A.Ene => "Ene"
      | A.Due => "Due"
      | A.Rabe => "Rabe"
    }
  }

  f2 (a : A) : string
  {
    match (a) { // W: matching is not exhaustive
      | A.Ene => "Ene" // OK
      | A.Due => "Due" // OK
    }
  }

  f3 (a : A) : string
  {
    match (a) { // OK
      | A.Ene => "Ene" // OK
      | A.Due => "Due"
      | A.Due => "Due" // W: unused
      | A.Rabe => "Rabe" // OK
    }
  }

  f4 (a : B) : string
  {
    match (a) { // W: matching is not exhaustive
      | B.Ene => "Ene" // OK
      | B.Due => "Due" // OK
      | B.Rabe => "Rabe" // OK
    }
  }

  f5 (a: B) : string {
    match (a) {
      | B.Ene => "ene"
      | B.Ene %| B.Due => "enedue"
      | B.Rabe => "rabe"
      | _ => "def"
    }
  }
  
  public run () : void {
    printf ("-------- Enums ---------\n");
    System.Console.WriteLine (f (A.Ene));
    System.Console.WriteLine (f (A.Due));
    System.Console.WriteLine (f (A.Rabe));
    System.Console.WriteLine (f4 (B.Ene));
    System.Console.WriteLine (f4 (B.Due));
    System.Console.WriteLine (f4 (B.Rabe));
    System.Console.WriteLine (f5 (B.Ene | B.Due));
    System.Console.WriteLine (f5 (B.Rabe));
    System.Console.WriteLine (f5 (B.Due));        
  }
}

module MatchWithConversion {
  void_to_object (x : int) : object {
    | 2 => object ()
    | 3 => ({ _ = 4; })
    | 1 => ()      
    | _ => null
  }
 
   mutable bul : bool;
 
  dummy () : void {
    bul = false;
    if(bul)
      ()
    else {
      match(bul) {
        | true =>     ()
        | false => ()
      }
    }
  }
  
  public run () : void  {
    assert (void_to_object (3) == null);
    assert (void_to_object (2) != null);
    assert (void_to_object (1) == null);
    assert (void_to_object (0) == null);        
  }
}

module NullMatch {
  public run () : void {
    printf ("-------- null match ---------\n");    
    match (RedBlackTree.Leaf(1)) {
      | null => ()
      | _x => printf ("blu\n");
    }
    match ("") {
      | null => ()
      | "" => printf ("blaempty\n");        
      | _x => printf ("bla\n");
    }
    match ("bla") {
      | null => ()
      | "" => printf ("blaempty\n");        
      | _x => printf ("bla\n");
    }
    match (null : RedBlackTree) {
      | null => printf ("blunull\n");
      | _x => printf ("blu\n");
    }
    match (null) {
      | null => printf ("blanull\n");
      | "" => printf ("blaempty\n");        
      | _x => printf ("bla\n");
    }
    try {
      match (null) {
        | RedBlackTree.Leaf => ()
        | RedBlackTree.Red => ()
        | RedBlackTree.Black => ()
      }
    } catch { _ is MatchFailureException => printf ("null match exception OK\n") }  
  }
}

public variant Token
{
  | EndOfFile
  
  public override ToString () : string {
    match (this) {
      | EndOfFile => "end"
    }
  }
}

public module Variants {
  variant MyVari {
    | A | B | C
    | D { x : MyVari; y : MyVari }
    | E { x : int }
  }

  guards_order () : void {
    match (MyVari.D (MyVari.C (), MyVari.C ())) {
      | MyVari.D (x, y)
      | MyVari.D (x, _) as y   // W: this match clause is unused
       =>
        System.Console.WriteLine ("{0} {1}", x, y)

      | _ => ()
    }
  }
  
  public Run () : void {
    printf ("----------- variants ------------------\n");
    match (RedBlackTree.Red (null, 0, null) : RedBlackTree) {
      | _x is RedBlackTree.Red => printf ("good1\n")
      | _x is RedBlackTree.Black => printf ("bad1a\n")
      | RedBlackTree.Leaf => printf ("bad1b\n")
      | _ => printf ("bad1c\n")// W: unused
    }
    match (RedBlackTree.Leaf (0) : RedBlackTree) {
      | _x is RedBlackTree.Red => ()
      | _x is RedBlackTree.Black => ()
      | RedBlackTree.Leaf => System.Console.WriteLine ("good2")
      | _ => ()// W: unused
    }
    match (MyVari.B() : MyVari) {
      | _x is MyVari.B => printf ("good3\n")
      | _x is MyVari.C => ()
      | MyVari.D => ()
      | _ => ()
    }
    guards_order ();

    match (MyVari.A() : MyVari) {// W: matching is not exhaustive, example unmatched value
      | B | E (1) => assert (false); 
      | D | A | C => ();
    }
  }
}


enum LowerCaseEnum {
  | a
  | b
  | c
}

class M8
{
        public static Mn () : void
        {
            def x = LowerCaseEnum.b;
            match (x) {
                | LowerCaseEnum.a => print ("bad1\n")
                | LowerCaseEnum.b => print ("good\n")
                | LowerCaseEnum.c => print ("bad2\n")
            }
        }
}


variant MType {
  | Class { tc : MType; args : list[MType]; }
  | Void
}

module M9 {
  public Run () : void
  {
    def b1 = MType.Class (null, []);
    def b2 = MType.Void ();

    _ = 
     match ((b1, b2)) {
        | (null, null) => "?"
        | (t, null) => $ "$t+"
        | (null, t)
        | (MType.Class (tc, []), t)
          when tc != null && tc.Equals (null) =>
          $ "$t-"
        | (t1, t2) => $ "($t1 TILL $t2)"
      }
   }
}

public enum Fruit { | Apple | Orange };

namespace IsMatches {

variant Foo {
  | A { x : int; }
  | B
}

module M {
  l () : list[int] { [42] }
  
  public Run () : void
  {
    def x = l ();
    def y = Foo.A (42) : Foo;
    
    assert (x is [42]);
    assert (! (x is []));
    assert (y is Foo.A (42));
    assert (! (y is Foo.A (10)));
    assert (! (y is Foo.B));  


    def lst = [Foo.A(10),Foo.B()];
    assert (lst.Exists (fun (x) { x is Foo.B }));
  }

  
  // matching compiler duplicates code in examples like below
    public make_evaluation_exprs (toks : list [int], parms : array [int]) 
    : void
    {
      def make_expressions (toks, i) {
        match (toks) {
          | [] when i == 0 => ()

          | _ when i == 1 =>
            ()
          | _ when i == 2 =>
            ()
          | _ when i == 3 =>
            ()
          | _ when i == 4 =>
            ()
          | _ when i == 5 =>
            ()
          | _ when i == 6 =>
            ()
          | _ when i == 7 =>
            ()
          | _ when i == 8 =>
            ()
          | _ when i == 9 =>
            ()
          | _ when i == 10 =>
            ()

          | _ when i == 11 =>
            ()

          | _ => 
            ()
        }
      };

      make_expressions (toks, parms.Length);
    }
}

}


namespace SwitchGoto {

  class TyVar {}
  class StaticTyVar {}
  variant MType : TyVar
  {
    #region Options
    | Class {
        tycon : string;
        args : list [TyVar];
      }
    | TyVarRef {
        tyvar : StaticTyVar;
      }
    | Fun {
        from : TyVar;
        to : TyVar;
      }
    | Tuple {
        args : list [TyVar];
      }
    | Array {
        t : TyVar;
        rank : int;
      }
    | Ref {
        t : TyVar;
      }
    | Out {
        t : TyVar;
      }
    | Void

      /* Used when given value is required to have all the listed types.

         Invariant 1: the types listed cannot be in the subtyping relation
         pairwise.

         Invariant 2: there can be only Class() objects inside.
         
         This type is not expressible in the .NET type system directly,
         it can be however expressed with type variable bounds in some
         cases. */
    | Intersection {
        types : list [MType];
      }
    #endregion

    #region Pretty printing
    public override ToString () : string
    {
      match (this) {
        | Class (tc, []) =>
          def trim = tc.Replace ("Nemerle.Core.", "");
          match (trim) {
            | "System.String" => "string"
            | "System.Int32" => "int"
            | "System.Single" => "float"
            | "System.Char" => "char"
            | "System.Boolean" => "bool"
            | "System.Void" => "void"
            | _ => trim
          }

        | Class (tc, args) => 
          def trim = tc.Replace ("Nemerle.Core.", "");
          trim + args.ToString ()
          
        | TyVarRef (s) => s.ToString ()
        
        | Fun (t1, t2) => $ "$t1 -> $t2"
          
        | Tuple (lst) => "(" + lst.ToString (" * ") + ")"

        | Ref (t) => $ "ref $t"
          
        | Out (t) => $ "out $t"
          
        | Array (t, 1) => $ "array [$t]"
          
        | Array (t, n) => $ "array.$n [$t]"
          
        | Void => "void"
        
        | Intersection (lst) => "(" + lst.ToString (" AND ") + ")"
      }
    }
    #endregion
    static head_variant () : void
    {
      def make_expressions (toks) {
        match (toks) {
          | [] => 1
          | [1] => 2
          | [1,2] => 21
          | [1,3] => 22
          | [1,4] => 23
          | [1,6] => 24
          | [2,2] => 212
          | [2,3] => 222
          | [2,4] => 23
          | [2,6] => 242
          | [22,2] => 2122
          | [22,3] => 2222
          | [22,4] => 232
          | [22,6] => 2422
          | _ => assert (false)
        }
      };

      assert (make_expressions ([22,6]) == 2422);
      assert (make_expressions ([2,3]) == 222);
    }
    
  public static go () : void
  {
     head_variant ();
     assert (MType.Class ("foo", []).ToString() == "foo");
  }
  }
}



public module TestRunner
{
  public Main () : void
  {
    M1.test_1 ();
    M2.test_2 ();
    M3.test_3 ();
    M4.test_4 ();
    M5.test_5 ();
    M6.test_6 ();
    M7.test_7 ();
    Enums.run ();
    MatchWithConversion.run ();
    NullMatch.run ();
    Variants.Run ();
    M8.Mn ();
    M9.Run ();

    System.Console.WriteLine(System.Convert.ToInt64((
      match ( Fruit.Orange) { 
        | _tmp : System.Enum => _tmp 
        | _ => null // W: this match clause is unused
    }) ) ); 

        def lst = [true, false];

        foreach (b in lst)
          System.Console.Write (b.ToString () + " ");
        System.Console.WriteLine (".");

    match (42M) {
      | 42M => print ("ok42\n");
      | _ => print ("oops\n");
    }

    IsMatches.M.Run ();

    // test closurising of the matched value

        def foo (l) {
          | [x]
          | [_, x] =>
            l.Map (fun (y : int) { x + y })
          | _ => l
        }

        System.Console.WriteLine (foo ([1,2]));

    SwitchGoto.MType.go();
  }
}

/*
BEGIN-OUTPUT
--------------------
Test 1...
123
321
123
321
123
321
123
321
123
321
123
321
--------------------
Test 2...
1065
1
2
2
1065
10
255
13
---
0
0
0
0
0
1
1
1
---
0
0
0
0
0
66
48
120
---
method_c OK
---
9
--------------------
Test 3...
Red
Black
RedBlackTree.Leaf
Red / RedBlackTree.Leaf: 0
Black / RedBlackTree.Leaf
Black / Red
Black / Black
RedBlackTree.Leaf: 8
two elements
one element
empty list
5 elements
TestD: -8
TestD: 7
TestD: -6
TestE: three
TestE: something else
TestF: three
TestF: four
TestF: something else
TestG: zero
TestG: one
TestG: three
TestG: something else
--------------------
Test 4...
0
1
2
2
10
13
12
-1
---
0
0
0
0
1
1
1
1
---
method_d OK
method_e OK
method_f OK
method_f OK
---
--------------------
Test 5...
1
2
0
minus-dwa
minus-odin
odin
dwa
czjetyrie
?
?
?
aj
bi
??
true
false
--------------------
Test 6...
OK/A
OK/B
OK/C
--------------------
Test 7...
1,2
2,3
2,3
+10
x>y
+3,3
?
14
33
+10
x>y
x>y
-------- Enums ---------
Ene
Due
Rabe
Ene
Due
Rabe
enedue
rabe
def
-------- null match ---------
blu
blaempty
bla
blunull
blanull
null match exception OK
----------- variants ------------------
good1
good2
good3
Variants+MyVari+C Variants+MyVari+C
good
1
True False .
ok42
[3, 4]
END-OUTPUT
*/
