/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Compiler.Typedtree;

namespace Nemerle.Compiler
{
  public class LocalValue : Located, Nemerle.IComparable [LocalValue]
  {
    name : string;
    is_mutable : bool;
    mutable ty : TyVar;
    kind : Kind;
    defined_in : Fun_header;

    id : int;
    mutable used_in : Set [Fun_header];

    mutable in_closure : bool;
    mutable ever_used : bool;
    mutable is_registered : bool;
    mutable closure_field : IField;

    internal InClosure : bool
    {
      set { in_closure = value; }
      get { in_closure }
    }


    public Id : int
    {
      get { id }
    }

    
    public DefinedIn : Fun_header
    {
      get { defined_in }
    }


    public UsedIn : Set [Fun_header]
    {
      get { used_in }
    }


    public EverUsed : bool
    { 
      set { ever_used = value; }
      get { ever_used }
    }


    internal UseFrom (fh : Fun_header) : void
    {
      ever_used = true;
      used_in = used_in.Replace (fh);
    }


    public Type : TyVar
    {
      get { ty }
    }


    public ValKind : Kind
    {
      get { kind }
    }


    public IsMutable : bool
    {
      get { is_mutable }
    }


    public Name : string
    {
      get { name }
    }

    
    public override ToString () : string
    {
      $ "$kind $(Name)"
    }


    public CompareTo (other : LocalValue) : int
    {
      id - other.id
    }

    internal UsedInPatternBody : bool
    {
      get {
        match (kind) {
          | Kind.PatternVariable (x) => x
          | _ => assert (false)
        }
      }
      set {
        (kind :> Kind.PatternVariable).used_in_body = value;
      }
    }


    internal IsRegistered : bool
    {
      get { is_registered }
    }


    internal Register () : void
    {
      assert (!IsRegistered);
      is_registered = true;
      DefinedIn.all_locals = this :: DefinedIn.all_locals;

      match (ValKind) {
        | Kind.Function (h, parents) =>   
          foreach (parent in parents)
            parent.children_funs = h :: parent.children_funs;
        | _ => {}
      }
    }


    internal ClosureField : IField
    {
      get { closure_field }
      set {
        assert (closure_field == null);
        closure_field = value;
      }
    }


    internal SetObjectType () : void
    {
      ty = InternalType.Object;
    }

    #region IL integration
    variant ILKind
    {
      | None
      | Parm { slot : int; }
      | ByRefParm { slot : int; }
      | Local { builder : System.Reflection.Emit.LocalBuilder; }
    }

    mutable ilkind : ILKind = ILKind.None ();

    CheckIL () : void
    {
      Util.cassert (! (ilkind is ILKind.None),
                    $ "ilkind is none for $this (id=$Id)");
    }

    internal IsByRefParm : bool
    {
      get {
        CheckIL ();
        ilkind is ILKind.ByRefParm
      }
    }
    
    internal IsParm : bool
    {
      get {
        CheckIL ();
        match (ilkind) {
          | Parm
          | ByRefParm => true
          | Local => false
          | None => Util.ice ()
        }
      }
    }

    internal LocalSlot : System.Reflection.Emit.LocalBuilder
    {
      get {
        CheckIL ();
        match (ilkind) {
          | Local (slot) => slot
          | _ => Util.ice ()
        }
      }
      set {
        Util.cassert (ilkind is ILKind.None);
        ilkind = ILKind.Local (value);
      }
    }

    internal ParmIndex : int
    {
      get {
        CheckIL ();
        match (ilkind) {
          | Parm (id)
          | ByRefParm (id) => id
          | _ => Util.ice ()
        }
      }
    }

    internal SetParmIndex (idx : int, is_by_ref : bool) : void
    {
      Util.cassert (ilkind is ILKind.None);
      ilkind = 
        if (is_by_ref)
          ILKind.ByRefParm (idx);
        else
          ILKind.Parm (idx)
    }
    #endregion


    
    public variant Kind
    {
      | Plain
      | Function {
          header : Fun_header; 
          uses_closure_of : list [Fun_header]; 
        }
      | FunParm {
          kind : ParmKind;
        }
      | ExceptionValue
      | PatternVariable {
          /** True iff variable is used in body of match, not only in guard. */
          mutable used_in_body : bool;
        }
      | BlockReturn {
          /// where to store result of the current block
          result_val : LocalValue;
          /// the exit label of the block
          block_label : int;
        }
      | ClosurisedThisPointer
      | MacroRedirection { subst : Parsetree.PExpr; }

      public override ToString () : string
      {
        match (this) {
          | Plain => "a local value"
          | Function => "a local function"
          | FunParm => "a function parameter"
          | ExceptionValue => "a caught exception"
          | PatternVariable => "a value bound in pattern"
          | BlockReturn => "a return from a block"
          | ClosurisedThisPointer => "a `this' pointer" // not reached?
          | MacroRedirection => "a macro invocation" // not reached?            
        }
      }
    }

    public this (defined_in : Fun_header, name : string, 
                 ty : TyVar, kind : Kind, is_mutable : bool)
    {
      this.defined_in = defined_in;
      this.name = name;
      this.ty = ty;
      this.kind = kind;
      this.is_mutable = is_mutable;
      
      this.id = Util.next_id ();
      this.used_in = Set ();
    }
  }

} // ns
