/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle;
using Nemerle.Collections;
using SRE = System.Reflection.Emit;

namespace Nemerle.Compiler 
{
  [System.Flags]
  enum SpecialConstraint {
    | Class = 0x001
    | Struct = 0x002
    | New    = 0x004
    | None = 0
  }
  
  /** Represents an explicitly bound type variable, like the one from
      polymorphic type or method definition. It is also used in
      generalizations of inferred types. */
  public class StaticTyVar
  {
    public override ToString () : string
    {
      // XXX include constraints?
      $ "$(Name).$(Id)"
    }


    /** Return lower constraints on given type variable (that is the
        [where] clauses). The constraints should be already normalized (that 
        is they should follow MType.Intersection invariants).
      */
    public Constraints : list [MType]
    {
      get {
        assert (constraints != null);
        constraints
      }

      set {
        assert (constraints == null);
        constraints = value;
        unless (value.IsEmpty)
          LowerBound.Validate ();
      }
    }


    public LowerBound : MType
    {
      get {
        assert (constraints != null);
        match (constraints) {
          | [] =>
            assert (InternalType.Object != null);
            InternalType.Object
          | [x] => x
          | lst => MType.Intersection (lst)
        }
      }
    }
    

    public Name : string
    {
      get { name }
    }


    public Id : int
    {
       get { id }
    }
    

    public this (name : string)
    {
      this.name = name;
      id = current_id;
      ++current_id;
    }

#if _GENERICS
    public this (name : string, gtype : System.Type) {
      this (name);
      this.gtype = gtype;
    }
#endif      
   
    
    public GetSystemType () : System.Type
    {
#if _GENERICS
      assert (gtype != null, ToString ());
      gtype
#else    
      match (Constraints) {
        | [] => SystemType.Object
        | t :: _ => t.GetSystemType ()
      }
#endif      
    }

    internal SpecialConstraints : SpecialConstraint
    {
      get { special }
    }
    
    internal SetConstraints (special : SpecialConstraint, subtype : list [MType]) : void
    {
      when ((special & SpecialConstraint.Struct) %&& (special & ~SpecialConstraint.Struct))
        Message.Error ("`struct' generic constraint cannot be used together with `class' or `new ()'");

      when (special %&& SpecialConstraint.Struct)
        Message.Warning ("`struct' generic constraint is currently ignored by compiler");
      when (special %&& SpecialConstraint.New)
        Message.Warning ("`new ()' generic constraint is currently ignored by compiler");
        
      this.special = special;
      Constraints = subtype;
    }
    
    
    [Nemerle.OverrideObjectEquals]
    public Equals (o : StaticTyVar) : bool
    {
      id == o.id
    }

#if _GENERICS
    internal SetGenericBuilder (x : SRE.GenericTypeParameterBuilder) : void
    {
      gtype = x;
    }

    mutable gtype : System.Type; 
#endif
    internal id : int;
    name : string;
    mutable special : SpecialConstraint;
    mutable constraints : list [MType];

    static mutable current_id : int;    
  }
}
