/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

using PT = Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler
{
  public partial class Typer
  {
    class PatternTyper
    {
      typer : Typer;
      matched_value_type : TyVar;
      patterns : list [PT.PExpr * option [PT.PExpr]];
      messenger : Messenger;

      mutable common_variables : NemerleMap [PT.Name, LocalValue];
      mutable current_pattern_variables : NemerleMap [PT.Name, LocalValue];


      public this (typer : Typer,
                   matched_value_type : TyVar,
                   patterns : list [PT.PExpr * option [PT.PExpr]])
      {
        this.typer = typer;
        this.matched_value_type = matched_value_type;
        this.patterns = patterns;
        messenger = typer.messenger;
      }


      public Run () : list [Pattern * TExpr]
      {
        mutable result = [];

        foreach ((pattern, guard) in patterns) {
          current_pattern_variables = NemerleMap ();
          def typed_pattern = TypePattern (matched_value_type, pattern);

          // FIXME: shouldn't this be in the library?
          def intersect (map1 : NemerleMap [PT.Name, _], map2 : NemerleMap [PT.Name, _]) {
            map1.Fold (NemerleMap (), fun (key, value, res : NemerleMap [PT.Name, _]) {
              if (map2.Member (key))
                res.Add (key, value)
              else
                res
            })
          }

          if (common_variables == null)
            common_variables = current_pattern_variables;
          else
            common_variables = intersect (common_variables,
                                          current_pattern_variables);

          typer.PushLocals ();
            current_pattern_variables.Iter (typer.AddLocal);

            def expr =
              match (guard) {
                | None => <[ true ]>
                | Some (expr) => expr
              }

            def typed_expr = typer.TypeExpr (expr, InternalType.Boolean);
          typer.PopLocals ();

          result = (typed_pattern, typed_expr) :: result;
        }

        common_variables.Iter (fun (_, v : LocalValue) {
                                v.UsedInPatternBody = true
                               });
        common_variables.Iter (typer.AddLocal);

        List.Rev (result)
      }


      TypeTuplePattern (matched_value_type : TyVar,
                        pats : list [PT.PExpr]) : Pattern
      {
        def is_assignment (expr) {
          expr is PT.PExpr.Assign (PT.PExpr.Ref, _) 
        }

        if (List.Exists (pats, is_assignment)) {
          if (List.ForAll (pats, is_assignment)) {
            match (matched_value_type.Hint) {
              | Some (MType.Class (ti, _)) when matched_value_type.IsFixed =>
                def is_instance_field (_ : IMember) {
                  | field is IField => !field.IsStatic
                  | prop is IProperty =>
                    !prop.IsStatic && !prop.IsIndexer && prop.GetGetter () != null
                  | _ => false
                }
                def lookup_field (expr) {
                  | PT.PExpr.Assign (PT.PExpr.Ref (name), pat) =>
                    def fields = List.Filter (ti.LookupMember (name.Id), 
                                              is_instance_field);
                    match (fields) {
                      | [] =>
                        ReportError (messenger,
                                     $"the type `$(ti)' has no field named `$(name)'");
                        (null, Pattern.Error ())

                      | [field] =>
                        field.HasBeenUsed = true;
                        def ty = matched_value_type.FixedValue.TypeOfMember (field);
                        (field, TypePattern (ty, pat))

                      | _ =>
                        // I don't thinks this can happen
                        ReportError (messenger,
                                     $"overload ambiguity during selection of `$ti.$(name)'");
                        (null, Pattern.Error ())
                    }

                  | _ => assert (false)
                }

                Pattern.Record (List.Map (pats, lookup_field))
                
                
              | _ =>
                ReportError (messenger,
                             "the (fieldname = ...) pattern is not allowed "
                             "here, try prefixing it with the class name");
                Pattern.Error ()
            }
          } else {
            ReportError (messenger, 
                         "not all, but some patterns in tuple are named");
            Pattern.Error ()
          }
        } else {
          match (matched_value_type.Hint) {
            // if we expecte to match on a class, transform the pattern
            // to include field names
            | Some (MType.Class (ti, _)) =>
              def mems = ti.GetFields (BindingFlags.DeclaredOnly %|
                                       BindingFlags.Public %|
                                       BindingFlags.Instance);
              def names = List.Map (mems, fun (x : IField) { x.Name });
              if (names.Length == pats.Length) {
                def assigns = List.Map2 (names, pats,
                  fun (n, p) {
                    PT.PExpr.Assign (PT.PExpr.Ref (PT.Name (n)), p)
                  });
                TypeTuplePattern (matched_value_type, assigns)
              } else {
                ReportError (messenger,
                             $ "pattern matches $(pats.Length) values, "
                               "while the type `$(ti)' has $(names.Length) "
                               "fields");
                Pattern.Error ()
              }
              
            | Some (MType.Tuple (types)) 
              when types.Length != pats.Length =>
              ReportError (messenger,
                           $ "pattern matches $(pats.Length)-tuples, "
                             "while the value matched is a "
                             "$(types.Length)-tuple");
              Pattern.Error ()
              
            | _ =>
              def types =
                List.Map (pats, fun (_) { FreshTyVar () });
              if (matched_value_type.Unify (MType.Tuple (types))) {
                def typed_pats = List.Map2 (types, pats, TypePattern);
                Pattern.Tuple (typed_pats)
              } else {
                ReportError (messenger,
                             $ "type clash in pattern typing");
                Pattern.Error ()
              }
          }
        }
      }
        
        
      TypePattern (matched_value_type : TyVar, pattern : PT.PExpr) : Pattern
      {
        Util.locate (pattern.loc, {
          def typed = DoTypePattern (matched_value_type, pattern);
          when (typed.ty == null)
            typed.ty = matched_value_type;
          typed
        })
      }


      TypeApplication (matched_value_type : TyVar, 
                       name : PT.PExpr, pattern : PT.PExpr) : Pattern
      {
        TypeApplication (matched_value_type, name, pattern, is_where = false)
      }

      
      TypeApplication (matched_value_type : TyVar, 
                       name : PT.PExpr, pattern : PT.PExpr,
                       is_where : bool) : Pattern
      {
        match (Util.QidOfExpr (name)) {
          | Some (([id], name)) when !is_where && !System.Char.IsUpper (id [0]) =>
            match (pattern) {
              | PT.PExpr.Wildcard =>
                TypePattern (matched_value_type,
                             PT.PExpr.As (PT.PExpr.Wildcard (), PT.Splicable.Name (name)))
              | _ =>
                ReportError (messenger, $ "`$(name)' cannot be variant option, "
                                          "since it starts with lowercase letter");
                Pattern.Error ()
            }

          | Some ((idl, name)) =>
            def env = name.GetEnv (typer.env);
            def is_proper_symbol (sym : IMember) {
              | fld is IField when !is_where =>
                fld.CanAccess (typer.current_type) &&
                fld.IsLiteral

              | ti is TypeInfo =>
                ti.CanAccess (typer.current_type)

              | _ => false
            }

            def symbols =
              match ((idl, matched_value_type.Hint)) {
                | ([name], Some (MType.Class (tc, _))) =>
                  tc.LookupMember (name).Filter (is_proper_symbol);
                | _ => []
              }
            def symbols = 
              if (symbols.IsEmpty) 
                env.LookupSymbol (idl, typer.current_type)
                   .Filter (is_proper_symbol)
              else symbols;

            def symbols = SquashDuplicates (symbols);

            match (symbols) {
              | [fld is IField] =>
                if (pattern is PT.PExpr.Wildcard) {
                  if (matched_value_type.Require (fld.GetMemType () : MType)) {
                    def val = ConstantFolder.FieldValueAsPureLiteral (fld);
                    if (fld.DeclaringType.GetTydecl () is TypeDeclaration.Enum)
                      Pattern.Enum (fld, val)
                    else
                      Pattern.Literal (val)
                  }
                  else {
                    ReportError (messenger, 
                                 $ "the matched value type "
                                   "$matched_value_type was required "
                                   "to have type $(fld.GetMemType ())");
                    Pattern.Error ()
                  }
                }
                else {
                  ReportError (messenger, 
                               "a pattern was supplied after enum field name");
                  Pattern.Error ()
                }

              | [ti is TypeInfo] =>
                def lookup (ti : TypeInfo) {
                  match (ti.GetTydecl ()) {
                    | TypeDeclaration.Alias (MType.Class (tc, _)) =>
                      lookup (tc)

                    | _ when is_where =>
                      def option_type = ti.GetFreshType ();
                      if (matched_value_type.Require (option_type)) {
                        TypePattern (option_type, pattern)
                      } else {
                        ReportError (messenger,
                                     $ "the matched value type "
                                       "$matched_value_type was expected "
                                       "to be compatible with $option_type");
                        Pattern.Error ()
                      }


                    | TypeDeclaration.VariantOption =>
                      // def parent_tycon = Option.UnSome (ti.SuperClass ());
                      def option_type = ti.GetFreshType ();
                      // Message.Debug ($ "$matched_value_type Provide $option_type");
                      if (matched_value_type.Provide (option_type)) {
                        def inpat = TypePattern (option_type, pattern);
                        Pattern.Application (ti, inpat)
                      } else {
                        ReportError (messenger,
                                     $ "the matched value type "
                                       "$matched_value_type was expected "
                                       "to be compatible with $option_type");
                        Pattern.Error ()
                      }

                      //else if (option_type >:> t)
                      //  Message.Warning ($ "matching of $t with $option_type is redundant");
                      //  Message.HintOnce ("refer to fields of this variant option without matching");

                    | _ =>
                      ReportError (messenger, $ "`$(ti.FullName)' is not a variant option");
                      Pattern.Wildcard ()
                  }
                }
                lookup (ti)
                
              | [] =>
                ReportError (messenger, $ "unbound type name $(idl.ToString (\".\"))");
                Pattern.Error ()

              | x =>
                ReportError (messenger, $ "overloading ambiguity $(x.ToString (\", \"))");
                Pattern.Error ()
            }

          | None =>
            ReportError (messenger, "expected qualified identifier in pattern");
            Pattern.Error ()
        }
      }


      DoTypePattern (matched_value_type : TyVar, pattern : PT.PExpr) : Pattern
      {
        match (pattern) {
          | PT.PExpr.ListLiteral (l) =>
            TypePattern (matched_value_type, Macros.Lift (l))

          | PT.PExpr.As (pat, PT.Splicable.Name (name)) =>
            def typed_pattern = TypePattern (matched_value_type, pat);
            def fixed_type =
              match (typed_pattern) {
                | Pattern.Application (ti, _) =>
                  def raw_type = ti.GetFreshType ();
                  when (!raw_type.Require (matched_value_type))
                    ReportError (messenger,
                                 $ "the matched value type "
                                   "$matched_value_type was expected "
                                   "to be compatible with $raw_type");
                  raw_type
                | _ => matched_value_type
              }
              
            when (current_pattern_variables.Member (name))
              ReportError (messenger, 
                           $ "pattern variable `$(name)' already seen "
                             "in this pattern");

            def decl =
              if (common_variables == null || !common_variables.Member (name)) {
                typer.DefineLocal (name, 
                                   fixed_type,
                                   LocalValue.Kind.PatternVariable (false),
                                   false)
              } else {
                def decl = common_variables.Get (name);
                unless (decl.Type.Provide (fixed_type)) {
                  ReportError (messenger,
                               $ "$decl used to have type $(decl.Type) but now it "
                                 "has type $fixed_type");
                  when (messenger.InErrorMode)
                    Message.Hint ("rename variable if they have no connection");
                }
                decl
              }

            current_pattern_variables =
              current_pattern_variables.Add (name, decl);
              
            Pattern.As (typed_pattern, decl)

            
          | PT.PExpr.As (_, PT.Splicable.Expression) =>
            ReportError (messenger, 
                         "$ operator used outside of quotation <[ ... ]> "
                         "context");
            Pattern.Error ()
            

          | PT.PExpr.TypeEnforcement (nested, needed_type) =>
            def needed_type' = typer.BindType (needed_type).FixedValue;

            if (messenger.LocalError) Pattern.Error ()
            else if (matched_value_type.TryRequire (needed_type')) { // just a hint?
              _ = matched_value_type.Require (needed_type');
              TypePattern (needed_type', nested)
            } else {
              when (messenger.NeedMessage)
                Message.Warning (602, "using ``:'' as a type tests is "
                                      "deprecated, please use ``is'' instead");
              TypePattern (matched_value_type, 
                           PT.PExpr.Is (pattern.loc, nested, needed_type))
            }


          | PT.PExpr.Is (nested, needed_type) =>
            def needed_type = typer.BindType (needed_type).FixedValue;
            def properly_subtypes = matched_value_type.TryProvide (needed_type);

            def res =
              if (properly_subtypes) {
                _ = matched_value_type.Provide (needed_type);
                null
              } else {
                if (needed_type.IsInterface ||
                    Option.UnSome (matched_value_type.Hint).IsInterface) {
                  // ok, we allow interfaces here
                  null
                } else if (matched_value_type.TryRequire (needed_type)) {
                  _ = matched_value_type.Require (needed_type);
                  when (messenger.NeedMessage)
                    Message.Warning ("using the ``is'' pattern here is redundant, please use ``:''");
                  TypePattern (needed_type, nested)
                } else {
                  ReportError (messenger,
                               $ "matched value has type $matched_value_type "
                                 "while the pattern enforces $needed_type");
                  Pattern.Error ()
                }
              }

            match (needed_type) {
              | _ when res != null => res
              | MType.Class (ti, _) =>
                def typed_pattern =
                  Pattern.HasType (pattern.loc, matched_value_type, ti);
                match (TypePattern (needed_type, nested)) {
                  | Pattern.As (Pattern.Wildcard, decl) =>
                    Pattern.As (pattern.loc, matched_value_type, 
                                typed_pattern, decl)
                  | Pattern.Wildcard => typed_pattern
                  | _ =>
                    ReportError (messenger,
                                 "only variable patterns are allowed "
                                 "(here : type)");
                    Pattern.Error ()
                }
              | _ =>
                ReportError (messenger,
                             $ "invalid type in (x : type) pattern: "
                               "$needed_type");
                Pattern.Error ()
            }


          | PT.PExpr.Wildcard => Pattern.Wildcard ()


          | PT.PExpr.Literal (lit) =>
            if (matched_value_type.IsFixed && 
                Typer.LiteralConversionPossible (lit, matched_value_type.FixedValue)) {
              Pattern.Literal (lit)
            } else {
              _ = typer.Expect (matched_value_type, TypeOfLiteral (lit),
                                       "matched value");
              Pattern.Literal (lit)
            }

          | PT.PExpr.Tuple (pats) =>
            TypeTuplePattern (matched_value_type, pats)


          | PT.PExpr.Assign as ex =>
            TypeTuplePattern (matched_value_type, [ex])


          | PT.PExpr.Sequence (pats) =>
            when (messenger.NeedMessage)
              Message.Warning (602, "using ``{ x = foo; y = 42 }'' to match "
                                    "fields is deprecated, please use ``SomeClass "
                                    "where (x = foo, y = 42)'' instead");
            TypeTuplePattern (matched_value_type, pats)
            

          | PT.PExpr.Ref
          | PT.PExpr.Member => 
            TypeApplication (matched_value_type, pattern, PT.PExpr.Wildcard ())

             
          | PT.PExpr.Call (PT.PExpr.Ref (n), _) when ConstantFolder.is_known_operator (n.Id) => 

          def folded = ConstantFolder.FoldConstants (typer.env, pattern);
          // constant folder will return the same object if it didn't do anything
          if ((folded : object) == pattern) {
            ReportError (messenger,
                         "couldn't fold arithmetic expression in pattern "
                         "to a constant");
            when (messenger.NeedMessage)
              Message.Hint ("you can use `| x when x == complex expression =>' "
                            " to do such things");
            Pattern.Error ()
          } else
            TypePattern (matched_value_type, folded)
                         

          | PT.PExpr.Call (f, []) =>
            TypeApplication (matched_value_type, f, PT.PExpr.Wildcard ())

          
          | PT.PExpr.Call (f, args) =>
            TypeApplication (matched_value_type, f, PT.PExpr.Tuple (args))
            

          | PT.PExpr.Where (e1, e2) =>
            TypeApplication (matched_value_type, e1, e2, is_where = true)
            

          | PT.PExpr.TypedPattern (body) => body

              
          | PT.PExpr.Quoted (parse_element) =>
            // we use Macros functions to translate quoted element
            Macros.in_pattern = true;          
            def lifted = 
              match (parse_element) {
                | PT.SyntaxElement.Expression (e) => Macros.quoted_expr (e)
                | PT.SyntaxElement.MatchCase (e) => Macros.QuotedMatchCase (e)
                | PT.SyntaxElement.Function (e) => Macros.quoted_fundecl (e)
                | PT.SyntaxElement.Parameter (e) => Macros.quoted_fparam (e)
                | PT.SyntaxElement.ClassMember (e) => Macros.quoted_member (e)                
                | PT.SyntaxElement.TType =>
                  Message.FatalError ("matching over typed types not supported")
                | PT.SyntaxElement.RawToken
                | PT.SyntaxElement.TypeBuilder
                | PT.SyntaxElement.MethodBuilder
                | PT.SyntaxElement.EventBuilder
                | PT.SyntaxElement.FieldBuilder
                | PT.SyntaxElement.PropertyBuilder
                | PT.SyntaxElement.ParameterBuilder => 
                  Util.ice ("strange syntax element appeared in quotation")
              };
            Macros.in_pattern = false;            
            TypePattern (matched_value_type, Macros.patternize_quotation (lifted))
            

          | PT.PExpr.Spliced =>
            ReportError (messenger, 
                         "$ operator may appear only within <[ ... ]> quotation");
            Pattern.Error ()
            

          | PT.PExpr.Ellipsis =>
            ReportError (messenger, 
                         ".. arguments list may appear only within <[ ... ]> quotation");
            Pattern.Error ()
            
            
          | _ =>
            ReportError (messenger, "invalid pattern");
            Pattern.Error ()
        }
      }
    }
  }
}
