/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

namespace Nemerle.Compiler
{
  internal class Typer2
  {
    top_level_fun : NemerleMethod;
    mutable this_ptr_decl : LocalValue;
    mutable current_fun : Fun_header;
    current_type : TypeBuilder;
    messenger : Messenger;

    [System.Flags]
    enum Context {
      | Clean              = 0x0000
      | IsTail             = 0x0001
      | NeedLValue         = 0x0002
      | IsCalledValue      = 0x0004
      | IsIndexerRef       = 0x0008
      | IsDelegeteCtorParm = 0x0010
      | AllowGoto          = 0x0020
      | AllowTry           = 0x0040
      | AllowTryAtFuncLev  = 0x0080
      
      | AllowGotoAndSuch   = AllowGoto %| AllowTry

      | TopLevel           = IsTail %| AllowGoto %| AllowTry
    }
    
    public this (ty : TypeBuilder, fn : NemerleMethod)
    {
      current_fun = fn.GetHeader ();
      top_level_fun = fn;
      current_type = ty;

      messenger = Passes.Solver.CurrentMessenger;
    }
    

    public Run () : void
    {
      foreach (parm in current_fun.parms)
        parm.decl.Register ();

      match (current_fun.body) {
        | FunBody.Typed (expr) =>
          current_fun.body = FunBody.Typed (Walk (Context.TopLevel, expr));
          FixupCurrentFunction ();
        | _ => {}
      }
    }


    Walk (expr : TExpr) : TExpr
    {
      Walk (Context.Clean, expr)
    }


    Walks (exprs : list [TExpr]) : list [TExpr]
    {
      List.Map (exprs, Walk)
    }

    
    Walk (ctx : Context, expr : TExpr) : TExpr
    {
      assert (expr != null);
      Util.locate (expr.loc, {
        messenger.CleanLocalError ();
        def expr = PushConversionDown (expr);
        def res = DoWalk (ctx, expr);
        def res =
          if (res == null)
            expr
          else {
            match (res) {
              | Sequence (_, e)
              | DefValIn (_, _, e) =>
                res.ty = e.Type;
              | _ =>
                res.ty = expr.ty;
            }
            res
          }
        when (ctx %&& Context.IsTail)
          match (res) {
            | TExpr.Call as call =>
              call.is_tail = true;
            | _ => ()
          }
        res
      })
    }
    

    IsSelfCall (func : TExpr) : bool
    {
      match (Unfold (func)) {
        | TExpr.LocalRef (decl) =>
          match (decl.ValKind) {
            | LocalValue.Kind.Function (hd, _) =>
              hd.id == current_fun.id
            | _ => false
          }
        | TExpr.StaticRef (meth is IMethod) =>
          meth.GetHeader ().id == current_fun.id
        | TExpr.MethodRef (th, meth : IMethod, _) =>
          meth.GetHeader ().id == current_fun.id &&
          Unfold (th) is TExpr.This
        | _ => false
      }
    }


    Unfold (expr : TExpr) : TExpr
    {
      match (expr) {
        | TExpr.Delayed (dt) =>
          match (dt.DtKind) {
            | Typer.DelayedTyping.Kind.Resolved (expr) =>
              Unfold (expr)
            | _ => assert (false)
                 //  TExpr.Error ()
          }
        | TExpr.TypeConversion =>
          PushConversionDown (expr)
        | _ => expr
      }
    }


    IsOperator (expr : TExpr) : bool
    {
      match (Unfold (expr)) {
        | TExpr.StaticRef (meth is IMethod) =>
          ! (meth.BuiltinKind is BuiltinMethodKind.NotBuiltin)
        | _ => false
      }
    }

    OperatorKind (expr : TExpr) : BuiltinMethodKind
    {
      match (Unfold (expr)) {
        | TExpr.StaticRef (meth is IMethod) =>
          meth.BuiltinKind
        | _ => assert (false)
      }
    }


    IsCurrentType (tv : TyVar) : bool
    {
      match (tv.Fix ()) {
        | MType.Class (ti, _) =>
          ti.Equals (current_type)
        | _ => false
      }
    }


    GetEventObj (expr : TExpr) : option [IEvent * TExpr]
    {
      match (Unfold (expr)) {
        | TExpr.StaticEventRef (ev) => Some ((ev, null))
        | TExpr.EventMember (obj, ev) => Some ((ev, obj))
        | _ => None ()
      }
    }



    TheSame (e1 : TExpr, e2 : TExpr) : bool
    {
      e1 : object == e2 : object ||
      match ((Unfold (e1), Unfold (e2))) {
        | (TExpr.LocalRef (d1), TExpr.LocalRef (d2)) =>
          d1.Equals (d2)
        | (TExpr.StaticRef (m1), TExpr.StaticRef (m2)) =>
          m1.Equals (m2)
        | (TExpr.This, TExpr.This) => true
        | _ => false
      }
    }

    PushConversionDown (expr : TExpr) : TExpr
    {
      match (expr) {
        | TExpr.Delayed =>
          PushConversionDown (Unfold (expr))
        | TExpr.TypeConversion (e, t, k) as conv =>
          if (k is ConversionKind.Implicit && 
              t.TryUnify (e.Type)) {
            _ = t.Unify (e.Type);
            e
          } else {
            def e = Unfold (e);
            match (e) {
              | TExpr.DefValIn as d =>
                conv.expr = VoidIfNull (d.body);
                d.body = PushConversionDown (conv);
                e

              | TExpr.DefFunctionsIn as d =>
                conv.expr = VoidIfNull (d.body);
                d.body = PushConversionDown (conv);
                e

              | TExpr.Sequence as d =>
                conv.expr = VoidIfNull (d.e2);
                d.e2 = PushConversionDown (conv);
                e

              | _ => expr
            }
          }
        | _ => expr
      }
    }


    ConvertEventExpr (target : TExpr, source : TExpr) : TExpr
    {
      match (GetEventObj (target)) {
        | Some ((ev1, obj1)) =>
          match (Unfold (source)) {
            | TExpr.Call (meth, [p1, p2], _) 
              when IsOperator (meth) =>
              match (OperatorKind (meth)) {
                | BuiltinMethodKind.CallWithCast (meth)
                  when meth.Equals (InternalType.Delegate_Combine) ||
                       meth.Equals (InternalType.Delegate_Remove) =>
                  match (GetEventObj (p1.expr)) {
                    | Some ((ev2, obj2))
                      when ev1.Equals (ev2) && TheSame (obj1, obj2) =>
                      def ev_meth =
                        if (meth.Equals (InternalType.Delegate_Combine))
                          ev1.GetAdder ()
                        else
                          ev2.GetRemover ();
                      assert (ev_meth != null);
                      def ev_meth_ref =
                      // XXX possibly wrong type here
                        if (obj1 == null)
                          TExpr.StaticRef (ev_meth.GetFreshType (), ev_meth)
                        else {
                          def obj1 = Walk (obj1); 
                          TExpr.MethodRef (ev_meth.GetFreshType (), 
                                           obj1, ev_meth, IsBaseRef (obj1));
                        }
                      p2.expr = Walk (p2.expr);
                      TExpr.Call (ev_meth_ref, [p2], false)
                    | _ => null
                  }
                | _ => null
              }
            | _ => null
          }
        | _ => null
      }
    }


    WalkPattern (pat : Pattern) : Pattern
    {
      pat.Walk (fun (_) {
        | Pattern.HasType (tc) =>
          when (pat.ty.TryRequire (tc.GetFreshType ()) && messenger.NeedMessage)
            Message.Warning (pat.loc, "using the ``is'' pattern here is redundant, "
                                      "please use ``:''");
          null
        
        | Pattern.As (_, decl) =>
          unless (decl.IsRegistered)
            decl.Register ();
          null

        | _ => null
      })
    }


    FixupCurrentFunction () : void
    {
      def warned = Hashtable (50);
      
      def locals = List.Rev (current_fun.all_locals);
      current_fun.all_locals = null; // GC it

      unless (current_type.IsDelegate) {
        foreach (v in locals)
          when (!v.EverUsed && v.Name[0] != '_' && !warned.Contains (v.Id)) {
            Message.Warning (168, v.loc, $ "$(v) was never used");
            Message.HintOnce (168, v.loc, "replace name with `_' or prefix it like"
                               " `_bar' to avoid the warning");
            warned.Set (v.Id, null)
          }
        when (this_ptr_decl == null && current_fun : object == top_level_fun.GetHeader ())
          unless (top_level_fun.Attributes %&& (NemerleAttributes.Static |
                                                NemerleAttributes.Virtual | 
                                                NemerleAttributes.Override |
                                                NemerleAttributes.SpecialName ) ||
                  top_level_fun.MemberType == System.Reflection.MemberTypes.Constructor)
            Message.Warning (10006, $"`this' was never used, consider making `$top_level_fun' static");
      }

      foreach (v in locals)
        match (v.ValKind) {
          | LocalValue.Kind.Function (h, _) when !h.closure_vars.IsEmpty =>
            // we cannot yet deal with expanding these to loops
            h.usage = FunctionUsage.UsedAsFirstClass
          | _ => ()
        }

      foreach (v in locals) {
        mutable in_clo = false;

        v.UsedIn.Iter (fun (h : Fun_header) {
          unless (in_clo) {
            // check if all functions from h to current_fun are used_just_once
            def find_defining (header : Fun_header) {
              match (header.usage) {
                | _ when header.id == current_fun.id => {}
                | FunctionUsage.UsedJustOnce =>
                  find_defining (header.decl.DefinedIn)
                | _ =>
                  in_clo = true;
              }
            }
            find_defining (h)
          }
        });

        when (in_clo) {
          // check if we are not closuring some invalid things
          match (v.ValKind) {
            | LocalValue.Kind.FunParm (k) when k != ParmKind.Normal =>
              Message.Error (v.loc, $ "cannot store ref/out parameters in"
                                      " closures ($v)");
            | _ => ()
          };
          v.InClosure = true;
          v.DefinedIn.closure_vars = v :: v.DefinedIn.closure_vars;
        }
      }
    }


    LambdaTransform (expr : TExpr, obj : TExpr, meth : IMethod, notvirt : bool) : TExpr
    {
      def obj_cache = 
        LocalValue (current_fun, "_N_obj_cache", obj.Type,
                    LocalValue.Kind.Plain (), is_mutable = false);
      def (ret_type, parms_types) =
        match (expr.Type.Fix ()) {
          | MType.Fun (from, to) =>
            (to, from.Fix ().GetFunctionArguments ())
          | _ => assert (false)
        }
      def parms = 
        List.Map (parms_types, fun (ty : TyVar) {
          Fun_parm ("fp", 0, ty, Modifiers.Empty, ParmKind.Normal)
        });
      def lambda_header = 
        Fun_header (obj.loc, "_N_method_lambda", ret_type, parms,
                    [], current_fun.tenv);
      def parms_refs =
        List.Map (parms, fun (fp : Fun_parm) {
          fp.decl =
            LocalValue (lambda_header, fp.name, fp.ty,
                        LocalValue.Kind.FunParm (ParmKind.Normal), 
                        is_mutable = false);
          Parm (TExpr.LocalRef (fp.ty, fp.decl))
        });
      def lambda_body =
        TExpr.Call (ret_type, 
           TExpr.MethodRef (expr.Type,
              TExpr.LocalRef (obj_cache.Type, obj_cache),
              meth,
              notvirt), parms_refs, false);
      def parents = current_fun :: current_fun.GetParents ();
      def lambda_value =
        LocalValue (current_fun, lambda_header.name, expr.Type,
                    LocalValue.Kind.Function (lambda_header, parents),
                    is_mutable = false);
      lambda_header.body = FunBody.Typed (lambda_body);
      lambda_header.decl = lambda_value;

      def res =
        TExpr.DefFunctionsIn (expr.Type, [lambda_header], 
                              TExpr.LocalRef (expr.Type, lambda_value));
      def res = TExpr.DefValIn (expr.Type, obj_cache, obj, res);

      Walk (res)
    }


    static VoidIfNull (expr : TExpr) : TExpr
    {
      if (expr == null)
        TExpr.Literal (InternalType.Void, Literal.Void ())
      else
        expr
    }


    IsBaseRef (expr : TExpr) : bool
    {
      match (Unfold (expr)) {
        | TExpr.This as ex => ! IsCurrentType (ex.Type)
          
        | TExpr.LocalRef (decl) as ex =>
          // assert (decl.ValKind != null);
          ! IsCurrentType (ex.Type) &&
          decl.ValKind is LocalValue.Kind.ClosurisedThisPointer
          
        | _ => false
      }
    }


    IgnoreExpr (expr : TExpr) : TExpr
    {
      def solver = Passes.Solver;
      solver.PushState ();
      def ok = expr.Type.Unify (InternalType.Void);
      solver.PopState ();
      if (ok) {
        def res = expr.Type.Unify (InternalType.Void);
        assert (res);
        expr
      } else {
        Message.Warning (10005, expr.Location, 
                         $ "ignored computed value of type $(expr.Type)");
        Message.HintOnce (10005, expr.Location, 
                          "use `_ = ...'; or -nowarn:10005 to avoid the warning");
        
        BuildEnforcement (expr, InternalType.Void);
      }
    }


    BuildEnforcement (expr : TExpr, target : MType) : TExpr
    {
      BuildConversion (expr, target, ConversionKind.UpCast ())
    }
    
    
    BuildConversion (expr : TExpr, target : TyVar, kind : ConversionKind) : TExpr
    {
      // Message.Debug ($ "bc: $expr -> $target ($kind)");

      assert (expr != null);

      def target = target.Fix ();
      def is_ignore = kind is ConversionKind.IgnoreValue;
      
      def kind =
        match (target) {
          | MType.Void when ! is_ignore => ConversionKind.IL (true)
          | _ => kind
        }

      if (kind is ConversionKind.Implicit && 
          target.TryUnify (expr.Type)) {
        _ = target.Unify (expr.Type);
        expr
      } else if (is_ignore) {
        IgnoreExpr (expr)
      } else
        match (kind) {
          | ConversionKind.MethodCall (meth) =>
            TExpr.Call (target, TExpr.StaticRef (meth.GetFreshType (), meth), 
                        [Parm (expr)], false)
          | ConversionKind.DownCast =>
            if (expr.Type.TryRequire (target)) {
              _ = expr.Type.Require (target);
              Message.Warning (10001, $"there is no check needed to cast $(expr.Type) to $target");
              Message.HintOnce (10001, "consider using : instead of :>");
              BuildEnforcement (expr, target)
            } else
              TExpr.TypeConversion (target, expr, target, kind)
          | ConversionKind.Unspecified =>
            if (expr.Type.TryRequire (target)) {
              _ = expr.Type.Require (target);
              BuildEnforcement (expr, target)
            } else
              TExpr.TypeConversion (target, expr, target, ConversionKind.DownCast ())
          | _ =>
            TExpr.TypeConversion (target, expr, target, kind)
        }
    }


    static BuildBlockReturn (ty : TyVar, decl : LocalValue, parms : list [Parm]) : TExpr
    {
      def (result_decl, label) =
        match (decl.ValKind) {
          | LocalValue.Kind.BlockReturn (res, lab) => (res, lab)
          | _ => Util.ice ()
        }

      match (parms) {
        | [] =>
          def ok = result_decl.Type.Unify (InternalType.Void);
          Util.cassert (ok);
          TExpr.Goto (ty, label)
        | [parm] =>
          def ok = parm.expr.Type.Require (result_decl.Type);
          Util.cassert (ok);
          TExpr.Sequence (ty, 
                          TExpr.Assign (InternalType.Void,
                                        TExpr.LocalRef (result_decl),
                                        parm.expr),
                          TExpr.Goto (ty, label))
        | _ => Util.ice ()
      }
    }


    static StripImplicitConversion (expr : TExpr) : TExpr
    {
      match (expr) {
        | TExpr.TypeConversion (e, _, ConversionKind.Implicit) => e
        | e => e
      }
    }


    DoWalk (ctx : Context, expr : TExpr) : TExpr
    {
      //Message.Debug ($ "dowalk: $expr $(ctx %&& Context.AllowGoto)");
      match (expr) {
        | TExpr.LocalRef (decl) =>
          when (decl.ValKind is LocalValue.Kind.BlockReturn) {
            ReportError (messenger, 
                         $ "$decl was used as a first class value, "
                           "this is not supported");
            decl.Register ();
          }
          
          unless (decl.IsRegistered) {
            Message.Warning ($ "unregistered local $decl, $(decl.Id)");
            assert (false);
          }
          decl.UseFrom (current_fun);

          match (decl.ValKind) {
            | LocalValue.Kind.Function (hd, _) =>
              if (ctx %&& Context.IsCalledValue) {
                match (hd.usage) {
                  | FunctionUsage.NotUsed =>
                    // we may set just-once flag only if referencing
                    // from the outer function
                    if (decl.DefinedIn.id == current_fun.id &&
                        (! hd.uses_try_block || ctx %&& Context.AllowTryAtFuncLev))
                      hd.usage = FunctionUsage.UsedJustOnce
                    else
                      hd.usage = FunctionUsage.Used;
                  | FunctionUsage.UsedJustOnce =>
                    hd.usage = FunctionUsage.Used
                  | FunctionUsage.Used | FunctionUsage.UsedAsFirstClass => ()
                }
              } else {
                // Message.Debug ($ "making $decl 1st class");
                hd.usage = FunctionUsage.UsedAsFirstClass;
              }
            | _ => {}
          }
          null
          
          
        | TExpr.StaticRef (mem) =>
          match (mem) {
            | f is IField when ctx & Context.NeedLValue != 0 =>
              f.HasBeenAssigned = true;

            | _ => mem.HasBeenUsed = true;
          }
          null
          
          
        | TExpr.DefFunctionsIn (funs, body) =>
          foreach (fn in funs)
            fn.decl.Register ();
          foreach (fn in funs) {
            def last_fun = current_fun;
            current_fun = fn;
            foreach (parm in fn.parms)
              parm.decl.Register ();
            match (fn.body) {
              | FunBody.Typed (expr) =>
                fn.body = FunBody.Typed (Walk (Context.TopLevel, expr))
              | _ => assert (false)
            }
            FixupCurrentFunction ();
            current_fun = last_fun;
          }
          def body = Walk (ctx & ~Context.IsCalledValue, VoidIfNull (body));
          TExpr.DefFunctionsIn (funs, body)

          
        | TExpr.ImplicitValueTypeCtor (tc) =>
          tc.HasBeenUsed = true;
          null
          

        | TExpr.FieldMember (obj, fld) =>
          if (ctx %&& Context.NeedLValue)
            fld.HasBeenAssigned = true;
          else
            fld.HasBeenUsed = true;
          TExpr.FieldMember (Walk (obj), fld)
          
          
        | TExpr.ConstantObjectRef (mem) =>
          mem.HasBeenUsed = true;
          null
          
          
        | TExpr.PropertyMember (obj, prop) =>
          prop.HasBeenUsed = true;
          def obj = Walk (obj);
          def meth =
            if (ctx %&& Context.NeedLValue) prop.GetSetter ()
            else prop.GetGetter ();
          when (meth == null) {
            assert (!(ctx %&& Context.NeedLValue), "should be rejected by Typer");
            ReportError (messenger, $ "the get accessor is unavailable for $prop");
          }

          unless (meth.CanAccess (current_type))
            Message.Error ($"property accessor $meth is inaccessible");
              
          // for setters the argument list is fixed later
          // XXX possibly wrong type here
          //Message.Debug ($ "prop $meth $obj $(IsBaseRef (obj))");
          def the_ref = TExpr.MethodRef (meth.GetFreshType (), obj, meth, IsBaseRef (obj));
          if (ctx %&& Context.IsIndexerRef)
            the_ref
          else if (prop.IsIndexer) {
            ReportError (messenger, $ "$prop was used as a first class value");
            the_ref
          } else
            TExpr.Call (the_ref, [], false)

        
        | TExpr.StaticPropertyRef (prop) =>
          prop.HasBeenUsed = true;
          def meth =
            if (ctx %&& Context.NeedLValue) prop.GetSetter ()
            else prop.GetGetter ();
          assert (meth != null);
          // XXX possibly wrong type here
          def the_ref = TExpr.StaticRef (meth.GetFreshType (), meth); 
          if (ctx %&& Context.IsIndexerRef)
            the_ref
          else if (prop.IsIndexer) {
            ReportError (messenger, $ "$prop was used as a first class value");
            the_ref
          } else
            TExpr.Call (the_ref, [], false)
          
        
        | TExpr.EventMember (obj, ev) =>
          ev.HasBeenUsed = true;
          def obj = Walk (obj);
          
          mutable field = null;
          when (ev.DeclaringType.Equals (current_type)) {
            field = (ev :> NemerleEvent).storage_field;
          }
          if (field != null)
            TExpr.FieldMember (obj, field);
          else {
            ReportError (messenger, $ "$(ev) can only appear on the left-side of a += or -=");
            TExpr.Error ()
          }
          
        | TExpr.StaticEventRef (ev) =>
          ev.HasBeenUsed = true;

          mutable field = null;
          when (ev.DeclaringType.Equals (current_type)) {
            field = (ev :> NemerleEvent).storage_field;
          }
          if (field != null)
            TExpr.StaticRef (field)            
          else {
            ReportError (messenger, $ "$(ev) can only appear on the left-side of a += or -=");
            TExpr.Error ()
          }

        | TExpr.MethodRef (obj, meth, notvirt) =>
          meth.HasBeenUsed = true;
          def notvirt = notvirt || IsBaseRef (obj);

          if (ctx %&& (Context.IsCalledValue | Context.IsDelegeteCtorParm))
            TExpr.MethodRef (Walk (obj), meth, notvirt)
          else
            LambdaTransform (expr, obj, meth, notvirt)
          

        | TExpr.Call (func, parms, false) =>
          match (Unfold (func)) {
            | TExpr.StaticRef (meth is IMethod) 
              when meth.DeclaringType.IsDelegate &&
                   meth.GetFunKind () is FunKind.Constructor =>
              match (parms) {
                | [parm] =>
                  parm.expr = Walk (Context.IsDelegeteCtorParm, parm.expr)
                | _ => Util.ice ()
              }
            | _ =>
              foreach (p in parms)
                p.expr = Walk (p.expr);
          }

          if (IsOperator (func))
            match (OperatorKind (func)) {
              | BuiltinMethodKind.CallWithCast (meth') =>
                assert (meth' != null);
                // XXX possibly wrong type
                BuildConversion (
                  TExpr.Call (meth'.ReturnType, 
                              TExpr.StaticRef (meth'.GetFreshType (), meth'), parms, false),
                  expr.Type.Fix (), ConversionKind.Unspecified ())
              
              | _ => assert (false)
            }
          else
            match (Unfold (func)) {
              | TExpr.PropertyMember (_, prop) when prop.IsIndexer
              | TExpr.StaticPropertyRef (prop) when prop.IsIndexer => 
                def ctx =
                  Context.IsCalledValue %| 
                  Context.IsIndexerRef  %|
                  (ctx & Context.NeedLValue);
                TExpr.Call (Walk (ctx, func), parms, false)
              | TExpr.ConstantObjectRef (mem) =>
                TExpr.StaticRef (mem) // skip the call
              | TExpr.ImplicitValueTypeCtor as ivtc =>
                ivtc

              | TExpr.LocalRef (LocalValue where (
                  ValKind = LocalValue.Kind.BlockReturn) as decl) =>
                Walk (ctx, BuildBlockReturn (expr.Type, decl, parms))

              | TExpr.OpCode ("==.ref")
              | TExpr.OpCode ("!=.ref") =>
                match (parms) {
                  | [p1, p2] =>
                    def e1 = StripImplicitConversion (p1.expr);
                    def e2 = StripImplicitConversion (p2.expr);

                    def t1 = e1.Type.Fix ();
                    def t2 = e2.Type.Fix ();
                    def prob =
                      if (t1.IsSystemObject && !t2.CanBeNull) t2
                      else if (t2.IsSystemObject && !t1.CanBeNull) t1
                      else null;

                    if (prob != null)
                      ReportError (messenger, 
                                   $ "comparing a value type $prob to "
                                     "System.Object (a `null' literal?) "
                                     "with reference equality");
                    else if (!t1.IsSystemObject && !t2.IsSystemObject) {
                      ReportError (messenger, 
                                   $ "comparing values of types $t1 and $t2 "
                                     "with reference equality");
                      Message.HintOnce ("upcast one of the values to object if "
                                        "this is desired");
                    } else {}

                    TExpr.Call (Unfold (func), parms, false)
                  | _ => assert (false)
                }

              | _ =>
                if ((ctx %&& Context.IsTail) && IsSelfCall (func))
                  TExpr.SelfTailCall (parms)
                else {
                  def ctx =
                    if (ctx %&& Context.AllowTry)
                      Context.AllowTryAtFuncLev | Context.IsCalledValue
                    else Context.IsCalledValue;
                  TExpr.Call (Walk (ctx, func), parms, false)
                }
            }
              
        
        | TExpr.Call
        | TExpr.SelfTailCall => assert (false)
        

        | TExpr.Assign (target, source) =>
          def event_expr = ConvertEventExpr (target, source);
          if (event_expr != null)
            event_expr
          else {
            def transform_call =
              match (target) {
                | TExpr.PropertyMember
                | TExpr.StaticPropertyRef
                | TExpr.Call => true
                | _ => false
              }

            def target = Walk (Context.NeedLValue, target);
            def src_ctx =
              if (target is TExpr.LocalRef) ctx & Context.AllowGotoAndSuch
              else Context.Clean;
            def source = Walk (src_ctx, source);

            match (target) {
              | TExpr.Call as t when transform_call =>
                t.parms += [Parm (source)];
                t
                
              | _ when transform_call => assert (false)
              | _ =>
                TExpr.Assign (target, source)
            }
          }
          
   
        | TExpr.DefValIn (name, val, body) =>
          name.Register ();
          // goto is allowed inside value definition, because
          // we do the store at the end anyway
          def val = Walk (ctx & Context.AllowGotoAndSuch, val);
          def body = Walk (ctx, VoidIfNull (body));
          TExpr.DefValIn (name, val, body)
          
          
        | TExpr.Match (matched_value, cases) =>
          def cast_to = expr.Type.Fix ();
          foreach (case in cases) {
            mutable pats = [];
            foreach ((pat, expr) in case.patterns)
              pats = (WalkPattern (pat), Walk (expr)) :: pats;
            case.patterns = List.Rev (pats);
            case.body = 
              BuildEnforcement (Walk (ctx, VoidIfNull (case.body)), cast_to);
          }
          // We allow try here because matched value is pushed on the 
          // stack first so it is clean (if it was).
          def matched = Walk (ctx & Context.AllowTry, matched_value);
          TExpr.Match (matched, cases)
          

        | TExpr.Throw (exn) =>
          // exception can be null for `throw;' rethrow expression
          if (exn != null)
            TExpr.Throw (Walk (exn))
          else
            expr

        | TExpr.TryWith (body, exn, handler) =>
          unless (ctx %&& Context.AllowTry)
            ReportError (messenger, 
                         "try block is not allowed inside expressions");
          
          exn.Register ();
          current_fun.uses_try_block = true;

          TExpr.TryWith (Walk (Context.AllowTry, body), exn, 
                         Walk (Context.AllowTry, handler))
          
          
        | TExpr.TryFinally (body, handler) =>
          unless (ctx %&& Context.AllowTry)
            ReportError (messenger, 
                         "try block is not allowed inside expressions");
          
          current_fun.uses_try_block = true;
          
          TExpr.TryFinally (Walk (Context.AllowTry, body), 
                            IgnoreExpr (Walk (Context.AllowTry, handler)))
          

        | TExpr.Literal (Literal.Integer as i) =>
          def t = expr.Type.Fix ();
          unless (i.treat_as.Equals (t)) {
            if (Typer.LiteralConversionPossible (i, t))
              i.treat_as = t :> MType.Class;
            else {
              Message.Warning ($ "literal type $(i.treat_as), "
                          "expression type $t, literal is $i");
              assert (false);
            }
          }
          null


        | TExpr.Literal =>
          null
          
        
        | TExpr.This =>
          when (this_ptr_decl == null) {
            this_ptr_decl = 
              LocalValue (
                top_level_fun.GetHeader (), 
                "_N_closurised_this_ptr",
                // cannot use expr.Type, because this can be base
                current_type.GetMemType (), 
                LocalValue.Kind.ClosurisedThisPointer (),
                is_mutable = false);
            this_ptr_decl.Register ();
          }
          this_ptr_decl.UseFrom (current_fun);
          TExpr.LocalRef (this_ptr_decl);
          
          
        | TExpr.Base (meth) =>
          meth.HasBeenUsed = true;
          null
          

        | TExpr.TypeConversion (expr, t, ConversionKind.UpCast)
          when t.Equals (InternalType.Void) && expr.Type.TryUnify (InternalType.Void)
        | TExpr.TypeConversion (expr, _, ConversionKind.IgnoreValue)
          when expr.Type.TryUnify (InternalType.Void) =>
          _ = expr.Type.Unify (InternalType.Void);
          Walk (ctx, expr)
        
        | TExpr.TypeConversion (expr, t, kind) =>
          BuildConversion (Walk (expr), t, kind)
          
          
        | TExpr.Sequence (e1, e2) =>
          def e1 = IgnoreExpr (Walk (ctx & Context.AllowGotoAndSuch, e1));
          TExpr.Sequence (e1, Walk (ctx & ~Context.IsCalledValue, e2))

          
        | TExpr.Tuple (args) =>
          TExpr.Tuple (Walks (args))


        | TExpr.TupleIndexer (obj, k, n) =>
          TExpr.TupleIndexer (Walk (obj), k, n)
          
          
        | TExpr.Array (args, dimensions) =>
          TExpr.Array (Walks (args), Walks (dimensions))
          
          
        | TExpr.TypeOf => null
          
          
        | TExpr.ArrayIndexer (obj, args) =>
          TExpr.ArrayIndexer (Walk (obj), Walks (args))
          
        
        | TExpr.OpCode => null
        
        | TExpr.Delayed =>
          Walk (ctx, Unfold (expr))


        | TExpr.Error =>
          null
          

        | TExpr.Goto =>
          unless (ctx %&& Context.AllowGoto)
            ReportError (messenger, "goto (block return?) is not allowed "
                                    "inside expressions");
          null
          
          
        | TExpr.Label (lab, body) =>
          TExpr.Label (lab, Walk (ctx, body))


        | TExpr.If (cond, e1, e2) =>
          TExpr.If (Walk (cond), Walk (ctx, e1), Walk (ctx, e2))


        | TExpr.HasType (e, t) =>
          TExpr.HasType (Walk (e), t)
          
        
        | TExpr.DefaultValue => null

       
        | TExpr.NotNull
        | TExpr.Switch
        | TExpr.MultipleAssign
        | TExpr.MethodAddress => assert (false)
      }
    }
  }
}
