/* Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 

namespace Sioux.Csl
{
  using Nemerle.Collections;
  using Nemerle.Xml;
  using Nemerle.IO;

  using System.Xml;
  using System.Xml.Xsl;

  using Sioux;


  /**
   * The submissions class
   */
  class Submission 
  {
    values : Hashtable [string, string];

    internal val (name : string) : string
    {
      Csl.get_val (values, name)
    }

    generate_id () : void
    {
      def sb = System.Text.StringBuilder (17);
      for (mutable i = 0; i < 16; i = i + 1) {
        ignore (sb.Append (((random.Next (24) + ('a' :> int)) :> char)))
      };
      values.Set ("id", sb.ToString ());
    }

    internal store () : void
    {
      lock (Csl.submissions) {
        match (Csl.submissions.Get (val ("id"))) {
          | Some (subm) => 
            def t = Csl.get_val (subm.values, "time_started");
            when (t != "")
              values.Set ("time_started", t);
          | None => ()
        };
        Csl.submissions.Set (val ("id"), this);
      }
    }

    internal read_post (pv : Hashtable [string, string], ignore_id : bool) : void
    {
      def old_id = val ("id");
      def read (name) { values.Set (name, Csl.get_val (pv, name).Trim ()) };
      iter_all (read);
      when (ignore_id || val ("id") == "")
        values.Set ("id", old_id);
        
      values.Set ("time_last_edit", System.DateTime.Now.ToString ());

      def id = val ("id");
      def loop (i) {
        if (i >= id.Length) true
        else {
          def ch = (id[i] :> int);
          if (ch >= ('a' :> int) && ch <= ('z' :> int)) loop (i + 1)
          else false
        }
      };
      if (id.Length != 16 || !loop (0)) {
        throw DieException ()
      } else {
        when (Option.IsNone (validate ()))
          store ()
      }
    }

    internal serialize (doc : XmlDocument) : XmlNode
    {
      def n = doc.CreateElement ("submission");
      def add_field (name) {
        def n' = doc.CreateElement (name);
        ignore (n.AppendChild (n'));
        ignore (n'.AppendChild (doc.CreateTextNode (val (name))));
      };
      iter_all (add_field);
      add_field ("time_started");
      add_field ("time_last_edit");
      n
    }

    internal unserialize (n : XmlNode) : void
    {
      def add_field (n : XmlNode) {
        when (n != null) {
          when (n.NodeType == XmlNodeType.Element) {
            values.Set (n.Name, n.InnerText)
          };
          add_field (n.NextSibling)
        }
      };
      add_field (n.FirstChild)
    }

    iter_all (f : string -> void) : void
    {
      List.Iter (text_fields, f);
      List.Iter (bool_fields, f);
      List.Iter (edit_fields, f);
      f ("id");
    }

    internal dump () : string
    {
      def res = System.Text.StringBuilder ();
      def read (name) { ignore (res.Append (sprintf ("%s='%s'\n", name, val (name)))) };
      iter_all (read);
      res.ToString ()
    }

    internal this ()
    {
      values = Hashtable ();
      values.Set ("time_started", System.DateTime.Now.ToString ());
      values.Set ("time_last_edit", System.DateTime.Now.ToString ());
      generate_id ();
    }
    
    internal get_submission_form (t : XmlTemplate) : XmlTemplate
    {
      List.Iter ("id" :: text_fields, fun (name) {
                  t.SetText (name + "/value", val (name))
                });
      t.SetText ("id2/value", val ("id"));
      List.Iter (bool_fields, fun (name) {
                  when (val (name) != "")
                    t.SetText (name + "/checked", "checked")
                });
      List.Iter (edit_fields, fun (name) {
                  t.SetText (name, val (name))
                });
      
      t
    }

    internal send_kill_email (admin_mode : bool) : void
    {
      def m = System.Web.Mail.MailMessage ();
      m.BodyEncoding = System.Text.Encoding.GetEncoding ("iso-8859-1");
      m.Subject = "Confirmation of removal of CSL sumbission.";
      def csl_email = "CSL'04 <cslsub@nemerle.org>";
      def real_name = sprintf ("%s %s <%s>", val ("first_name"), 
                               val ("last_name"), val ("email"));
      if (admin_mode)
        m.To = csl_email
      else {
        m.To = real_name;
//        m.Cc = csl_email;
      };
      m.From = csl_email;
      m.Body = "Submission removed.\n" +
                "Submission data:\n" + dump ();
      System.Web.Mail.SmtpMail.Send (m)
    }

    private calculate_sum () : int
    {
      def accomp = (if (val ("accompany") == "on") true else false);
      def stay = (if (val ("depart_stay") == "on") 25 else 0);
      def stay = (if (accomp) 2 * stay else stay);
      def accomp = (if (accomp) 250 else 0);
      match ((val ("fee_regular"), val ("fee_student"))) {
        | ("on", "on") => -1
        | ("on", _) =>
	  440 + accomp + stay
	| (_, "on") =>
          300 + accomp + stay
        | _ => -2
      }
    }

    internal send_confirmation_email (admin_mode : bool, edit_mode : bool) : void
    {
      def m = System.Web.Mail.MailMessage ();
      m.BodyEncoding = System.Text.Encoding.GetEncoding ("iso-8859-1");
      m.Subject = 
        if (edit_mode)
          "Confirmation of edition of CSL'04 submission"
        else
          "Confirmation of participation submission for CSL'04";
      def csl_email = "CSL'04 <cslsub@nemerle.org>";
      if (admin_mode)
        m.To = csl_email
      else {
        m.To = sprintf ("%s %s <%s>", val ("first_name"), 
                          val ("last_name"), val ("email"));
//        m.Cc = csl_email;
      };
      m.From = csl_email;

      def mees = System.Text.StringBuilder ();
      if (edit_mode) 
        ignore (mees.Append ("Received changes to CSL'04 submission.\n\n"))
      else 
        ignore (mees.Append ("Received sumbission for CSL'04.\n\n"));
      def _ = mees.Append ("Your fee (");
      def _ = mees.Append (calculate_sum ().ToString ());
      def _ = mees.Append (" Euro) must be transfered, within 10 days from the");
      def _ = mees.Append ("registration, to the account:\n\nowned by:");
      def _ = mees.Append (" Polskie Stowarzyszenie dla Maszyn Liczacych\n\n");
      def _ = mees.Append ("in: Bank Zachodni WBK S.A. IV/O Wroclaw ul. Kuznicza ");
      def _ = mees.Append ("17/19 Wroclaw\n\nAccount No: 65 1090 2503 0000 0006 3000 0597");
      def _ = mees.Append ("\n\nMake sure your name appears on the bank transfer. ");
      def _ = mees.Append ("The transfer must be free of charge for the beneficiary.\n\n");
      def _ = mees.Append ("To change your submission visit:\n");
      def _ = mees.Append ("  http://lilith.ii.uni.wroc.pl:8001/edit.xml?id=" + val ("id") + "\n\n");
      def _ = mees.Append ("Submission data:\n");
      def _ = mees.Append (dump ());

      m.Body = mees.ToString ();
      System.Web.Mail.SmtpMail.Send (m)
    }

    internal validate () : option [XmlTemplate]
    {
      def templ = XmlTemplate ("csl04/error.xml");
      def broken (n) { val (n).Trim () == "" };
      if (List.Exists (["email", "first_name", "last_name"], broken)) {
        templ.SetText ("msg", "There are not filled fields.");
        Some (templ)
      } 
      else if (val ("email").LastIndexOf ('@') == -1) {
        templ.SetText ("msg", "Invalid email address.");
	Some (templ)
      }
      else if (calculate_sum () < 0) {
        match (calculate_sum ()) {
          | -1 => 
	    templ.SetText ("msg", "Both regular and student fee were chosen");
          | _ =>
            templ.SetText ("msg", "Regular or student fee must be chosen");
        };
        Some (templ)
      }
      else None ();
    }

    internal get_submission_confirm (t : XmlTemplate, 
                                     admin_mode : bool, 
                                     edit_mode : bool) : XmlTemplate
    {
      match (validate ()) {
        | Some (t) => t
        | None =>
          if (admin_mode)
            t.SetText ("email", "CSL")
          else
            t.SetText ("email", val ("email"));
          t.SetText ("fee", calculate_sum ().ToString ());
          t.SetText ("dump_here", dump ());
          t.SetText ("edit/href", (if (admin_mode) Csl.secret_password else "") 
                                  + "/edit.xml?id=" + val ("id"));
          send_confirmation_email (admin_mode, edit_mode);
          
          t
      }
    }

    static text_fields : list [string];
    static bool_fields : list [string];
    static edit_fields : list [string];
    static random : System.Random;

    static this ()
    {
           
      text_fields = [
        "first_name",
        "last_name",
        "email",
        "title",
        "organization",
        "address",
        "zip_city",
        "country",
        "fax",
        "fee_roommate",
        "accompany_name",
        "arrive_airport_time",
        "arrive_center_time",
        "depart_friday_time",
        "depart_saturday_time"
      ];
      bool_fields = [
        "depart_book",
        "arrive_center",
        "depart_friday",
        "depart_stay",
        "depart_saturday",
        "fee_regular",
        "fee_student",
        "vega",
        "accompany",
        "accompany_vega",
        "arrive_directly",
        "arrive_book",
        "arrive_airport"
      ];
      edit_fields = ["remarks"];
      random = System.Random ();
    }
  }
}
