/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Xml;
using Nemerle.Collections;

namespace Nemerle.Xml
{
    public module Transformers
    {
        /**
         *  Transformer that rewrites included urls, it doesn't rewrites excluded urls
         */
        public EncodeUrlTransformer ( include : string, exclude : string) : XmlTransformer
        {            
            def encode_url_transformer(template : XmlTemplate) : XmlTemplate
            {
                def doc = template.GetDocument();

                def make_list(s : string) : list [string * string]
                {
                    def delimiter     = array ['|'];              
                    def expressions   = s.Split(delimiter);

                    mutable exprs_list = [];
                    
                    foreach(expr : string in expressions)
                    {
                        def delimiter = array['/'];
                        def expr      = expr.Split(delimiter);
                        when(expr.Length == 2)
                            exprs_list = (expr[0],expr[1])::exprs_list;
                    }

                    exprs_list
                }

                def include_expressions = make_list(include);
                def exclude_expressions = make_list(exclude);
                    
                def walk (n : XmlNode,expr_list : list [string * string],
                          modify : (XmlElement * string * string) -> void) 
                {
                    mutable attributes_to_modify = [];
                    
                    when (n.NodeType == XmlNodeType.Element)
                    {
                        def element = (n :> XmlElement);
                        def attributes = element.Attributes;

                        foreach(attribute :> XmlAttribute in attributes)
                        {
                            def select_attribute(exprs : list [string * string])
                            {
                              | [] => ()
                              | (e,a) :: rest =>
                                {
                                  if((e == element.Name || e == "*") && (a == attribute.Name || a == "*"))
                                    attributes_to_modify = (attribute) :: attributes_to_modify;
                                  else
                                    select_attribute(rest);
                                }
                            }

                            select_attribute(expr_list)
                        }

                        def modify_selected_attributes(selected_attributes : list [XmlAttribute])
                        {
                            match(selected_attributes)
                            {
                                | [] => ()
                                | attr :: rest =>
                                  {
                                      modify(element,attr.Name,attr.Value);
                                      modify_selected_attributes(rest);
                                  }
                            }
                        }

                        modify_selected_attributes(attributes_to_modify);
                     }

                     def walk_children (cur : XmlNode,lst,modify)
                     {
                         if (cur == null)
                             ()
                         else 
                         {
                             walk (cur,lst, modify);
                             walk_children (cur.NextSibling,lst,modify);
                         }
                     }

                     walk_children (n.FirstChild,expr_list,modify)
                 }

                 def modify_excluded (element : XmlElement,attribute_name : string,attribute_value : string)
                 {
                     def delimiter = array[';'];
                     def split = attribute_value.Split(delimiter);
                     
                     when(split.Length == 2)
                         element.SetAttribute(attribute_name,split[0])
                 }
                 
                 def modify_included (element : XmlElement,attribute_name : string,attribute_value : string)
                 {                    
                     when(attribute_name != "href" || !attribute_value.StartsWith("#"))
                         element.SetAttribute(attribute_name,attribute_value + ";EncodeUrl()")
                 }
                
                 walk(doc,include_expressions,modify_included);
                 walk(doc,exclude_expressions,modify_excluded);

                 template
            }

            XmlTransformer.Handler(encode_url_transformer)
        }

        /**
         * Default transformer, includes all href, action attributes and src attribute of frame field
         * Excludes from rewriting src attribute of img field
         */
        public EncodeUrlTransformer () : XmlTransformer
        {
            EncodeUrlTransformer ("*/href|*/action|frame/src","img/src")
        }
    }
}
