using System.Globalization;

using Nemerle.Compiler;
using Nemerle.IO;

class LexerDumper : LexerFile {
  mutable last_loc : Location = Location.Default;

  public this (file : string) {
    base (file);
  }

  private Escape (str : string) : string {
    str.Replace ("&", "&amp;").Replace ("<", "&lt;").Replace (">", "&gt;")
  }
  
  public override GetToken () : Token {
    def tok = base.GetToken ();
    def token_str = 
      match (tok) {
        | Token.Identifier (name) => Escape (name)

        | Token.Keyword (name) => $"<span class='key'>$name</span>"

        | Token.Operator (name) => $"<span class='op'>$(Escape (name))</span>"
        | Token.Semicolon => "<span class='op'>;</span>"
        | Token.Comma => "<span class='op'>,</span>"          
        | Token.BeginBrace  => "<span class='op'>{</span>"            
        | Token.EndBrace    => "<span class='op'>}</span>"  
        | Token.BeginRound  => "<span class='op'>(</span>"  
        | Token.EndRound    => "<span class='op'>)</span>"  
        | Token.BeginSquare => "<span class='op'>[</span>"  
        | Token.EndSquare   => "<span class='op'>]</span>"  
        | Token.BeginQuote  => "<span class='op'><[</span>"  
        | Token.EndQuote    => "<span class='op'>]></span>"  

        | Token.StringLiteral (value) =>
          def value = Escape (value.Replace ("\n", "\\n"));
          $"<span class='str'>\"$value\"</span>"

        | Token.CharLiteral (value) =>  $"<span class='char'>'$(value)'</span>"

        | Token.SByteLiteral (value) => $"<span class='num'>$(value)SB</span>"

        | Token.ByteLiteral (value) =>  $"<span class='num'>$(value)B</span>"

        | Token.ShortLiteral (value) => $"<span class='num'>$(value)S</span>"

        | Token.UShortLiteral (value) => $"<span class='num'>$(value)US</span>"

        | Token.IntLiteral (value) => $"<span class='num'>$(value)</span>"

        | Token.UIntLiteral (value) => $"<span class='num'>$(value)U</span>"

        | Token.LongLiteral (value) => $"<span class='num'>$(value)L</span>"

        | Token.ULongLiteral (value) => $"<span class='num'>$(value)UL</span>"

        | Token.FloatLiteral (value) =>
          def str = value.ToString (CultureInfo.InvariantCulture);
          $"<span class='num'>$(str)f</span>"

        | Token.DoubleLiteral (value) =>
          def str = value.ToString (CultureInfo.InvariantCulture);
          $"<span class='num'>$(str)</span>"

        | Token.DecimalLiteral (value) =>
          def str = value.ToString (CultureInfo.InvariantCulture);
          $"<span class='num'>$(str)M</span>"

        | Token.Comment (value) =>
          def value = value.TrimEnd (array [' ', '\n']);
          def height_loop (i, sum) {
            if (i < value.Length)
              if (value [i] == '\n') height_loop (i + 1, sum + 1)
              else height_loop (i + 1, sum)
            else
              sum
          }
          if (height_loop (0, 1) > 1)
            $"<span class='comment'>/* $(value) */</span>"
          else
            $"<span class='comment'>// $(value)\n</span>"

        | Token.EndOfFile => ""
        | x => Util.ice ("this token should not come from lexer: " + x.ToString ());
      }
    unless ((tok matches Token.EndOfFile) || (defines.Get ("OMMIT") matches Some (true))) {
      mutable cur_loc = tok.Location;
      if (cur_loc.Line != last_loc.EndLine) {
        repeat (cur_loc.Line - last_loc.EndLine) print ("\n");
        repeat (cur_loc.Column) print (" ");
      }
      else
        repeat (cur_loc.Column - last_loc.EndColumn) print (" ");

//      print (tok.Location.ToString ());
      /// dump current token string            
      print (token_str);

      when (tok matches Token.Comment)
        print (" ");

      // remember current location as the previous one
      last_loc = cur_loc;
    }
    tok
  }
}

module Dumper {
  Main (args : array [string]) : void {
    Message.InitOutput (System.Console.Out);
    
    if (args.Length < 1) printf ("usage: dumper.exe filename.n DEF1 DEF2 ...\n");
    else {
      Options.Sources = [args [0]];
      Options.XmlDoc = true;
      LexerFile.store_comments = true;
      Options.ProgressBar = false;
      Passes.LexingPipeline = LexerDumper;
      Passes.ScanningPipeline = fun (_) { 
        print ("\n</pre>\n");
        System.Environment.Exit (0);
      };

      for (mutable i = 1; i < args.Length; ++i)
        LexerDumper.command_defines.Set (args [i], true);

      print ("<pre class='code'>\n");
      Passes.Run ();
    }
  }
}

