/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2009 Sun Microsystems, Inc.
 */

package org.netbeans.modules.cnd.discovery.project;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.modules.cnd.api.project.NativeFileItem;
import org.netbeans.modules.cnd.api.toolchain.CompilerSet;
import org.netbeans.modules.cnd.api.toolchain.CompilerSetManager;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.CsmInclude;
import org.netbeans.modules.cnd.api.model.CsmModel;
import org.netbeans.modules.cnd.api.model.CsmModelAccessor;
import org.netbeans.modules.cnd.api.model.CsmProject;
import org.netbeans.modules.cnd.api.model.util.UIDs;
import org.netbeans.modules.cnd.api.project.NativeProject;
import org.netbeans.modules.cnd.api.remote.RemoteFileUtil;
import org.netbeans.modules.nativeexecution.api.util.Path;
import org.netbeans.modules.cnd.discovery.projectimport.ImportProject;
import org.netbeans.modules.cnd.makeproject.MakeOptions;
import org.netbeans.modules.cnd.makeproject.MakeProjectTypeImpl;
import org.netbeans.modules.cnd.makeproject.api.wizards.WizardConstants;
import org.netbeans.modules.cnd.modelimpl.csm.core.ModelImpl;
import org.netbeans.modules.cnd.modelimpl.csm.core.ProjectBase;
import org.netbeans.modules.cnd.modelimpl.test.ModelBasedTestCase;
import org.netbeans.modules.cnd.repository.support.RepositoryTestUtils;
import org.netbeans.modules.cnd.test.CndCoreTestUtils;
import org.netbeans.modules.cnd.utils.FSPath;
import org.netbeans.modules.cnd.utils.cache.CndFileUtils;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironmentFactory;
import org.netbeans.modules.nativeexecution.api.NativeProcess;
import org.netbeans.modules.nativeexecution.api.NativeProcessBuilder;
import org.netbeans.modules.remote.spi.FileSystemProvider;
import org.openide.WizardDescriptor;
import org.openide.util.Exceptions;
import org.openide.util.RequestProcessor;
import org.openide.util.Utilities;

/**
 *
 * @author Alexander Simon
 */
public abstract class MakeProjectTestBase extends ModelBasedTestCase { //extends NbTestCase
    protected static final String LOG_POSTFIX = ".discoveryLog";
    private static final boolean TRACE = false;
    private final Logger logger1;

    public MakeProjectTestBase(String name) {
        super(name);
        if (TRACE) {
            System.setProperty("cnd.discovery.trace.projectimport", "true"); // NOI18N
            System.setProperty("org.netbeans.modules.cnd.test.CndTestIOProvider.traceout","true"); // NOI18N
        }
        //System.setProperty("org.netbeans.modules.cnd.makeproject.api.runprofiles", "true"); // NOI18N
//        System.setProperty("cnd.modelimpl.assert.notfound", "true");
        System.setProperty("cnd.mode.unittest", "true");
        System.setProperty("org.netbeans.modules.cnd.apt.level","OFF"); // NOI18N
        //System.setProperty("cnd.modelimpl.timing","true"); // NOI18N
        System.setProperty("parser.report.include.failures","true"); // NOI18N
        //System.setProperty("cnd.modelimpl.timing.per.file.flat","true"); // NOI18N
        //System.setProperty("cnd.dump.native.file.item.paths","true"); // NOI18N
        logger1 = Logger.getLogger("org.netbeans.modules.editor.settings.storage.Utils");
        logger1.setLevel(Level.SEVERE);
        //System.setProperty("org.netbeans.modules.cnd.apt.level","WARNING"); // NOI18N
        //Logger.getLogger("org.netbeans.modules.cnd.apt").setLevel(Level.WARNING);
        //MockServices.setServices(MakeProjectType.class);
    }

    protected boolean optimizeNativeExecutions() {
        return false;
    }

    protected boolean optimizeSimpleProjects() {
        return false;
    }

    @Override
    protected void setUp() throws Exception {
        super.setUp();
        //MockServices.setServices(MakeProjectType.class);
        MakeOptions.getInstance().setFixUnresolvedInclude(false);
        startupModel();
    }

    @Override
    protected List<Class<?>> getServices() {
        List<Class<?>> list = new ArrayList<>();
        list.add(MakeProjectTypeImpl.class);
        list.addAll(super.getServices());
        return list;
    }
 
    @Override
    protected void setUpMime() {
        // setting up MIME breaks other services
        super.setUpMime();
    }

    private void startupModel() {
        ModelImpl model = (ModelImpl) CsmModelAccessor.getModel();
        model.startup();
        RepositoryTestUtils.deleteDefaultCacheLocation();
    }

    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
        shutdownModel();
    }

    private void shutdownModel() {
        ModelImpl model = (ModelImpl) CsmModelAccessor.getModel();
        model.shutdown();
        RepositoryTestUtils.deleteDefaultCacheLocation();
    }

    protected File detectConfigure(String path){
        File configure = new File(path, "configure");
        if (configure.exists()) {
            return configure;
        }
        configure = new File(path, "CMakeLists.txt");
        if (configure.exists()) {
            return configure;
        }
        File base = new File(path);
        File[] files = base.listFiles();
        if (files != null){
            for(File file : files) {
                if (file.getAbsolutePath().endsWith(".pro")){
                    return file;
                }
            }
        }
        if (new File(path, "configure").exists()) {
            return new File(path, "configure");
        }
        if (files != null){
            for(File file : files) {
                if (file.isDirectory()) {
                    File res = new File(file,"configure");
                    if (res.exists()) {
                        return res;
                    }
                }
            }
        }
        return new File(path, "configure");
    }
    
    protected ExecutionEnvironment getEE(){
        return ExecutionEnvironmentFactory.getLocal();
    }

    public void performTestProject(String URL, List<String> additionalScripts, boolean useSunCompilers, final String subFolder) throws Exception {
        Map<String, String> tools = findTools();
        final ExecutionEnvironment ee = getEE();
        CompilerSetManager csm = CompilerSetManager.get(ee);
        while (csm.isPending()) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException ex) {
                // skip
            }
        }
        CompilerSet def = csm.getDefaultCompilerSet();
        if (useSunCompilers) {
            if (def != null && def.getCompilerFlavor().isGnuCompiler()) {
                for(CompilerSet set : CompilerSetManager.get(ee).getCompilerSets()){
                    if (set.getCompilerFlavor().isSunStudioCompiler()) {
                        CompilerSetManager.get(ee).setDefault(set);
                        break;
                    }
                }
            }
        } else {
            if (def != null && def.getCompilerFlavor().isSunStudioCompiler()) {
                for(CompilerSet set : CompilerSetManager.get(ee).getCompilerSets()){
                    if (set.getCompilerFlavor().isGnuCompiler()) {
                        CompilerSetManager.get(ee).setDefault(set);
                        break;
                    }
                }
            }
        }
        def = CompilerSetManager.get(ee).getDefaultCompilerSet();
        final boolean isSUN = def != null ? def.getCompilerFlavor().isSunStudioCompiler() : false;
        if (tools == null) {
            assertTrue("Please install required tools.", false);
            System.err.println("Test did not run because required tools do not found");
            return;
        }
        try {
            String aPath = download(URL, additionalScripts, tools)+subFolder;

            final File configure = detectConfigure(aPath);
            final String path;
            if (configure.exists()) {
                path = configure.getParent();
            } else {
                path = aPath;
            }
            final File makeFile = new File(path, "Makefile");
            if (!configure.exists()) {
                if (!makeFile.exists()){
                    assertTrue("Cannot find configure or Makefile in folder "+path, false);
                }
            }
            if (Utilities.isWindows()){
                // cygwin does not allow test discovery in real time, so disable tests on windows
                //return;
            }

            // For simple configure-make projects store logs for reuse
            boolean generateLogs = false;
            if (optimizeSimpleProjects() && "configure".equals(configure.getName())) {
                generateLogs = !hasLogs(new File(path));
            }
            
            WizardDescriptor wizard = new WizardDescriptor() {
                @Override
                public synchronized Object getProperty(String name) {
                    if (WizardConstants.PROPERTY_SIMPLE_MODE.equals(name)) {
                        return Boolean.TRUE;
                    } else if (WizardConstants.PROPERTY_NATIVE_PROJ_DIR.equals(name)) {
                        return path;
                    } else if (WizardConstants.PROPERTY_NATIVE_PROJ_FO.equals(name)) {
                        return CndFileUtils.toFileObject(path);
                    } else if (WizardConstants.PROPERTY_PROJECT_FOLDER.equals(name)) {
                        ExecutionEnvironment ee = ExecutionEnvironmentFactory.getLocal();
                        return new FSPath(FileSystemProvider.getFileSystem(ee), RemoteFileUtil.normalizeAbsolutePath(path, ee));
                    } else if (WizardConstants.PROPERTY_TOOLCHAIN.equals(name)) {
                        return CompilerSetManager.get(ee).getDefaultCompilerSet();
                    } else if (WizardConstants.PROPERTY_HOST_UID.equals(name)) {
                        return ExecutionEnvironmentFactory.toUniqueID(ee);
                    } else if (WizardConstants.PROPERTY_CONFIGURE_SCRIPT_PATH.equals(name)) {
                        if (optimizeNativeExecutions() && makeFile.exists()){// && !configure.getAbsolutePath().endsWith("CMakeLists.txt")) {
                            // optimization on developer computer:
                            // run configure only once
                            return null;
                        } else {
                            return configure.getAbsolutePath();
                        }
                    } else if ("realFlags".equals(name)) {
                        if (path.indexOf("cmake-")>0 && subFolder.isEmpty()) {
                            if (isSUN) {
                                return "CMAKE_C_COMPILER=cc CMAKE_CXX_COMPILER=CC CFLAGS=-g CXXFLAGS=-g CMAKE_BUILD_TYPE=Debug CMAKE_CXX_FLAGS_DEBUG=-g CMAKE_C_FLAGS_DEBUG=-g";
                            } else {
                                return "CFLAGS=\"-g3 -gdwarf-2\" CXXFLAGS=\"-g3 -gdwarf-2\" CMAKE_BUILD_TYPE=Debug CMAKE_CXX_FLAGS_DEBUG=\"-g3 -gdwarf-2\" CMAKE_C_FLAGS_DEBUG=\"-g3 -gdwarf-2\"";
                            }
                        } else {
                            if (configure.getAbsolutePath().endsWith("configure")) {
                                if (isSUN) {
                                    return "CC=cc CXX=CC CFLAGS=-g CXXFLAGS=-g";
                                } else {
                                    return "CFLAGS=\"-g3 -gdwarf-2\" CXXFLAGS=\"-g3 -gdwarf-2\"";
                                }
                            } else if (configure.getAbsolutePath().endsWith("CMakeLists.txt")) {
                                if (isSUN) {
                                    return "-G \"Unix Makefiles\" -DCMAKE_BUILD_TYPE=Debug -DCMAKE_C_COMPILER=cc -DCMAKE_CXX_COMPILER=CC -DCMAKE_CXX_FLAGS_DEBUG=-g -DCMAKE_C_FLAGS_DEBUG=-g -DCMAKE_EXPORT_COMPILE_COMMANDS=ON";
                                } else {
                                    return "-G \"Unix Makefiles\" -DCMAKE_BUILD_TYPE=Debug -DCMAKE_CXX_FLAGS_DEBUG=\"-g3 -gdwarf-2\" -DCMAKE_C_FLAGS_DEBUG=\"-g3 -gdwarf-2\" -DCMAKE_EXPORT_COMPILE_COMMANDS=ON";
                                }
                            } else if (configure.getAbsolutePath().endsWith(".pro")) {
                                if (isSUN) {
                                    return "-spec solaris-cc QMAKE_CC=cc QMAKE_CXX=CC QMAKE_CFLAGS=-g QMAKE_CXXFLAGS=-g";
                                } else {
                                    if (Utilities.getOperatingSystem() == Utilities.OS_MAC) {
                                        return "-spec macx-g++ QMAKE_CFLAGS=\"-g3 -gdwarf-2\" QMAKE_CXXFLAGS=\"-g3 -gdwarf-2\"";
                                    } else {
                                        if (Utilities.isWindows()) {
                                            for (CompilerSet set : CompilerSetManager.get(ee).getCompilerSets()){
                                                if (set.getCompilerFlavor().isMinGWCompiler()) {
                                                    CompilerSetManager.get(ee).setDefault(set);
                                                    break;
                                                }
                                            }
                                        }
                                        return "QMAKE_CFLAGS=\"-g3 -gdwarf-2\" QMAKE_CXXFLAGS=\"-g3 -gdwarf-2\"";
                                    }
                                }
                            }
                        }
                    } else if ("buildProject".equals(name)) {
                        if (optimizeNativeExecutions() && makeFile.exists() && findObjectFiles(path)) {
                            // optimization on developer computer:
                            // make only once
                            return Boolean.FALSE;
                        } else {
                            return Boolean.TRUE;
                        }
                    }
                    return null;
                }
            };

            ImportProject importer = new ImportProject(wizard);
            importer.setUILessMode();
            if (generateLogs) {
                importer.setConfigureLog(new File(path, "configure" + LOG_POSTFIX));
                importer.setMakeLog(new File(path, "Makefile" + LOG_POSTFIX));
            }
            importer.create();
            OpenProjects.getDefault().open(new Project[]{importer.getProject()}, false);
            int i = 0;
            while(!importer.isFinished()) {
                try {
                    Thread.sleep(100);
                } catch (InterruptedException ex) {
                    Exceptions.printStackTrace(ex);
                }
                if (i > 10 && !OpenProjects.getDefault().isProjectOpen(importer.getProject())){
                    break;
                }
                i++;
            }

            if (generateLogs) {
                hackConfigure(configure);
                hackMakefile(makeFile);
            }

            assertEquals("Failed configure", ImportProject.State.Successful, importer.getState().get(ImportProject.Step.Configure));
            assertEquals("Failed build", ImportProject.State.Successful, importer.getState().get(ImportProject.Step.Make));
            CsmModel model = CsmModelAccessor.getModel();
            Project makeProject = importer.getProject();
            assertTrue("Not found model", model != null);
            assertTrue("Not found make project", makeProject != null);
            NativeProject np = makeProject.getLookup().lookup(NativeProject.class);
            assertTrue("Not found native project", np != null);
            CsmProject csmProject = model.getProject(np);
            assertTrue("Not found model project", csmProject != null);
            csmProject.waitParse();
            perform(csmProject);
            OpenProjects.getDefault().close(new Project[]{makeProject});
        } catch (IOException ex) {
            Exceptions.printStackTrace(ex);
            assertTrue(ex.getMessage(), false);
        }
    }

    protected boolean findObjectFiles(String path){
        return findObjectFiles(new File(path));
    }

    private boolean findObjectFiles(File file){
        if (file.isDirectory()) {
            File[] ff = file.listFiles();
            if (ff != null) {
                for(File f : ff){
                    if (f.isDirectory()) {
                        boolean b = findObjectFiles(f);
                        if (b) {
                            return true;
                        }
                    } else if (f.isFile() && f.getName().endsWith(".o")) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    protected List<String> requiredTools(){
        List<String> list = new ArrayList<>();
        list.add("wget");
        list.add("gzip");
        list.add("tar");
        list.add("rm");
        list.add("cp");
        return list;
    }

    protected Map<String, String> findTools(){
        Map<String, String> map = new HashMap<>();
        for(String t: requiredTools()){
            map.put(t, null);
        }
        if (findTools(map)){
            return map;
        }
        return null;
    }

    private boolean findTools(Map<String, String> map){
        if (map.isEmpty()) {
            return true;
        }
        for (String path : Path.getPath()) {
            for(Map.Entry<String, String> entry : map.entrySet()){
                if (entry.getValue() == null) {
                    String task = path+File.separatorChar+entry.getKey();
                    File tool = new File(task);
                    if (tool.exists() && tool.isFile()) {
                        entry.setValue(task);
                    } else if (Utilities.isWindows()) {
                        task = task+".exe";
                        tool = new File(task);
                        if (tool.exists() && tool.isFile()) {
                            entry.setValue(task);
                        }   
                    }
                }
            }
        }
        boolean res = true;
        for(Map.Entry<String, String> entry : map.entrySet()){
           if (entry.getValue() == null) {
               System.err.println("Not found required tool: "+entry.getKey());
               res =false;
           } else {
              if (TRACE) {
                   System.err.println("Found required tool: "+entry.getKey()+"="+entry.getValue());
              }
           }
        }
        return res;
    }

    protected void perform(CsmProject csmProject) throws Exception {
        try {
            Thread.sleep(1000);
        } catch (InterruptedException ex) {
            Exceptions.printStackTrace(ex);
        }
        csmProject.waitParse();
        Collection<CsmFile> col = csmProject.getAllFiles();
        if (TRACE) {
            System.err.println("Model has "+col.size()+" files");
        }
        boolean hasInvalidFiles = false;
        for (CsmFile file : col) {
            if (TRACE) {
                //System.err.println("\t"+file.getAbsolutePath());
            }
            boolean fileHasUnresolved = false;
            for(CsmInclude include : file.getIncludes()){
                if (include.getIncludeFile() == null) {
                    hasInvalidFiles = true;
                    fileHasUnresolved = true;
                    System.err.println("Not resolved include directive "+include.getIncludeName()+" in file "+file.getAbsolutePath());
                }
            }
            if (fileHasUnresolved) {
                NativeFileItem nativeFileItem = ((ProjectBase) file.getProject()).getNativeFileItem(UIDs.get(file));
                if (nativeFileItem != null) {
                    for(FSPath path : nativeFileItem.getUserIncludePaths()) {
                        System.err.println("\tSearch path "+path.getPath());
                    }
                }
            }
        }
        assertNoExceptions();
        assertFalse("Model has unresolved include directives", hasInvalidFiles);
    }

    protected String download(String urlName, List<String> additionalScripts, Map<String, String> tools) throws IOException {
        String zipName = urlName.substring(urlName.lastIndexOf('/')+1);
        String tarName = zipName.substring(0, zipName.lastIndexOf('.'));
        String packageName = tarName.substring(0, tarName.lastIndexOf('.'));
        File fileDataPath = CndCoreTestUtils.getDownloadBase();
        String dataPath = fileDataPath.getAbsolutePath();
        File localFilesStorage = new File(System.getProperty("user.home"), "cnd-test-files-storage");
        File fileFromStorage = new File(localFilesStorage, zipName);

        File fileCreatedFolder = new File(fileDataPath, packageName);
        String createdFolder = fileCreatedFolder.getAbsolutePath();
        if (!fileCreatedFolder.exists()){
            fileCreatedFolder.mkdirs();
        } else {
            if (!optimizeNativeExecutions() &&
                    (!optimizeSimpleProjects() || !hasLogs(fileCreatedFolder))) {
                execute(tools, "rm", dataPath, "-rf", packageName);
                fileCreatedFolder.mkdirs();
            }
        }
        if (fileCreatedFolder.list().length == 0){
            if (!new File(fileDataPath, tarName).exists()) {
                if (fileFromStorage.exists()) {
                    execute(tools, "cp", dataPath, fileFromStorage.getAbsolutePath(), dataPath);
                } else {
                    if (urlName.startsWith("http")) {
                        execute(tools, "wget", dataPath, "-qN", urlName);
                    } else {
                        execute(tools, "cp", dataPath, urlName, dataPath);
                    }
                }
                execute(tools, "gzip", dataPath, "-d", zipName);
            }
            execute(tools, "tar", dataPath, "xf", tarName);
            execAdditionalScripts(createdFolder, additionalScripts, tools);
        } else {
            final File configure = new File(fileCreatedFolder, "configure");
            final File makeFile = detectConfigure(createdFolder);
            if (!configure.exists()) {
                if (!makeFile.exists()){
                    execAdditionalScripts(createdFolder, additionalScripts, tools);
                }
            }
        }
        execute(tools, "rm", createdFolder, "-rf", "nbproject");
        return createdFolder;
    }

    private void execute(Map<String, String> tools, String cmd, String folder, String ... arguments){
        String command = tools.get(cmd);
        StringBuilder buf = new StringBuilder();
        for(String arg : arguments) {
            buf.append(' ');
            buf.append(arg);
        }
        if (TRACE) {
            System.err.println(folder+"#"+command+buf.toString());
        }
        NativeProcessBuilder ne = NativeProcessBuilder.newProcessBuilder(ExecutionEnvironmentFactory.getLocal())
        .setWorkingDirectory(folder)
        .setExecutable(command)
        .setArguments(arguments);
        ne.redirectError();
        waitExecution(ne);
    }

    private void execAdditionalScripts(String createdFolder, List<String> additionalScripts, Map<String, String> tools) throws IOException {
        if (additionalScripts != null) {
            for(String s: additionalScripts){
                int i = s.indexOf(' ');
                String command = s.substring(0,i);
                String arguments = s.substring(i+1);
                command = tools.get(command);
                if (TRACE) {
                    System.err.println(createdFolder+"#"+command+" "+arguments);
                }
                //NativeExecutor ne = new NativeExecutor(createdFolder, tools.get(command), arguments, new String[0], command, "run", false, false);
                NativeProcessBuilder ne = NativeProcessBuilder.newProcessBuilder(ExecutionEnvironmentFactory.getLocal())
                .setWorkingDirectory(createdFolder)
                .setCommandLine(command+" "+arguments);
                waitExecution(ne);
            }
        }
    }

    private void waitExecution(NativeProcessBuilder ne){
        try {
            NativeProcess process = ne.call();
            final BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            RequestProcessor RP = new RequestProcessor("command", 2);
            RP.post(new Runnable() {
                @Override
                public void run() {
                    try {
                        while (true) {
                            String line = reader.readLine();
                            if (line == null) {
                                break;
                            } else {
                                if (TRACE) {
                                  System.out.println(line);
                                }
                            }
                        }
                    } catch (IOException ex) {
                        Exceptions.printStackTrace(ex);
                    } finally {
                        try {
                            reader.close();
                        } catch (IOException ex) {
                            Exceptions.printStackTrace(ex);
                        }
                    }
                }
            });
            final BufferedReader reader2 = new BufferedReader(new InputStreamReader(process.getErrorStream()));
            RP.post(new Runnable() {
                @Override
                public void run() {
                    try {
                        while (true) {
                            String line = reader2.readLine();
                            if (line == null) {
                                break;
                            } else {
                                System.out.println(line);
                            }
                        }
                    } catch (IOException ex) {
                        Exceptions.printStackTrace(ex);
                    } finally {
                        try {
                            reader2.close();
                        } catch (IOException ex) {
                            Exceptions.printStackTrace(ex);
                        }
                    }
                }

            });
            int rc = process.waitFor();
        } catch (InterruptedException ex) {
            Exceptions.printStackTrace(ex);
        } catch (IOException ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    protected static boolean hasLogs(File projectDir) {
        File configureLog = new File(projectDir, "configure" + LOG_POSTFIX);
        if (!configureLog.exists()) {
            return false;
        }
        File makeLog = new File(projectDir, "Makefile" + LOG_POSTFIX);
        return makeLog.exists();
    }

    protected static void hackConfigure(File file) {
        try {
            BufferedReader in = new BufferedReader(new FileReader(file));
            String firstLine = in.readLine();
            in.close();
            BufferedWriter out = new BufferedWriter(new FileWriter(file));
            out.write(firstLine);
            out.newLine();
            out.write("cat \"" + file.getAbsolutePath() + LOG_POSTFIX + "\"");
            out.close();
        } catch (IOException e) {
        }
    }

    protected static void hackMakefile(File file) {
        try {
            BufferedWriter out = new BufferedWriter(new FileWriter(file));
            out.write("all:");
            out.newLine();
            out.write("\tcat \"" + file.getAbsolutePath() + LOG_POSTFIX + "\"");
            out.close();
        } catch (IOException e) {
        }
    }
}
