/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <netinet/ip.h>

#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd.h>

#include <nd_udp.h>
#include <nd_udp_callbacks.h>

#ifdef LINUX
struct ipovly
{
  u_char           ih_x1[9];
  u_char           ih_pr;
  u_int16_t        ih_len;
  struct in_addr   ih_src;
  struct in_addr   ih_dst;
};
#endif


static ND_ProtoField udp_fields[] = {
  { ND_VAL_FIELD, N_("Src. port (%u)"),    N_("Source port number"),      16, nd_udp_sport_cb  },
  { ND_VAL_FIELD, N_("Dst. port (%u)"),    N_("Destination port number"), 16, nd_udp_dport_cb  },
  { ND_VAL_FIELD, N_("Length (%u)"),       N_("UDP packet length"),       16, nd_udp_length_cb },
  { ND_VAL_FIELD, N_("Checksum (0x%.4x)"), N_("UDP checksum"),            16, nd_udp_csum_cb   },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData udp_menu_data[] = {
  { N_("Fix Checksums"),  N_("Fixes the UDP checksums of the current selection"), 0, nd_udp_cksum_fix_cb },
  { NULL, NULL, 0, NULL}
};


static ND_Protocol *udp;

static ND_Protocol *
udp_get_ip(void)
{
  static ND_Protocol *ip = NULL;

  if (!ip)
    {
      ip = nd_proto_registry_find(ND_PROTO_LAYER_NET, 0x0800);
      
      if (ip->id == 1)
	ip = NULL;
    }

  return ip;
}

static gboolean
udp_header_complete(guchar *data, guchar *data_end)
{
  D(("UDP header complete? %p %p --> %i\n", data, data_end, (data + 8 <= data_end)));
  return (data + 8 <= data_end);
}

static gboolean
udp_get_first(const ND_Packet *packet, struct ip **iphdr, struct udphdr **udphdr)
{
  GList        *l;
  ND_Protocol  *ip;
  ND_ProtoData *pd, *pd_prev;

  if (!packet || !iphdr || !udphdr)
    return FALSE;

  if (! (ip = udp_get_ip()))
    return FALSE;

  /* Walk the proto data and find the first UDP
     (we don't support nesting UDP) with a preceding IP
  */
  
  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (g_list_previous(l) && pd->inst.proto == udp)
	{
	  pd_prev = (ND_ProtoData *) g_list_previous(l)->data;

	  if (pd_prev->inst.proto == ip)
	    {
	      *iphdr  = (struct ip *) pd_prev->data;
	      *udphdr = (struct udphdr *) pd->data;

	      return TRUE;
	    }
	}
    }
  
  return FALSE;
}

/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("UDP Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the UDP protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  udp = nd_proto_new("UDP", ND_PROTO_LAYER_TRANS, IPPROTO_UDP);
			     		
  udp->create_gui      = nd_udp_create_gui;
  udp->set_gui         = nd_udp_set_gui;
  udp->init_packet     = nd_udp_init_packet;
  udp->header_complete = nd_udp_header_complete;
  udp->fix_packet      = nd_udp_fix_packet;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  udp->fields = udp_fields;
  udp->header_width = 32;

  /* We provide a little menu that allows us to fix checksums. */
  udp->menu = nd_gui_create_menu(udp_menu_data);
	     
  return udp;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_udp_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_udp_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct udphdr *udphdr;
  
  if (pinf->inst.nesting != 0)
    D(("WARNING -- UDP plugin doesn't support nested UDP\n"));

  udphdr = (struct udphdr*) nd_packet_get_data(packet, udp, 0);

  nd_udp_set_gui_sport(pinf, udphdr);
  nd_udp_set_gui_dport(pinf, udphdr);
  nd_udp_set_gui_len(pinf, udphdr);
  nd_udp_set_gui_csum(pinf, udphdr, packet);
}


void       
nd_udp_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  struct udphdr *udphdr;
  ND_Protocol   *payload_proto;

  udphdr = (struct udphdr *) data;
  
  if (!udp_header_complete(data, data_end))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }
  
  nd_packet_add_proto_data(packet, udp, data, data_end);  
  
  /* If UDP contains any payload, check the appriopriate header
     field value and demultiplex packet initialization up to the
     next correct protocol: */

  payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_APP, ntohs(udphdr->uh_dport));
  payload_proto->init_packet(packet, data + 8, data_end);
}


gboolean
nd_udp_header_complete(const ND_Packet *packet, guint nesting)
{
  ND_ProtoData *pd;

  if (!packet)
    return FALSE;

  if (! (pd = nd_packet_get_proto_data(packet, udp, nesting)))
    return FALSE;

  return udp_header_complete(pd->data, pd->data_end);
}


gboolean   
nd_udp_datagram_complete(const ND_Packet *packet, guint nesting)
{
  ND_ProtoData  *pd;
  struct udphdr *udphdr;

  if (!packet)
    return FALSE;

  if (! (pd = nd_packet_get_proto_data(packet, udp, nesting)))
    return FALSE;

  udphdr = (struct udphdr *) pd->data;

  return (pd->data + ntohs(udphdr->uh_ulen) <= nd_packet_get_end(packet));
}


gboolean
nd_udp_fix_packet(ND_Packet *packet, int index)
{
  struct udphdr *udphdr;
  guint16        correct_sum;
  
  if (!packet)
    return FALSE;
  
  if (!nd_udp_csum_correct(packet, &correct_sum))
    {
      udphdr = (struct udphdr*) nd_packet_get_data(packet, udp, 0);
      udphdr->uh_sum = correct_sum;
      nd_packet_modified_at_index(packet, index);
      return TRUE;
    }

  return FALSE;
}


guint16
nd_udp_checksum(const ND_Packet *packet)
{
  ND_Protocol    *ip;
  struct ip*      iphdr = NULL;
  struct udphdr  *udphdr = NULL;
  struct ipovly   ipovly;
  guint16        *w;
  guint           i, preadd;
  guint16         old, result = 0;

  if (!packet || !(ip = udp_get_ip()))
    return 0;

  if (!udp_get_first(packet, &iphdr, &udphdr))
    return 0;
  
  w  = (guint16 *) &ipovly;
  preadd = 0;
      
  memset(&ipovly, 0, sizeof(struct ipovly));
  ipovly.ih_pr  = iphdr->ip_p;
  ipovly.ih_len = udphdr->uh_ulen;
  ipovly.ih_src = iphdr->ip_src;
  ipovly.ih_dst = iphdr->ip_dst;
      
  for (i = 0; i < sizeof(struct ipovly)/sizeof(guint16) ; i++)
    preadd += *w++;

  old = udphdr->uh_sum;
  udphdr->uh_sum = 0;
  result = nd_misc_in_cksum((u_short*)udphdr, ntohs(udphdr->uh_ulen), preadd);
  udphdr->uh_sum = old;

  return result;
}

  
/* Misc helper stuff below --------------------------------------------- */

void
nd_udp_set_gui_sport(ND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[0], DATA_TO_PTR(ntohs(udphdr->uh_sport)));
}


void
nd_udp_set_gui_dport(ND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[1], DATA_TO_PTR(ntohs(udphdr->uh_dport)));
}


void
nd_udp_set_gui_len(ND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[2], DATA_TO_PTR(ntohs(udphdr->uh_ulen)));
}


void
nd_udp_set_gui_csum(ND_ProtoInfo *pinf, struct udphdr *udphdr, const ND_Packet *packet)
{
  nd_proto_field_set(pinf, &udp_fields[3], DATA_TO_PTR(ntohs(udphdr->uh_sum)));
  
  if (!nd_udp_datagram_complete(packet, 0))
    {
      nd_proto_info_field_set_state(pinf,
				    &udp_fields[3],
				    ND_FIELD_STATE_UNKN);
      return;
    }
  
  if (!nd_udp_csum_correct(packet, NULL))
    nd_proto_info_field_set_state(pinf,
				  &udp_fields[3],
				  ND_FIELD_STATE_ERROR);
  else
    nd_proto_info_field_set_state(pinf,
				  &udp_fields[3],
				  ND_FIELD_STATE_NORMAL);
}


ND_Protocol *
nd_udp_get(void)
{
  return udp;
}


gboolean   
nd_udp_csum_correct(const ND_Packet *packet, guint16 *correct_sum)
{
  struct udphdr *udphdr;
  guint16 old_sum, correct_sum_tmp;

  if (!packet)
    return FALSE;

  udphdr = (struct udphdr *) nd_packet_get_data(packet, nd_udp_get(), 0);

  old_sum = udphdr->uh_sum;
  correct_sum_tmp = nd_udp_checksum(packet);

  if (correct_sum)
    *correct_sum = correct_sum_tmp;

  return (old_sum == correct_sum_tmp);
}
