/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>

#include <netdude/nd.h>
#include <netdude/nd_clipboard.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_packet.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_trace_registry.h>
#include <support.h>

static ND_Packet *cb = NULL;
static ND_Packet *cb_end = NULL;
static int        cb_size;
static char      *cb_content_string = NULL;

static void
nd_clipboard_free(void)
{
  ND_Packet *p, *p2;

  if (cb)
    {
      p = cb;
      
      while (p)
	{
	  p2 = p;
	  p = p->next;
	  nd_packet_free(p2);
	}

      cb = cb_end = NULL;
      cb_size = 0;
    }
}


static void
clipboard_create_content_string(ND_Trace *trace)
{
  GtkCList *list;
  GList *sel, *sel_copy = NULL;
  gchar *text, *ptr;
  int    strlength = 0, len;

  if (!trace || !trace->list)
    return;

  g_free(cb_content_string);
  list = GTK_CLIST(trace->list);

  /* Go through all selected rows and record the string lengths,
     also build a copy of the list, we'll sort that one later. */

  for (sel = list->selection; sel; sel = g_list_next(sel))
    {
      sel_copy = g_list_prepend(sel_copy, GINT_TO_POINTER(GTK_CLIST_ROW(sel)));
      gtk_clist_get_text(list, (int) GTK_CLIST_ROW(sel), 0, &text);
      strlength += strlen(text) + 1;
    }

  cb_content_string = g_new0(gchar, strlength + 1);
  if (!cb_content_string)
    return;

  sel_copy = g_list_sort(sel_copy, nd_misc_int_cmp);

  ptr = cb_content_string;
  
  /* We now have a string of sufficient length and a list of rows
     sorted by increasing row number. Iterate again and put together
     our selection content! */

  for (sel = sel_copy; sel; sel = g_list_next(sel), ptr++)
    {
      gtk_clist_get_text(list, (int) GTK_CLIST_ROW(sel), 0, &text);
      len = strlen(text);
      memcpy(ptr, text, len);
      ptr += len;
      *ptr = '\n';
    }
  
  *ptr = '\0';


  /* Make sure that we now own the selection */

  gtk_selection_owner_set(toplevel,
			  GDK_SELECTION_PRIMARY,
			  GDK_CURRENT_TIME);

  /* And clean our mess up. */

  g_list_free(sel_copy);
}


/* This callback gets called when a different app wants to
   know the contents of the primary selection and we've
   signaled that we possess it. It creates a string out of
   all currently selected packets in the current trace. */
static void
clipboard_selection_handle(GtkWidget *list,
			   GtkSelectionData *selection_data,
			   gpointer data)
{
  /* Let the requesting app know of our selection */

  if (!cb_content_string)
    return;

  gtk_selection_data_set(selection_data, GDK_SELECTION_TYPE_STRING, 8,
			 cb_content_string, strlen(cb_content_string));


  return;
  TOUCH(list);
  TOUCH(data);
}


void
nd_clipboard_init(void)
{
  /* Selection stuff -- register our callback that returns
     our concept of NEtdude's current selection, i.e. the
     tcpdump output of the currently selected packets: */

  gtk_selection_add_target(toplevel,
			   GDK_SELECTION_PRIMARY,
			   GDK_SELECTION_TYPE_STRING, 1);

  gtk_signal_connect(GTK_OBJECT(toplevel), "selection_get",
		     GTK_SIGNAL_FUNC(clipboard_selection_handle),
		     NULL);
}


void  
nd_clipboard_cut(void)
{
  ND_Packet *p, *c = NULL;
  ND_Trace  *trace;

  return_if_no_current_trace(trace);
  nd_clipboard_free();

  if ( (p = nd_trace_sel_get(trace, TRUE)))
    {
      for ( ; p; p = p->sel_next)
	{
	  if (p->prev)
	    {
	      if (p->next)
		{
		  p->prev->next = p->next;
		  p->next->prev = p->prev;
		}
	      else
		{
		  p->prev->next = NULL;
		}
	    }
	  else if (p->next)
	    {
	      trace->pl = p->next;
	      p->next->prev = NULL;
	    }
	  else
	    {
	      trace->pl = NULL;
	      trace->pl_end = NULL;
	    }

	  if (c)
	    {
	      c->next = p;
	      p->prev = c;
	      c = p;
	    }
	  else
	    {
	      cb = c = p;
	    }
	}

      cb_end = c;
      cb_end->next = NULL;

      trace->num_packets -= nd_trace_sel_size(trace);
      cb_size = nd_trace_sel_size(trace);
      
      trace->sel.sel = NULL;
      nd_trace_clear_selection(trace);

      nd_trace_set_dirty(trace, TRUE);
      nd_gui_list_remove_selected_rows(trace->list);
    }
}


void  
nd_clipboard_copy(void)
{
  ND_Packet *p, *p2, *c, *s;
  ND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_clipboard_free();

  if ( (s = nd_trace_sel_get(trace, TRUE)))
    {
      char message[MAXPATHLEN];
      int  copied = 0;
      
      c = nd_packet_duplicate(s);
      
      if (!c)
	return;
      
      cb = c;
      copied++;
      clipboard_create_content_string(trace);
            
      for (p = s->sel_next, cb_end = cb; p; p = p->sel_next)
	{
	  if ( (p2 = nd_packet_duplicate(p)))
	    {
	      c->next = p2;
	      p2->prev = c;
	      c = p2;
	      cb_end = c;
	      
	      copied++;
	    }
	}
      
      cb_size = copied;
      
      g_snprintf(message, MAXPATHLEN, _("Copied %i packet(s) into clipboard."), cb_size);
      nd_gui_statusbar_set(message);
    }  
}


void  
nd_clipboard_paste(void)
{
  ND_Trace  *trace;
  GtkCList  *clist;
  ND_Packet *p, *p2, *c, *cb_copy = NULL, *cb_copy_end = NULL, *current;
  int        index = 0, pasted = 0;
  char       line[MAXPATHLEN];
  char      *s[2];
  char       message[MAXPATHLEN];

  if (!cb)
    return;

  return_if_no_current_trace(trace);

  current = nd_trace_get_current_packet(trace);
  clist = GTK_CLIST(trace->list);
  D_ASSERT_PTR(clist);

  /* If the trace we're pasting to is initialized
   * already, make sure the pasted packets have the
   * same linktype. Also adjust the maximum snaplen
   * if the clipboard packets are larger than the
   * ones already present:
   */

  if (nd_trace_initialized(trace))
    {
      if (cb->trace->tcpdump.pfh.linktype != 
	  trace->tcpdump.pfh.linktype)
	{
	  nd_gui_statusbar_set(_("Packets in clipboard are of different "
				 "link type, pasting aborted."));
	  return;
	}

      if (cb->trace->tcpdump.pfh.snaplen >
	  trace->tcpdump.pfh.linktype)
	{
	  D(("Adjusting snaplen to %u\n", cb->trace->tcpdump.pfh.snaplen));
	  trace->tcpdump.pfh.linktype = cb->trace->tcpdump.pfh.snaplen;
	}
    }
  else
    {
      D(("First paste --> initializing trace.\n"));
      nd_trace_init_header(trace,
			   cb->trace->tcpdump.pfh.snaplen,
			   cb->trace->tcpdump.pfh.linktype);

      /* FIXME -- revisit this. In order to not crash when
       * saving this new trace, we need a pcap_t, but where
       * do we take it from? For now just open the original
       * file a second time.
       *
       * What happens if we edit a file, create a new one,
       * paste stuff, delete the first file,
       * --> we end up with a stale pcap_t ... and it should be
       * easy to just save a tcpdump save file without actually
       * having a pcap_t around!!!
       */
      
      trace->pcap = pcap_open_offline(cb->trace->filename, pcap_errbuf);
    }
  
  if (current)
    index = nd_packet_get_index(current);
  
  cb_copy = c = nd_packet_duplicate(cb);

  if (!cb_copy)
    return;

  cb_copy->trace = trace;
  
  for (p = cb->next, cb_copy_end = cb_copy; p; p = p->next)
    {
      if ( (p2 = nd_packet_duplicate(p)))
	{
	  c->next = p2;
	  p2->prev = c;
	  c = p2;
	  cb_copy_end = c;
	  c->trace = trace;
	}
    }
  
  s[0] = line;
  s[1] = "";
  
  for (c = cb_copy; c; c = c->next)
    {
      pasted++;
      nd_tcpdump_get_packet_line(c, line, TRUE);
      gtk_clist_insert(clist, index++, s);
    }
  
  if (current)
    {
      p = current->prev;
      if (p)
	{
	  p->next = cb_copy;
	  cb_copy->prev = p;
	}
      else
	{
	  trace->pl = cb_copy;
	  cb_copy->prev = NULL;
	}
      
      cb_copy_end->next = current;
      current->prev = cb_copy_end;
    }
  else
    {
      trace->pl = cb_copy;
      cb_copy->prev = NULL;      
    }
  
  trace->num_packets += pasted;
  nd_gui_num_packets_set();
  nd_trace_set_dirty(trace, TRUE);
  
  g_snprintf(message, MAXPATHLEN, _("Pasted %i packet(s) from clipboard."), pasted);
  nd_gui_statusbar_set(message);
}


int   
nd_clipboard_occupied(void)
{
  return (cb != NULL);
}


