/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_packet_h
#define __nd_packet_h

#include <netdude/nd.h>

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>

#include <pcap.h>

/* Netdude's view of a packet: */


/* Remember to update nd_packet_duplicate() when adding
   stuff here !!! */
struct nd_packet
{
  /* The pcap header */
  struct pcap_pkthdr    ph;

  guchar               *data;

  /* The trace this packet belongs to */
  ND_Trace             *trace;

  /* Protocols at various layers into the packet */
  GList                *pd; /* GList<ND_ProtoData*>*/
  
  gint64                protocols;

  /* Whether the packet is currently hidden
     in the GUI display. */ 
  int                   is_hidden     : 1;

  /* Pointers to the next/previous selected
     packets, for quicker iteration. */
  struct nd_packet     *sel_next;
  struct nd_packet     *sel_prev;

  /* Linked list pointers. */
  struct nd_packet     *next;
  struct nd_packet     *prev;

};


/**
 * ND_PacketFunc - callback prototype for iterating over protocols.
 * @packet: the packet over whose protocol data is iterated.
 * @pd: iterated protocol data.
 * @user_data: arbitrary data passed through to this callback.
 *
 * Functions of this signature are used to iterate over the
 * protocols that are contained in packtes. See
 * nd_packet_foreach_proto() and nd_packet_foreach_proto_backward().
 */
typedef void (*ND_PacketFunc)(ND_Packet    *packet,
			      ND_ProtoData *pd,
			      void         *user_data);

/**
 * nd_packet_new - allocates a packet.
 * @trace: trace the new packet will belong to.
 *
 * The function creates a packet and returns it. Each packet must
 * belong to a trace. The function does not register the packet
 * in the trace, because a packet could be inserted anywhere.
 *
 * Returns: fresh packet.
 */
ND_Packet      *nd_packet_new(ND_Trace *trace);


/**
 * nd_packet_pcap_read_handle -  convenience pcap read handler
 * @data: an #ND_Trace pointer.
 * @h: pointer to a pcap packet header
 * @pdata: packet data
 *
 * This is a pcap read handler that expects the data ptr to be an
 * ND_Trace * and adds the captured packet to the trace. The GUI
 * is not updated -- use nd_pcap_read_handler_gui() if you need to.
 */
void            nd_packet_pcap_read_handler(u_char *data,
					    const struct pcap_pkthdr *h,
					    const u_char *pdata);

/**
 * nd_packet_pcap_read_handle -  convenience pcap read handler
 * @data: an #ND_Trace pointer.
 * @h: pointer to a pcap packet header
 * @pdata: packet data
 *
 * This is a pcap read handler that expects the data ptr to be an
 * ND_Trace * and adds the captured packet to the trace. Additionally,
 * it updates the tcpdump list of the trace with each packet
 * received.
 */
void            nd_packet_pcap_read_handler_gui(u_char *data,
						const struct pcap_pkthdr *h,
						const u_char *pdata);

/**
 * nd_packet_free - destructor.
 * @packet: packet to clean up.
 *
 * This cleans up the memory used by a packet. It does
 * not take care of removing the packet from the tcpdump
 * list etc -- you likely want to use nd_packet_delete()!
 */
void            nd_packet_free(ND_Packet *packet);


/**
 * nd_packet_duplicate - deep-copy packet
 * @packet: packet to copy.
 *
 * Returns a deep copy of a packet, but with the
 * list pointers set to NULL (i.e. the copy does not
 * belong to any packet list).
 *
 * Returns: packet copy or NULL when error occured.
 */
ND_Packet      *nd_packet_duplicate(ND_Packet *packet);


/**
 * nd_packet_delete - cleans up a packet
 * @packet: deleted packet.
 * @update_gui: whether to update the GUI or not.
 *
 * This function removes @packet from its trace's
 * packet lists, removes the GUI line if @gui_update
 * is TRUE, then frees @packet.
 */
void            nd_packet_delete(ND_Packet *packet, gboolean update_gui);


/**
 * nd_packet_init - initializes packet internally.
 * @packet: packet to initialize.
 *
 * The function initializes a packet, its data offset pointers,
 * and the protocol types for the protcol stack. It cleans
 * up before adjusting internal settings, so you can call this
 * repeatedly.
 */
void            nd_packet_init(ND_Packet *packet);


/**
 * nd_packet_init_from_pcap - initializes packet internally.
 * @packet: packet to initialize.
 * @pcap: pcap handle.
 *
 * The function initializes a packet using information provided
 * in the given pcap handle. Use this if you want to initialize
 * a standalone packet that is not part of an ND_Trace. Other
 * than that, it's just like nd_packet_init().
 */
void            nd_packet_init_from_pcap(ND_Packet *packet, pcap_t *pcap);


/**
 * nd_packet_is_complete - was packet completely captured?
 * @packet: packet to check.
 *
 * The predicate returns %TRUE if the packet was captured
 * completely (i.e. pcap's caplen == len) and %FALSE otherwise.
 *
 * Returns: completeness result.
 */
gboolean        nd_packet_is_complete(const ND_Packet *packet);


/**
 * nd_packet_update - updates partial packet state
 * @packet: packet to update.
 * @proto: protocol where to start update.
 * @nesting: at what nesting of @proto to update.
 *
 * This is nd_packet_init()'s little brother. It doesn't initialize
 * the whole packet, but only the part starting at @proto with
 * nesting level @nesting.
 */
void            nd_packet_update(ND_Packet *packet, ND_Protocol *proto, guint nesting);


/**
 * nd_packet_set_gui - syncs GUI.
 * @packet: packet that provides settings.
 *
 * The function adjusts the GUI to packet contents, for all
 * protocols contained in the packet.
 */
void            nd_packet_set_gui(const ND_Packet *packet);


/**
 * nd_packet_get_data - data accessing function
 * @packet: packet to query.
 * @proto: protocol to look up.
 * @nesting: nesting level of @proto to use.
 *
 * The function returns a pointer to the packet data containing
 * the start of @proto's data at the given nesting level. If the
 * packet doesn't contain any such data, %NULL is returned.
 *
 * Returns: data pointer.
 */
guchar         *nd_packet_get_data(const ND_Packet *packet,
				   const ND_Protocol *proto,
				   guint nesting);

/**
 * nd_packet_get_data_end - data accessing function
 * @packet: packet to query.
 * @proto: protocol to look up.
 * @nesting: nesting level of @proto to use.
 *
 * The function returns a pointer to the first byte after the
 * data containing the start of @proto's data at the given nesting
 * level. If the packet doesn't contain any such data, %NULL
 * is returned.
 *
 * Returns: data pointer.
 */
guchar         *nd_packet_get_data_end(const ND_Packet *packet,
				       const ND_Protocol *proto,
				       guint nesting);


/**
 * nd_packet_add_proto_data - adds internal protocol information.
 * @packet: packet to update.
 * @proto: protocol type of data chunk
 * @data: start of protocol data.
 * @data_end: first byte of data after protocol data.
 *
 * Each packet internally maintains a list of data offsets which
 * store info about the protocol type, nesting etc. This function
 * appends a new chunk information to that list. It is called from
 * nd_packet_init() and nd_packet_update(), chances are you want
 * these functions instead.
 */
void            nd_packet_add_proto_data(ND_Packet *packet,
					 ND_Protocol *proto,
					 guchar *data,
					 guchar *data_end);

/**
 * nd_packet_get_proto_data - protocol information accessor.
 * @packet: packet to query.
 * @proto: protocol to look up.
 * @nesting: nesting level for @proto.
 *
 * This function is like nd_packet_get_data, but does not only
 * return the data pointer but the full #ND_ProtoData structure,
 * which yields nesting info, data start end end pointers etc.
 *
 * Returns: protocol data or %NULL if n/a.
 */
ND_ProtoData   *nd_packet_get_proto_data(const ND_Packet *packet,
					 const ND_Protocol *proto,
					 guint nesting);

/**
 * nd_packet_get_end -  returns pointer to end of packet data.
 * @packet: packet to query.
 *
 * The function returns a ponter to the byte following the last
 * byte of the captured data.
 *
 * Returns: data pointer.
 */
guchar         *nd_packet_get_end(const ND_Packet *packet);


/**
 * nd_packet_has_complete_header - whether a given protocol's header is present.
 * @packet: packet to query.
 * @proto: protocol to check for.
 * @nesting: nesting level for @proto.
 *
 * The predicate returns %TRUE when a packet contains a complete header
 * of the requested protocol, %FALSE otherwise. The implementation of
 * the check itself is up to the implementation of @proto's plug-in.
 * If you only need to know whether a protocol is present in the packet
 * at all, use nd_packet_has_proto(), which is faster.
 *
 * Returns: check result.
 */
gboolean        nd_packet_has_complete_header(const ND_Packet *packet,
					      const ND_Protocol *proto,
					      guint nesting);

/**
 * nd_packet_has_proto - whether a given protocol is present in packet.
 * @packet: packet to query.
 * @proto: protocol to check for.
 *
 * The predicate returns %TRUE when a packet contains data of the
 * given protocol. It may contain multiple instances of a protocol,
 * but the function only checks if a protocol is present at all.
 * If you need to find out whether a protocol is present at a given
 * nesting level (e.g. whether a packet contains IP in IP), use
 * nd_packet_has_complete_header() or nd_packet_has_proto_nested().
 *
 * Returns: check result.
 */
gboolean        nd_packet_has_proto(const ND_Packet *packet,
				    const ND_Protocol *proto);


/**
 * nd_packet_has_proto_nested - whether a protocol is present in packet at given nesting level.
 * @packet: packet to query.
 * @proto: protocol to check for.
 * @nesting: nesting level of @proto.
 *
 * The predicate returns %TRUE when a packet contains data of the
 * given protocol at the requested nesting level.
 * If you only need to find out whether a protocol is present at all,
 * use nd_packet_has_proto() instead, which is faster.
 *
 * Returns: check result.
 */
gboolean        nd_packet_has_proto_nested(const ND_Packet *packet,
					   const ND_Protocol *proto,
					   guint nesting);

/**
 * nd_packet_update_proo_state - updates the state of a packet.
 * @packet: packet to update.
 * @index: index of packet.
 *
 * The function iterates over the protocols contained in its data,
 * and calls the update_state() callback of the stateful protocols.
 * It's faster when the index of the packet is already known, so
 * pass it if you happen to know it. Otherwise, pass -1 for @index.
 */
void            nd_packet_update_proto_state(ND_Packet *packet,
					     int index);

/**
 * nd_packet_foreach_proto - protocol iterator.
 * @packet: packet whose protocols to iterate.
 * @cb: callback to call.
 * @user_data: arbitrary data passed to @cb.
 *
 * The function iterates over the protocols in the packet, from
 * outer- to the innermost, and calls @cb with that the corresponding
 * packet, protocol data and @user_data.
 */
void            nd_packet_foreach_proto(ND_Packet *packet,
					ND_PacketFunc cb,
					void *user_data);

/**
 * nd_packet_foreach_proto_backward - backward protocol iterator.
 * @packet: packet whose protocols to iterate.
 * @cb: callback to call.
 * @user_data: arbitrary data passed to @cb.
 *
 * The function iterates over the protocols in the packet, from
 * inner- to the outermost, and calls @cb with that the corresponding
 * packet, protocol data and @user_data.
 */
void            nd_packet_foreach_proto_backward(ND_Packet *packet,
						 ND_PacketFunc cb,
						 void *user_data);

/**
 * nd_packet_modified - marks a packet as modified.
 * @packet: packet to mark.
 *
 * The function marks a packet as modified and updates the GUI
 * accordingly.
 */
void            nd_packet_modified(ND_Packet *packet);


/**
 * nd_packet_modified - marks a packet as modified.
 * @packet: packet to mark.
 *
 * The function marks a packet as modified and updates the GUI
 * accordingly. If you know the packet's index, pass it, otherwise
 * pass -1 for @index or use nd_packet_modified().
 */
void            nd_packet_modified_at_index(ND_Packet *packet, int index);


/**
 * nd_packet_get_index -- returns index of packet.
 * @packet: packet to query.
 * 
 * The function returns the index of the packet in it's trace,
 * the first packet has index 0.
 *
 * Returns: index, or -1 if error occurred.
 */
int             nd_packet_get_index(const ND_Packet *packet);


/**
 * nd_packet_get_proto_nesting - returns nesting level of protocol at given data offset.
 * @packet: packet to query.
 * @proto: protocol whose nesting level to find.
 * @data: data pointer.
 *
 * The function returns the nesting level of @proto at the given
 * data offset. The first occurrence of a protocol has nesting level 0.
 * So, if you call this for IP and the data pointer points to somewhere
 * in or after the IP header in an ICMP error message, the nesting level
 * will be 1.
 *
 * Returns: nesting level, or -1 when an error occurred.
 */
int             nd_packet_get_proto_nesting(const ND_Packet *packet,
					    const ND_Protocol *proto,
					    guchar *data);

#endif
