#ifndef FILE_AUTODIFF
#define FILE_AUTODIFF

/**************************************************************************/
/* File:   autodiff.hpp                                                   */
/* Author: Joachim Schoeberl                                              */
/* Date:   24. Oct. 02                                                    */
/**************************************************************************/

// Automatic differentiation datatype


/**
   Object for automatic differentiation.
   Contains function value and D derivatives. Algebraic
   operations are overloaded by using product-rule etc. etc. 
**/
template <int D>
class AutoDiff
{
  double val;
  double dval[D];
public:
  /// initial object as zero
  AutoDiff  () throw() { ; }

  AutoDiff  (const AutoDiff & ad2) throw()
  {
    val = ad2.val;
    for (int i = 0; i < D; i++)
      dval[i] = ad2.dval[i];
  }

  /// initial object with constant value
  AutoDiff  (double aval) throw()
  {
    val = aval;
    for (int i = 0; i < D; i++)
      dval[i] = 0;
  }

  /// init object with (val, e_diffindex)
  AutoDiff  (double aval, int diffindex)  throw()
  {
    val = aval;
    for (int i = 0; i < D; i++)
      dval[i] = 0;
    dval[diffindex] = 1;
  }

  /// assign constant value
  AutoDiff & operator= (double aval) throw()
  {
    val = aval;
    for (int i = 0; i < D; i++)
      dval[i] = 0;
    return *this;
  }

  /// returns value
  double Value() const throw() { return val; }

  /// returns partial derivative
  double DValue (int i) const throw() { return dval[i]; }

  /// access value
  double & Value() throw() { return val; }

  /// accesses partial derivative 
  double & DValue (int i) throw() { return dval[i]; }

  ///
  AutoDiff<D> & operator+= (const AutoDiff<D> & y) throw()
  {
    val += y.val;
    for (int i = 0; i < D; i++)
      dval[i] += y.dval[i];
    return *this;
  }

  ///
  AutoDiff<D> & operator-= (const AutoDiff<D> & y) throw()
  {
    val -= y.val;
    for (int i = 0; i < D; i++)
      dval[i] -= y.dval[i];
    return *this;

  }

  ///
  AutoDiff<D> & operator*= (const AutoDiff<D> & y) throw()
  {
    for (int i = 0; i < D; i++)
      {
	dval[i] *= y.val;
	dval[i] += val * y.dval[i];
      }
    val *= y.val;
    return *this;
  }

  /// 
  bool operator== (double val2) throw()
  {
    return val == val2;
  }

  ///
  bool operator!= (double val2) throw()
  {
    return val != val2;
  }

  ///
  bool operator< (double val2) throw()
  {
    return val < val2;
  }
  
  ///
  bool operator> (double val2) throw()
  {
    return val > val2;
  }
};


//@{  AutoDiff helper functions.

/// 
template<int D>
inline ostream & operator<< (ostream & ost, const AutoDiff<D> & x)
{
  ost << x.Value() << ", D = ";
  for (int i = 0; i < D; i++)
    ost << x.DValue(i) << " ";
  return ost;
}

///
template<int D>
inline AutoDiff<D> operator+ (const AutoDiff<D> & x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  res.Value () = x.Value()+y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x.DValue(i) + y.DValue(i);
  return res;
}


/// 
template<int D>
inline AutoDiff<D> operator- (const AutoDiff<D> & x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  res.Value() = x.Value()-y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x.DValue(i) - y.DValue(i);
  return res;
}

/*
template<int D>
inline AutoDiff<D> Minus (const AutoDiff<D> & x, const AutoDiff<D> & y)
{
  AutoDiff<D> res;
  res.Value() = x.Value()-y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x.DValue(i) - y.DValue(i);
  return res;
}
*/

///
template<int D>
inline AutoDiff<D> operator+ (double x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  res.Value() = x+y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = y.DValue(i);
  return res;
}

///
template<int D>
inline AutoDiff<D> operator+ (const AutoDiff<D> & y, double x) throw()
{
  AutoDiff<D> res;
  res.Value() = x+y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = y.DValue(i);
  return res;
}


///
template<int D>
inline AutoDiff<D> operator- (const AutoDiff<D> & x) throw()
{
  AutoDiff<D> res;
  res.Value() = -x.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = -x.DValue(i);
  return res;
}

///
template<int D>
inline AutoDiff<D> operator- (const AutoDiff<D> & x, double y) throw()
{
  AutoDiff<D> res;
  res.Value() = x.Value()-y;
  for (int i = 0; i < D; i++)
    res.DValue(i) = x.DValue(i);
  return res;
}

///
template<int D>
inline AutoDiff<D> operator- (double x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  res.Value() = x-y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = -y.DValue(i);
  return res;
}


///
template<int D>
inline AutoDiff<D> operator* (double x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  res.Value() = x*y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x*y.DValue(i);
  return res;
}

///
template<int D>
inline AutoDiff<D> operator* (const AutoDiff<D> & y, double x) throw()
{
  AutoDiff<D> res;
  res.Value() = x*y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x*y.DValue(i);
  return res;
}

///
template<int D>
inline AutoDiff<D> operator* (const AutoDiff<D> & x, const AutoDiff<D> & y) throw()
{
  AutoDiff<D> res;
  double hx = x.Value();
  double hy = y.Value();

  res.Value() = hx*hy;
  for (int i = 0; i < D; i++)
    res.DValue(i) = hx*y.DValue(i) + hy*x.DValue(i);
  /*
  res.Value() = x.Value()*y.Value();
  for (int i = 0; i < D; i++)
    res.DValue(i) = x.Value()*y.DValue(i) + x.DValue(i) * y.Value();
  */
  return res;
}


/*
template<int D>
inline AutoDiff<D> Mult (const AutoDiff<D> & x, const AutoDiff<D> & y)
{
  AutoDiff<D> res;
  double hx = x.Value();
  double hy = y.Value();

  res.Value() = hx*hy;
  for (int i = 0; i < D; i++)
    res.DValue(i) = hx*y.DValue(i) + hy*x.DValue(i);

  return res;
}
*/



template<int D>
inline AutoDiff<D> Inv (const AutoDiff<D> & x)
{
  AutoDiff<D> res(1.0 / x.Value());
  for (int i = 0; i < D; i++)
    res.DValue(i) = -x.DValue(i) / (x.Value() * x.Value());
  return res;
}


template<int D>
inline AutoDiff<D> operator/ (const AutoDiff<D> & x, const AutoDiff<D> & y)
{
  return x * Inv (y);
}

template<int D>
inline AutoDiff<D> operator/ (const AutoDiff<D> & x, double y)
{
  return (1/y) * x;
}

template<int D>
inline AutoDiff<D> operator/ (double x, const AutoDiff<D> & y)
{
  return x * Inv(y);
}

//@}

#endif
