/*
                                  NETWIB
                             Network library
                Copyright(c) 1999-2005 Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
                   http://laurentconstantin.free.fr/
                   http://membres.lycos.fr/lauconstantin/
  [my current email address is on the web servers]

                                  -----
  This file is part of Netwib.

  Netwib is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  version 2 as published by the Free Software Foundation.

  Netwib is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details (http://www.gnu.org/).

------------------------------------------------------------------------
*/

#include <netwib/inc/maininc.h>

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_init_uint32(netwib_uint32 in,
                                     netwib_uint64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_uint64)in;
#else
    pout->high = 0;
    pout->low = in;
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_init_uintmax(netwib_uintmax in,
                                      netwib_uint64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_uint64)in;
#else
    pout->high = 0;
    pout->low = (netwib_uint32)in;
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_init_uintptr(netwib_uintptr in,
                                      netwib_uint64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_uint64)in;
#else
    pout->high = 0;
    pout->low = (netwib_uint32)in;
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_init_int32(netwib_int32 in,
                                   netwib_int64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_int64)in;
#else
    if (in >= 0) {
      pout->high = 0;
      pout->low = in;
    } else {
      pout->high = 0xFFFFFFFFu;
      pout->low = in;
    }
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_init_intmax(netwib_intmax in,
                                    netwib_int64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_int64)in;
#else
    if (in >= 0) {
      pout->high = 0;
      pout->low = (netwib_int32)in;
    } else {
      pout->high = 0xFFFFFFFFu;
      pout->low = (netwib_int32)in;
    }
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_init_intptr(netwib_intptr in,
                                    netwib_int64 *pout)
{

  if (pout != NULL) {
#if NETWIB_INT64_FAKE == 0
    *pout = (netwib_int64)in;
#else
    if (in >= 0) {
      pout->high = 0;
      pout->low = (netwib_int32)in;
    } else {
      pout->high = 0xFFFFFFFFu;
      pout->low = (netwib_int32)in;
    }
#endif
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint32_init_uint64(netwib_uint64 in,
                                     netwib_uint32 *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (in > 0xFFFFFFFFu) {
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) *pout = (netwib_uint32)in;
#else
  if (in.high) {
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) *pout = in.low;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uintmax_init_uint64(netwib_uint64 in,
                                      netwib_uintmax *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (pout != NULL) *pout = (netwib_uintmax)in;
#else
  if (in.high) {
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) *pout = (netwib_uintmax)in.low;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uintptr_init_uint64(netwib_uint64 in,
                                      netwib_uintptr *pout)
{

#if NETWIB_INT64_FAKE == 0
 #if NETWIB_INTPTR_BITS == 32
  if (in > 0xFFFFFFFFu) {
    return(NETWIB_ERR_PATOOHIGH);
  }
 #endif
  if (pout != NULL) *pout = (netwib_uintptr)in;
#else
  if (in.high) {
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) *pout = (netwib_uintptr)in.low;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int32_init_int64(netwib_int64 in,
                                   netwib_int32 *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (in >= 0) {
    if (in > 0x7FFFFFFF) {
      return(NETWIB_ERR_PATOOHIGH);
    }
  } else {
    if (in < -0x7FFFFFFF-1) {
      return(NETWIB_ERR_PATOOLOW);
    }
  }
  if (pout != NULL) *pout = (netwib_int32)in;
#else
  if (!(in.high & 0x80000000u)) { /* >= 0 */
    if (in.high != 0) { /* > 0xFFFFFFFF */
      return(NETWIB_ERR_PATOOHIGH);
    }
    if (in.low & 0x80000000u) { /* > 0x7FFFFFFF */
      return(NETWIB_ERR_PATOOHIGH);
    }
  } else {
    if (in.high != 0xFFFFFFFFu) { /* < -0xFFFFFFFF */
      return(NETWIB_ERR_PATOOLOW);
    }
    if (!(in.low & 0x80000000u)) { /* < -0x80000000 */
      return(NETWIB_ERR_PATOOLOW);
    }
  }
  if (pout != NULL) *pout = (netwib_int32)in.low;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_intmax_init_int64(netwib_int64 in,
                                    netwib_intmax *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (pout != NULL) *pout = (netwib_intmax)in;
#else
  netwib_er(netwib_int32_init_int64(in, (netwib_int32*)pout));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_intptr_init_int64(netwib_int64 in,
                                    netwib_intptr *pout)
{

#if NETWIB_INT64_FAKE == 0
 #if NETWIB_INTPTR_BITS == 32
  if (in >= 0) {
    if (in > 0x7FFFFFFF) {
      return(NETWIB_ERR_PATOOHIGH);
    }
  } else {
    if (in < -0x7FFFFFFF-1) {
      return(NETWIB_ERR_PATOOLOW);
    }
  }
 #endif
  if (pout != NULL) *pout = (netwib_intptr)in;
#else
  netwib_er(netwib_int32_init_int64(in, (netwib_int32*)pout));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_init_int64(netwib_int64 in,
                                    netwib_uint64 *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (in < 0) {
    return(NETWIB_ERR_PATOOLOW);
  }
  if (pout != NULL) *pout = (netwib_uint64)in;
#else
  if (in.high & 0x80000000u) { /* < 0 */
    return(NETWIB_ERR_PATOOLOW);
  }
  if (pout != NULL) {
    pout->high = in.high;
    pout->low = in.low;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_init_uint64(netwib_uint64 in,
                                    netwib_int64 *pout)
{

#if NETWIB_INT64_FAKE == 0
  if (in > NETWIB_UINT_LL(0x7FFFFFFFFFFFFFFF)) {
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) *pout = (netwib_int64)in;
#else
  if (in.high & 0x80000000u) { /* > 0x7FFFFFFFFFFFFFFFu */
    return(NETWIB_ERR_PATOOHIGH);
  }
  if (pout != NULL) {
    pout->high = in.high;
    pout->low = in.low;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_add(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_uint64 *paplusb)
{

#if NETWIB_INT64_FAKE == 0
  if (paplusb != NULL) *paplusb = a + b;
#else
  netwib_uint32 tmp, carry;
  if (paplusb != NULL) {
    tmp = a.low + b.low;
    /* If x + y overflows, the result is < x and < y. For example,
       using a byte:
         0x03 + 0xFF = 0x02
         We have: 0x02 < 0x03 and 0x02 < 0xFF
         So, this is a fast way to determine the value of carry. We use
         "tmp < a.low", but we could also use "tmp < b.low".
    */
    carry = (tmp < a.low) ? 1 : 0;
    paplusb->low = tmp;
    /* this operation might again overflow, but we ignore */
    paplusb->high = a.high + b.high + carry;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_sub(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_uint64 *paminusb)
{

#if NETWIB_INT64_FAKE == 0
  if (paminusb != NULL) *paminusb = a - b;
#else
  netwib_uint32 tmp, carry;
  if (paminusb != NULL) {
    tmp = a.low - b.low;
    /* If x - y underflows, the result is > x. For example,
       using a byte:
         0x03 - 0xFF = 0x04
         We have: 0x04 > 0x03
         So, this is a fast way to determine the value of carry.
    */
    carry = (tmp > a.low) ? 1 : 0;
    paminusb->low = tmp;
    /* this operation might again underflow, but we ignore */
    paminusb->high = a.high - b.high - carry;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
#if NETWIB_INT64_FAKE == 1
static netwib_err netwib_priv_uint32_mul(netwib_uint32 a,
                                         netwib_uint32 b,
                                         netwib_uint64 *pamulb)
{
  netwib_uint32 tmp, c0, c1, c2;

#define netwib_priv_uint16_high(x) ((x)>>16)
#define netwib_priv_uint16_low(x) ((x)&0xFFFF)

  /* A multiplication is :
                                             a.high         a.low
       *                                     b.high         b.low
       =============================================================
       +                |               | b.low*a.high | b.low*a.low
       +                | b.high*a.high | b.high*a.low |
       =============================================================
                               c2       |     c1       |     c0
     We will compute 3 columns c0, c1 and c2.
     To prove overflow, we use those values:
       MM = max(mul) = 0xFFFF * 0xFFFF = 0xFFFE0001
       MH = max(high(MM)) = 0xFFFE
       ML = max(low(mul)) = 0xFFFF
  */
  c2 = netwib_priv_uint16_high(b) * netwib_priv_uint16_high(a);
  tmp = netwib_priv_uint16_low(b) * netwib_priv_uint16_low(a);
  c0 = netwib_priv_uint16_low(tmp);
  c1 = netwib_priv_uint16_high(tmp);
  /* c1 will not overflow because:
      max(c1) = old_max(c1) + MM = MH + MM = 0xFFFE + 0xFFFE0001
              = 0xFFFEFFFF < 0xFFFFFFFF
  */
  c1 += netwib_priv_uint16_low(b) * netwib_priv_uint16_high(a);
  tmp = netwib_priv_uint16_high(b) * netwib_priv_uint16_low(a);
  /* c1 might overflow because:
       max(c1) = old_max(c1) + MM = 0xFFFEFFFF + 0xFFFE0001
               = 0x1FFFD0000
  */
  c1 += tmp;
  if (c1 < tmp) {
    /* overflow occurred in c1, so put a carry in c2 (this carry
       will not cause an overflow in c2) */
    c2 += 0x10000;
  }
  /* to sum up, we have:
       max(c0) = ML = 0xFFFF
       max(c1)[if_over] = 0x[1]FFFD0000
       max(c1)[if_no_over] = 0xFFFFFFFF
       max(c2)[if_over] = MM + 0x10000 = 0xFFFE0001 + 0x10000 = 0xFFFF0001
       max(c2)[if_no_over] = MM = 0xFFFE0001
  */
  /* now, separate c1 and store it in c0 and c2 */
  /* c0 will not overflow because:
       max(c0) = old_max(c0) + low(c1)<<16 = 0xFFFF + 0xFFFF<<16
               = 0xFFFFFFFF
  */
  c0 += netwib_priv_uint16_low(c1) << 16;
  /* c2 will not overflow because:
       max(c2)[if_over] = old_max(c2)[if_over] + high(c1)[if_over]
                        = 0xFFFF0001 + 0xFFFD = 0xFFFFFFFE
       max(c2)[if_no_over] = old_max(c2)[if_no_over] + high(c1)[if_no_over]
                        = 0xFFFE0001 + 0xFFFF = 0xFFFF0000
  */
  c2 += netwib_priv_uint16_high(c1);
  /* set values */
  pamulb->high = c2;
  pamulb->low = c0;

  return(NETWIB_ERR_OK);
}
#endif

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_mul(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_uint64 *pamulb)
{

#if NETWIB_INT64_FAKE == 0
  if (pamulb != NULL) *pamulb = a * b;
#else
  /* A multiplication is :
                            a.high         a.low
       *                    b.high         b.low
       =============================================
       +                | b.low*a.high | b.low*a.low
       +  b.high*a.high | b.high*a.low |
     To sum up :
        pamulb->low = b.low*a.low
        pamulb->high = b.low*a.high + b.high*a.low
        first column is not needed in this function so it is ignored
   */
  netwib_uint64 tmp;
  if (pamulb != NULL) {
    netwib_er(netwib_priv_uint32_mul(b.low, a.low, pamulb));
    netwib_er(netwib_priv_uint32_mul(b.low, a.high, &tmp));
    pamulb->high += tmp.low; /* might overflow, but ignore */
    netwib_er(netwib_priv_uint32_mul(b.high, a.low, &tmp));
    pamulb->high += tmp.low; /* might overflow, but ignore */
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_div(netwib_uint64 n,
                             netwib_uint64 d,
                             netwib_uint64 *pq,
                             netwib_uint64 *pr)
{

#if NETWIB_INT64_FAKE == 0
  if (d == 0) {
    return(NETWIB_ERR_PATOOLOW);
  }
  if (pq != NULL) *pq = n / d;
  if (pr != NULL) *pr = n % d;
#else
  netwib_uint64 q, r, tmp;
  netwib_uint32 sh;
  netwib_cmp cmp;
  if (d.high == 0 && d.low == 0) {
    return(NETWIB_ERR_PATOOLOW);
  }
  /*
     A binary division is for example:
        100110010 || 10101 = 21
        .         ||======
        -10101    || 0111* = 1110 = 14
         10001010 ||
         -10101   ||
           110110 ||
          -10101  ||
             1100 ||
                * ||
     => n=306, d=21, q=14, r=12  (21*14+12=306)
     This implementation is slow, but it works.
     If you have time, don't hesitate to send me a better algorithm.
  */
  netwib_er(netwib_uint64_init_uint32(0, &q));
  sh = 64;
  while(NETWIB_TRUE) {
    netwib_er(netwib_uint64_cmp(n, d, &cmp));
    if (cmp == NETWIB_CMP_LT) {
      netwib_er(netwib_uint64_shl(q, sh, &q));
      r = n;
      break;
    }
    if (sh == 0) {
      return(NETWIB_ERR_LOINTERNALERROR);
    }
    sh--;
    netwib_er(netwib_uint64_shl(q, 1, &q));
    netwib_er(netwib_uint64_shr(n, sh, &tmp));
    netwib_er(netwib_uint64_cmp(tmp, d, &cmp));
    if (cmp != NETWIB_CMP_LT) {
      netwib_er(netwib_uint64_shl(d, sh, &tmp));
      netwib_er(netwib_uint64_sub(n, tmp, &n));
      q.low |= 1;
    }
  }
  if (pq != NULL) *pq = q;
  if (pr != NULL) *pr = r;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_div(netwib_int64 n,
                            netwib_int64 d,
                            netwib_int64 *pq,
                            netwib_int64 *pr)
{

#if NETWIB_INT64_FAKE == 0
  if (d == 0) {
    return(NETWIB_ERR_PATOOLOW);
  }
  if (pq != NULL) *pq = n / d;
  if (pr != NULL) *pr = n % d;
  return(NETWIB_ERR_OK);
#else
  netwib_uint64 n2, d2, q, r;
  netwib_bool changesign;
  changesign = NETWIB_FALSE;
  n2 = n;
  if (n.high & 0x80000000u) {
    changesign = ! changesign;
    netwib_er(netwib_int64_neg(n, &n2));
  }
  d2 = d;
  if (d.high & 0x80000000u) {
    changesign = ! changesign;
    netwib_er(netwib_int64_neg(d, &d2));
  }
  netwib_er(netwib_uint64_div(n2, d2, &q, &r));
  if (changesign) {
    netwib_er(netwib_int64_neg(q, &q));
  }
  if (n.high & 0x80000000u) {
    netwib_er(netwib_int64_neg(r, &r));
  }
  if (pq != NULL) *pq = q;
  if (pr != NULL) *pr = r;
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_inc(netwib_uint64 *pui)
{

#if NETWIB_INT64_FAKE == 0
  if (pui != NULL) *pui = *pui + 1;
#else
  if (pui != NULL) {
    if (pui->low == 0xFFFFFFFFu) {
      pui->high++;
      pui->low = 0;
    } else {
      pui->low++;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_dec(netwib_uint64 *pui)
{

#if NETWIB_INT64_FAKE == 0
  if (pui != NULL) *pui = *pui - 1;
#else
  if (pui != NULL) {
    if (pui->low == 0) {
      pui->high--;
      pui->low = 0xFFFFFFFF;
    } else {
      pui->low--;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_cmp(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_cmp *pcmp)
{

#if NETWIB_INT64_FAKE == 0
  if (pcmp != NULL) {
    if (a > b) {
      *pcmp = NETWIB_CMP_GT;
    } else if (a < b) {
      *pcmp = NETWIB_CMP_LT;
    } else {
      *pcmp = NETWIB_CMP_EQ;
    }
  }
#else
  if (pcmp != NULL) {
    if (a.high == b.high) {
      if (a.low == b.low) {
        *pcmp = NETWIB_CMP_EQ;
      } else if (a.low < b.low) {
        *pcmp = NETWIB_CMP_LT;
      } else {
        *pcmp = NETWIB_CMP_GT;
      }
    } else if (a.high < b.high) {
      *pcmp = NETWIB_CMP_LT;
    } else {
      *pcmp = NETWIB_CMP_GT;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_cmp(netwib_int64 a,
                            netwib_int64 b,
                            netwib_cmp *pcmp)
{

#if NETWIB_INT64_FAKE == 0
  if (pcmp != NULL) {
    if (a > b) {
      *pcmp = NETWIB_CMP_GT;
    } else if (a < b) {
      *pcmp = NETWIB_CMP_LT;
    } else {
      *pcmp = NETWIB_CMP_EQ;
    }
  }
#else
  netwib_bool aneg, bneg;

  if (pcmp != NULL) {
    aneg = a.high & 0x80000000u;
    bneg = b.high & 0x80000000u;
    if (aneg && !bneg) {
      *pcmp = NETWIB_CMP_LT;
    } else if (!aneg && bneg) {
      *pcmp = NETWIB_CMP_GT;
    } else if (aneg && bneg) {
      if (a.high == b.high) {
        if (a.low == b.low) {
          *pcmp = NETWIB_CMP_EQ;
        } else if (a.low < b.low) {
          *pcmp = NETWIB_CMP_LT;
        } else {
          *pcmp = NETWIB_CMP_GT;
        }
      } else if (a.high < b.high) {
        *pcmp = NETWIB_CMP_GT;
      } else {
        *pcmp = NETWIB_CMP_LT;
      }
    } else {
      if (a.high == b.high) {
        if (a.low == b.low) {
          *pcmp = NETWIB_CMP_EQ;
        } else if (a.low < b.low) {
          *pcmp = NETWIB_CMP_LT;
        } else {
          *pcmp = NETWIB_CMP_GT;
        }
      } else if (a.high < b.high) {
        *pcmp = NETWIB_CMP_LT;
      } else {
        *pcmp = NETWIB_CMP_GT;
      }
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_int64_neg(netwib_int64 a,
                            netwib_int64 *pnega)
{

#if NETWIB_INT64_FAKE == 0
  if (pnega != NULL) *pnega = -a;
#else
  if (pnega != NULL) {
    pnega->high = ~a.high;
    pnega->low = ~a.low;
    if (pnega->low == 0xFFFFFFFFu) {
      pnega->high++;
      pnega->low = 0;
    } else {
      pnega->low++;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_shl(netwib_uint64 a,
                             netwib_uint32 b,
                             netwib_uint64 *pashlb)
{

#if NETWIB_INT64_FAKE == 0
  if (pashlb != NULL) *pashlb = a << b;
#else
  if (pashlb != NULL) {
    /* this is a bit strange, but with gcc shifting more than
       processor size is like shift was moduled :
      int f(int a) {
        printf("%08X l\n", 0x101 << a);
        printf("%08X r\n", 0x101 >> a);
      }
      int main(void) {
        f(31);
        f(32);
        f(33);
        f(34);
      }
      80000000 l
      00000000 r
      00000101 l
      00000101 r
      00000202 l
      00000080 r
      00000404 l
      00000040 r
    */
    b = b % 64;
    if (b == 0) {
      pashlb->high = a.high;
      pashlb->low = a.low;
    } else if (b < 32) {
      pashlb->high = a.high << b | (a.low >> (32 - b));
      pashlb->low = a.low << b;
    } else {
      b -= 32;
      pashlb->high = a.low << b;
      pashlb->low = 0;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_shr(netwib_uint64 a,
                             netwib_uint32 b,
                             netwib_uint64 *pashrb)
{

#if NETWIB_INT64_FAKE == 0
  if (pashrb != NULL) *pashrb = a >> b;
#else
  if (pashrb != NULL) {
    b = b % 64;
    if (b == 0) {
      pashrb->high = a.high;
      pashrb->low = a.low;
    } else if (b < 32) {
      pashrb->low = a.low >> b | (a.high << (32 - b));
      pashrb->high = a.high >> b;
    } else {
      b -= 32;
      pashrb->low = a.high >> b;
      pashrb->high = 0;
    }
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_and(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_uint64 *paandb)
{

#if NETWIB_INT64_FAKE == 0
  if (paandb != NULL) *paandb = a & b;
#else
  if (paandb != NULL) {
    paandb->high = a.high & b.high;
    paandb->low = a.low & b.low;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_or(netwib_uint64 a,
                            netwib_uint64 b,
                            netwib_uint64 *paorb)
{

#if NETWIB_INT64_FAKE == 0
  if (paorb != NULL) *paorb = a | b;
#else
  if (paorb != NULL) {
    paorb->high = a.high | b.high;
    paorb->low = a.low | b.low;
  }
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_xor(netwib_uint64 a,
                             netwib_uint64 b,
                             netwib_uint64 *paxorb)
{

#if NETWIB_INT64_FAKE == 0
  if (paxorb != NULL) *paxorb = a ^ b;
#else
  if (paxorb != NULL) {
    paxorb->high = a.high ^ b.high;
    paxorb->low = a.low ^ b.low;
  }
#endif


  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_uint64_not(netwib_uint64 a,
                             netwib_uint64 *pnot)
{

#if NETWIB_INT64_FAKE == 0
  if (pnot != NULL) *pnot = ~ a;
#else
  if (pnot != NULL) {
    pnot->high = ~ a.high;
    pnot->low = ~ a.low;
  }
#endif

  return(NETWIB_ERR_OK);
}

