/* This file is part of Netwib.
*/

/*-------------------------------------------------------------*/
typedef struct {
  int readfd;
  int writefd;
  int pid;
  netwib_bool killonclose;
  netwib_bool *pexitednormally;
  netwib_uint32 *preturnedvalue;
} netwib_priv_io_exec;

/*-------------------------------------------------------------*/
#if NETWIBDEF_LIBPTHREADFROMMIT == 0
static netwib_err netwib_priv_io_exec_close_descriptors(netwib_uint32 excepted)
{
  netwib_uint32 themax, i;

 #if NETWIBDEF_HAVEFUNC_GETRLIMIT==1
  {
    struct rlimit lim;
    int reti;
    reti = getrlimit(RLIMIT_NOFILE, &lim);
    if (reti == -1) {
      return(NETWIB_ERR_FUGETRLIMIT);
    }
    themax = lim.rlim_cur;
  }
 #elif NETWIBDEF_HAVEFUNC_SYSCONF==1
  themax = sysconf(_SC_OPEN_MAX);
 #else
  themax = OPEN_MAX;
 #endif

  /* we could use "fcntl(fd, F_SETFD, FD_CLOEXEC);", but we have to check
     if fd is open. So, I choose to close everything. */
  for (i = 3 ; i < excepted; i++) {
    /*ignore*/close(i);
  }
  for (i = excepted+1 ; i < themax; i++) {
    /*ignore*/close(i);
  }

  return(NETWIB_ERR_OK);
}
#endif

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_init_child(netwib_conststring filename,
                                                 netwib_string const argv[],
                                                 int childtofather[2],
                                                 int fathertochild[2],
                                                 int checkexec[2])
{
  int reti, devnull;

  /* close unneeded file descriptors */
  if (childtofather != NULL) {
    reti = close(childtofather[0]);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  }
  if (fathertochild != NULL) {
    reti = close(fathertochild[1]);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  }

  /* child becomes session leader : like this, a kill will kill
     every sub-child.
  */
  /*ignore*/setsid();

  /* open /dev/null */
  devnull = open("/dev/null", O_RDWR);
  if (devnull < 0) return(NETWIB_ERR_FUOPEN);

  /* associate file descriptors */
  if (fathertochild != NULL) {
    reti = dup2(fathertochild[0], STDIN_FILENO);
    if (reti == -1) return(NETWIB_ERR_FUDUP2);
    reti = close(fathertochild[0]);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  } else {
    reti = dup2(devnull, STDIN_FILENO);
    if (reti == -1) return(NETWIB_ERR_FUDUP2);
  }

  if (childtofather != NULL) {
    reti = dup2(childtofather[1], STDOUT_FILENO);
    if (reti == -1) return(NETWIB_ERR_FUDUP2);
    reti = close(childtofather[1]);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  } else {
    reti = dup2(devnull, STDOUT_FILENO);
    if (reti == -1) return(NETWIB_ERR_FUDUP2);
  }

  reti = dup2(devnull, STDERR_FILENO);
  if (reti == -1) return(NETWIB_ERR_FUDUP2);

  /* prepare our trick to check if exec succeeded */
  reti = close(checkexec[0]);
  if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  reti = fcntl(checkexec[1], F_SETFD, FD_CLOEXEC);
  if (reti == -1) return(NETWIB_ERR_FUFCNTL);

  /* close unneeded file descriptors */
#if NETWIBDEF_LIBPTHREADFROMMIT == 0
  /* except for MIT POSIX threads, because a private fd is used, and we
     don't know which one is */
  netwib_er(netwib_priv_io_exec_close_descriptors(checkexec[1]));
#endif

  /* exec */
  reti = execve(filename, argv, NULL);
  if (reti < 0) return(NETWIB_ERR_LOCANTEXEC);

  /* should not be reached */
  return(NETWIB_ERR_LOINTERNALERROR);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_init3(netwib_string filename,
                                            netwib_string argv[],
                                            int childtofather[2],
                                            int fathertochild[2],
                                            int *ppid)
{
  netwib_err ret, retchild;
  int checkexec[2];
  int pid;
  int reti;

  /* this pipe is used to check if exec worked */
  reti = pipe(checkexec);
  if (reti == -1) return(NETWIB_ERR_FUPIPE);

  /* try to launch child */
  pid = fork();
  if (pid == -1) {
    close(checkexec[0]);
    close(checkexec[1]);
    return(NETWIB_ERR_FUFORK);
  }

  if (pid == 0) {
    /* child */
    ret = netwib_priv_io_exec_init_child(filename, argv,
                                         childtofather, fathertochild,
                                         checkexec);
    /* error number is written to checkexec pipe */
    /*ignore*/netwib_priv_fdpipe_write_uint32(checkexec[1], ret);
    _exit(NETWIB_ERR_DATAEND);
    return(NETWIB_ERR_LOINTERNALERROR);
    /* never reached */
  }

  /* wait to ensure exec is ok */
  reti = close(checkexec[1]);
  if (reti == -1) return(NETWIB_ERR_FUCLOSE);
#if NETWIBDEF_LIBPTHREADFROMMIT == 1
  /* read implementation in MIT POSIX threads does not detect the close.
     So, we first wait for one second */
 {
   netwib_time t;
   netwib_bool event;
   netwib_er(netwib_time_init_now(&t));
   netwib_er(netwib_time_plus_sec(&t, 1));
   netwib_er(netwib_priv_fd_wait(checkexec[0], NETWIB_IO_WAYTYPE_READ,
                                 &t, &event));
   if (event) {
     ret = netwib_priv_fd_read_uint32(checkexec[0], (netwib_uint32*)&retchild);
   } else {
     ret = NETWIB_ERR_DATAEND;
   }
 }
#else
  ret = netwib_priv_fd_read_uint32(checkexec[0], (netwib_uint32*)&retchild);
#endif
  close(checkexec[0]);
  if (ret == NETWIB_ERR_OK) {
    if (retchild == NETWIB_ERR_OK) {
      return(NETWIB_ERR_LOERROROKKO);
    } else {
      return(retchild);
    }
  } else if (ret != NETWIB_ERR_DATAEND) {
    return(ret);
  }

  /* store information */
  *ppid = pid;
  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_init2(netwib_string filename,
                                            netwib_string argv[],
                                            netwib_io_waytype providedway,
                                            int *ppid,
                                            int *preadfd,
                                            int *pwritefd)
{
  netwib_bool provideread, providewrite;
  netwib_err ret;
  int fathertochild[2];
  int childtofather[2];
  int reti;

  switch(providedway) {
    case NETWIB_IO_WAYTYPE_READ :
      provideread = NETWIB_TRUE;
      providewrite = NETWIB_FALSE;
      break;
    case NETWIB_IO_WAYTYPE_WRITE :
      provideread = NETWIB_FALSE;
      providewrite = NETWIB_TRUE;
      break;
    case NETWIB_IO_WAYTYPE_RDWR :
      provideread = NETWIB_TRUE;
      providewrite = NETWIB_TRUE;
      break;
    case NETWIB_IO_WAYTYPE_NONE :
      provideread = NETWIB_FALSE;
      providewrite = NETWIB_FALSE;
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
  }

  /* we use pipes to communicate */
  if (provideread) {
    reti = pipe(childtofather);
    if (reti == -1) return(NETWIB_ERR_FUPIPE);
  }
  if (providewrite) {
    reti = pipe(fathertochild);
    if (reti == -1) {
      if (provideread) {
        close(childtofather[0]);
        close(childtofather[1]);
      }
      return(NETWIB_ERR_FUPIPE);
    }
  }

  /* try to launch child */
  ret = netwib_priv_io_exec_init3(filename, argv,
                                  provideread?childtofather:NULL,
                                  providewrite?fathertochild:NULL,
                                  ppid);
  if (ret != NETWIB_ERR_OK) {
    if (provideread) {
      close(childtofather[0]);
      close(childtofather[1]);
    }
    if (providewrite) {
      close(fathertochild[0]);
      close(fathertochild[1]);
    }
    return(ret);
  }

  /* close unneeded file descriptors */
  if (provideread) {
    reti = close(childtofather[1]);
    if (reti == -1) {
      close(childtofather[0]);
      if (providewrite) {
        close(fathertochild[0]);
        close(fathertochild[1]);
      }
      return(NETWIB_ERR_FUCLOSE);
    }
  }
  if (providewrite) {
    reti = close(fathertochild[0]);
    if (reti == -1) {
      close(fathertochild[1]);
      if (provideread) {
        close(childtofather[0]);
        close(childtofather[1]);
      }
      return(NETWIB_ERR_FUCLOSE);
    }
  }

  /* store information */
  *preadfd = provideread?childtofather[0]:-1;
  *pwritefd = providewrite?fathertochild[1]:-1;
  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_init(netwib_constbuf *pbufcommand,
                                           netwib_io_waytype providedway,
                                           netwib_bool killonclose,
                                           netwib_bool *pexitednormally,
                                           netwib_uint32 *preturnedvalue,
                                           netwib_bool *preadinitialized,
                                           netwib_bool *pwriteinitialized,
                                           netwib_priv_io_exec *ptr)
{
  netwib_string filename, *argv;
  netwib_err ret;
  int pid=0, readfd=0, writefd=0;

  /* try to launch child */
  netwib_er(netwib_priv_cmdline_init(pbufcommand, &filename, NULL, &argv));
  ret = netwib_priv_io_exec_init2(filename, argv, providedway,
                                  &pid, &readfd, &writefd);
  netwib_er(netwib_priv_cmdline_close(&filename, &argv));
  if (ret != NETWIB_ERR_OK) {
    return(ret);
  }

  /* store information */
  ptr->readfd = readfd;
  ptr->writefd = writefd;
  ptr->pid = pid;
  ptr->killonclose = killonclose;
  ptr->pexitednormally = pexitednormally;
  ptr->preturnedvalue = preturnedvalue;
  *preadinitialized = (readfd==-1)?NETWIB_FALSE:NETWIB_TRUE;
  *pwriteinitialized = (writefd==-1)?NETWIB_FALSE:NETWIB_TRUE;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_close(netwib_priv_io_exec *ptr)
{
  int reti, status;

  if (ptr->killonclose) {
    /* smooth then violent kill of the child */
#if NETWIBDEF_LIBPTHREADFROMMIT == 1
    /* waitpid implementation in MIT POSIX threads does not support WNOHANG */
    /* note we send signal to the group */
    /* ignore */ kill( - ptr->pid, SIGTERM);
    netwib_er(netwib_priv_pause());
    /* ignore */ kill( - ptr->pid, SIGKILL);
    reti = waitpid(ptr->pid, &status, 0);
#else
    reti = waitpid(ptr->pid, &status, WNOHANG);
    if (reti == 0) {
      /* note we send signal to the group */
      /* ignore */ kill( - ptr->pid, SIGTERM);
      netwib_er(netwib_priv_pause());
      reti = waitpid(ptr->pid, &status, WNOHANG);
      if (reti == 0) {
        /* ignore */ kill( - ptr->pid, SIGKILL);
        reti = waitpid(ptr->pid, &status, 0);
      }
    }
#endif
  } else {
    reti = waitpid(ptr->pid, &status, 0);
  }
  if (reti != ptr->pid) {
    return(NETWIB_ERR_FUWAITPID);
  }

  if (WIFEXITED(status)) {
    if (ptr->pexitednormally != NULL) {
      *(ptr->pexitednormally) = NETWIB_TRUE;
    }
    if (ptr->preturnedvalue != NULL) {
      *(ptr->preturnedvalue) = WEXITSTATUS(status);
    }
  } else {
    if (ptr->pexitednormally != NULL) {
      *(ptr->pexitednormally) = NETWIB_FALSE;
    }
  }

  /* close descriptors */
  if (ptr->readfd != -1) {
    reti = close(ptr->readfd);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  }
  if (ptr->writefd != -1) {
    reti = close(ptr->writefd);
    if (reti == -1) return(NETWIB_ERR_FUCLOSE);
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_read(netwib_io *pio,
                                           netwib_buf *pbuf)
{
  netwib_priv_io_exec *ptr = (netwib_priv_io_exec *)pio->pcommon;
  return(netwib_priv_fd_read(ptr->readfd, pbuf));
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_write(netwib_io *pio,
                                            netwib_constbuf *pbuf)
{
  netwib_priv_io_exec *ptr = (netwib_priv_io_exec *)pio->pcommon;
  return(netwib_priv_fdpipe_write(ptr->writefd, pbuf));
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_wait(netwib_io *pio,
                                           netwib_io_waytype way,
                                           netwib_consttime *pabstime,
                                           netwib_bool *pevent)
{
  netwib_priv_io_exec *ptr = (netwib_priv_io_exec *)pio->pcommon;

  switch (way) {
    case NETWIB_IO_WAYTYPE_READ :
      netwib_er(netwib_priv_fd_wait(ptr->readfd, way, pabstime, pevent));
      break;
    case NETWIB_IO_WAYTYPE_WRITE :
      netwib_er(netwib_priv_fd_wait(ptr->writefd, way, pabstime, pevent));
      break;
    case NETWIB_IO_WAYTYPE_RDWR :
    case NETWIB_IO_WAYTYPE_SUPPORTED :
      return(NETWIB_ERR_PLEASECONSTRUCT);
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_ctl_set(netwib_io *pio,
                                              netwib_io_waytype way,
                                              netwib_io_ctltype type,
                                              netwib_ptr p,
                                              netwib_uint32 ui)
{
  netwib_priv_io_exec *ptr = (netwib_priv_io_exec *)pio->pcommon;
  int reti;

  switch(type) {
    case NETWIB_IO_CTLTYPE_RES:
      return(NETWIB_ERR_PAINVALIDTYPE);
      break;
    case NETWIB_IO_CTLTYPE_END:
      if (way != NETWIB_IO_WAYTYPE_WRITE) return(NETWIB_ERR_PAINVALIDTYPE);
      if (ptr->writefd != -1) {
        reti = close(ptr->writefd);
        if (reti == -1) return(NETWIB_ERR_FUCLOSE);
        ptr->writefd = -1;
      }
      pio->wr.supported = NETWIB_FALSE;
      return(NETWIB_ERR_OK);
      break;
    default:
      return(NETWIB_ERR_PLEASETRYNEXT);
  }

  p = p; /* for compiler warning */
  ui = ui; /* for compiler warning */
  return(NETWIB_ERR_PLEASETRYNEXT);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_ctl_get(netwib_io *pio,
                                              netwib_io_waytype way,
                                              netwib_io_ctltype type,
                                              netwib_ptr p,
                                              netwib_uint32 *pui)
{

  switch(type) {
    case NETWIB_IO_CTLTYPE_RES:
      if (pui != NULL) *pui = NETWIB_IO_RESTYPE_EXEC;
      return(NETWIB_ERR_OK);
    case NETWIB_IO_CTLTYPE_END:
      return(NETWIB_ERR_PAINVALIDTYPE);
    default:
      return(NETWIB_ERR_PLEASETRYNEXT);
  }

  pio = pio; /* for compiler warning */
  way = way; /* for compiler warning */
  p = p; /* for compiler warning */
  return(NETWIB_ERR_PLEASETRYNEXT);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_io_exec_fclose(netwib_io *pio)
{
  netwib_priv_io_exec *ptr = (netwib_priv_io_exec *)pio->pcommon;

  netwib_er(netwib_priv_io_exec_close(ptr));
  netwib_er(netwib_ptr_free(&pio->pcommon));
  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_io_init_exec(netwib_constbuf *pbufcommand,
                               netwib_io_waytype providedway,
                               netwib_bool killonclose,
                               netwib_bool *pexitednormally,
                               netwib_uint32 *preturnedvalue,
                               netwib_io **ppio)
{
  netwib_bool rdsup, wrsup;
  netwib_ptr pcommon;
  netwib_err ret;

  netwib_er(netwib_ptr_malloc(sizeof(netwib_priv_io_exec), &pcommon));
  ret = netwib_priv_io_exec_init(pbufcommand, providedway, killonclose,
                                 pexitednormally, preturnedvalue,
                                 &rdsup, &wrsup,
                                 (netwib_priv_io_exec *)pcommon);
  if (ret != NETWIB_ERR_OK) {
    netwib_er(netwib_ptr_free(&pcommon));
    return(ret);
  }

  netwib_er(netwib_io_init(rdsup, wrsup,
                           pcommon,
                           &netwib_priv_io_exec_read,
                           &netwib_priv_io_exec_write,
                           &netwib_priv_io_exec_wait,
                           NULL,
                           &netwib_priv_io_exec_ctl_set,
                           &netwib_priv_io_exec_ctl_get,
                           &netwib_priv_io_exec_fclose,
                           ppio));

  return(NETWIB_ERR_OK);
}
