/*
                                  NETWOX
                             Network toolbox
                Copyright(c) 1999-2005 Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
                   http://laurentconstantin.free.fr/
                   http://membres.lycos.fr/lauconstantin/
  [my current email address is on the web servers]

                                  -----
  This file is part of Netwox.

  Netwox is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  version 2 as published by the Free Software Foundation.

  Netwox is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details (http://www.gnu.org/).

------------------------------------------------------------------------
*/

/*-------------------------------------------------------------*/
#include "../../netwox.h"

/*-------------------------------------------------------------*/
netwib_err netwox_md_init(netwox_mdversion mdversion,
                          netwox_md_context *pcontext)
{
  pcontext->mdversion = mdversion;

  pcontext->a = 0x67452301u;
  pcontext->b = 0xefcdab89u;
  pcontext->c = 0x98badcfeu;
  pcontext->d = 0x10325476u;

  netwib__uint64_init_uint32(0, pcontext->totalbytes);

  pcontext->bufsize = 0;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
/* update a 64 byte block */
static netwib_err netwox_priv_md_update64(netwox_md_context *pcontext,
                                          netwib_constdata data)
{
  register netwib_uint32 a, b, c, d;
  netwib_uint32 in[16], i;

  a = pcontext->a;
  b = pcontext->b;
  c = pcontext->c;
  d = pcontext->d;

  for (i = 0; i < 16; i++) {
    netwib__data_decode_uint32_le(data, in[i]);
  }

  if (pcontext->mdversion == NETWOX_MDVERSION_4) {
#define MD4_F(x,y,z) ((z) ^ ((x) & ((y) ^ (z))))
#define MD4_G(x,y,z) (((x) & (y)) | ((x) & (z)) | ((y) & (z)))
#define MD4_H(x,y,z) ((x) ^ (y) ^ (z))
#define MD4_ROUND1(v,x,y,z,ui,s) v = netwib_c2_uint32_rol(v+MD4_F(x,y,z)+ui, s)
#define MD4_ROUND2(v,x,y,z,ui,s) v = netwib_c2_uint32_rol(v+MD4_G(x,y,z)+ui+0x5a827999u, s)
#define MD4_ROUND3(v,x,y,z,ui,s) v = netwib_c2_uint32_rol(v+MD4_H(x,y,z)+ui+0x6ed9eba1u, s)
    MD4_ROUND1(a, b, c, d, in[0], 3);
    MD4_ROUND1(d, a, b, c, in[1], 7);
    MD4_ROUND1(c, d, a, b, in[2], 11);
    MD4_ROUND1(b, c, d, a, in[3], 19);
    MD4_ROUND1(a, b, c, d, in[4], 3);
    MD4_ROUND1(d, a, b, c, in[5], 7);
    MD4_ROUND1(c, d, a, b, in[6], 11);
    MD4_ROUND1(b, c, d, a, in[7], 19);
    MD4_ROUND1(a, b, c, d, in[8], 3);
    MD4_ROUND1(d, a, b, c, in[9], 7);
    MD4_ROUND1(c, d, a, b, in[10], 11);
    MD4_ROUND1(b, c, d, a, in[11], 19);
    MD4_ROUND1(a, b, c, d, in[12], 3);
    MD4_ROUND1(d, a, b, c, in[13], 7);
    MD4_ROUND1(c, d, a, b, in[14], 11);
    MD4_ROUND1(b, c, d, a, in[15], 19);
    MD4_ROUND2(a, b, c, d, in[0], 3);
    MD4_ROUND2(d, a, b, c, in[4], 5);
    MD4_ROUND2(c, d, a, b, in[8], 9);
    MD4_ROUND2(b, c, d, a, in[12], 13);
    MD4_ROUND2(a, b, c, d, in[1], 3);
    MD4_ROUND2(d, a, b, c, in[5], 5);
    MD4_ROUND2(c, d, a, b, in[9], 9);
    MD4_ROUND2(b, c, d, a, in[13], 13);
    MD4_ROUND2(a, b, c, d, in[2], 3);
    MD4_ROUND2(d, a, b, c, in[6], 5);
    MD4_ROUND2(c, d, a, b, in[10], 9);
    MD4_ROUND2(b, c, d, a, in[14], 13);
    MD4_ROUND2(a, b, c, d, in[3], 3);
    MD4_ROUND2(d, a, b, c, in[7], 5);
    MD4_ROUND2(c, d, a, b, in[11], 9);
    MD4_ROUND2(b, c, d, a, in[15], 13);
    MD4_ROUND3(a, b, c, d, in[0], 3);
    MD4_ROUND3(d, a, b, c, in[8], 9);
    MD4_ROUND3(c, d, a, b, in[4], 11);
    MD4_ROUND3(b, c, d, a, in[12], 15);
    MD4_ROUND3(a, b, c, d, in[2], 3);
    MD4_ROUND3(d, a, b, c, in[10], 9);
    MD4_ROUND3(c, d, a, b, in[6], 11);
    MD4_ROUND3(b, c, d, a, in[14], 15);
    MD4_ROUND3(a, b, c, d, in[1], 3);
    MD4_ROUND3(d, a, b, c, in[9], 9);
    MD4_ROUND3(c, d, a, b, in[5], 11);
    MD4_ROUND3(b, c, d, a, in[13], 15);
    MD4_ROUND3(a, b, c, d, in[3], 3);
    MD4_ROUND3(d, a, b, c, in[11], 9);
    MD4_ROUND3(c, d, a, b, in[7], 11);
    MD4_ROUND3(b, c, d, a, in[15], 15);
  } else if (pcontext->mdversion == NETWOX_MDVERSION_5) {
#define MD5_F(x,y,z) ((z) ^ ((x) & ((y) ^ (z))))
#define MD5_G(x,y,z) MD5_F(z, x, y)
#define MD5_H(x,y,z) ((x) ^ (y) ^ (z))
#define MD5_I(x,y,z) ((y) ^ ((x) | ~(z)))
#define MD5_ROUND1(v,x,y,z,ui,s) v+=MD5_F(x,y,z)+ui, v=netwib_c2_uint32_rol(v,s), v+=x
#define MD5_ROUND2(v,x,y,z,ui,s) v+=MD5_G(x,y,z)+ui, v=netwib_c2_uint32_rol(v,s), v+=x
#define MD5_ROUND3(v,x,y,z,ui,s) v+=MD5_H(x,y,z)+ui, v=netwib_c2_uint32_rol(v,s), v+=x
#define MD5_ROUND4(v,x,y,z,ui,s) v+=MD5_I(x,y,z)+ui, v=netwib_c2_uint32_rol(v,s), v+=x
    MD5_ROUND1(a, b, c, d, in[0] + 0xd76aa478u, 7);
    MD5_ROUND1(d, a, b, c, in[1] + 0xe8c7b756u, 12);
    MD5_ROUND1(c, d, a, b, in[2] + 0x242070db, 17);
    MD5_ROUND1(b, c, d, a, in[3] + 0xc1bdceeeu, 22);
    MD5_ROUND1(a, b, c, d, in[4] + 0xf57c0fafu, 7);
    MD5_ROUND1(d, a, b, c, in[5] + 0x4787c62a, 12);
    MD5_ROUND1(c, d, a, b, in[6] + 0xa8304613u, 17);
    MD5_ROUND1(b, c, d, a, in[7] + 0xfd469501u, 22);
    MD5_ROUND1(a, b, c, d, in[8] + 0x698098d8, 7);
    MD5_ROUND1(d, a, b, c, in[9] + 0x8b44f7afu, 12);
    MD5_ROUND1(c, d, a, b, in[10] + 0xffff5bb1u, 17);
    MD5_ROUND1(b, c, d, a, in[11] + 0x895cd7beu, 22);
    MD5_ROUND1(a, b, c, d, in[12] + 0x6b901122, 7);
    MD5_ROUND1(d, a, b, c, in[13] + 0xfd987193u, 12);
    MD5_ROUND1(c, d, a, b, in[14] + 0xa679438eu, 17);
    MD5_ROUND1(b, c, d, a, in[15] + 0x49b40821, 22);
    MD5_ROUND2(a, b, c, d, in[1] + 0xf61e2562u, 5);
    MD5_ROUND2(d, a, b, c, in[6] + 0xc040b340u, 9);
    MD5_ROUND2(c, d, a, b, in[11] + 0x265e5a51, 14);
    MD5_ROUND2(b, c, d, a, in[0] + 0xe9b6c7aau, 20);
    MD5_ROUND2(a, b, c, d, in[5] + 0xd62f105du, 5);
    MD5_ROUND2(d, a, b, c, in[10] + 0x02441453, 9);
    MD5_ROUND2(c, d, a, b, in[15] + 0xd8a1e681u, 14);
    MD5_ROUND2(b, c, d, a, in[4] + 0xe7d3fbc8u, 20);
    MD5_ROUND2(a, b, c, d, in[9] + 0x21e1cde6, 5);
    MD5_ROUND2(d, a, b, c, in[14] + 0xc33707d6u, 9);
    MD5_ROUND2(c, d, a, b, in[3] + 0xf4d50d87u, 14);
    MD5_ROUND2(b, c, d, a, in[8] + 0x455a14ed, 20);
    MD5_ROUND2(a, b, c, d, in[13] + 0xa9e3e905u, 5);
    MD5_ROUND2(d, a, b, c, in[2] + 0xfcefa3f8u, 9);
    MD5_ROUND2(c, d, a, b, in[7] + 0x676f02d9, 14);
    MD5_ROUND2(b, c, d, a, in[12] + 0x8d2a4c8au, 20);
    MD5_ROUND3(a, b, c, d, in[5] + 0xfffa3942u, 4);
    MD5_ROUND3(d, a, b, c, in[8] + 0x8771f681u, 11);
    MD5_ROUND3(c, d, a, b, in[11] + 0x6d9d6122, 16);
    MD5_ROUND3(b, c, d, a, in[14] + 0xfde5380cu, 23);
    MD5_ROUND3(a, b, c, d, in[1] + 0xa4beea44u, 4);
    MD5_ROUND3(d, a, b, c, in[4] + 0x4bdecfa9, 11);
    MD5_ROUND3(c, d, a, b, in[7] + 0xf6bb4b60u, 16);
    MD5_ROUND3(b, c, d, a, in[10] + 0xbebfbc70u, 23);
    MD5_ROUND3(a, b, c, d, in[13] + 0x289b7ec6, 4);
    MD5_ROUND3(d, a, b, c, in[0] + 0xeaa127fau, 11);
    MD5_ROUND3(c, d, a, b, in[3] + 0xd4ef3085u, 16);
    MD5_ROUND3(b, c, d, a, in[6] + 0x04881d05, 23);
    MD5_ROUND3(a, b, c, d, in[9] + 0xd9d4d039u, 4);
    MD5_ROUND3(d, a, b, c, in[12] + 0xe6db99e5u, 11);
    MD5_ROUND3(c, d, a, b, in[15] + 0x1fa27cf8, 16);
    MD5_ROUND3(b, c, d, a, in[2] + 0xc4ac5665u, 23);
    MD5_ROUND4(a, b, c, d, in[0] + 0xf4292244u, 6);
    MD5_ROUND4(d, a, b, c, in[7] + 0x432aff97, 10);
    MD5_ROUND4(c, d, a, b, in[14] + 0xab9423a7u, 15);
    MD5_ROUND4(b, c, d, a, in[5] + 0xfc93a039u, 21);
    MD5_ROUND4(a, b, c, d, in[12] + 0x655b59c3, 6);
    MD5_ROUND4(d, a, b, c, in[3] + 0x8f0ccc92u, 10);
    MD5_ROUND4(c, d, a, b, in[10] + 0xffeff47du, 15);
    MD5_ROUND4(b, c, d, a, in[1] + 0x85845dd1u, 21);
    MD5_ROUND4(a, b, c, d, in[8] + 0x6fa87e4f, 6);
    MD5_ROUND4(d, a, b, c, in[15] + 0xfe2ce6e0u, 10);
    MD5_ROUND4(c, d, a, b, in[6] + 0xa3014314u, 15);
    MD5_ROUND4(b, c, d, a, in[13] + 0x4e0811a1, 21);
    MD5_ROUND4(a, b, c, d, in[4] + 0xf7537e82u, 6);
    MD5_ROUND4(d, a, b, c, in[11] + 0xbd3af235u, 10);
    MD5_ROUND4(c, d, a, b, in[2] + 0x2ad7d2bb, 15);
    MD5_ROUND4(b, c, d, a, in[9] + 0xeb86d391u, 21);
  } else {
    return(NETWOX_ERR_INTERNALERROR);
  }

  pcontext->a += a;
  pcontext->b += b;
  pcontext->c += c;
  pcontext->d += d;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
/* take data from context */
static netwib_err netwox_priv_md_update64_ctx(netwox_md_context *pcontext)
{

  if (pcontext->bufsize != NETWOX_MD_BLOCK_MAXLEN) {
    return(NETWOX_ERR_INTERNALERROR);
  }

  netwib_er(netwox_priv_md_update64(pcontext, pcontext->buf));
  pcontext->bufsize = 0;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwox_md_update(netwox_md_context *pcontext,
                            netwib_constbuf *pbuf)
{
  netwib_data data;
  netwib_uint32 datasize;
  netwib_uint64 datasize64;

  datasize = netwib__buf_ref_data_sizenull(pbuf);
  if (datasize == 0) {
    return(NETWIB_ERR_OK);
  }
  data = netwib__buf_ref_data_ptr(pbuf);

  netwib__uint64_init_uint32(datasize, datasize64);
  netwib__uint64_add(datasize64, pcontext->totalbytes, pcontext->totalbytes);

  if (pcontext->bufsize) {
    while (datasize && (pcontext->bufsize < NETWOX_MD_BLOCK_MAXLEN)) {
      pcontext->buf[pcontext->bufsize++] = *data++;
      datasize--;
    }
    if (pcontext->bufsize == NETWOX_MD_BLOCK_MAXLEN) {
      netwib_er(netwox_priv_md_update64_ctx(pcontext));
    }
  }

  while (datasize >= NETWOX_MD_BLOCK_MAXLEN) {
    netwib_er(netwox_priv_md_update64(pcontext, data));
    data += NETWOX_MD_BLOCK_MAXLEN;
    datasize -= NETWOX_MD_BLOCK_MAXLEN;
  }
  while (datasize) {
    pcontext->buf[pcontext->bufsize++] = *data++;
    datasize--;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwox_md_final(netwox_md_context *pcontext,
                           netwib_buf *pmd)
{
  netwib_uint64 totalbits;
  netwib_data data;

  /* create a 56 byte block */
  if (pcontext->bufsize < 56) {
    pcontext->buf[pcontext->bufsize++] = 0x80; /* first pad character */
    while (pcontext->bufsize < 56) {
      pcontext->buf[pcontext->bufsize++] = 0; /* next pad characters */
    }
  } else {
    pcontext->buf[pcontext->bufsize++] = 0x80; /* first pad character */
    while (pcontext->bufsize < 64) {
      pcontext->buf[pcontext->bufsize++] = 0; /* next pad characters */
    }
    netwib_er(netwox_priv_md_update64_ctx(pcontext));
    netwib_c_memset(pcontext->buf, 0, 56);
    pcontext->bufsize = 56;
  }

  /* append the 64 bit count
     Note: this overflows if totalbytes > 0x1FFFFFFFFFFFFFFF, but we ignore
     because netwox will never use such a big value.
  */
  netwib__uint64_shl(pcontext->totalbytes, 3, totalbits);
  pcontext->buf[56] = netwib_c2_uint64_7(totalbits);
  pcontext->buf[57] = netwib_c2_uint64_6(totalbits);
  pcontext->buf[58] = netwib_c2_uint64_5(totalbits);
  pcontext->buf[59] = netwib_c2_uint64_4(totalbits);
  pcontext->buf[60] = netwib_c2_uint64_3(totalbits);
  pcontext->buf[61] = netwib_c2_uint64_2(totalbits);
  pcontext->buf[62] = netwib_c2_uint64_1(totalbits);
  pcontext->buf[63] = netwib_c2_uint64_0(totalbits);
  pcontext->bufsize += 8;
  netwib_er(netwox_priv_md_update64_ctx(pcontext));

  /* copy in pmd */
  netwib_er(netwib_buf_wantspace(pmd, NETWOX_MD4_HASH_LEN, &data));
  netwib__data_append_uint32_le(data, pcontext->a);
  netwib__data_append_uint32_le(data, pcontext->b);
  netwib__data_append_uint32_le(data, pcontext->c);
  netwib__data_append_uint32_le(data, pcontext->d);
  pmd->endoffset += NETWOX_MD4_HASH_LEN;

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwox_md_compute(netwox_mdversion mdversion,
                             netwib_constbuf *pbuf,
                             netwib_buf *pmd)
{
  netwox_md_context ctx;

  netwib_er(netwox_md_init(mdversion, &ctx));
  netwib_er(netwox_md_update(&ctx, pbuf));
  netwib_er(netwox_md_final(&ctx, pmd));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/

/*-------------------------------------------------------------*/
static netwib_err netwox_priv_md_test(netwox_mdversion mdversion,
                                      netwib_conststring data,
                                      netwib_conststring mdwanted)
{
  netwib_buf buf, md;

  netwib_er(netwib_buf_init_ext_text(data, &buf));
  netwib_er(netwib_buf_init_mallocdefault(&md));
  netwib_er(netwox_md_compute(mdversion, &buf, &md));
  netwib_er(netwox_buf_check_strmixed(&md, mdwanted));
  netwib_er(netwib_buf_close(&md));

  return(NETWIB_ERR_OK);
}
#define netwox_priv_md4_test(str,strwanted) netwox_priv_md_test(NETWOX_MDVERSION_4,str,strwanted)
#define netwox_priv_md5_test(str,strwanted) netwox_priv_md_test(NETWOX_MDVERSION_5,str,strwanted)

/*-------------------------------------------------------------*/
netwib_err netwox_md_test()
{
  /* rfc test suite */
  netwib_er(netwox_priv_md4_test("", "31d6cfe0d16ae931b73c59d7e0c089c0"));
  netwib_er(netwox_priv_md4_test("a", "bde52cb31de33e46245e05fbdbd6fb24"));
  netwib_er(netwox_priv_md4_test("abc", "a448017aaf21d8525fc10ae87aa6729d"));
  netwib_er(netwox_priv_md4_test("message digest", "d9130a8164549fe818874806e1c7014b"));
  netwib_er(netwox_priv_md4_test("abcdefghijklmnopqrstuvwxyz", "d79e1c308aa5bbcdeea8ed63df412da9"));
  netwib_er(netwox_priv_md4_test("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789", "043f8582f241db351ce627e153e7f0e4"));
  netwib_er(netwox_priv_md4_test("12345678901234567890123456789012345678901234567890123456789012345678901234567890", "e33b4ddc9c38f2199c3e7b164fcc0536"));
  /* created compiling rfc program and using it on aaaa strings */
  netwib_er(netwox_priv_md4_test("a", "bde52cb31de33e46245e05fbdbd6fb24"));
  netwib_er(netwox_priv_md4_test("aa", "0de97e6bacb92b24d7578ffb8d58f51e"));
  netwib_er(netwox_priv_md4_test("aaa", "918d7099b77c7a06634c62ccaf5ebac7"));
  netwib_er(netwox_priv_md4_test("aaaa", "30fdb877509c742c0ef3d63ddbec5146"));
  netwib_er(netwox_priv_md4_test("aaaaa", "54485d61c2bf8519c3997d2c17d41b43"));
  netwib_er(netwox_priv_md4_test("aaaaaa", "9135d5535d445a5adc299d227d3bdbfb"));
  netwib_er(netwox_priv_md4_test("aaaaaaa", "eb393983d7223a7271398da9cd13f13c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaa", "23008f046fc579f2d373339ec07f1ef1"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaa", "a38217d543726545e70685379586f249"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaa", "55aee4317cf6626378bdd590e1a10009"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaa", "528bca944a4fc5f156765b0c415a0aea"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaa", "8f919c346c23b06b46c872be5f80d919"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaa", "eb50153829a34a8de50ecceec7d44bae"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaa", "0af8eb203f383dcf6a9d888de443572f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaa", "c531cb0a83667b164886e6c1538ad95e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaa", "877a3d1769c7fa80a74e7bd9d7602ef3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaa", "df84f880a964489d9832af34fa58e591"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaa", "80e3d8a01982aa1e14994e453d33dd26"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaa", "f2f147fb12388bece57eca1dcc5ed53f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaa", "1d9db7a8b873e64a5c62727edf6d4bbd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaa", "cbbe5c1d394bb0b081e960fef4e7ca15"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaa", "0641e7cd13c7fa26f6da39e83cd31252"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaa", "76d25193130828accf4d771acb1e51e3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaa", "ab434803006332ab606b8c9d284579c9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaa", "ea31d4cd2d48469501e09c62da35fdbd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaa", "9a374b8b9dd4d3d02ac55036236e7a4d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaa", "cc678cd190cbd158e2a111a6a8e6eb4b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaa", "dd3d0c638699b8db7d4776a7bf415394"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ad4914d6703ec452117852fe99d45e83"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d4450595903614027ba328eefa0ea601"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b439b841fd3becff4e2dac49d19ed7cf"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7dfef9b2eb78b2367246c381c8856478"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b3f634cc931234dedf1e51b0015914f9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c9ee7f5964094201ee080b572ef135e0"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e02f85b1a7838b905e90e279f27febc8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "1254586bfd14e030ce4086fa961ce782"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e93b0ebe0fe3c688419faf37511c8f5b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d6d79128936f4b32d01e395aecf29d82"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7a0ac9f4f25a7c47afa9aa7df30d3221"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2f195c997aada83926fe22847cd3b37c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "09354a0a378cfda1ff95a8885d38c4a8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c2256534bfeae9fa1ee7e86187bb965a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fe8f4ae6501ca2898981f60da8c7f6ae"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "46140f97efbd88928ff112f5367b526a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9d403d371c315ff969baadd8623bc8b3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "068d234494f92f646ba378bf505f8c47"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "af7c0bbed49c6211f1ff4b1739e7ac27"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "14d946cc28ac58f8c5f210a06c1c6f25"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "eb8702358201cdace81aaa2db0c6584e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fb2a7c151e17ec3df8502062d86135e3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2d52d26552cbc27cb68eb829e35dd24d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "38ab80b7c2b45b568488244adf334410"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bd3ed6f7a3a4dd4705360984a18577e5"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "10993f670d6d785f3e87bc46e8da89dc"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c889c81dd86c4d2e025778944ea02881"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d5f9a9e9257077a5f08b0b92f348b0ad"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "872097e6f78e3b53f890459d03bc6fb7"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "277f5f559a60c0af69efda466786fb30"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a70ae7f83d838cce274d7491aa915028"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8c6b85becab240ca5db17955c4d39782"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "672a99ba40462771641359dcc4cb1ddd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5ae7b0c20144bc35483e8d7c16297658"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7ea3da77432d44c323671097d1348fc8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "52f5076fabd22680234a3fa9f9dc5732"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "330e377bf231f3cacfecc2c182fe7e5b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "095ba42e17c00f9336f807d8bdae72a5"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b714fe2e2d4ebc2d801a481ffae39fa9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "769051239bb45773c87c19f35071178a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "49311d7bb7cc3c078f932e873d7769d2"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "df01fc1e5dd0bfc600db67201c977efc"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "09751a7e990fb1d82c0a1293e5f5b3cc"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "040e619a227c013b5201a9796246d4ae"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3470ce6363ed22e5496f138aa7108416"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "26a8c2b51dc60d23597cca9025119030"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e82acdf62a2512470b9580b53df18a2b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c5b92b27da91d2267c23446ecb6a912c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cde8af463ff6018ae7b99ac9de24ea36"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a883a850600df1eef28c573e034e7d18"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a7cce750192ac057036f1b4c5a2605c8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "721a93b051049c47487b06a59acc7d64"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f28aa8607f27e972e483638794c1c5ff"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "577ab2592e92823d26788493457afb35"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "157bb5e384bbfd04719cbb1eacbac84b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "66385a9301518dd05b0f565f08a600ea"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0b87dd13cdf6541f400fabe41fa5ba78"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a6446864a8bf8d07d57d96dd908ea956"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6979b8ecfe581790ac7cd990e8e0736e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f0e85bd3ba0e224fdc2306c256cd5f3a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "60fa15155478d3c8a76e5abbdb77cfbe"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fcff0a17bd61381b77355cef66808308"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "828c52051a9693a1b54be9352268955d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "53a6b8d4dd7d0770a5f6dc9874e7b88c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "00f8653f803627b70ef2e7e1654576c0"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "14a4d10648330012fe672b650c196021"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a6a0b64c05fcd2e57d8cbbc59a1a00b7"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ddc02b8e0a315ba8ee08851668a081a9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4067061356fa1e283ec5f3610e7eacf0"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "717d2ef3060ca3208decae86f9bdcfd9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7b625df18dc2fff7f5244a4c50915893"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a2a3c7c3ee6088bad252bfbbae229bb6"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "547401415a107a8147d3bab71991be0e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e1c162a95ebe24d4a78da81faa6a451b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2a3d6778231dc7ef4ab0d96db648d128"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "89e6cf2b88c9328a4c348a731d317d25"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cf71fec4631db55308ad80186b8dcf37"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "343ca55fc783302ef9a0b33757e5ef19"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "aad33b8ff079a18d6425470d011b4d31"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "357c138b5498b531cb174127fcf14a0f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "73b22be5dbad1d26bd9071afebc35856"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "74101d5e7a9321dad687b4c2ac7e7551"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "69daeffb60dd1dcfc8a0ddf5ed4dea4f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ccbf3dfa0fac8c6e5c2504cf15777e71"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6ee2ad0a2a06e975c2fa8887333de734"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8a7defd65211a52a20cbd989bcb079d7"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "650a6088c41b5951ef46b09f8a8f7a16"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f731acbd40496a63fd33c72bff4adc4c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "64279e932b0a6cf7fbebc12969ad85f9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5257d42ae36dcfc8418fa40600696e16"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e65dd227ccef97fa1d34d70189120f76"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b03ddbd470b47c013e0c7ab2ddd763db"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e5601aa6994470f918405d745ede163c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6baf506a6e6a525e9ef9bbf7e6b4f45b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d312f30d9fff78e5404f8eac3f0b665f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a7a1c6286070e9a7afa4831d2bf7bafd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "941b80acd86c9d9c3f27380591507ded"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "85c05a6bb4b2cf906813652c68686361"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9733b046ad770b4e093b35de4e09e828"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cb4a20a561558e29460190c91dced59f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2adcd303c29f93a3ee33a560ece91cd2"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "52b8ce960bb64e4ec2b579d4047b175e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6ef49aaa109b8120004ffcc801218caf"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "370ed97adf490f75693cd5fc73a8e3e0"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "54de78d79ad53da4ce46f945160b591f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0d9014c7b4a9edb3d594056e78d25b9d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4ae5f06e7a0ab2b7142583873acfccfd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c4ca41e447a27ecee443370b002b6459"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9a64358c2602da3f21d2e79b21e94bf1"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5761a624a7bdedad64e543bc73213e64"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d301a78cb6959f11e81bd7a3c6bf5bb3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5d726c762665398737c34803095e91f3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "db62b01151a01d5e4a00d87f2a48b98a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c310b6e1016ecb9f5a5c5a4b89f17a76"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "33c7d6e29f904b27272e75144be07d18"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "835048e983d82fb0fa151bb8b6fa636e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b9ff2575260e2ad08557eeba52b27cdd"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bcccbcfeab174bddb81cc74dd97984f6"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9b98a75eded6b5af8c449b75a74c30b3"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5f9f642231152dd8cd5caa9b5fc59b5d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "84d82189c5458f8647d338fd62ef1667"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e23de568fd6f45821048ce04e6cb9936"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "258036bafb8c30a22f91b0fcd617bf38"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ede010c66d6a6ca93d3edc90b930ecd2"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "dacacfe5c93b5fe54b1f8051a01d8986"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d2be8e7befe337165b35df1d83169f34"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "de24f3df2037a8f5d20bea65f80e615d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "068d0da7c47d9f5af21c7b5347835b97"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d3bc3aa7ee6ba91ebd05f5686ef740d8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "621c8b6b8a7098984f4c778dc6ccf096"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e6b498b596ae5f284956366fdb96384a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "80ac492cec23bd6cff4911a18787fea4"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d363f07a0fb40f6657c37a536bb98c60"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "712c5ebcfed54d144d91065613f7de21"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ab77975376dd2fd57822a3b2cddaed06"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "493d022f09d8af7823e7d7c92b19e93b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c10f1d67988c74217fd2fd6b2c33f384"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0d7185c73b32317950fbc4b438e3d52c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "81cb2c1d50e4eece6f5ed929dee4da3b"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "482f946f2042cbde505ba38f9251946e"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "65c7faa81a04b94cbd80b84e00440618"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "883576e0cc0fcf4de604ca0294a86fe9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9efef17aca12b845b00923a2aeceaf78"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a5dda3582d56628509233cf67f7ef49d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7860a3216f5564133a8952283b06421f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f4eb7356766bf5f0318c18692c47f092"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "04393082c4f4ad13ed2ebe664d7a4519"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a1f930630262a1bc9f19c938eee72643"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "964c7780c9f338c4813a100232d596c9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "200f2bbae41b56d134f0785460993439"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "08560f4bc732d31d8a7290a5e9048a33"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e8860b7b8bb5dc8fd3cb3a19c9d9bb3c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "975de623f3ed6f11c1bb02a21baf2197"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d4af4b526f0f61c07b6aea3c3007ff7f"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f8f0dde8f284b682c8552977d122ec4d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "dd5adf7607bfcb67c48ed80cd0b4661d"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3a0113eeab8041b50b9f58e30d9aea11"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "016e615d9e08877dff9236deaac234b1"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "281e3750ac6ee2f6a1aed6c1f8685554"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "10750af3fcb469eb01f8559421d885b9"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c7323157e95025f459e86b6dc77c685c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6ffa642403dfcc2ded2e1f16497fe451"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3caa58f9016fc89c4bd3b438f85d50f2"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b49a47e312d187405ca777d26e32c73c"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fa56d708b1e7c1466c0c0708945e4fe8"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d8dad8805f1a97a390e13a4a0f7d0e5a"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a6144bd4950b8b75b6e280d11d68ed65"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3f4711020812cfa7bbcf089cc1ca0dbb"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ed7f41fedd5ae2a9ffb3230de77ed066"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "af49a9860e821528a1ea04c1d520db42"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9e113ce4969caf150eb8997cbcd8f0da"));
  netwib_er(netwox_priv_md4_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e4720c787792df1ec36423b1f3fb0ee6"));

  /* rfc test suite */
  netwib_er(netwox_priv_md5_test("", "d41d8cd98f00b204e9800998ecf8427e"));
  netwib_er(netwox_priv_md5_test("a", "0cc175b9c0f1b6a831c399e269772661"));
  netwib_er(netwox_priv_md5_test("abc", "900150983cd24fb0d6963f7d28e17f72"));
  netwib_er(netwox_priv_md5_test("message digest", "f96b697d7cb7938d525a2f31aaf161d0"));
  netwib_er(netwox_priv_md5_test("abcdefghijklmnopqrstuvwxyz", "c3fcd3d76192e4007dfb496cca67e13b"));
  netwib_er(netwox_priv_md5_test("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789", "d174ab98d277d9f5a5611c2c9f419d9f"));
  netwib_er(netwox_priv_md5_test("12345678901234567890123456789012345678901234567890123456789012345678901234567890", "57edf4a22be3c955ac49da2e2107b67a"));
  /* created with unix md5sum on aaaa strings */
  netwib_er(netwox_priv_md5_test("a", "0cc175b9c0f1b6a831c399e269772661"));
  netwib_er(netwox_priv_md5_test("aa", "4124bc0a9335c27f086f24ba207a4912"));
  netwib_er(netwox_priv_md5_test("aaa", "47bce5c74f589f4867dbd57e9ca9f808"));
  netwib_er(netwox_priv_md5_test("aaaa", "74b87337454200d4d33f80c4663dc5e5"));
  netwib_er(netwox_priv_md5_test("aaaaa", "594f803b380a41396ed63dca39503542"));
  netwib_er(netwox_priv_md5_test("aaaaaa", "0b4e7a0e5fe84ad35fb5f95b9ceeac79"));
  netwib_er(netwox_priv_md5_test("aaaaaaa", "5d793fc5b00a2348c3fb9ab59e5ca98a"));
  netwib_er(netwox_priv_md5_test("aaaaaaaa", "3dbe00a167653a1aaee01d93e77e730e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaa", "552e6a97297c53e592208cf97fbb3b60"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaa", "e09c80c42fda55f9d992e59ca6b3307d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaa", "d57f21e6a273781dbf8b7657940f3b03"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaa", "45e4812014d83dde5666ebdf5a8ed1ed"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaa", "c162de19c4c3731ca3428769d0cd593d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaa", "451599a5f9afa91a0f2097040a796f3d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaa", "12f9cf6998d52dbe773b06f848bb3608"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaa", "23ca472302f49b3ea5592b146a312da0"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaa", "88e42e96cc71151b6e1938a1699b0a27"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaa", "2c60c24e7087e18e45055a33f9a5be91"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaa", "639d76897485360b3147e66e0a8a3d6c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaa", "22d42eb002cefa81e9ad604ea57bc01d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaa", "bd049f221af82804c5a2826809337c9b"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaa", "ff49cfac3968dbce26ebe7d4823e58bd"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaa", "d95dbfee231e34cccb8c04444412ed7d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaa", "40edae4bad0e5bf6d6c2dc5615a86afb"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaa", "a5a8bfa3962f49330227955e24a2e67c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaa", "ae791f19bdf77357ff10bb6b0e97e121"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaa", "aaab9c59a88bf0bdfcb170546c5459d6"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b0f0545856af1a340acdedce23c54b97"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f7ce3d7d44f3342107d884bfa90c966a"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "59e794d45697b360e18ba972bada0123"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3b0845db57c200be6052466f87b2198a"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5eca9bd3eb07c006cd43ae48dfde7fd3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b4f13cb081e412f44e99742cb128a1a5"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4c660346451b8cf91ef50f4634458d41"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "11db24dc3f6c2145701db08625dd6d76"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "80dad3aad8584778352c68ab06250327"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "1227fe415e79db47285cb2689c93963f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8e084f489f1bdf08c39f98ff6447ce6d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "08b2f2b0864bac1ba1585043362cbec9"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4697843037d962f62a5a429e611e0f5f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "10c4da18575c092b486f8ab96c01c02f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "af205d729450b663f48b11d839a1c8df"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0d3f91798fac6ee279ec2485b25f1124"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4c3c7c067634daec9716a80ea886d123"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d1e358e6e3b707282cdd06e919f7e08c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8c6ded4f0af86e0a7e301f8a716c4363"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4c2d8bcb02d982d7cb77f649c0a2dea8"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bdb662f765cd310f2a547cab1cfecef6"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "08ff5f7301d30200ab89169f6afdb7af"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6eb6a030bcce166534b95bc2ab45d9cf"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "1bb77918e5695c944be02c16ae29b25e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b6fe77c19f0f0f4946c761d62585bfea"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e9e7e260dce84ffa6e0e7eb5fd9d37fc"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "eced9e0b81ef2bba605cbc5e2e76a1d0"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ef1772b6dff9a122358552954ad0df65"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3b0c8ac703f828b04c6c197006d17218"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "652b906d60af96844ebd21b674f35e93"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "dc2f2f2462a0d72358b2f99389458606"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "762fc2665994b217c52c3c2eb7d9f406"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cc7ed669cf88f201c3297c6a91e1d18d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cced11f7bbbffea2f718903216643648"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "24612f0ce2c9d2cf2b022ef1e027a54f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b06521f39153d618550606be297466d5"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "014842d480b571495a4a0363793f7367"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c743a45e0d2e6a95cb859adae0248435"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "def5d97e01e1219fb2fc8da6c4d6ba2f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "92cb737f8687ccb93022fdb411a77cca"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a0d1395c7fb36247bfe2d49376d9d133"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ab75504250558b788f99d1ebd219abf2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0f5c6c4e740bfcc08c3c26ccb2673d46"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cddd19bec7f310d8c87149ef47a1828f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "96b39b8b95e016c79d104d83395b8133"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f1fc0b14ff8fa674b02344577e23eeb1"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0e8d28a1cafa3ffcff22afd480cce7d8"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "448539ffc17e1e81005b65581855cef4"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "61e39aae7c53e6e77db2e4405d9fb157"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "618a426895ee6133a372bebd1129b63e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "046c90690c9e36578b9d4a7e1d249c75"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "aadab38075c43296ee7e12466ebb03e3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b15af9cdabbaea0516866a33d8fd0f98"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "986e6938ed767a8ae9530eef54bfe5f1"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7ae25a72b71a42ccbc5477fd989cd512"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "98d34e50d4aa7a893cc7919a91acb0e3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3fc53fc22ea40f1a0afd78fc2cd9aa0f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "923e37c738b9d7b1526f70b65229cc3d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b3966b7a08e5d46fd0774b797ba78dc2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f50c7286b540bb181db1d6e05a51a296"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4efd6c8826e65a61f82af954d431b59b"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ef1031e79e7a15a4470a5e98b23781b5"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "067876bfd0df0f4c5002780ec85e6f8c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "789851dfa4c03563e9cef5f7bc050a7e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "baf934720818ee49477e74fc644faa5e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9a0ea77ca26d2c121ddcc179edb76308"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "20c825561572e33d026f99ddfd999538"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "464c461455c5a927079a13609c20b637"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cf37d42f89b6adb0e1a9e99104501b82"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d266af45e3d06b70d9f52e2df4344186"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f8b59fa22eb0ba944e2b7aa24d67b681"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0918d7c2f9062743450a86eae9dde1a3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "36a92cc94a9e0fa21f625f8bfb007adf"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "681d73898dad5685d48b5e8438bc3a66"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "337ccef058459c3c16411381778da0c4"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6ccdfcc742862036ce07583633c5f77e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ddfa1adc974649dc5b414be86def7457"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "650ebc28ad85f11aa4b63b6ee565b89d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e4571793bcaba284017eeabd8df85697"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4fc040d354ad9ba5e4f62862109d3e17"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "25814274e02aa7cc03d6314eb703e655"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "11378ecaee0089c840d26352704027e3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "86f950bfcd824d5546da01c40576db31"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "089f243d1e831c5879aa375ee364a06e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9146ef3527c7cfcc66dc615c3986e391"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d727cfdfc9ed0347e6917a68b982f7bc"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "da8f45e1fdc12deecfe56aeb5288796e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "29cfcf52d8250a253a535cf7989c7bd2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0f6eb555b8e3c35411eebe9348594193"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a922439f963e7e59040e4756992c6f1b"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "81f8453cf3f7e5ee5479c777e5a8d80c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8a7bd0732ed6a28ce75f6dabc90e1613"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5f61c0ccad4cac44c75ff505e1f1e537"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f6acfca2d47c87f2b14ca038234d3614"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "269fc62c517f3d55c368152addca57e7"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "50587cb16413da779b35508018721647"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5e4a3ecfdaa4636b84a39b6a7be7c047"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c5339dc2af6bf595580281ffb07353f6"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e51176a47347e167ed0ed766b6de1a0c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "020406e1d05cdc2aa287641f7ae2cc39"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e510683b3f5ffe4093d021808bc6ff70"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b325dc1c6f5e7a2b7cf465b9feab7948"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e016e4ccc7fdaea56fc377600b58c4cb"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3870ec709d2fc64b255d65be3123ad69"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "a92bde1f862c3fe797ecd69510bbd266"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "04daa146f3a2256fdcbf015c0f67e168"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3d13c8bf627421ccc937aa1c9ac87bf1"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "247dc7ffc545e4dda64ae12def481c4e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2dfd4def392ee9563241b7db7eb7c346"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d11a18a4743a1a0a699d1704efb74a0d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "55b62fabd9c77d44d86e992eeeb093e6"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9a72cf7d0bd5ae2907c79f91837e3ced"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "d3828cce1835534475029202ebd799e4"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b0bebbf0015658d4740679f263a3f01f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "02368ebf1f53bc4634211b1693021666"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "04960f7d18960e348372949e4baa9752"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c6041e7a86d407e9402b175670519260"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "439fd4c056bec1d14acd393746f6ae59"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "81a855120e04494c5a6c874a2360fd57"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ef57bd47a964dc3aadd959c4131e64ac"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0b0ab27b16cbba267c141fe0f4ee9189"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "abccd84f340bfe4ba59095cc3d5ca6ad"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bc620e8c15265f195c8818e2f3e3c58b"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fdcd84c4143286f6fc70c69208acd18d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "50e05071e773b1e9f3009a4a559ce6b2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9e69c7a6c1863fbba2532f09ba665bde"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "47a962111aa5187eeef3d17a278d95f2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "c13e57e33526bc713b5a1825f92651bc"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "72b392f15593e42404b38e5c889fa75e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "5327acd3278274265d44e22ccfc4042c"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "930dcac6da160b2a4c51879da76d3417"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "41292c326f926f1534ead47fe302f0a0"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "2bdecb5cf6b69a00f7832299ef2fb5a5"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8bf93e9e8a3e4396de3f211c788e177e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "eea9cb566e19d6a7f55fbae78d94ef2a"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3b8452700a829dec78397aa5c0458dd3"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7950059f699eaea1e0a1759340d7c153"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "40840c5f1de00f17a8e70d5bd4d00af2"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "80f86f6af38be9ca8e40c2dc44491a0a"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7aab2c2e72c77163e7102412dc332125"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bfd6869ae2ee2fe2675846d341eaa67d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7e4d976f6d552d1d5bac7e2693dc8759"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "37d9884c32abfc6f372ee899434e64ad"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e362cd83a4b49d81ac6788b7839a56fd"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "9203cbb93b25d80b9d1b75e3c6c4b0dc"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "77441eda11554ec5b915d942605f66ed"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e0fe0c02b5c9c5afe10ab9d6a3769efe"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "cc7682cf11b214e928f3df899772e789"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "ade0901d347afb25ecf9df4955bb8061"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "987379587cbe8e94b7057269232ff826"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fd44a60101b04b7ddbc2b4e9b509ca1f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "53107a7f1e6f13a2e63239b6f2bf0ef1"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "0b82cdd562f26aaa2459610a7ba8cd76"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "79f12de7255e9c8c0ec9a9be45ee6210"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "92338d8de02ed7aa8b3adc9120b94e71"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8fc48efda580fce85b8705d540e8382e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "63642b027ee89938c922722650f2eb9b"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "fe54daa473502e9cc2c26dd66d564eab"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "b90f3d4b7dcd8cdd8d96cb14695f4793"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "3e73392e7a03bca45b67650d79a8fc63"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7fe51f2642dffbabc33eea2fcc2039ba"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bc33790e52f99718cf920329961ee753"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "54d1e41ebac5db7886f01ab0afb65b17"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "16e2824f7a3f00ef0028994182071953"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "234c07907df5019d5f40f03936939bce"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "8ea3af1d9476fa0b6c04ce4f3a336c03"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "e95b69eae07d498d484afc771d1c45fc"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "f22a673abbc4372544ba37b51a5f5a91"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "7e6161eb1be7b06928c536fada91b7f1"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "4dfe3c301e88fff67822e1cfcfece43f"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "edda210ac6645fbf5815eb4c58821f6d"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "6a514de2bf1926129b08f9234cd0115e"));
  netwib_er(netwox_priv_md5_test("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "887f30b43b2867f4a9accceee7d16e6c"));

  return(NETWIB_ERR_OK);
}
