/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgCertProperties.cpp: implementation of the DlgCertProperties class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgCertProperties.h"
#include "dlgs_wdr.h"
#include "clintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


BEGIN_EVENT_TABLE(DlgCertPropertiesListCtrl, wxListCtrl)
    EVT_LIST_ITEM_SELECTED(IDC_LISTCTRL, DlgCertPropertiesListCtrl::OnItemClick)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(DlgCertProperties, wxDialog)
    EVT_BUTTON(IDC_OK, DlgCertProperties::OnOk)
    EVT_BUTTON(IDC_SAVE_CERT, DlgCertProperties::OnSaveCert)
END_EVENT_TABLE()

void DlgCertPropertiesListCtrl::OnItemClick(wxListEvent& event)
{
	DlgCertProperties * wParent = (DlgCertProperties *)GetParent();
	wParent->OnItemClick(event);
}


DlgCertProperties::DlgCertProperties(wxWindow * wParent, const PKI_CERT & cert):wxDialog(wParent, (wxWindowID)-1, _("Certificate properties"), wxDefaultPosition, wxDefaultSize, wxSYSTEM_MENU | wxMINIMIZE_BOX | wxCAPTION), m_cert(cert)
{
	m_imageListSmall = NULL;
	InitDatas();
}
DlgCertProperties::~DlgCertProperties()
{
	void * value;
	long SelectedItem = -1;
	while( (SelectedItem = m_listCtrl->GetNextItem(SelectedItem, wxLIST_NEXT_ALL, wxLIST_STATE_DONTCARE)) != -1)
	{
		value = (void*)m_listCtrl->GetItemData(SelectedItem);
		if(value)
		{
			free(value);
		}
	}
	if(m_imageListSmall) delete m_imageListSmall;
}


void DlgCertProperties::InitDatas()
{
	int IconIdValue;
	int IconIdExtCrit;
	int IconIdExtNorm;

	mString datas;
	mString strDatas1;
	mString strDatas2;



	DlgCertProperties_SetWindow(this);
	((wxTextCtrl *)FindWindow(IDC_VALUE))->SetBackgroundColour(GetBackgroundColour());

	m_listCtrl = ((DlgCertPropertiesListCtrl *)FindWindow(IDC_LISTCTRL));
	m_imageListSmall = new wxImageList(16, 16, TRUE);

	wxIcon ico;
	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_VALUE));
	IconIdValue = m_imageListSmall->Add( ico );
	
	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_EXT_CRIT));
	IconIdExtCrit = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_EXT));
	IconIdExtNorm = m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	m_listCtrl->InsertColumn(0, _("Field"), wxLIST_FORMAT_LEFT, 135);
	m_listCtrl->InsertColumn(1, _("Value"), wxLIST_FORMAT_LEFT, 198);




	// The version
	datas.sprintf("V%ld", m_cert.GetVersion() + 1);
	InserItem(_("Version"), datas.c_str(), datas.c_str(), IconIdValue);


	//The serial
	datas.sprintf("%ld", m_cert.GetSerial());
	InserItem(_("Serial"), datas.c_str(), datas.c_str(), IconIdValue);


	//The signature algo
	datas = m_cert.GetSignatureAlg();
	InserItem(_("Signature algorithm"), datas.c_str(), datas.c_str(), IconIdValue);

	//The issuer
	strDatas1 = FormatDN(m_cert.GetIssuerDN(), true);
	strDatas2 = FormatDN(m_cert.GetIssuerDN(), false);
	InserItem(_("Issuer"), strDatas1.c_str(), strDatas2.c_str(), IconIdValue);

	//The valid from
	datas = FormatTime(m_cert.GetStartDate());
	InserItem(_("Valid from"), datas.c_str(), datas.c_str(), IconIdValue);

	//The valid until
	datas = FormatTime(m_cert.GetEndDate());
	InserItem(_("Valid until"), datas.c_str(), datas.c_str(), IconIdValue);

	//The subject
	strDatas1 = FormatDN(m_cert.GetCertDN(), true);
	strDatas2 = FormatDN(m_cert.GetCertDN(), false);
	InserItem(_("Subject"), strDatas1.c_str(), strDatas2.c_str(), IconIdValue);

	//The key len
	datas.sprintf("%ld Bits", m_cert.GetKeyLen());
	strDatas1 = GetPubKey(m_cert.GetPublicKey());
	InserItem(_("Public key"), datas.c_str(), strDatas1.c_str(), IconIdValue);




	const char * name;
	const char * value;
	long i;
	int IconId;

	for(i=0; i < m_cert.GetExtensions().EntriesCount(); i++)
	{
		name = m_cert.GetExtensions().GetName(i);
		value = m_cert.GetExtensions().Get(i);
		if(!name || ! value) continue;

		if(strstr(value, "critical,"))
		{
			IconId = IconIdExtCrit;
			value += strlen("critical,");
			while(isspace(*value)) value++;
		}
		else
		{
			IconId = IconIdExtNorm;
		}

		InserItem(name, value, value, IconId);
	}

	//The thumbprint algo
	datas = m_cert.GetThumprintAlgo();
	InserItem(_("Thumbprint Algotithm"), datas.c_str(), datas.c_str(), IconIdValue);

	//The thumbprint
	datas = m_cert.GetThumprint();
	InserItem(_("Thumbprint"), datas.c_str(), datas.c_str(), IconIdValue);

	CenterOnScreen();
	ShowModal();
}


mString DlgCertProperties::GetPubKey(const EVP_PKEY * pkey)
{
	mString retValue;
	char * datas;
	BIO * membio = BIO_new(BIO_s_mem());
	if(!membio) return "";
	int KeyLen;
	BIGNUM * PubKey = NULL;;
	int i;


	if (pkey->type == EVP_PKEY_RSA)
	{
		PubKey = pkey->pkey.rsa->n;
	}
	if(!PubKey)
	{
		BIO_free_all(membio);
		return retValue;
	}

	KeyLen = BN_num_bytes(PubKey);

	datas = (char *)malloc(KeyLen + 1);
	if(!datas)
	{
		BIO_free_all(membio);
		return retValue;
	}

	if(BN_bn2bin(PubKey, (unsigned char *)datas) <= 0)
	{
		BIO_free_all(membio);
		return retValue;
	}



	for (i=0; i<KeyLen; i++)
	{
		if ((i%18) == 0 && i)
			BIO_write(membio,"\n",2);
	
		BIO_printf(membio,"%02x%s",(unsigned char)datas[i], ((i+1) == KeyLen)?"":":");
	}
	free(datas);
	

	datas = (char *)malloc(BIO_number_written(membio) + 1);
	if(!datas)
	{
		BIO_free_all(membio);
		return retValue;
	}

	BIO_read(membio, datas, BIO_number_written(membio));
	datas[BIO_number_written(membio)]=0;
	BIO_free_all(membio);

	retValue = datas;
	free(datas);

	return retValue;
}

void DlgCertProperties::OnItemClick(wxListEvent& event)
{
	char * value;
	long SelectedItem;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	value = (char*)m_listCtrl->GetItemData(SelectedItem);

	((wxTextCtrl *)FindWindow(IDC_VALUE))->SetValue(value);
}

void DlgCertProperties::OnOk(wxCommandEvent& event)
{
	Close(TRUE);
}

void DlgCertProperties::OnSaveCert(wxCommandEvent& event)
{
	long pos;
	const char * cn;
	pos = m_cert.GetCertDN().SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
	if(pos != HASHTABLE_NOT_FOUND)
	{
		cn = m_cert.GetCertDN().Get(pos);
		if(!cn)
			cn = "";
	}
	else
	{
		cn = "";
	}
	wxFileDialog dialog(this, _("Save certificate"), "", cn, _("Certificate File (*.cer)|*.cer"), wxSAVE|wxOVERWRITE_PROMPT);

	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}

	FILE * fd;

	fd = fopen(dialog.GetPath() , "wb");
	
	if(!fd)
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}

	
	if (PEM_write_X509(fd, m_cert.GetX509()) <= 0)
	{
		HandleError(NULL , this);
	}

	fclose(fd);
}

void DlgCertProperties::InserItem(const char *name, const char *value, const char *datas, int icon_id)
{
	int RowId = m_listCtrl->GetItemCount();

	m_listCtrl->InsertItem(RowId, name, icon_id);
	m_listCtrl->SetItemData(RowId, (long)strdup(datas));
	m_listCtrl->SetItem(RowId, 1, value);
}
