/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CRL.cpp: implementation of the PKI_CRL class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_CRL.h"
#include <openssl/ocsp.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PKI_CRL PKI_CRL::EmptyInstance;

PKI_CRL::PKI_CRL(const PKI_CRL & other)
{
	Reset();
	*this = other;
}


PKI_CRL::PKI_CRL()
{
	Reset();
}

PKI_CRL::~PKI_CRL()
{
	Clear();
}

bool PKI_CRL::X509CrlToString()
{
	int dsize=0;
	unsigned char *p,*data=NULL;
	PEM_DER pem_cvrt;
	
	if ((dsize=i2d_X509_CRL(crl,NULL)) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_X509CRL_CRL);
		return false;
	}

	/* dzise + 8 bytes are needed */
	data=(unsigned char *)malloc(dsize+20);
	if (!data)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p=data;
	if((dsize=i2d_X509_CRL(crl,&p))<0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_X509CRL_CRL);
		return false;
	}

	if(!lePEM_CRL.FromDER(data, dsize))
	{
		free(data);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	free(data);
	return true;
}

bool PKI_CRL::StringToX509Crl(const char * CrlPem)
{
	int dsize=0;
	unsigned char *data=NULL;
	unsigned char *tmp_data=NULL;
	PEM_DER der_cvrt;
	BIO * sout;

	if(*CrlPem == '-')
	{
		sout = BIO_new_mem_buf((char*)CrlPem, strlen(CrlPem));
		if(!sout)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
		if(!PEM_read_bio_X509_CRL(sout, &crl, NULL, NULL))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			BIO_free_all(sout);
			return false;
		}
		BIO_free_all(sout);
		return true;
	}

	if(!der_cvrt.Pem2Der(CrlPem, strlen(CrlPem), (char **)&data, &dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	tmp_data=data;
	if(!d2i_X509_CRL(&crl,&data,dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CRL_X509CRL);
		free(tmp_data);
		return false;
	}

	free(tmp_data);
	return true;
}

const mString & PKI_CRL::GetPemCRL() const
{
	return lePEM_CRL;
}

X509_CRL * PKI_CRL::GetX509_CRL(bool dup) const
{
	if(!crl)
	{
		return NULL;
	}
	if(dup)
	{
		CRYPTO_add(&crl->references, 1, CRYPTO_LOCK_X509_CRL);
	}
	return crl;
}

void PKI_CRL::Reset()
{
	crl=NULL;
}

void PKI_CRL::Clear()
{
	InternalExts.Clear();
	IssuerDN.Clear();
	RevokedCerts.clear();
	ClearPointers();
}


void PKI_CRL::AddRevokedCert(REVOCATION_INFO * rev_inf)
{
	RevokedCerts.push_back(*rev_inf);
}

bool PKI_CRL::Generate(const PKI_CERT & CaCert, const char * md, unsigned long ValidityLen, const HashTable_String * Exts)
{
	ClearPointers();

	size_t i;

	X509_REVOKED *r;
	ASN1_TIME *revDate;
	ASN1_TIME * tim;
	ASN1_ENUMERATED *rtmp = NULL;
	ASN1_OBJECT *hold = NULL;

	const EVP_MD * digest;
	ASN1_INTEGER *serial;
	const X509 * CertCA;

	//We test the validity of the CA certificate/key
	if(!CaCert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	const PKI_RSA & CaKey = CaCert.GetPrivateKey();
	if(!CaKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}




	CertCA = CaCert.GetX509();
	if(!CertCA)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	// Starting to create CRL
	if ( !(crl=X509_CRL_new()) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	// CRL v2
	if(!X509_CRL_set_version(crl, 1))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}
	//Set issuer
	if ( !X509_CRL_set_issuer_name(crl, X509_get_subject_name((X509*)CertCA)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}
	
	//Set validity length information
	tim = ASN1_TIME_new();
	if (!tim)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	X509_gmtime_adj(tim,0);
	X509_CRL_set_lastUpdate(crl, tim);	
	X509_gmtime_adj(tim,ValidityLen * 60 * 60);
	X509_CRL_set_nextUpdate(crl, tim);	
	ASN1_TIME_free(tim);

	for(i=0; i<RevokedCerts.size(); i++)
	{
		if((r=X509_REVOKED_new()) == NULL)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}

		//Revocation date
		revDate = ASN1_TIME_set_localtime(NULL, RevokedCerts[i].rev_date);
		if(!revDate)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			return false;
		}

		if(!X509_REVOKED_set_revocationDate(r, revDate))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			ASN1_TIME_free(revDate);
			return false;
		}
		ASN1_TIME_free(revDate);

		
		
		//Revoked serial
		if( !(serial = ASN1_INTEGER_new()) )
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			return false;
		}
		if(!ASN1_INTEGER_set(serial, RevokedCerts[i].serial))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			ASN1_INTEGER_free(serial);
			return false;
		}
		if(!X509_REVOKED_set_serialNumber(r, serial))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			ASN1_INTEGER_free(serial);
			return false;
		}
		ASN1_INTEGER_free(serial);

		// Set revocation reason
		if(RevokedCerts[i].reason != OCSP_REVOKED_STATUS_NOSTATUS)
		{
			rtmp = ASN1_ENUMERATED_new();
			if(!rtmp)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
				X509_REVOKED_free(r);
				return false;
			}
			if(!ASN1_ENUMERATED_set(rtmp, RevokedCerts[i].reason))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
				X509_REVOKED_free(r);
				ASN1_ENUMERATED_free(rtmp);
				return false;
			}
			if(!X509_REVOKED_add1_ext_i2d(r, NID_crl_reason, rtmp, 0, 0))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
				X509_REVOKED_free(r);
				ASN1_ENUMERATED_free(rtmp);
				return false;
			}
			hold = OBJ_nid2obj(NID_hold_instruction_call_issuer);
			if(!hold)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
				X509_REVOKED_free(r);
				ASN1_ENUMERATED_free(rtmp);
				return false;
			}
			if(!X509_REVOKED_add1_ext_i2d(r, NID_hold_instruction_code, hold, 0, 0))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
				X509_REVOKED_free(r);
				ASN1_ENUMERATED_free(rtmp);
				return false;
			}
			ASN1_ENUMERATED_free(rtmp);
		}

		//Add entry to CRL
		if(!X509_CRL_add0_revoked(crl,r))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
			X509_REVOKED_free(r);
			return false;
		}
	}

	//Sort CRL by serial number
	X509_CRL_sort(crl);


	if (md != NULL)
		digest=(EVP_MD*)EVP_get_digestbyname(md);
	else
		digest=(EVP_MD *)EVP_md5();
	if (!digest)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}

	//Add CRL extension if any
	if(Exts)
	{
		X509V3_CTX crlctx;		
		X509V3_set_ctx(&crlctx, (X509*)CertCA, NULL, NULL, crl, 0);

		if(!InternalExts.Add_CrlExtensions(Exts, &crlctx, crl))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	//Sign CRL with CA's private key
	if (!X509_CRL_sign(crl, (EVP_PKEY*)CaKey.GetRsaKey(), digest))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}

	//Convert CRL to PEM
	if(!X509CrlToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}

	if(!LoadIssuerDN())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}

	if(!InternalExts.Load(crl->crl->extensions))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_CRL);
		return false;
	}

	return true;
}


time_t PKI_CRL::GetStartTime() const
{
	if(!crl) return 0;

	return ASN1_TIME_to_timet((char*)crl->crl->lastUpdate->data);
}

time_t PKI_CRL::GetEndTime() const
{
	if(!crl) return 0;

	return ASN1_TIME_to_timet((char*)crl->crl->nextUpdate->data);
}

bool PKI_CRL::CommonLoad()
{
	//Convert CRL to PEM
	if(!X509CrlToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}


	X509_REVOKED * r;
	REVOCATION_INFO rev_inf;
	int j;
	ASN1_OBJECT *obj;
	X509_EXTENSION * ex;
	unsigned char *p;
	X509V3_EXT_METHOD *method;	
	ASN1_ENUMERATED * rtmp;

	for(int i=0; i < sk_X509_REVOKED_num(crl->crl->revoked); i++)
	{
		r = sk_X509_REVOKED_value(crl->crl->revoked, i);
		if(!r) continue;

		rev_inf.serial = ASN1_INTEGER_get(r->serialNumber);
		if(rev_inf.serial == (unsigned long)-1)
			continue;
		rev_inf.rev_date = ASN1_TIME_to_timet((char*)r->revocationDate->data);
		rev_inf.reason = OCSP_REVOKED_STATUS_NOSTATUS;

		// Load extensions to get the revocation reason
		if(r->extensions)
		{
			for (j=0; j<sk_X509_EXTENSION_num(r->extensions); j++)
			{
				ex=sk_X509_EXTENSION_value(r->extensions, j);

				obj=X509_EXTENSION_get_object(ex);
				if(!obj || OBJ_obj2nid(obj) != NID_crl_reason)
					continue;
				if(!(method = X509V3_EXT_get(ex)))
					continue;
				p = ex->value->data;
				if(method->it)
					rtmp = (ASN1_ENUMERATED*)ASN1_item_d2i(NULL, &p, ex->value->length, ASN1_ITEM_ptr(method->it));
				else
					rtmp = (ASN1_ENUMERATED*)method->d2i(NULL, &p, ex->value->length);
				if(!rtmp)
					continue;
				rev_inf.reason = ASN1_ENUMERATED_get(rtmp);
				ASN1_ENUMERATED_free(rtmp);
				break;
			}
		}
		RevokedCerts.push_back(rev_inf);
	}

	if(!LoadIssuerDN())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!InternalExts.Load(crl->crl->extensions))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool PKI_CRL::SetCRL(const X509_CRL * Crl)
{
	CRYPTO_w_lock(CRYPTO_LOCK_X509_CRL);
	if(!Crl)
	{
		Clear();
		CRYPTO_w_unlock(CRYPTO_LOCK_X509_CRL);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	// We ignore the same certificate
	if(crl == Crl)
	{
		CRYPTO_w_unlock(CRYPTO_LOCK_X509_CRL);
		return true;
	}
	((X509_CRL *)Crl)->references++;
	CRYPTO_w_unlock(CRYPTO_LOCK_X509_CRL);
	
	
	Clear();

	crl = (X509_CRL *)Crl;

	return CommonLoad();
}



bool PKI_CRL::SetCRL(const char * CrlPem)
{
	Clear();

	if(! (crl = X509_CRL_new()) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Convert CRL from PEM
	if(!StringToX509Crl(CrlPem))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return CommonLoad();
}

void PKI_CRL::ClearPointers()
{
	if(crl) X509_CRL_free(crl);
	lePEM_CRL = "";
	Reset();
}

size_t PKI_CRL::RevokedCertsCount() const
{
	return RevokedCerts.size();
}

const REVOCATION_INFO * PKI_CRL::GetRevokedCert(size_t pos) const
{
	if(pos >= RevokedCerts.size())
		return NULL;
	return &RevokedCerts[pos];
}

const HashTable_String & PKI_CRL::GetExtensions() const
{
	return InternalExts.GetExts();
}

const char * PKI_CRL::GetSignatureAlg() const
{
	int nid;
	nid = OBJ_obj2nid(crl->crl->sig_alg->algorithm);
	if(nid == NID_undef) return NULL;
	return (char*)OBJ_nid2sn(nid);
}

const HashTable_Dn & PKI_CRL::GetIssuerDN() const
{
	return IssuerDN;
}

unsigned long PKI_CRL::GetVersion() const
{
	return X509_CRL_get_version(crl);
}

bool PKI_CRL::LoadIssuerDN()
{
	X509_NAME * name;
	name = crl->crl->issuer;
	if(!name)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!IssuerDN.From_X509_NAME(name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}


bool PKI_CRL::IsRevoked(const PKI_CERT &cert) const 
{
	size_t i;
	unsigned long serial;

	if(!cert)
		return true;
	
	serial = cert.GetSerial();

	for(i=0; i<RevokedCerts.size(); i++)
	{
		if(RevokedCerts[i].serial == serial)
		{
			return true;
		}
	}
	return false;
}

bool PKI_CRL::operator=(const PKI_CRL & other)
{
	// Trying to copy myself on me
	if(&other.crl == &crl)
		return true;

	Clear();
	if(!other.crl)
		return false;

	CRYPTO_add(&other.crl->references, 1, CRYPTO_LOCK_X509);
	crl = other.crl;

	return CommonLoad();
}

PKI_CRL::operator int() const
{
	if(!crl)
		return 0;
	else
		return 1;
}

bool PKI_CRL::load_Datas(const X509_CRL *Crl)
{
	return SetCRL(Crl);
}

bool PKI_CRL::give_Datas(X509_CRL **Crl) const
{
	if(*Crl)
	{
		X509_CRL_free(*Crl);
	}
	if(!crl)
	{
		*Crl = NULL;
	}
	else
	{
		*Crl = GetX509_CRL(true);
		if(!*Crl)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
		}
	}
	return true;
}

bool PKI_CRL::LoadFromFile(const mString &File)
{
	FILE * fp;
	X509_CRL * x509crl;

	fp = fopen(File.c_str(), "rb");
	if(!fp)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		ERR_add_error_data(2, File.c_str(), strerror(errno));
		return false;
	}
	if( !(x509crl = d2i_X509_CRL_fp(fp, NULL)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ERR_add_error_data(1, File.c_str());
		fclose(fp);
		return false;
	}
	fclose(fp);
	if(!SetCRL(x509crl))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ERR_add_error_data(1, File.c_str());
		X509_CRL_free(x509crl);
		return false;
	}
	X509_CRL_free(x509crl);
	return true;
}
