/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CSR.h: interface for the PKI_CSR class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_CSR_H
#define PKI_CSR_H


#include <stdio.h>
#include <string.h>
#include <Objects.h>
#include <mString.h>



#include <PKI_ERR.h>
#include <PKI_EXT.h>
#include <HashTable/HashTable_Dn.h>
#include <PEM_DER.h>

#include "PKI_RSA.h"

#include <openssl/x509.h>

/*!
  This class represents a CSR (Certificate Signing Request)
*/
class PKI_CSR  
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_CSR();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other csr to copy from.
	 */
	PKI_CSR(const PKI_CSR & other);

	/*! \brief This is the constructor (it will generate the CSR).
	 *  \param RequestDN [IN] The DN of th e PKI_CSR.
	 *  \param mRsaKey [IN] A RSA bikey, il will be integrated to the CSR and sign it.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_CSR(const HashTable_Dn & RequestDN, const PKI_RSA & mRsaKey);

	/*! \brief This is the constructor (it will parse the CSR).
	 *  \param PEM_CSR [IN] A pointer to a PEM encoded CSR.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PKI_CSR(const char * PEM_CSR);
	
	/*! \brief This is the destructor.
	 */
	virtual ~PKI_CSR();

	/*! \brief This function sets the CSR in PEM format.
	 *  \param PEM_CSR [IN] A pointer to a PEM encoded CSR.
	 *  \return true on success, false on failure.
	 */
	bool SetCSR(const char * PEM_CSR);

	/*! \brief This function sets the CSR.
	 *  \param x509csr [IN] A pointer to the CSR.
	 *  \return true on success, false on failure.
	 */
	bool SetCSR(const X509_REQ * x509csr);

	/*! \brief This function sets the CSR.
	 *  \param x509csr [IN] A pointer to the CSR.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const X509_REQ * x509csr);

	/*! \brief This function copies the internal csr.
	 *  \param Csr [OUT] The csr.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(X509_REQ ** Csr) const;

	/*! \brief This function generates the CSR.
	 *  \param RequestDN [IN] The DN of the PKI_CSR.
	 *  \param mRsaKey [IN] A RSA bikey, il will be integrated to the PKI_CSR and sign it.
	 *  \return true on success, false on failure.
	 */
	bool GenerateCSR(const HashTable_Dn & RequestDN, const PKI_RSA & mRsaKey);
	
	/*! \brief This function validates the signature of a PKI_CSR.
	 *  \return true when signature is valid, false when signature is not valid.
	 */
	bool CheckSignature() const;

	/*! \brief This function returns the CSR in the OpenSSL X509_REQ format.
	 *  \param duplicate [IN] Should the X509_REQ be duplicated, then freed by calling function.
	 *  \return The CSR in the OpenSSL X509_REQ format.
	 */
	X509_REQ * GetX509_REQ(bool duplicate=false) const;

	/*! \brief This function return the Dn of the CSR.
	 *  \return The Dn of the CSR.
	 */
	const HashTable_Dn & GetRequestDN() const;

	/*! \brief This function return the RSA bi-key of the CSR.
	 *  \return The RSA bi-key of the CSR.
	 */
	const PKI_RSA & GetRsaKey() const;

	/*! \brief This function returns the CSR in a PEM format.
	 *  \return The CSR in a PEM format.
	 */
	const mString & GetPemCSR() const;
	
	/*! \brief This function returns the extensions of the certificate.
	 *  \return The extensions of the certificate.
	 */
	const HashTable_String & GetExtensions() const;

	/*! \brief This function returns the version of the CSR.
	 *  \return The version of the CSR.
	 */
	unsigned long GetVersion() const;
	
	/*! \brief This function returns the signature algorithm of the CSR.
	 *  \return The signature algorithm of the CSR.
	 */
	const char * GetSignatureAlg() const;

	/*! \brief This function returns the length of the key of the CSR.
	 *  \return The length of the key of the CSR.
	 */
	unsigned long GetKeyLen() const;

	/*! \brief This function clears the internal datas.
	 */
	void Clear();

	/*! \brief This function returns the public key in the OpenSSL EVP_PKEY format.
	 *  \return The public key in the OpenSSL EVP_PKEY format.
	 */
	const EVP_PKEY * GetPublicKey() const;
	
	/*! \brief This function returns the public key of the certificate.
	 *  \return The public key of the certificate.
	 */
	const X509_PUBKEY * GetX509_PUBKEY() const;


	/*! \brief This operator copies one PKI_CSR into another.
	 *  \param other [IN] The other csr to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_CSR &other );

	/*! \brief This operator is used to know if the class is set.
	 *  \return return 1 when key is operable and 0 when not.
	 */
	operator int() const;

	/*! \brief This member is an empty instance of this class.
	 */
	static PKI_CSR EmptyInstance;
private:
	bool LoadAll();
	void Reset();
	bool LoadDN();
	bool StringToX509Req(const char * pem);
	bool X509ReqToString();
	bool MakeRequest (const HashTable_Dn & RequestDN);
	mString m_pemCsr;
	HashTable_Dn m_dnCsr;
	PKI_EXT m_extsCsr;
	X509_REQ * m_csr;
	PKI_RSA m_keyCsr;
	EVP_PKEY * m_pubKeyCsr;
};

#endif //PKI_CSR_H

