/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



#include "newpki_threads.h"
#include <string.h>
#include <stdio.h>
#include <openssl/err.h>
#include <time.h>


map< NewpkiThread*, bool > NewpkiThread::m_threads;
CriticalSection NewpkiThread::m_threadsLock;
bool NewpkiThread::m_isSignaledStarted = false;
bool NewpkiThread::m_isSignaledStopped = false;

NewpkiThread::NewpkiThread()
{
	m_ShouldStop = false;
	m_IsRunning = false;
	m_Param = NULL;
	m_thread_id = NEWPKI_THREAD_NULL;
	m_StartRoutine = NULL;
}

NewpkiThread::~NewpkiThread()
{
	Stop();
}


void NewpkiThread::Create(THREAD_START_ROUTINE * StartRoutine, void * Param)
{
	m_Param=Param;
	m_StartRoutine = StartRoutine;
}


bool NewpkiThread::DoStart()
{
	m_ShouldStop = false;
	#ifdef _WIN32
		m_thread_id = ::_beginthread(tmp_ThreadProc, 0, (void *)this);
		if(m_thread_id == -1)
		{
			m_thread_id = NEWPKI_THREAD_NULL;
			return false;
		}
	#else
		pthread_attr_t attr;

		::pthread_attr_init(&attr);
		::pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
		
		if(::pthread_create(&m_thread_id, &attr, &tmp_ThreadProc, (void *)this))
		{
			m_thread_id = NEWPKI_THREAD_NULL;
			::pthread_attr_destroy(&attr);
			return false;
		}
		::pthread_attr_destroy(&attr);
	#endif
	m_IsRunning = true;

	return true;
}


bool NewpkiThread::Start()
{
	m_threadsLock.EnterCS();
	if(m_isSignaledStopped)
	{
		m_threadsLock.LeaveCS();
		return false;
	}
	if(m_isSignaledStarted)
	{
		Stop();
		if(!DoStart())
		{
			m_threadsLock.LeaveCS();
			return false;
		}
	}
	m_threads[this] = true;
	m_threadsLock.LeaveCS();

	return true;
}

void NewpkiThread::Stop()
{
	std::map< NewpkiThread*, bool >::iterator i;

	m_ShouldStop = true;

	m_threadsLock.EnterCS();
	i = m_threads.find(this);
	if(i != m_threads.end())
	{
		m_threads.erase(i);
	}
	m_threadsLock.LeaveCS();

	// Remove it from the queue
	if(m_thread_id != NEWPKI_THREAD_NULL)
	{
		#ifdef _WIN32
			WaitForSingleObject((HANDLE)m_thread_id, INFINITE);
		#else
			pthread_join(m_thread_id, NULL);
		#endif

		m_thread_id = NEWPKI_THREAD_NULL;
		m_IsRunning = false;
	}
}

void NewpkiThread::SignalStart()
{
	std::map< NewpkiThread*, bool >::iterator i;

	// Start all waiting threads
	m_threadsLock.EnterCS();
	for(i = m_threads.begin(); i != m_threads.end(); i++)
	{
		i->first->DoStart();
		Sleep(1000);
	}
	m_isSignaledStarted = true;
	m_isSignaledStopped = false;
	m_threadsLock.LeaveCS();
}

void NewpkiThread::SignalStop()
{
	std::map< NewpkiThread*, bool >::iterator i;

	// Signal to all threads that they should stop
	m_threadsLock.EnterCS();
	for(i = m_threads.begin(); i != m_threads.end(); i++)
	{
		i->first->m_ShouldStop = true;
	}
	m_isSignaledStarted = false;
	m_isSignaledStopped = true;
	m_threads.clear();
	m_threadsLock.LeaveCS();
}


void NewpkiThread::Sleep(time_t ms_sleep)
{
	#ifdef _WIN32
		::Sleep(ms_sleep);
	#else
		::usleep( (time_t) ms_sleep * 1000);
	#endif
}

NEWPKI_THREAD_ID NewpkiThread::CurrentThreadId()
{
	#ifdef _WIN32
		return ::GetCurrentThreadId();
	#else
		return ::pthread_self();
	#endif
}

NEWPKI_THREAD_ID NewpkiThread::GetThreadId() const
{
	return m_thread_id;
}

#ifdef _WIN32
void NewpkiThread::tmp_ThreadProc(void * Param)
#else
void  * NewpkiThread::tmp_ThreadProc(void * Param)
#endif
{
	NewpkiThread * me_this = (NewpkiThread *)Param;
	if(me_this)
	{
		me_this->m_StartRoutine(me_this, me_this->m_Param);
	}
	me_this->m_IsRunning = false;


	::ERR_clear_error();
	::ERR_remove_state(0);
	#ifdef _WIN32
		::_endthread();
		return;
	#else
		::pthread_exit(NULL);
		return NULL;
	#endif
}

bool NewpkiThread::IsRunning() const
{
	return (m_thread_id != NEWPKI_THREAD_NULL && m_IsRunning);
}

bool NewpkiThread::ShouldStop() const
{
	return m_ShouldStop;
}

bool NewpkiThread::SleepInterrupt(time_t s_sleep) const
{
	time_t lastTime;
	time_t currTime;

	time(&lastTime);
	do
	{
		NewpkiThread::Sleep(500);
		time(&currTime);
		if(ShouldStop())
			return false;
	}
	while((currTime - lastTime) < s_sleep);

	return true;
}
