# -*- coding: utf-8 -*-

# Copyright © 2012-2014 Roberto Alsina and others.

# Permission is hereby granted, free of charge, to any
# person obtaining a copy of this software and associated
# documentation files (the "Software"), to deal in the
# Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice
# shall be included in all copies or substantial portions of
# the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
# OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

from __future__ import print_function, absolute_import, unicode_literals
import codecs
import datetime
import os
try:
    from urlparse import urljoin, urlparse
    import robotparser as robotparser
except ImportError:
    from urllib.parse import urljoin, urlparse  # NOQA
    import urllib.robotparser as robotparser  # NOQA

from nikola.plugin_categories import LateTask
from nikola.utils import config_changed


urlset_header = """<?xml version="1.0" encoding="UTF-8"?>
<urlset
    xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9
                        http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd">
"""

loc_format = """ <url>
  <loc>{0}</loc>
  <lastmod>{1}</lastmod>
 </url>
"""

urlset_footer = "</urlset>"

sitemapindex_header = """<?xml version="1.0" encoding="UTF-8"?>
<sitemapindex
    xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9
                        http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd">
"""

sitemap_format = """ <sitemap>
  <loc>{0}</loc>
  <lastmod>{1}</lastmod>
 </sitemap>
"""

sitemapindex_footer = "</sitemapindex>"


def get_base_path(base):
    """returns the path of a base URL if it contains one.

    >>> get_base_path('http://some.site') == '/'
    True
    >>> get_base_path('http://some.site/') == '/'
    True
    >>> get_base_path('http://some.site/some/sub-path') == '/some/sub-path/'
    True
    >>> get_base_path('http://some.site/some/sub-path/') == '/some/sub-path/'
    True
    """
    # first parse the base_url for some path
    base_parsed = urlparse(base)

    if not base_parsed.path:
        sub_path = ''
    else:
        sub_path = base_parsed.path
    if sub_path.endswith('/'):
        return sub_path
    else:
        return sub_path + '/'


class Sitemap(LateTask):
    """Generate a sitemap."""

    name = "sitemap"

    def gen_tasks(self):
        """Generate a sitemap."""
        kw = {
            "base_url": self.site.config["BASE_URL"],
            "site_url": self.site.config["SITE_URL"],
            "output_folder": self.site.config["OUTPUT_FOLDER"],
            "strip_indexes": self.site.config["STRIP_INDEXES"],
            "index_file": self.site.config["INDEX_FILE"],
            "sitemap_include_fileless_dirs": self.site.config["SITEMAP_INCLUDE_FILELESS_DIRS"],
            "mapped_extensions": self.site.config.get('MAPPED_EXTENSIONS', ['.html', '.htm', '.xml', '.rss']),
            "robots_exclusions": self.site.config["ROBOTS_EXCLUSIONS"]
        }

        output = kw['output_folder']
        base_url = kw['base_url']
        mapped_exts = kw['mapped_extensions']

        output_path = kw['output_folder']
        sitemapindex_path = os.path.join(output_path, "sitemapindex.xml")
        sitemap_path = os.path.join(output_path, "sitemap.xml")
        base_path = get_base_path(kw['base_url'])
        sitemapindex = {}
        urlset = {}

        def scan_locs():
            for root, dirs, files in os.walk(output, followlinks=True):
                if not dirs and not files and not kw['sitemap_include_fileless_dirs']:
                    continue  # Totally empty, not on sitemap
                path = os.path.relpath(root, output)
                # ignore the current directory.
                path = (path.replace(os.sep, '/') + '/').replace('./', '')
                lastmod = self.get_lastmod(root)
                loc = urljoin(base_url, base_path + path)
                if kw['index_file'] in files and kw['strip_indexes']:  # ignore folders when not stripping urls
                    urlset[loc] = loc_format.format(loc, lastmod)
                for fname in files:
                    if kw['strip_indexes'] and fname == kw['index_file']:
                        continue  # We already mapped the folder
                    if os.path.splitext(fname)[-1] in mapped_exts:
                        real_path = os.path.join(root, fname)
                        path = os.path.relpath(real_path, output)
                        if path.endswith(kw['index_file']) and kw['strip_indexes']:
                            # ignore index files when stripping urls
                            continue
                        if not robot_fetch(path):
                            continue
                        if path.endswith('.html') or path.endswith('.htm'):
                            try:
                                if u'<!doctype html' not in codecs.open(real_path, 'r', 'utf8').read(1024).lower():
                                    # ignores "html" files without doctype
                                    # alexa-verify, google-site-verification, etc.
                                    continue
                            except UnicodeDecodeError:
                                # ignore ancient files
                                # most non-utf8 files are worthless anyways
                                continue
                        """ put RSS in sitemapindex[] instead of in urlset[], sitemap_path is included after it is generated """
                        if path.endswith('.xml') or path.endswith('.rss'):
                            if u'<rss' in codecs.open(real_path, 'r', 'utf8').read(512) or u'<urlset'and path != sitemap_path:
                                path = path.replace(os.sep, '/')
                                lastmod = self.get_lastmod(real_path)
                                loc = urljoin(base_url, base_path + path)
                                sitemapindex[loc] = sitemap_format.format(loc, lastmod)
                                continue
                            else:
                                continue  # ignores all XML files except those presumed to be RSS
                        post = self.site.post_per_file.get(path)
                        if post and (post.is_draft or post.is_private or post.publish_later):
                            continue
                        path = path.replace(os.sep, '/')
                        lastmod = self.get_lastmod(real_path)
                        loc = urljoin(base_url, base_path + path)
                        urlset[loc] = loc_format.format(loc, lastmod)

        def robot_fetch(path):
            for rule in kw["robots_exclusions"]:
                robot = robotparser.RobotFileParser()
                robot.parse(["User-Agent: *", "Disallow: {0}".format(rule)])
                if not robot.can_fetch("*", '/' + path):
                    return False  # not robot food
            return True

        def write_sitemap():
            # Have to rescan, because files may have been added between
            # task dep scanning and task execution
            with codecs.open(sitemap_path, 'wb+', 'utf8') as outf:
                outf.write(urlset_header)
                for k in sorted(urlset.keys()):
                    outf.write(urlset[k])
                outf.write(urlset_footer)
            sitemap_url = urljoin(base_url, base_path + "sitemap.xml")
            sitemapindex[sitemap_url] = sitemap_format.format(sitemap_url, self.get_lastmod(sitemap_path))

        def write_sitemapindex():
            with codecs.open(sitemapindex_path, 'wb+', 'utf8') as outf:
                outf.write(sitemapindex_header)
                for k in sorted(sitemapindex.keys()):
                    outf.write(sitemapindex[k])
                outf.write(sitemapindex_footer)

        # Yield a task to calculate the dependencies of the sitemap
        # Other tasks can depend on this output, instead of having
        # to scan locations.
        def scan_locs_task():
            scan_locs()
            return {'locations': list(urlset.keys()) + list(sitemapindex.keys())}

        yield {
            "basename": "_scan_locs",
            "name": "sitemap",
            "actions": [(scan_locs_task)]
        }

        yield self.group_task()
        yield {
            "basename": "sitemap",
            "name": sitemap_path,
            "targets": [sitemap_path],
            "actions": [(write_sitemap,)],
            "uptodate": [config_changed(kw)],
            "clean": True,
            "task_dep": ["render_site"],
            "calc_dep": ["_scan_locs:sitemap"],
        }
        yield {
            "basename": "sitemap",
            "name": sitemapindex_path,
            "targets": [sitemapindex_path],
            "actions": [(write_sitemapindex,)],
            "uptodate": [config_changed(kw)],
            "clean": True,
            "file_dep": [sitemap_path]
        }

    def get_lastmod(self, p):
        if self.site.invariant:
            return '2014-01-01'
        else:
            return datetime.datetime.fromtimestamp(os.stat(p).st_mtime).isoformat().split('T')[0]

if __name__ == '__main__':
    import doctest
    doctest.testmod()
