# -*- coding: ascii -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2007 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re
import urlparse
import random
import gc

if 'DISPLAY' in os.environ:
    import gtk
    import cairo

import ninix.seriko
import ninix.menu

import pix

range_scale = [(' 100%',  100),
               ('  90%',   90),
               ('  80%',   80),
               ('  70%',   70),
               ('  60%',   60),
               ('  50%',   50),
               ('  40%',   40),
               (' 200%',  200),
               (' 300%',  300),
               ('1000%', 1000)]


class Surface:

    # keyval/name mapping
    if 'DISPLAY' in os.environ:
        from ninix.keymap import keymap_old, keymap_new
    else:
        keymap_old = {}
        keymap_new = {}

    def __init__(self, sakura, debug=0):
        self.sakura = sakura
        self.debug = debug
        self.window = []
        self.__mikire = 0
        self.__kasanari = 0
        self.__menu = None
        self.__scale = 100 # %
        self.desc = None
        self.__use_pna = 0
        self.__alpha_channel = None

    def set_debug(self, debug):
        self.debug = debug
        for surface_window in self.window:
            surface_window.set_debug(debug)
            surface_window.reset_surface()

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        for surface_window in self.window:
            surface_window.set_use_pna(flag)
            surface_window.reset_surface()

    def set_alpha_channel(self, alpha):
        if not 0.1 <= alpha <= 1.0 or alpha is None:
            alpha = 1.0
        self.__alpha_channel = alpha
        for surface_window in self.window:
            surface_window.set_alpha_channel(self.__alpha_channel)

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        ##if self.__scale == scale:
        ##    return
        self.__scale = scale # %
        for surface_window in self.window:
            surface_window.set_scale(scale)
            surface_window.reset_surface()

    def set_animation_quality(self, quality):
        for surface_window in self.window:
            surface_window.set_animation_quality(quality)

    def finalize(self):
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        ##self.__menu.finalize()
        self.__menu = None

    def create_gtk_window(self, title, skip_taskbar):
        window = gtk.Window()
        pix.set_rgba_colormap(window)
        window.set_title(title)
        window.set_decorated(False)
        window.set_resizable(False)
        if skip_taskbar:
            window.set_skip_taskbar_hint(True)
        window.connect('delete_event', self.delete)
        window.connect('key_press_event', self.key_press)
        window.connect('window_state_event', self.window_state)
        window.set_events(gtk.gdk.KEY_PRESS_MASK)
        window.realize()
        return window

    def window_stayontop(self, flag):
        for surface_window in self.window:
            gtk_window = surface_window.window
            gtk_window.set_keep_above(flag)
                
    def window_iconify(self, flag):
        gtk_window = self.window[0].window
        if flag:
            if not gtk_window.window.get_state() & \
               gtk.gdk.WINDOW_STATE_ICONIFIED:
                gtk_window.iconify()
        else:
            if gtk_window.window.get_state() & gtk.gdk.WINDOW_STATE_ICONIFIED:
                gtk_window.deiconify()

    def window_state(self, window, event):
        if not self.sakura.running: ## FIXME
            return
        if event.new_window_state & gtk.gdk.WINDOW_STATE_ICONIFIED:
            if window == self.window[0].window:
                self.sakura.ghost.notify_iconified() ## FIXME
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   not gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.iconify()
        else:
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and \
                   gtk_window.window.get_state() & \
                   gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.deiconify()
            if window == self.window[0].window:
                self.sakura.ghost.notify_deiconified() ## FIXME
        return

    def delete(self, window, event):
        self.sakura.ghost.quit() ## FIXME
        return False

    def key_press(self, window, event):
        name = self.keymap_old.get(event.keyval, event.string)
        keycode = self.keymap_new.get(event.keyval, event.string)
        if name or keycode:
            self.sakura.notify_event('OnKeyPress', name, keycode)
        return True

    def window_stick(self, action):
        stick = self.__menu.get_stick()
        for window in self.window:
            if stick:
                window.window.stick()
            else:
                window.window.unstick()

    def open_popup_menu(self, button, side):
        self.__menu.popup(button, side)

    re_surface_id = re.compile('^surface([0-9]+)$')

    def get_seriko(self, surface):
        seriko = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            seriko[key] = ninix.seriko.get_actors(config)
        return seriko

    def new(self, desc, alias, surface, name, prefix):
        self.desc = desc
        self.name = name
        self.prefix = prefix
        if alias is None:
            alias0 = alias1 = None
        else:
            alias0 = alias.get('sakura.surface.alias')
            alias1 = alias.get('kero.surface.alias')
        # load pixmap
        pixbufs = {}
        elements = {}
        for basename, (path, config) in surface.iteritems():
            if path is None:
                continue
            if not os.path.exists(path):
                name, suffix = os.path.splitext(path)
                dgp_path = ''.join((name, '.dgp'))
                if not os.path.exists(dgp_path):
                    ddp_path = ''.join((name, '.ddp'))
                    if not os.path.exists(ddp_path):
                        print '%s: file not found (ignored)' % path
                        continue
                    else:
                        path = ddp_path
                else:
                    path = dgp_path
            elements[basename] = [path]
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            pixbufs[key] = elements[basename]
        # compose surface elements
        composite_pixbuf = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            if 'element0' in config:
                if self.debug & 8192:
                    print 'surface', key
                composite_pixbuf[key] = self.compose_elements(elements, config)
        pixbufs.update(composite_pixbuf)
        # check if necessary surfaces have been loaded
        for key in ['0', '10']:
            if key not in pixbufs:
                raise SystemExit, 'cannot load surface #%s (abort)\n' % key
        self.__pixbufs = pixbufs
        # arrange surface configurations
        region = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define collision areas
            buf = []
            for n in range(256):
                # "redo" syntax
                rect = config.get(''.join(('collision', str(n))))
                if rect is None:
                    continue
                values = rect.split(',')
                if len(values) != 5:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in values[:4]]
                except ValueError:
                    continue
                buf.append((values[4].strip(), x1, y1, x2, y2))
            for part in ['head', 'face', 'bust']:
                # "inverse" syntax
                rect = config.get(''.join(('collision.', part)))
                if not rect:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in rect.split(',')]
                except ValueError:
                    pass
                buf.append((part.capitalize(), x1, y1, x2, y2))
            region[key] = buf
        self.__region = region
        # MAYUNA
        mayuna = {}
        for basename, (path, config) in surface.iteritems():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            # define animation patterns
            mayuna[key] = ninix.seriko.get_mayuna(config)
        bind = {}
        for side in ['sakura', 'kero']:
            bind[side] = {}
            for index in range(128):
                name = self.desc.get(
                    '%s.bindgroup%d.name' % (side, index), None)
                default = self.desc.get(
                    '%s.bindgroup%d.default' % (side, index), 0)
                if name is not None:
                    bind[side][index] = [name, default]
        self.mayuna = {}
        for side in ['sakura', 'kero']:
            self.mayuna[side] = []
            for index in range(128):
                key = self.desc.get('%s.menuitem%d' % (side, index), None)
                if key == '-':
                    self.mayuna[side].append([key, None, 0])
                else:
                    try:
                        key = int(key)
                    except:
                        pass
                    else:
                        if key in bind[side]:
                            name = bind[side][key][0].split(',')
                            self.mayuna[side].append([key, name[1],
                                                      bind[side][key][1]])
        # create surface windows
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        for name, side, default, alias in [('sakura', 0, '0', alias0),
                                           ('kero', 1, '10', alias1)]:
            if name == 'sakura':
                title = self.sakura.get_selfname() or \
                        ''.join(('surface.', name))
            else:
                title = self.sakura.get_keroname() or \
                        ''.join(('surface.', name))
            if side >= 1:
                skip_taskbar = 1
            else:
                skip_taskbar = 0
            gtk_window = self.create_gtk_window(title, skip_taskbar)
            seriko = self.get_seriko(surface)
            surface_window = SurfaceWindow(
                gtk_window, side, self.sakura, desc, alias, surface,
                pixbufs, seriko, region, mayuna, bind[name],
                default, self.__use_pna, self.__alpha_channel, self.debug)
            self.window.append(surface_window)
        self.__surface = surface
        self.__menu = ninix.menu.Menu(self.sakura.ghost, self) ## FIXME
        top_dir = self.prefix
        name = self.desc.get('menu.background.bitmap.filename',
                             'menu_background.png')
        name = name.replace('\\', '/')
        path = os.path.join(top_dir, name)
        if not os.path.exists(path):
            top_dir = os.path.join(self.sakura.ghost.get_prefix(),
                                   'ghost', 'master') ## FIXME
            path = os.path.join(top_dir, 'menu_background.png')
            if os.path.exists(path):
                path_background = path
            else:
                path_background = None
            path = os.path.join(top_dir, 'menu_sidebar.png')
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
            path = os.path.join(top_dir, 'menu_foreground.png')
            if os.path.exists(path):
                path_foreground = path
            else:
                path_foreground = None
        else:
            path_background = path
            name = self.desc.get('menu.sidebar.bitmap.filename',
                                 'menu_sidebar.png')
            name = name.replace('\\', '/')
            path = os.path.join(top_dir, name)
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
            name = self.desc.get('menu.foreground.bitmap.filename',
                                 'menu_foreground.png')
            name = name.replace('\\', '/')
            path = os.path.join(top_dir, name)
            if os.path.exists(path):
                path_foreground = path
            else:
                path_foreground = None
        if path_background:
            self.__menu.set_pixmap(
                path_background, path_sidebar, path_foreground)
        fontcolor_r = self.desc.getint('menu.background.font.color.r', 0)
        fontcolor_g = self.desc.getint('menu.background.font.color.g', 0)
        fontcolor_b = self.desc.getint('menu.background.font.color.b', 0)
        background = (fontcolor_r, fontcolor_g, fontcolor_b)
        fontcolor_r = self.desc.getint('menu.foreground.font.color.r', 0)
        fontcolor_g = self.desc.getint('menu.foreground.font.color.g', 0)
        fontcolor_b = self.desc.getint('menu.foreground.font.color.b', 0)
        foreground = (fontcolor_r, fontcolor_g, fontcolor_b)
        self.__menu.set_fontcolor(background, foreground)
        self.__menu.create_mayuna_menu(self.get_mayuna_menu())

    def add_window(self, side, default):
        assert side >= 2 and len(self.window) == side ## FIXME
        name = 'char%d' % side
        title = ''.join(('surface.', name))
        gtk_window = self.create_gtk_window(title, 1)
        seriko = self.get_seriko(self.__surface)
        surface_window = SurfaceWindow(
            gtk_window, side, self.sakura, self.desc, None, self.__surface,
            self.__pixbufs, seriko, self.__region, {}, {},
            default, self.__use_pna, self.__alpha_channel, self.debug)
        self.window.append(surface_window)
        surface_window.set_scale(self.__scale)
        ##surface_window.set_surface(default)

    def get_mayuna_menu(self):
        for side, index in [('sakura', 0), ('kero', 1)]:
            for menu in self.mayuna[side]:
                if menu[0] != '-':
                    menu[2] = self.window[index].bind[menu[0]][1]
        return self.mayuna

    def compose_elements(self, elements, config):
        error = None
        for n in range(256):
            key = ''.join(('element', str(n)))
            if key not in config:
                break
            spec = [value.strip() for value in config[key].split(',')]
            try:
                method, filename, x, y = spec
                x = int(x)
                y = int(y)
            except ValueError:
                error = 'invalid element spec for %s: %s' % (key, config[key])
                break
            basename, suffix = os.path.splitext(filename)
            if suffix.lower() not in ['.png', '.dgp', '.ddp']:
                error = 'unsupported file format for %s: %s' % (key, filename)
                break
            basename = basename.lower()
            if basename not in elements:
                error = '%s file not found: %s' % (key, filename)
                break
            pixbuf = elements[basename][0]
            if n == 0: # base surface
                pixbuf_list = [pixbuf]
            elif method == 'overlay':
                pixbuf_list.append((pixbuf, x, y))
            elif method == 'base':
                pixbuf_list.append((pixbuf, x, y))
            else:
                error = 'unknown method for %s: %s' % (key, method)
                break
            if self.debug & 8192:
                w = pixbuf.get_width()
                h = pixbuf.get_height()
                print '%s: %s %s, x=%d, y=%d, w=%d, h=%d' % (
                    key, method, filename, x, y, w, h)
        if error is not None:
            print error
            pixbuf_list = None
            gc.collect()
            pixbuf_list = []
        return pixbuf_list

    def get_window(self, side):
        if len(self.window) > side:
            return self.window[side].window # FIXME
        else: 
            return None

    def reset_surface(self, side):
        if len(self.window) > side:
            self.window[side].reset_surface()

    def set_surface_default(self, side):
        if side is None:
            for side in range(len(self.window)):
                self.window[side].set_surface_default()
        elif 0 <= side < len(self.window):
            self.window[side].set_surface_default()

    def set_surface(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].set_surface(surface_id)

    def get_surface(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface()
        else:
            return 0

    def get_surface_size(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_size()
        else:
            return 0, 0

    def get_surface_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_offset()
        else:
            return 0, 0

    def get_touched_region(self, side, x, y):
        if len(self.window) > side:
            return self.window[side].get_touched_region(x, y)
        else:
            return ''

    def get_direction(self, side):
        if len(self.window) > side:
            return self.window[side].get_direction()
        else:
            return 0

    def set_direction(self, side, direction):
        if len(self.window) > side:
            self.window[side].set_direction(direction)

    def reset_balloon_position(self):
        top_margin = self.sakura.ghost.get_top_margin() ## FIXME
        bottom_margin = self.sakura.ghost.get_bottom_margin() ## FIMXE
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height() - bottom_margin
        for side in range(len(self.window)):
            x, y = self.get_position(side)
            sox, soy = self.window[side].get_surface_offset()
            w, h = self.get_surface_size(side)
            bw, bh = self.sakura.ghost.get_balloon_size(side) ## FIXME
            ox, oy = self.get_balloon_offset(side)
            direction = self.get_direction(side)
            align = self.get_alignment(side)
            if direction == 0:
                bx = max(x + sox - bw + ox, 0)
            else:
                bx =  min(x + sox + w - ox, scrn_w - bw)
            if align == 0:
                by = min(y + soy + oy, scrn_h - bh)
            elif align == 1:
                by = max(y + soy + oy, 0 + top_margin)
            else:
                if y + soy < scrn_h / 2:
                    by = max(y + soy + oy, 0 + top_margin)
                else:
                    by = min(y + soy + oy, scrn_h - bh)
            self.set_position(side, x, y)
            self.set_direction(side, direction)
            self.sakura.ghost.set_balloon_position(side, bx, by) ## FIXME

    def reset_position(self):
        top_margin = self.sakura.ghost.get_top_margin() ## FIXME
        bottom_margin = self.sakura.ghost.get_bottom_margin() ## FIMXE
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height() - bottom_margin
        for side in range(len(self.window)):
            align = self.get_alignment(side)
            if side == 0: # sakura
                w, h = self.get_surface_size(side)
                x = scrn_w - w
                if align == 1: # top
                    y = 0 + top_margin
                else:
                    y = scrn_h - h
                bw, bh = self.sakura.ghost.get_balloon_size(side) ## FIXME
                ox, oy = self.get_balloon_offset(side)
                direction = 0 # left
                bx = max(x - bw + ox, 0)
            else:
                b0w, b0h = self.sakura.ghost.get_balloon_size(side - 1) ## FIMXE
                b1w, b1h = self.sakura.ghost.get_balloon_size(side) ## FIMXE
                o0x, o0y = self.get_balloon_offset(side - 1)
                o1x, o1y = self.get_balloon_offset(side)
                w, h = self.get_surface_size(side)
                offset = max(0, b1w - (b0w - o0x))
                if (s0x + o0x - b0w) - offset - w + o1x < 0:
                    x = 0
                else:
                    x = (s0x + o0x - b0w) - offset - w + o1x
                if align == 1: # top
                    y = 0 + top_margin
                else:
                    y = scrn_h - h
                direction = 1 # right
                bx =  min(x + w - o1x, scrn_w - b1w)
            if align == 0:
                by = min(y + oy, scrn_h - bh)
            elif align == 1:
                by = max(y + oy, 0 + top_margin)
            else:
                if y < scrn_h / 2:
                    by = max(y + oy, 0 + top_margin)
                else:
                    by = min(y + oy, scrn_h - bh)
            self.set_position(side, x, y)
            self.set_direction(side, direction)
            self.sakura.ghost.set_balloon_position(side, bx, by) ## FIXME
            s0x, s0y, s0w, s0h = x, y, w, h

    def set_position(self, side, x, y):
        if len(self.window) > side:
            self.window[side].set_position(x, y)

    def get_position(self, side):
        if len(self.window) > side:
            return self.window[side].get_position()
        else:
            return 0, 0

    def set_alignment_current(self):
        for side in range(len(self.window)):
            self.window[side].set_alignment_current()

    def set_alignment(self, side, align):
        if len(self.window) > side:
            self.window[side].set_alignment(align)

    def get_alignment(self, side):
        if len(self.window) > side:
            return self.window[side].get_alignment()
        else:
            return 0

    def reset_alignment(self):
        if self.desc.get('seriko.alignmenttodesktop') == 'free':
            align = 2
        else:
            align = 0
        for side in range(len(self.window)):
            self.set_alignment(side, align)

    def is_shown(self, side):
        if len(self.window) > side:
            if self.window[side].is_shown():
                return 1
            else:
                return 0
        else:
            return 0

    def show(self, side):
        if len(self.window) > side:
            self.window[side].show()

    def hide_all(self):
        for side in range(len(self.window)):
            self.window[side].hide()

    def hide(self, side):
        if len(self.window) > side:
            self.window[side].hide()

    def raise_all(self):
        for side in range(len(self.window)):
            self.window[side].raise_()

    def raise_(self, side):
        if len(self.window) > side:
            self.window[side].raise_()

    def lower_all(self):
        for side in range(len(self.window)):
            self.window[side].lower()

    def lower(self, side):
        if len(self.window) > side:
            self.window[side].lower()

    def invoke(self, side, actor_id):
        if len(self.window) > side:
            self.window[side].invoke(actor_id)

    def invoke_yen_e(self, side, surface_id):
        if len(self.window) > side:
            self.window[side].invoke_yen_e(surface_id)

    def invoke_talk(self, side, surface_id, count):
        if len(self.window) > side:
            return self.window[side].invoke_talk(surface_id, count)
        else:
            return 0

    def set_icon(self, path):
        pixbuf = None
        if path is not None:
            try:
                pixbuf = pix.create_pixbuf_from_file(path, is_pnr=0)
            except:
                pixbuf = None
        for window in self.window:
            window.window.set_icon(pixbuf)
        del pixbuf
        gc.collect()

    def __check_mikire_kasanari(self):
        if not self.is_shown(0):
            self.__mikire = 0
            self.__kasanari = 0
            return
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height()
        x0, y0 = self.get_position(0)
        s0w, s0h = self.get_surface_size(0)
        if x0 + s0w / 3 < 0 or x0 + s0w * 2 / 3 > scrn_w or \
           y0 + s0h / 3 < 0 or y0 + s0h * 2 / 3 > scrn_h:
            self.__mikire = 1
        else:
            self.__mikire = 0
        if self.is_shown(1):
            x1, y1 = self.get_position(1)
            s1w, s1h = self.get_surface_size(1)
            if (x0 < x1 + s1w / 2 < x0 + s0w and y0 < y1 + s1h / 2 < y0 + s0h) or \
               (x1 < x0 + s0w / 2 < x1 + s1w and y1 < y0 + s0h / 2 < y1 + s1h):
                self.__kasanari = 1
            else:
                self.__kasanari = 0
        else:
            self.__kasanari = 0

    def get_mikire_kasanari(self):
        self.__check_mikire_kasanari()
        return self.__mikire, self.__kasanari

    def get_name(self):
        return self.name

    def get_username(self):
        if self.desc is None:
            return
        else:
            return self.desc.get('user.defaultname')

    def get_selfname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get('sakura.name')

    def get_selfname2(self):
        if self.desc is None:
            return
        else:
            return self.desc.get('sakura.name2')

    def get_keroname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get('kero.name')

    def get_friendname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get('sakura.friend.name')

    def get_balloon_offset(self, side):
        if side == 0:
            x, y = self.window[side].get_balloon_offset()
            if x is None:
                x = self.desc.getint('sakura.balloon.offsetx', 0)
            if y is None:
                y = self.desc.getint('sakura.balloon.offsety', 0)
        elif side == 1:
            x, y = self.window[side].get_balloon_offset()
            if x is None:
                x = self.desc.getint('kero.balloon.offsetx', 0)
            if y is None:
                y = self.desc.getint('kero.balloon.offsety', 0)
        else:
            x, y = None, None
            if len(self.window) > side:
                x, y = self.window[side].get_balloon_offset()
            if x is None:
                x = self.desc.getint('char%d.balloon.offsetx' % side, 0)
            if y is None:
                y = self.desc.getint('char%d.balloon.offsety' % side, 0)
        if self.__scale != 100:
            x = x * self.__scale / 100
            y = y * self.__scale / 100
        return x, y

    def toggle_bind(self, event, args):
        side, bind_id = args
        self.window[side].toggle_bind(bind_id)

    def get_collision_area(self, side, part):
        if len(self.window) > side:
            for p, x1, y1, x2, y2 in self.window[side].collisions:
                if p == part:
                    if self.__scale != 100:
                        x1 = x1 * self.__scale / 100
                        x2 = x2 * self.__scale / 100
                        y1 = y1 * self.__scale / 100
                        y2 = y2 * self.__scale / 100
                    return x1, y1, x2, y2
        return None

    def get_config_int(self, side, name):
        if len(self.window) > side:
            basename = ''.join(('surface', self.window[side].surface_id))
            path, config = self.window[side].surface[basename]
            return config.getint(name)
        else:
            return 0

class SurfaceWindow:

    # DnD data types
    dnd_targets = [
        ('text/plain', 0, 0),
        ]

    def __init__(self, window, side, sakura, desc, alias, surface,
                 pixbuf, seriko, region, mayuna, bind,
                 default_id, use_pna, alpha, debug):
        self.window = window
        self.side = side
        self.sakura = sakura
        self.desc = desc
        self.alias = alias
        self.align = 0
        self.__scale = 100 # %
        self.__use_pna = use_pna
        self.__alpha_channel = alpha
        self.__position_dirty = False
        if self.alias is not None:
            default_id = self.alias.get(default_id, [default_id])[0]
        self.surface = surface
        self.surface_id = default_id
        self.pixbuf = pixbuf
        self.seriko = ninix.seriko.Controler(seriko)
        self.region = region
        self.mayuna = mayuna
        self.bind = bind
        self.reset_pixbuf_cache()
        self.frame_buffer = []
        self.default_id = default_id
        self.debug = debug
        self.__shown = 0
        self.window_offset = (0, 0)
        self.set_position(0, 0)
        self.set_direction(0)
        self.window.connect('leave_notify_event', self.window_leave_notify) # XXX
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.BUTTON_RELEASE_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.POINTER_MOTION_HINT_MASK|
                              gtk.gdk.SCROLL_MASK)
        self.callbacks = []
        for signal, func in [('expose_event',         self.redraw),
                             ('button_press_event',   self.button_press),
                             ('button_release_event', self.button_release),
                             ('motion_notify_event',  self.motion_notify),
                             ('drag_data_received',   self.drag_data_received),
                             ('scroll_event',         self.scroll),
                             ]:
            self.callbacks.append(self.darea.connect(signal, func))
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.window.add(self.darea)
        self.darea.realize()
        self.darea.window.set_back_pixmap(None, False)
        self.set_surface(None)

    def set_debug(self, debug):
        self.debug = debug

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        self.reset_pixbuf_cache()

    def set_alpha_channel(self, value):
        if self.__alpha_channel == value:
            return
        self.__alpha_channel = value
        self.update()

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        ##if self.__scale == scale:
        ##    return
        self.__scale = scale # %

    def set_animation_quality(self, quality):
        self.seriko.set_animation_quality(quality, self)

    def drag_data_received(self, widget, context, x, y, data, info, time):
        if self.debug & 8192:
            print 'Content-type:', data.type
            print 'Content-length:', data.length
            print repr(data.data)
        if str(data.type) == 'text/plain':
            filelist = []
            for line in data.data.split('\r\n'):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                if scheme == 'file' and os.path.exists(path):
                    filelist.append(path)
            if filelist:
                self.sakura.notify_file_drop(filelist, self.side)
        return True

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def invoke_yen_e(self, surface_id):
        self.seriko.invoke_yen_e(self, surface_id)

    def invoke_talk(self, surface_id, count):
        return self.seriko.invoke_talk(self, surface_id, count)

    def reset_surface(self):
        surface_id = self.get_surface()
        self.set_surface(surface_id)

    def set_surface_default(self):
        self.set_surface(self.default_id)

    def set_surface(self, surface_id):
        if self.alias is not None and surface_id in self.alias:
            aliases = self.alias.get(surface_id)
            if aliases:
                surface_id = random.choice(aliases)
        if surface_id == '-2':
            self.seriko.terminate(self)
        if surface_id in ['-1', '-2']:
            pass
        elif surface_id not in self.pixbuf:
            self.surface_id = self.default_id
        else:
            self.surface_id = surface_id
        self.seriko.reset(self, surface_id)
        # define collision areas
        self.collisions = self.region[self.surface_id]
        # update window offset
        x, y = self.get_position()
        w, h = self.get_surface_size(self.surface_id)
        dw, dh = self.get_surface_size(self.default_id) # default surface size
        xoffset = (dw - w) / 2
        if self.get_alignment() == 0:
            yoffset = dh - h
            scrn_h = gtk.gdk.screen_height() - \
                     self.sakura.ghost.get_bottom_margin() ## FIXME
            y = scrn_h - dh
        elif self.get_alignment() == 1:
            yoffset = 0
        else:
            yoffset = (dh - h) / 2
        self.window_offset = (xoffset, yoffset)
        self.surface_pixbuf = self.get_pixbuf(self.surface_id)
        # resize window
        self.darea.set_size_request(w, h)
        self.update()
        self.seriko.start(self)
        # relocate window
        self.set_position(x, y)
        self.sakura.ghost.notify_surface_change(self.side) ## FIXME

    def iter_mayuna(self, surface_width, surface_height, mayuna, done):
        for surface, interval, method, args in mayuna.patterns:
            if method in ['bind', 'add']:
                if surface in self.pixbuf:
                    x, y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    # overlay surface pixbuf
                    if x + w > surface_width:
                        w = surface_width - x
                    if y + h > surface_height:
                        h = surface_height - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'reduce':
                if surface in self.pixbuf:
                    dest_x, dest_y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    x = y = 0 # XXX
                    yield method, pixbuf, dest_x, dest_y, w, h, x, y
            elif method == 'insert':
                index = args[0]
                for actor in self.mayuna[self.surface_id]:
                    actor_id = actor.get_id()
                    if actor_id == index:
                        if actor_id in self.bind and self.bind[actor_id][1] and \
                           actor_id not in done:
                            done.append(actor_id)
                            for result in self.iter_mayuna(surface_width, surface_height, actor, done):
                                yield result
                        else:
                            break
            else:
                raise RuntimeError, 'should not reach here'

    def reset_pixbuf_cache(self):
        self.pixbuf_cache = None
        self.mayuna_cache = None
        gc.collect()
        self.pixbuf_cache = {}
        self.mayuna_cache = {}

    def prefetch(self, surface_ids):
        for surface_id in surface_ids:
            if surface_id < 0:
                continue
            if self.alias is not None and surface_id in self.alias:
                aliases = []
                aliases.extend(self.alias.get(surface_id))
                if surface_id in aliases:
                    aliases.remove(surface_id)
                if aliases:
                    self.prefetch(aliases)
            if surface_id not in self.pixbuf:
                continue
            if surface_id not in self.pixbuf_cache:
                self.get_pixbuf(surface_id)

    def create_pixbuf_from_file(self, pixbuf_id):
        assert pixbuf_id in self.pixbuf
        try:
            pixbuf = pix.create_pixbuf_from_file(
                self.pixbuf[pixbuf_id][0], use_pna=self.__use_pna)
        except:
            if self.debug & 4:
                print 'cannot load surface #%d' % pixbuf_id
            return pix.create_blank_pixbuf(8, 8)
        for element, x, y in self.pixbuf[pixbuf_id][1:]:
            try:
                overlay = pix.create_pixbuf_from_file(
                    element, use_pna=self.__use_pna)
            except:
                continue
            w = overlay.get_width()
            h = overlay.get_height()
            sw = pixbuf.get_width()
            sh = pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            overlay.composite(pixbuf, dest_x, dest_y,
                              w, h, x, y, 1.0, 1.0,
                              gtk.gdk.INTERP_BILINEAR, 255)
            del overlay
        ##else:
        ##    gc.collect() # XXX
        return pixbuf

    def get_pixbuf(self, pixbuf_id):
        if pixbuf_id not in self.pixbuf:
            return pix.create_blank_pixbuf(8, 8)
        if pixbuf_id in self.pixbuf_cache:
            pixbuf = self.pixbuf_cache[pixbuf_id][0]
        else:
            pixbuf = self.create_pixbuf_from_file(pixbuf_id)
            self.pixbuf_cache[pixbuf_id] = (pixbuf, (
                pixbuf.get_width(), pixbuf.get_height()))
        return pixbuf.copy()

    def draw_region(self):
        surface_gc = self.darea.window.new_gc()
        surface_gc.function = gtk.gdk.INVERT
        for part, x1, y1, x2, y2 in self.collisions:
            if self.__scale != 100:
                x1 = x1 * self.__scale / 100
                x2 = x2 * self.__scale / 100
                y1 = y1 * self.__scale / 100
                y2 = y2 * self.__scale / 100
            self.darea.window.draw_rectangle(
                surface_gc, 0, x1, y1, x2 - x1, y2 - y1)

    def create_surface_pixbuf(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id in self.mayuna and self.mayuna[surface_id]:
            if surface_id in self.mayuna_cache:
                surface_pixbuf = self.mayuna_cache[surface_id].copy()
            else:
                surface_pixbuf = self.get_pixbuf(surface_id)
                done = []
                for actor in self.mayuna[surface_id]:
                    actor_id = actor.get_id()
                    if actor_id in self.bind and self.bind[actor_id][1] and \
                       actor_id not in done:
                        done.append(actor_id)
                        #surface_pixbuf = self.compose_surface(
                        #    surface_pixbuf, actor, done)
                        surface_width = surface_pixbuf.get_width()
                        surface_height = surface_pixbuf.get_height()
                        for method, pixbuf, dest_x, dest_y, w, h, x, y in self.iter_mayuna(surface_width, surface_height, actor, done):
                            if method in ['bind', 'add']:
                                pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                                 w, h, x, y, 1.0, 1.0,
                                                 gtk.gdk.INTERP_BILINEAR, 255)
                            elif method == 'reduce':
                                if pixbuf.get_has_alpha():
                                    dest_w = surface_width
                                    dest_h = surface_height
                                    surface_array = surface_pixbuf.get_pixels_array()
                                    array = pixbuf.get_pixels_array()
                                    for i in range(h):
                                        for j in range(w):
                                            if array[i][j][3] == 0: # alpha
                                                x = j + dest_x
                                                y = i + dest_y
                                                if 0 <= x < dest_w and 0 <= y < dest_h:
                                                    surface_array[y][x][3] = 0 # alpha
                            else:
                                raise RuntimeError, 'should not reach here'
                self.mayuna_cache[surface_id] = surface_pixbuf.copy()
        else:
            surface_pixbuf =self.get_pixbuf(surface_id)
        return surface_pixbuf

    def clear_frame_buffer(self):
        self.frame_buffer = []

    def update_frame_buffer(self, seriko, frame, move, dirty):
        if dirty:
            surface_pixbuf = self.create_surface_pixbuf(seriko.base_id)
            if surface_pixbuf is not None:
                # draw overlays
                for pixbuf_id, x, y in seriko.iter_overlays():
                    try:
                        pixbuf = self.get_pixbuf(pixbuf_id)
                        w = pixbuf.get_width()
                        h = pixbuf.get_height()
                    except:
                        continue
                    # overlay surface pixbuf
                    sw = surface_pixbuf.get_width()
                    sh = surface_pixbuf.get_height()
                    if x + w > sw:
                        w = sw - x
                    if y + h > sh:
                        h = sh - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                     w, h, x, y, 1.0, 1.0,
                                     gtk.gdk.INTERP_BILINEAR, 255)
                    del pixbuf
                ##gc.collect() # XXX
        else:
            surface_pixbuf = None
        self.frame_buffer.append((frame, move, surface_pixbuf, dirty))

    def update(self):
        if len(self.frame_buffer) > 0:
            frame, move, surface_pixbuf, dirty = self.frame_buffer.pop(0)
            if move is not None:
                self.move_surface(*move)
            if not dirty:
                return
        else:
            surface_pixbuf = self.create_surface_pixbuf()
        if self.__scale != 100:
            w = max(1, surface_pixbuf.get_width() * self.__scale / 100)
            h = max(1, surface_pixbuf.get_height() * self.__scale / 100)
            surface_pixbuf = surface_pixbuf.scale_simple(
                w, h, gtk.gdk.INTERP_BILINEAR)
        surface_pixmap, mask_pixmap = surface_pixbuf.render_pixmap_and_mask(1)
        self.window.shape_combine_mask(mask_pixmap, 0, 0)
        self.current_surface_pixbuf = surface_pixbuf
        self.darea.queue_draw()

    def redraw(self, darea, event):
        cr = darea.window.cairo_create()
        cr.set_source_rgba(1.0, 1.0, 1.0, 0.0)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()
        cr.set_source_pixbuf(self.current_surface_pixbuf, 0, 0)
        cr.paint_with_alpha(self.__alpha_channel)
        if self.debug & 4096:
            self.draw_region()

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def add_overlay(self, actor, pixbuf_id, x, y):
        self.seriko.add_overlay(self, actor, pixbuf_id, x, y)

    def __move(self, xoffset=0, yoffset=0):
        x, y = self.get_position()
        p, q = self.window_offset
        self.window.move(x + p + xoffset, y + q + yoffset)        

    def move_surface(self, xoffset, yoffset):
        self.__move(xoffset, yoffset)
        self.sakura.ghost.notify_surface_move(self.side, xoffset, yoffset) ## FIXME

    def get_balloon_offset(self):
        path, config = self.surface[''.join(('surface', self.surface_id))]
        if self.side == 0:
            x = config.getint('sakura.balloon.offsetx')
            y = config.getint('sakura.balloon.offsety')
        else:
            x = config.getint('kero.balloon.offsetx')
            y = config.getint('kero.balloon.offsety')
        return x, y

    def get_surface(self):
        return self.surface_id

    def get_surface_size(self, surface_id=None):
        if surface_id is None:
            surface_id = self.surface_id
        if surface_id in self.pixbuf_cache:
            w, h = self.pixbuf_cache[surface_id][1]
        else: # XXX
            self.get_pixbuf(surface_id)
            w, h = self.pixbuf_cache[surface_id][1]
        if self.__scale != 100:
            w = int(w * self.__scale / 100)
            h = int(h * self.__scale / 100)
        return w, h

    def get_surface_offset(self):
        return self.window_offset

    def get_touched_region(self, x, y):
        for part, x1, y1, x2, y2 in self.collisions:
            if x1 <= x <= x2 and y1 <= y <= y2:
                ##print part, 'touched'
                return part
        return ''

    def get_direction(self):
        return self.direction

    def set_direction(self, direction):
        self.direction = direction # 0: left, 1: right
        self.sakura.ghost.set_balloon_direction(self.side, direction) ## FIXME

    def set_position(self, x, y):
        self.position = (x, y)
        if self.__shown:
            self.__move()
        if x > gtk.gdk.screen_width() / 2:
            direction = 0
        else:
            direction = 1
        self.set_direction(direction)

    def get_position(self):
        return self.position

    def set_alignment_current(self):
        self.set_alignment(self.get_alignment())

    def set_alignment(self, align):
        if align == 0:
            scrn_h = gtk.gdk.screen_height() - \
                     self.sakura.ghost.get_bottom_margin() ## FIXME
            sw, sh = self.get_surface_size()
            sx, sy = self.get_position()
            sy = scrn_h - sh
            self.set_position(sx, sy)
        elif align == 1:
            sx, sy = self.get_position()
            sy = 0 + self.sakura.ghost.get_top_margin() ## FIXME
            self.set_position(sx, sy)
        else: # free
            pass
        if align in [0, 1, 2]:
            self.align = align

    def get_alignment(self):
        return self.align

    def destroy(self):
        self.reset_pixbuf_cache()
        for tag in self.callbacks:
            self.darea.disconnect(tag)
        self.seriko.terminate(self)
        self.window.remove(self.darea)
        self.darea.destroy()
        self.window.destroy()

    def is_shown(self):
        if self.__shown:
            return 1
        else:
            return 0

    def show(self):
        if not self.__shown:
            self.darea.show()
            self.window.show()
            self.__move()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            ##self.window.hide()
            self.darea.hide()
            self.__shown = 0

    def raise_(self):
        self.window.window.raise_()

    def lower(self):
        self.window.window.lower()

    def button_press(self, window, event):
        self.sakura.reset_idle_time()
        x = int(event.x)
        y = int(event.y)
        if self.__scale != 100:
            x = int(x * 100 / self.__scale)
            y = int(y * 100 / self.__scale)
        if event.button == 2:
            if event.type == gtk.gdk.BUTTON_PRESS:
                self.__position_dirty = True
                self.window.begin_move_drag(
                    event.button, int(event.x_root), int(event.y_root),
                    gtk.get_current_event_time())
        else:
            if event.type == gtk.gdk.BUTTON_PRESS:
                click = 1
            else:
                click = 2
            self.sakura.ghost.notify_surface_click(
                event.button, click, self.side, x, y) ## FIXME
        return True

    def button_release(self, window, event):
        return True

    def motion_notify(self, darea, event):
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        if not self.sakura.busy():
            if state & gtk.gdk.BUTTON1_MASK or \
               state & gtk.gdk.BUTTON2_MASK or \
               state & gtk.gdk.BUTTON3_MASK:
                pass
            else:
                if self.__scale != 100:
                    x = int(x * 100 / self.__scale)
                    y = int(y * 100 / self.__scale)
                part = self.get_touched_region(x, y)
                if part:
                    cursor = gtk.gdk.Cursor(gtk.gdk.HAND1)
                    self.darea.window.set_cursor(cursor)
                else:
                    self.darea.window.set_cursor(None)
                self.sakura.notify_surface_mouse_motion(self.side, x, y)
        return True

    def scroll(self, darea, event):
        x = int(event.x)
        y = int(event.y)
        if self.__scale != 100:
            x = int(x * 100 / self.__scale)
            y = int(y * 100 / self.__scale)
        if event.direction == gtk.gdk.SCROLL_UP:
            count = 1
        elif event.direction == gtk.gdk.SCROLL_DOWN:
            count = -1
        else:
            count = 0
        if count != 0:
            part = self.get_touched_region(x, y)
            self.sakura.notify_event('OnMouseWheel',
                                     x, y, count, self.side, part)
        return True

    def toggle_bind(self, bind_id):
        if bind_id in self.bind:
            current = self.bind[bind_id][1]
            self.bind[bind_id][1] = not current
            self.reset_pixbuf_cache()
            self.reset_surface()

    def window_leave_notify(self, window, event):
        if not self.__position_dirty:
            return
        self.__position_dirty = False
        x, y = self.window.get_position()
        p, q = self.window_offset
        w, h = self.get_surface_size()
        top_margin = self.sakura.ghost.get_top_margin() ## FIXME
        bottom_margin = self.sakura.ghost.get_bottom_margin() ## FIMXE
        scrn_h = gtk.gdk.screen_height() - bottom_margin
        new_x = x - p
        if self.align == 0: # bottom
            new_y = scrn_h - h
        elif self.align == 1: # top
            new_y = 0 + top_margin
        else: # free
            new_y = y
        new_y -= q
        old_x, old_y = self.get_position()
        if new_x != old_x or new_y != old_y or new_y != y - q:
            self.set_position(new_x, new_y)


def test():
    pass

if __name__ == '__main__':
    test()
