# -*- coding: utf-8 -*-
#
#  kawari.py - a "華和梨" compatible Shiori module for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import __builtin__
import base64
import os
import re
import time
import random
import sys
import StringIO

import codecs

###   READER   ###
charset = 'Shift_JIS' # default

def read_dict(path, debug=0):
    global charset
    f = __builtin__.open(path)
    lineno = 0
    buf = []
    for line in f:
        lineno = lineno + 1
        position = (lineno, path)
        if line.startswith('!KAWA0000'):
            line = decrypt(line[9:])
        if not line.strip() or \
           line.startswith('#') or \
           line.startswith(':crypt') or \
           line.startswith(':endcrypt'):
            continue
        if ':' not in line:
            if debug & 4:
                print 'kawari.py: syntax error at line %d in %s:' % position
                print line.strip()
            continue
        entries, phrases = [x.strip() for x in line.split(':', 1)]
        if not entries:
            if debug & 4:
                print 'kawari.py: syntax error at line %d in %s:' % position
                print line.strip()
            continue
        if not phrases:
            continue
        if entries == 'locale':
            try:
                codecs.lookup(phrases)
            except:
                print 'kawari.py: unsupported charset %s' % phrases
            else:
                charset = phrases
        else:
            entries = unicode(entries, charset, 'ignore')
            phrases = unicode(phrases, charset, 'ignore')
        buf.append((entries, phrases, position))
    return buf

def decrypt(data):
    buf = []
    for c in base64.decodestring(data):
        buf.append(chr(ord(c) ^ 0xcc))
    return ''.join(buf)

def encrypt(data):
    buf = []
    for c in data:
        buf.append(chr(ord(c) ^ 0xcc))
    line = ''.join(('!KAWA0000', base64.encodestring(''.join(buf))))
    return line.replace('\n', '')

def create_dict(buf, debug=0):
    rdict = {} # rules
    kdict = {} # keywords
    for entries, phrases, position in buf:
        parsed_entries = parse_entries(entries)
        parsed_phrases = parse_phrases(phrases)
        if parsed_phrases is None:
            if debug & 4:
                print 'kawari.py: syntax error at line %d in %s:' % position
                print phrases.strip()
            continue
        if entries.startswith('['):
            add_phrases(kdict, tuple(parsed_entries), parsed_phrases)
            continue
        for entry in parsed_entries:
            add_phrases(rdict, entry, parsed_phrases)
    return rdict, kdict

def add_phrases(dic, entry, phrases):
    if entry not in dic:
        dic[entry] = []
    for phrase in phrases:
        if phrase:
            phrase[0]  = phrase[0].lstrip()
            phrase[-1] = phrase[-1].rstrip()
        dic[entry].append(tuple(phrase))

def parse_entries(data):
    if data.startswith('[') and data.endswith(']'):
        entries = []
        i = 0
        j = len(data)
        while i < j:
            if data[i] == '"':
                i, text = parse_quotes(data, i)
                entries.append(text)
            else:
                i += 1
    else:
        entries = [s.strip() for s in data.split(',')]
    return entries

re_comma = re.compile(',')

def parse_phrases(data):
    buf = []
    i = 0
    j = len(data)
    while i < j:
        if data[i] == ',':
            i += 1
        i, phrase = parse(data, i, re_comma)
        if phrase:
            buf.append(phrase)
    return buf

def parse(data, start, stop_pattern=None):
    buf = []
    i = start
    j = len(data)
    while i < j:
        if stop_pattern and stop_pattern.match(data, i):
            break
        elif data[i] == '"':
            i, text = parse_quotes(data, i)
            buf.append('"%s"' % text)
        elif data[i:i + 2] == '${':
            i, text = parse_reference(data, i)
            buf.append(text)
        elif data[i:i + 2] == '$(':
            i, text = parse_inline_script(data, i)
            buf.append(text)
        elif data[i] == '$':
            buf.append(data[i])
            i += 1
        elif data[i] == ';':
            buf.append(data[i])
            i += 1
        else:
            i, text = parse_text(data, i, stop_pattern)
            buf.append(text)
    if buf:
        if is_space(buf[0]):
            del buf[0]
        else:
            buf[0] = buf[0].lstrip()
    if buf:
        if is_space(buf[-1]):
            del buf[-1]
        else:
            buf[-1] = buf[-1].rstrip()
    return i, buf

def parse_quotes(data, start):
    buf = []
    i = start + 1
    j = len(data)
    while i < j:
        if data[i] == '"':
            i += 1
            break
        elif data[i] == '\\':
            i += 1
            if i < j and data[i] == '"':
                buf.append(''.join(('\\', data[i])))
                i += 1
            else:
                buf.append('\\')
        else:
            buf.append(data[i])
            i += 1
    return i, ''.join(buf)

def parse_reference(data, start):
    i = start
    j = len(data)
    while i < j:
        if data[i] == '}':
            i += 1
            break
        else:
            i += 1
    return i, data[start:i]

def parse_inline_script(data, start):
    buf = ['$']
    i = start + 1
    j = len(data)
    npar = 0
    while i < j:
        #begin specification bug work-around (1/3)
        if data[i] == ')':
            buf.append(data[i])
            i += 1
            break
        #end
        if data[i] == '"':
            i, text = parse_quotes(data, i)
            buf.append('"%s"' % text)
        elif data[i:i + 2] == '${':
            i, text = parse_reference(data, i)
            buf.append(text)
        elif data[i:i + 2] == '$(':
            i, text = parse_inline_script(data, i)
            buf.append(text)
        else:
            if data[i] == '(':
                npar = npar + 1
            elif data[i] == ')':
                npar = npar - 1
            buf.append(data[i])
            i += 1
        if npar == 0:
            break
    return i, ''.join(buf)

def is_space(s):
    return not s.strip()

def parse_text(data, start, stop_pattern=None):
    condition = is_space(data[start])
    i = start
    j = len(data)
    while i < j:
        if stop_pattern and stop_pattern.match(data, i):
            break
        elif data[i] in ['$', '"']:
            break
        elif is_space(data[i]) != condition:
            break
        elif data[i] == ';':
            if i == start:
                i += 1
            break
        else:
            i += 1
    return i, data[start:i]

def read_local_script(path):
    f = __builtin__.open(path)
    rdict = {}
    kdict = {}
    for line in f.readline():
        if line.startswith('#'):
            rdict[unicode(line.strip(), charset, 'ignore')] = \
                 [unicode(f.readline().strip(), charset, 'ignore')]
    return rdict, kdict


###   KAWARI   ###

class Kawari:

    MAXDEPTH = 30

    def __init__(self, prefix, pathlist, rdictlist, kdictlist, debug=0):
        self.prefix = prefix
        self.pathlist = pathlist
        self.rdictlist = rdictlist
        self.kdictlist = kdictlist
        self.system_entries = {}
        self.debug = debug
        self.expr_parser = ExprParser()
        #begin specification bug work-around (2/3)
        self.expr_parser.kawari = self
        #end
        self.otherghost = {}
        self.get_system_entry('OnLoad')

    def finalize(self):
        self.get_system_entry('OnUnload')

    # SHIORI/1.0 API
    def getaistringrandom(self):
        return self.get('sentence').encode(charset, 'ignore')

    def getaistringfromtargetword(self, word):
        word = unicode(word, charset, 'ignore')
        return self.get('sentence').encode(charset, 'ignore') # XXX

    def getdms(self):
        return self.getword('dms')

    def getword(self, word_type):
        for delimiter in ['.', '-']:
            name = ''.join(('compatible', delimiter, word_type))
            script = self.get(name, default=None)
            if script is not None:
                return script.strip().encode(charset, 'ignore')
        return ''

    def getstring(self, name):
        name = unicode(name, charset, 'ignore')
        return self.get('resource.%s' % name).encode(charset, 'ignore')

    # SHIORI/2.2 API
    def get_event_response(self, event,
                           ref0=None, ref1=None, ref2=None, ref3=None,
                           ref4=None, ref5=None, ref6=None, ref7=None): ## FIXME
        def proc(ref):
            if ref is not None:
                ref = unicode(str(ref), charset, 'ignore')
            return ref
        ref = [proc(ref) for ref in [ref0, ref1, ref2, ref3, ref4,
                                     ref5, ref6, ref7]]
        for i in range(8):
            if ref[i] is not None:
                value = ref[i]
                self.system_entries['system.Reference%d' % i] = value
                self.system_entries['system-Reference%d' % i] = value
        script = None
        if event == 'OnCommunicate':
            self.system_entries['system.Sender'] = ref[0]
            self.system_entries['system-Sender'] = ref[0]
            self.system_entries['system.Sender.Path'] = 'local' # (local/unknown/external)
            self.system_entries['system-Sender.Path'] = 'local' # (local/unknown/external)
            if 'system.Age' not in self.system_entries:
                self.system_entries['system.Age'] = '0'
                self.system_entries['system-Age'] = '0'
            self.system_entries['system.Sentence'] = ref[1]
            self.system_entries['system-Sentence'] = ref[1]
            if ref[0] in self.otherghost:
                s0, s1 = self.otherghost[ref[0]]
                self.system_entries['system.Surface'] = ','.join((str(s0), str(s1)))
                self.system_entries['system-Surface'] = ','.join((str(s0),
                                                                  str(s1)))
            script = self.get_system_entry('OnResponse')
            if not script:
                for dic in self.kdictlist:
                    for entry in dic:
                        for word in entry:
                            if word not in ref[1]:
                                break
                        else:
                            script = self.expand(random.choice(dic[entry]))
                            break
            if not script:
                script = self.get_system_entry('OnResponseUnknown')
            if script is not None:
                script = script.strip()
        else:
            for delimiter in ['.', '-']:
                name = ''.join(('event', delimiter, event))
                script = self.get(name, default=None)
                if script is not None:
                    script = script.strip()
                    break
        if script is not None:
            script = script.encode(charset, 'ignore')
        return script

    # SHIORI/2.4 API
    def teach(self, word):
        word = unicode(word, charset, 'ignore')
        self.system_entries['system.Sentence'] = word
        self.system_entries['system-Sentence'] = word
        return self.get_system_entry('OnTeach').encode(charset, 'ignore')

    def get_system_entry(self, entry):
        for delimiter in ['.', '-']:
            name = ''.join(('system', delimiter, entry))
            script = self.get(name, default=None)
            if script is not None:
                return script.strip()
        return None

    def otherghostname(self, ghost_list):
        ghosts = []
        for ghost in ghost_list:
            name, s0, s1 = ghost.split(chr(1))
            ghosts.append([unicode(name, charset, 'ignore'), s0, s1])
            self.otherghost[name] = [s0, s1]
        otherghost_name = []
        for ghost in ghosts:
            otherghost_name.append(ghost[0])
        self.system_entries['system.OtherGhost'] = otherghost_name
        self.system_entries['system-OtherGhost'] = otherghost_name
        otherghost_ex = []
        for ghost in ghosts:
            otherghost_ex.append(chr(1).join(ghost))
        self.system_entries['system.OtherGhostEx'] = otherghost_ex
        self.system_entries['system-OtherGhostEx'] = otherghost_ex
        if ghosts:
            self.get_system_entry('OnNotifyOther')
        return ''

    def communicate_to(self):
        communicate = self.get_system_entry('communicate')
        if communicate:
            if communicate == 'stop':
                self.system_entries['system.Age'] = '0'
                self.system_entries['system-Age'] = '0'
                communicate = None
            else:
                if 'system.Age' in self.system_entries:
                    age = int(self.system_entries['system.Age']) + 1
                    self.system_entries['system.Age'] = str(age)
                    self.system_entries['system-Age'] = str(age)
                else:
                    self.system_entries['system.Age'] = '0'
                    self.system_entries['system-Age'] = '0'
        self.clear('system.communicate')
        if communicate is not None:
            communicate = communicate.encode(charset, 'ignore')
        return communicate

    # internal
    def clear(self, name):
        if self.debug & 128:
            print '*** clear("%s")' % name.encode('UTF-8', 'ignore')
        for dic in self.rdictlist:
            if name in dic:
                del dic[name]

    def get_internal_dict(self, name):
        for dic in self.rdictlist:
            if name in dic:
                break
        else:
            dic = self.rdictlist[0]
            dic[name] = []
        return dic

    def unshift(self, name, value):
        if self.debug & 128:
            print ('*** unshift("%s", "%s")' % (name, value)).encode('UTF-8',
                                                                     'ignore')
        dic = self.get_internal_dict(name)
        i, segments = parse(value, 0)
        dic[name].insert(0, segments)

    def shift(self, name):
        dic = self.get_internal_dict(name)
        value = self.expand(dic[name].pop(0))
        if self.debug & 128:
            print ('*** shift("%s") => "%s"' % (name, value)).encode('UTF-8',
                                                                     'ignore')
        return value

    def push(self, name, value):
        if self.debug & 128:
            print ('*** push("%s", "%s")' % (name, value)).encode('UTF-8',
                                                                  'ignore')
        dic = self.get_internal_dict(name)
        i, segments = parse(value, 0)
        dic[name].append(segments)

    def pop(self, name):
        dic = self.get_internal_dict(name)
        value = self.expand(dic[name].pop())
        if self.debug & 128:
            print ('*** pop("%s") => "%s"' % (name, value)).encode('UTF-8',
                                                                   'ignore')
        return value

    def set(self, name, value):
        self.clear(name)
        self.push(name, value)

    def get(self, name, context=None, depth=0, default=''):
        if depth == self.MAXDEPTH:
            return ''
        if name and name.startswith('@'):
            segments = random.choice(context[name])
        else:
            if '&' not in name:
                selection = self.select_simple_phrase(name)
            else:
                selection = self.select_compound_phrase(name)
            if selection is None:
                if self.debug & 8:
                    print '${%s} not found' % name.encode('UTF-8', 'ignore')
                return default
            segments, context = selection
        if self.debug & 16:
            print name.encode('UTF-8', 'ignore'), '=>', \
                  ''.join(segments).encode('UTF-8', 'ignore')
        return self.expand(segments, context, depth)

    def parse_all(self, data, start=0):
        i, segments = parse(data, start)
        return self.expand(segments)

    def parse_sub(self, data, start=0, stop_pattern=None):
        i, segments = parse(data, start, stop_pattern)
        return i, self.expand(segments)

    def expand(self, segments, context=None, depth=0):
        buf = []
        references = []
        i = 0
        j = len(segments)
        while i < j:
            segment = segments[i]
            if not segment:
                pass
            elif segment.startswith('${') and segment.endswith('}'):
                newname = segment[2:-1]
                if self.is_number(newname):
                    try:
                        segment = references[int(newname)]
                    except IndexError:
                        pass
                elif self.is_system_entry(newname):
                    if newname in ['system.OtherGhost', 'system-OtherGhost',
                                   'system.OtherGhostEx',
                                   'system-OtherGhostEx']:
                        segment_list = self.system_entries.get(newname)
                        if segment_list:
                            segment = random.choice(segment_list)
                        else:
                            segment = ''
                    elif newname == 'system.communicate':
                        segment = self.get_system_entry('communicate')
                    else:
                        segment = self.system_entries.get(newname, segment)
                else:
                    segment = self.get(newname, context, depth + 1)
                references.append(segment)
            elif segment.startswith('$(') and segment.endswith(')'):
                i, segment = self.eval_inline_script(segments, i)
            elif segment.startswith('"') and segment.endswith('"'):
                segment = segment[1:-1].replace(r'\"', '"')
            buf.append(segment)
            i += 1
        return ''.join(buf)

    def atoi(self, s):
        try:
            return int(s)
        except ValueError:
            return 0

    def is_number(self, s):
        try:
            int(s)
        except ValueError:
            return 0
        return 1

    def is_system_entry(self, s):
        return s.startswith('system-') or s.startswith('system.')

    def select_simple_phrase(self, name):
        n = 0
        buf = []
        for d in self.rdictlist:
            c = d.get(name, [])
            n += len(c)
            buf.append((c, d))
        if n == 0:
            return None
        n = random.randrange(0, n)
        for c, d in buf:
            m = len(c)
            if n < m:
                break
            n -= m
        return c[n], d        

    def select_compound_phrase(self, name):
        buf = []
        for name in [s.strip() for s in name.split('&')]:
            cp_list = []
            for d in self.rdictlist:
                cp_list.extend([tuple(e) for e in d.get(name, [])])
            buf.append(cp_list)
        buf[:] = [(len(x), x) for x in buf]
        buf.sort()
        buf[:] = [x for len_x, x in buf]
        candidates = []
        for item in buf.pop(0):
            for cp_list in buf:
                if item not in cp_list:
                    break
            else:
                candidates.append(item)
        if not candidates:
            return None
        return random.choice(candidates), None

    def eval_inline_script(self, segments, i):
        # check old 'if' syntax
        if segments[i].startswith('$(if ') and i + 1 < len(segments) and \
           segments[i + 1].startswith('$(then '):
            if_block = segments[i][5:-1].strip()
            i += 1
            then_block = segments[i][7:-1].strip()
            if i + 1 < len(segments) and segments[i + 1].startswith('$(else '):
                i += 1
                else_block = segments[i][7:-1].strip()
            else:
                else_block = ''
            if i + 1 < len(segments) and segments[i + 1] == '$(endif)':
                i += 1
            else:
                if self.debug & 32:
                    print 'kawari.py: syntax error: $(endif) expected'
                return i, '' # syntax error
            return i, self.exec_old_if(if_block, then_block, else_block)
        # execute command(s)
        values = []
        for command in self.split_commands(segments[i][2:-1]):
            argv = self.parse_argument(command)
            argv[0] = self.expand(argv[0])
            if argv[0] == 'silent':
                if len(argv) == 1:
                    values = []
                elif self.debug & 32:
                    print 'kawari.py: syntax error:', \
                          segments[i].encode('UTF-8', 'ignore')
                continue
            handler = self.kis_commands.get(argv[0])
            try:
                if handler is None:
                    raise RuntimeError, 'invalid command'
                values.append(handler(self, argv))
            except RuntimeError, message:
                if self.debug & 32:
                    print 'kawari.py: %s: %s' % \
                          (message, segments[i].encode('UTF-8', 'ignore'))
        result = ''.join(values)
        if self.debug & 64:
            print '>>>', segments[i].encode('UTF-8', 'ignore')
            print ''.join(('"', result.encode('UTF-8', 'ignore'), '"'))
        return i, result

    def split_commands(self, data):
        i, segments = parse(data, 0)
        # find multiple commands separated by semicolons
        buf = []
        command = []
        for segment in segments:
            if segment == ';':
                if command:
                    buf.append(command)
                    command = []
            else:
                command.append(segment)
        if command:
            buf.append(command)
        # strip white space before and after each command
        for command in buf:
            if is_space(command[0]):
                del command[0]
            if is_space(command[-1]):
                del command[-1]
        return buf

    def parse_argument(self, segments):
        buf = [[]]
        for segment in segments:
            if is_space(segment):
                buf.append([])
            else:
                buf[-1].append(segment)
        return buf

    def exec_new_if(self, argv):
        if len(argv) == 3:
            return self.exec_if(argv[1], argv[2], None)
        elif len(argv) == 4:
            return self.exec_if(argv[1], argv[2], argv[3])
        else:
            raise RuntimeError, 'syntax error'

    def exec_old_if(self, if_block, then_block, else_block):
        # convert [...] into $([...])
        if if_block and if_block.startswith('[') and if_block.endswith(']'):
            if_block = ''.join(('$(', if_block, ')'))
        # parse arguments
        i, if_block = parse(if_block, 0)
        i, then_block = parse(then_block, 0)
        if else_block:
            i, else_block = parse(else_block, 0)
        result = self.exec_if(if_block, then_block, else_block)
        if self.debug & 64:
            if else_block:
                print '>>> $(if %s)$(then %s)$(else %s)$(endif)' % (
                    ''.join(if_block).encode('UTF-8', 'ignore'),
                    ''.join(then_block).encode('UTF-8', 'ignore'),
                    ''.join(else_block).encode('UTF-8', 'ignore'))
            else:
                print '>>> $(if %s)$(then %s)$(endif)' % (
                    ''.join(if_block).encode('UTF-8', 'ignore'),
                    ''.join(then_block).encode('UTF-8', 'ignore'))
            print ''.join(('"', result.encode('UTF-8', 'ignore'), '"'))
        return result

    def exec_if(self, if_block, then_block, else_block):
        if self.expand(if_block) not in ['', '0', 'false', 'False']:
            return self.expand(then_block)
        elif else_block:
            return self.expand(else_block)
        return ''

    def exec_foreach(self, argv):
        if len(argv) != 4:
            raise RuntimeError, 'syntax error'
        temp = self.expand(argv[1])
        name = self.expand(argv[2])
        buf = []
        for dic in self.rdictlist:
            for segments in dic.get(name, []):
                self.set(temp, self.expand(segments))
                buf.append(self.expand(argv[3]))
        self.clear(temp)
        return ''.join(buf)

    def exec_loop(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        try:
            n = int(self.expand(argv[1]))
        except ValueError:
            raise RuntimeError, 'invalid argument'
        buf = []
        for i in range(n):
            buf.append(self.expand(argv[2]))
        return ''.join(buf)

    def exec_while(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        buf = []
        while self.expand(argv[1]) not in ['', '0', 'false', 'False']:
            buf.append(self.expand(argv[2]))
        return ''.join(buf)

    def exec_until(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        buf = []
        while self.expand(argv[1]) in ['', '0', 'false', 'False']:
            buf.append(self.expand(argv[2]))
        return ''.join(buf)

    def exec_set(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        self.set(self.expand(argv[1]), self.expand(argv[2]))
        return ''

    def exec_adddict(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        self.push(self.expand(argv[1]), self.expand(argv[2]))
        return ''

    def exec_array(self, argv): # XXX experimental
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        n = self.atoi(self.expand(argv[2]))
        for d in self.rdictlist:
            c = d.get(name, [])
            if n < len(c):
                return ''.join([self.expand(s) for s in c[n]])
            n -= len(c)
        else:
            raise RuntimeError, 'invalid argument'

    def exec_clear(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        self.clear(self.expand(argv[1]))
        return ''

    def exec_enumerate(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        return ' '.join([self.expand(s) for s in self.enumerate(name)])

    def enumerate(self, name):
        buf = []
        for dic in self.rdictlist:
            for segments in dic.get(name, []):
                buf.append(segments)
        return buf

    def exec_size(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        n = 0
        for d in self.rdictlist:
            c = d.get(name, [])
            n += len(c)
        return str(n)

    def exec_get(self, argv): # XXX experimental
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        n = self.atoi(self.expand(argv[2]))
        for d in self.rdictlist:
            c = d.get(name, [])
            if n < len(c):
                return ''.join(c[n])
            n -= len(c)
        else:
            raise RuntimeError, 'invalid argument'

    def exec_unshift(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        self.unshift(self.expand(argv[1]), self.expand(argv[2]))
        return ''

    def exec_shift(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        return self.shift(self.expand(argv[1]))

    def exec_push(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        self.push(self.expand(argv[1]), self.expand(argv[2]))
        return ''

    def exec_pop(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        return self.pop(self.expand(argv[1]))

    def exec_pirocall(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        selection = self.select_simple_phrase(self.expand(argv[1]))
        if selection is None:
            return ''
        return selection[0]

    def exec_split(self, argv):
        if len(argv) != 4:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        word_list = self.expand(argv[2]).split(self.expand(argv[3]))
        n = 0
        for word in word_list:
            n += 1
            entry = '%s.%d' % (name, n)
            self.set(entry, word)
        self.set(''.join((name, '.size')), str(n))
        return ''

    def get_dict_path(self, path):
        path = path.replace('\\', '/').lower()
        if not path:
            raise RuntimeError, 'invalid argument'
        if path.startswith('/'):
            return path
        return os.path.join(self.prefix, path)

    def exec_load(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            rdict, kdict = create_dict(read_dict(path, self.debug))
        except IOError:
            raise RuntimeError, 'cannot read file'
        if path in self.pathlist:
            i = self.pathlist.index(path)
            self.rdictlist[i].update(rdict)
            self.kdictlist[i].update(kdict)
        else:
            self.pathlist.insert(0, path)
            self.rdictlist.insert(0, rdict)
            self.kdictlist.insert(0, kdict)
        return ''

    def exec_save(self, argv, crypt=0):
        if len(argv) < 2:
            raise RuntimeError, 'syntax error'
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            f = __builtin__.open(path, 'w')
            f.write('#\r\n# Kawari save file\r\n#\r\n')
            for i in range(2, len(argv)):
                name = self.expand(argv[i])
                if not name.strip():
                    continue
                buf = []
                for segments in self.enumerate(name):
                    buf.append(''.join(segments))
                name = name.encode(charset, 'ignore')
                line = ''.join((name, ' : ',
                                ' , '.join(buf).encode(charset, 'ignore')))
                if crypt:
                    line = encrypt(line)
                f.write('# Entry %s\r\n%s\r\n' % (name, line))
            f.close()
        except IOError:
            raise RuntimeError, 'cannot write file'
        return ''

    def exec_savecrypt(self, argv):
        return self.exec_save(argv, 1)

    def exec_textload(self, argv):
        if len(argv) != 3:
            raise RuntimeError, 'syntax error'
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            linelist = __builtin__.open(path).readlines()
        except IOError:
            raise RuntimeError, 'cannot read file'
        name = self.expand(argv[2])
        n = 0
        for line in linelist:
            n += 1
            entry = '%s.%d' % (name, n)
            if line.endswith('\r\n'):
                line = line[:-2]
            elif line.endswith('\r') or line.endswith('\n'):
                line = line[:-1]
            if not line:
                self.clear(entry)
            else:
                self.set(entry, unicode(line, charset, 'replace'))
        self.set(''.join((name, '.size')), str(n))
        return ''

    def exec_escape(self, argv):
        data = ' '.join([self.expand(s) for s in argv[1:]])
        data = data.replace('\\', r'\\')
        data = data.replace('%', r'\%')
        return data

    def exec_echo(self, argv):
        return ' '.join([self.expand(s) for s in argv[1:]])

    def exec_tolower(self, argv):
        return self.exec_echo(argv).lower()

    def exec_toupper(self, argv):
        return self.exec_echo(argv).upper()

    def exec_eval(self, argv):
        return self.parse_all(self.exec_echo(argv))

    def exec_entry(self, argv):
        if len(argv) == 2:
            return self.get(self.expand(argv[1]))
        elif len(argv) == 3:
            return self.get(self.expand(argv[1])) or self.expand(argv[2])
        else:
            raise RuntimeError, 'syntax error'

    def exec_null(self, argv):
        if len(argv) != 1:
            raise RuntimeError, 'syntax error'
        return ''

    def exec_chr(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        num = self.atoi(self.expand(argv[1]))
        if num < 256:
            return chr(num)
        return ''.join((chr((num >> 8) & 0xff), chr(num & 0xff)))

    def exec_choice(self, argv):
        if len(argv) == 1:
            return ''
        i = random.randrange(1, len(argv))        
        return self.expand(argv[i])

    def exec_rand(self, argv):
        if len(argv) != 2:
            raise RuntimeError, 'syntax error'
        bound = self.atoi(self.expand(argv[1]))
        if bound == 0:
            return str(0)
        elif bound > 0:
            return str(random.randrange(0, bound))
        else:
            return str(random.randint(bound + 1, 0))

    def exec_date(self, argv):
        if len(argv) == 1:
            format = '%y/%m/%d %H:%M:%S'
        else:
            format = ' '.join([self.expand(s) for s in argv[1:]])
        buf = []
        i = 0
        j = len(format)
        now = time.localtime(time.time())
        while i < j:
            if format[i] == '%':
                i += 1
                if i < j:
                    c = format[i]
                    i += 1
                else:
                    break
                if c in ['y', 'Y']: # year (4 columns)
                    buf.append('%04d' % now[0])
                elif c == 'm': # month (01 - 12)
                    buf.append('%02d' % now[1])
                elif c == 'n': # month (1 - 12)
                    buf.append(str(now[1]))
                elif c == 'd': # day (01 - 31)
                    buf.append('%02d' % now[2])
                elif c == 'e': # day (1 - 31)
                    buf.append(str(now[2]))
                elif c == 'H': # hour (00 - 23)
                    buf.append('%02d' % now[3])
                elif c == 'k': # hour (0 - 23)
                    buf.append(str(now[3]))
                elif c == 'M': # minute (00 - 59)
                    buf.append('%02d' % now[4])
                elif c == 'N': # minute (0 - 59)
                    buf.append(str(now[4]))
                elif c == 'S': # second (00 - 59)
                    buf.append('%02d' % now[5])
                elif c == 'r': # second (0 - 59)
                    buf.append(str(now[5]))
                elif c == 'w': # weekday (0 = Sunday)
                    buf.append(str([1, 2, 3, 4, 5, 6, 0][now[6]]))
                elif c == 'j': # Julian day (001 - 366)
                    buf.append('%03d' % now[7])
                elif c == 'J': # Julian day (1 - 366)
                    buf.append(str(now[7]))
                elif c == '%':
                    buf.append('%')
                else:
                    buf.append('%')
                    i -= 1
            else:
                buf.append(format[i])
                i += 1
        return ''.join(buf)

    def exec_inc(self, argv):
        def _inc(value, step, bound):
            value += step
            if bound is not None and value > bound:
                return bound
            return value
        self.apply_counter_op(_inc, argv)
        return ''

    def exec_dec(self, argv):
        def _dec(value, step, bound):
            value -= step
            if bound is not None and value < bound:
                return bound
            return value
        self.apply_counter_op(_dec, argv)
        return ''

    def apply_counter_op(self, func, argv):
        if len(argv) < 2 or len(argv) > 4:
            raise RuntimeError, 'syntax error'
        name = self.expand(argv[1])
        value = self.atoi(self.get(name))
        if len(argv) >= 3:
            step = self.atoi(self.expand(argv[2]))
        else:
            step = 1
        if len(argv) == 4:
            bound = self.atoi(self.expand(argv[3]))
        else:
            bound = None
        self.set(name, str(func(value, step, bound)))

    def exec_test(self, argv):
        if argv[0] == 'test' and len(argv) == 4 or \
           argv[0] == '[' and len(argv) == 5 and self.expand(argv[4]) == ']':
            op1 = self.expand(argv[1])
            op  = self.expand(argv[2])
            op2 = self.expand(argv[3])
        else:
            raise RuntimeError, 'syntax error'
        if op in ['=', '==']:
            result = str(op1 == op2)
        elif op == '!=':
            result = str(op1 != op2)
        elif op == '<=':
            result = str(op1 <= op2)
        elif op == '>=':
            result = str(op1 >= op2)
        elif op == '<':
            result = str(op1 < op2)
        elif op == '>':
            result = str(op1 > op2)
        elif op == '-eq':
            result = str(self.atoi(op1) == self.atoi(op2))
        elif op == '-ne':
            result = str(self.atoi(op1) != self.atoi(op2))
        elif op == '-le':
            result = str(self.atoi(op1) <= self.atoi(op2))
        elif op == '-ge':
            result = str(self.atoi(op1) >= self.atoi(op2))
        elif op == '-lt':
            result = str(self.atoi(op1) < self.atoi(op2))
        elif op == '-gt':
            result = str(self.atoi(op1) > self.atoi(op2))
        else:
            raise RuntimeError, 'unknown operator'
        return result

    def exec_expr(self, argv):
        tree = self.expr_parser.parse(
            ' '.join([''.join(e) for e in argv[1:]]))
        if tree is None:
            raise RuntimeError, 'syntax error'
        try:
            value = self.interp_expr(tree)
        except ExprError:
            raise RuntimeError, 'runtime error'
        return value

    def interp_expr(self, tree):
        if tree[0] == ExprParser.OR_EXPR:
            for subtree in tree[1:]:
                value = self.interp_expr(subtree)
                if value and value not in ['0', 'False']:
                    break
            return value
        elif tree[0] == ExprParser.AND_EXPR:
            buf = []
            for subtree in tree[1:]:
                value = self.interp_expr(subtree)
                if not value or value in ['0', 'False']:
                    return '0'
                buf.append(value)
            return buf[0]
        elif tree[0] == ExprParser.CMP_EXPR:
            op1 = self.interp_expr(tree[1])
            op2 = self.interp_expr(tree[3])
            if self.is_number(op1) and self.is_number(op2):
                op1 = int(op1)
                op2 = int(op2)
            if tree[2] in ['=', '==']:
                return str(op1 == op2)
            elif tree[2] == '!=':
                return str(op1 != op2)
            elif tree[2] == '<=':
                return str(op1 <= op2)
            elif tree[2] == '>=':
                return str(op1 >= op2)
            elif tree[2] == '<':
                return str(op1 < op2)
            elif tree[2] == '>':
                return str(op1 > op2)
            else:
                raise RuntimeError, 'unknown operator'
        elif tree[0] == ExprParser.ADD_EXPR:
            for i in range(1, len(tree), 2):
                tree[i] = self.interp_expr(tree[i])
                if not self.is_number(tree[i]):
                    raise ExprError
            value = int(tree[1])
            for i in range(2, len(tree), 2):
                if tree[i] == '+':
                    value += int(tree[i + 1])
                elif tree[i] == '-':
                    value -= int(tree[i + 1])
            return str(value)
        elif tree[0] == ExprParser.MUL_EXPR:
            for i in range(1, len(tree), 2):
                tree[i] = self.interp_expr(tree[i])
                if not self.is_number(tree[i]):
                    raise ExprError
            value = int(tree[1])
            for i in range(2, len(tree), 2):
                if tree[i] == '*':
                    value *= int(tree[i + 1])
                elif tree[i] == '/':
                    try:
                        value /= int(tree[i + 1])
                    except ZeroDivisionError:
                        raise ExprError
                elif tree[i] == '%':
                    try:
                        value %= int(tree[i + 1])
                    except ZeroDivisionError:
                        raise ExprError
            return str(value)
        elif tree[0] == ExprParser.STR_EXPR:
            if tree[1] == 'length':
                length = len(self.get_characters(self.interp_expr(tree[2])))
                return str(length)
            elif tree[1] == 'index':
                s = self.get_characters(self.interp_expr(tree[2]))
                c = self.get_characters(self.interp_expr(tree[3]))
                for pos in range(len(s)):
                    if s[pos] in c:
                        break
                else:
                    pos = 0
                return str(pos)
            elif tree[1] == 'match':
                try:
                    match = re.match(self.interp_expr(tree[3]),
                                     self.interp_expr(tree[2]))
                except re.error:
                    match = None
                if match:
                    length = match.end() - match.start()
                else:
                    length = 0
                return str(length)
            elif tree[1] == 'find':
                s = self.interp_expr(tree[3])
                if self.interp_expr(tree[2]).find(s) < 0:
                    return ''
                return s
            elif tree[1] == 'findpos':
                s = self.interp_expr(tree[3])
                pos = self.interp_expr(tree[2]).find(s);
                if pos < 0:
                    return ''
                return str(pos + 1)
            elif tree[1] == 'substr':
                s = self.interp_expr(tree[2])
                p = self.interp_expr(tree[3])
                n = self.interp_expr(tree[4])
                if self.is_number(p) and self.is_number(n):
                    p = int(p) - 1
                    n = p + int(n)
                    if 0 <= p <= n:
                        characters = self.get_characters(s)
                        return ''.join(characters[p:n])
                return ''
        elif tree[0] == ExprParser.LITERAL:
            return self.expand(tree[1:])

    def get_characters(self, s):
        buf = []
        i = 0
        j = len(s)
        while i < j:
            buf.append(s[i])
            i += 1
        return buf

    kis_commands = {
        # flow controls
        'if':          exec_new_if,
        'foreach':     exec_foreach,
        'loop':        exec_loop,
        'while':       exec_while,
        'until':       exec_until,
        # dictionary operators
        'adddict':     exec_adddict,
        'array':       exec_array,
        'clear':       exec_clear,
        'enumerate':   exec_enumerate,
        'set':         exec_set,
        'load':        exec_load,
        'save':        exec_save,
        'savecrypt':   exec_savecrypt,
        'textload':    exec_textload,
        'size':        exec_size,
        'get':         exec_get,
        # list operators
        'unshift':     exec_unshift,
        'shift':       exec_shift,
        'push':        exec_push,
        'pop':         exec_pop,
        # counter operators
        'inc':         exec_inc,
        'dec':         exec_dec,
        # expression evaluators
        'expr':        exec_expr,
        'test':        exec_test,
        '[':           exec_test,
        'entry':       exec_entry,
        'eval':        exec_eval,
        # utility functions
        'NULL':        exec_null,
        '?':           exec_choice,
        'date':        exec_date,
        'rand':        exec_rand,
        'echo':        exec_echo,
        'escape':      exec_escape,
        'tolower':     exec_tolower,
        'toupper':     exec_toupper,
        'pirocall':    exec_pirocall,
        'split':       exec_split,
        'urllist':     None,
        'chr':         exec_chr,
        'help':        None,
        'ver':         None,
        'searchghost': None,
        'saoriregist': None,
        'saorierase':  None,
        'callsaori':   None,
        'callsaorix':  None,
        }

###   EXPR PARSER   ###

class ExprError(ValueError):
    pass

class ExprParser:

    def __init__(self):
        self.debug = 0
        # bit 0 = trace get_expr()
        # bit 1 = trace all "get_" functions

    def show_progress(self, func, buf):
        if buf is None:
            print '%s() -> syntax error' % func
        else:
            print '%s() -> %s' % (func, buf)

    re_token = re.compile('[():|&*/%+-]|[<>]=?|[!=]?=|match|index|findpos|find|substr|length|quote|(\\s+)')

    def tokenize(self, data):
        buf = []
        i = 0
        j = len(data)
        while i < j:
            match = self.re_token.match(data, i)
            if match:
                buf.append(match.group())
                i = match.end()
            else:
                i, segments = parse(data, i, self.re_token)
                buf.extend(segments)
        return buf

    def parse(self, data):
        self.tokens = self.tokenize(data)
        try:
            return self.get_expr()
        except ExprError:
            return None # syntax error

    # internal
    def done(self):
        return not self.tokens

    def pop(self):
        try:
            return self.tokens.pop(0)
        except IndexError:
            raise ExprError

    def look_ahead(self, index=0):
        try:
            return self.tokens[index]
        except IndexError:
            raise ExprError

    def match(self, s):
        if self.pop() != s:
            raise ExprError

    def match_space(self):
        if not is_space(self.pop()):
            raise ExprError

    def check(self, s, index=0):
        return self.look_ahead(index) == s

    def check_space(self, index=0):
        return is_space(self.look_ahead(index))

    # tree node types
    OR_EXPR  = 1
    AND_EXPR = 2
    CMP_EXPR = 3
    ADD_EXPR = 4
    MUL_EXPR = 5
    STR_EXPR = 6
    LITERAL  = 7

    def get_expr(self):
        buf = self.get_or_expr()
        if not self.done():
            raise ExprError
        if self.debug & 1:
            self.show_progress('get_expr', buf)
        return buf

    def get_or_expr(self):
        buf = [self.OR_EXPR]
        while 1:
            buf.append(self.get_and_expr())
            if not self.done() and \
               self.check_space() and self.check('|', 1):
                self.pop() # space
                self.pop() # operator
                self.match_space()
            else:
                break
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_or_expr', buf)
        return buf

    def get_and_expr(self):
        buf = [self.AND_EXPR]
        while 1:
            buf.append(self.get_cmp_expr())
            if not self.done() and \
               self.check_space() and self.check('&', 1):
                self.pop() # space
                self.pop() # operator
                self.match_space()
            else:
                break
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_and_expr', buf)
        return buf

    def get_cmp_expr(self):
        buf = [self.CMP_EXPR]
        buf.append(self.get_add_expr())
        if not self.done() and \
           self.check_space() and \
           self.look_ahead(1) in ['<=', '>=', '<', '>', '=', '==', '!=']:
            self.pop() # space
            buf.append(self.pop()) # operator
            self.match_space()
            buf.append(self.get_add_expr())
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_cmp_expr', buf)
        return buf

    def get_add_expr(self):
        buf = [self.ADD_EXPR]
        while 1:
            buf.append(self.get_mul_expr())
            if not self.done() and \
               self.check_space() and self.look_ahead(1) in ['+', '-']:
                self.pop() # space
                buf.append(self.pop()) # operator
                self.match_space()
            else:
                break
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_add_expr', buf)
        return buf

    def get_mul_expr(self):
        buf = [self.MUL_EXPR]
        while 1:
            buf.append(self.get_mat_expr())
            if not self.done() and \
               self.check_space() and self.look_ahead(1) in ['*', '/', '%']:
                self.pop() # space
                buf.append(self.pop()) # operator
                self.match_space()
            else:
                break
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_mul_expr', buf)
        return buf

    def get_mat_expr(self):
        buf = [self.STR_EXPR]
        buf.append(self.get_str_expr())
        if not self.done() and \
           self.check_space() and self.check(':', 1):
            buf.insert(1, 'match')
            self.pop() # space
            self.pop() # ':'
            self.match_space()
            buf.append(self.get_str_expr())
        if len(buf) == 2:
            buf = buf[1]
        if self.debug & 2:
            self.show_progress('get_mat_expr', buf)
        return buf

    def get_str_expr(self):
        argc = 0
        if self.check('length'):
            argc = 1
        elif self.look_ahead() in ['match', 'index', 'find', 'findpos']:
            argc = 2
        elif self.check('substr'):
            argc = 3
        if argc > 0:
            buf = [self.STR_EXPR, self.pop()] # fuction
            for i in range(argc):
                self.match_space()
                buf.append(self.get_str_expr())
        elif self.check('quote'):
            buf = [self.LITERAL]
            self.pop()
            self.match_space()
            if self.re_token.match(self.look_ahead()):
                buf.append(self.pop())
            else:
                buf.extend(self.get_str_seq())
        else:
            buf = self.get_sub_expr()
        if self.debug & 2:
            self.show_progress('get_str_expr', buf)
        return buf

    def get_sub_expr(self):
        if self.check('('):
            self.pop()
            if self.check_space():
                self.pop()
            buf = self.get_or_expr()
            if self.check_space():
                self.pop()
            #begin specification bug work-around (3/3)
            self.tokens[0] = self.kawari.parse_all(self.tokens[0])
            #end
            self.match(')')
        else:
            buf = [self.LITERAL]
            buf.extend(self.get_str_seq())
        if self.debug & 2:
            self.show_progress('get_sub_expr', buf)
        return buf

    def get_str_seq(self):
        buf = []
        while not self.done() and \
              not self.re_token.match(self.look_ahead()):
            buf.append(self.pop())
        if not buf:
            raise ExprError
        return buf

# <<< EXPR SYNTAX >>>
# expr     := or-expr
# or-expr  := and-expr (sp or-op sp and-expr)*
# or-op    := '|'
# and-expr := cmp-expr (sp and-op sp cmp-expr)*
# and-op   := '&'
# cmp-expr := add-expr (sp cmp-op sp add-expr)?
# cmp-op   := <= | >= | < | > | == | = | !=
# add-expr := mul-expr (sp add-op sp mul-expr)*
# add-op   := '+' | '-'
# mul-expr := mat-expr (sp mul-op sp mat-expr)*
# mul-op   := '*' | '/' | '%'
# mat-expr := str-expr (sp ':' sp str-expr)?
# str-expr := 'quote' sp (OPERATOR | str-seq) |
#             'match' sp str-expr sp str-expr |
#             'index' sp str-expr sp str-expr |
#             'find' sp str-expr sp str-expr |
#             'findpos' sp str-expr sp str-expr |
#             'substr' sp str-expr sp str-expr sp str-expr |
#             'length' sp str-expr |
#             sub-expr
# sub-expr := '(' sp? or-expr sp? ')' | str-seq
# sp       := SPACE+ (white space)
# str-seq  := STRING+ (literal, "...", ${...}, and/or $(...))


###   API   ###

DICT_FILE, INI_FILE = range(2)

def list_dict(kawari_dir, saori_ini={}, debug=0):
    return scan_ini(kawari_dir, 'kawari.ini', saori_ini, debug)

def scan_ini(kawari_dir, filename, saori_ini, debug=0):
    buf = []
    ini_path = os.path.join(kawari_dir, filename)
    try:
        line_list = read_dict(ini_path)
    except IOError:
        line_list = []
    read_as_dict = 0
    for entry, value, position in line_list:
        if entry == 'dict':
            filename = value.replace('\\', '/').lower().encode('utf-8')
            path = os.path.join(kawari_dir, filename)
            try:
                __builtin__.open(path).read(64)
            except IOError, (errno, message):
                if debug & 4:
                    print 'kawari.py: read error:', path
                continue
            buf.append((DICT_FILE, path))
        elif entry == 'include':
            filename = value.replace('\\', '/').lower().encode('utf-8')
            buf.extend(scan_ini(kawari_dir, filename, saori_ini, debug))
        elif entry == 'set':
            read_as_dict = 1
        elif entry in ['randomseed', 'debug', 'security', 'set']:
            pass
        elif entry == 'saori':
            saori_list = value.split(',')
            path = saori_list[0].strip().encode('utf-8')
            alias = saori_list[1].strip()
            if len(saori_list) == 3:
                option = saori_list[2].strip()
            else:
                option = 'loadoncall'
            saori_ini[alias] = [path, option]
        elif debug & 4:
            print 'kawari.py: unknown entry:', entry
    if read_as_dict:
        buf.append((INI_FILE, ini_path))
    return buf

def read_ini(path):
    buf = []
    try:
        line_list = read_dict(path)
    except IOError:
        line_list = []
    for entry, value, position in line_list:
        if entry == 'set':
            try:
                entry, value = value.split(None, 1)
            except ValueError:
                continue
            buf.append((entry.strip(), value.strip(), position))
    return buf


class Shiori(Kawari):

    def __init__(self, dll_name, debug=0):
        self.debug = debug
        self.dll_name = dll_name
        self.saori_list = {}
        self.kis_commands['saoriregist'] = self.exec_saoriregist
        self.kis_commands['saorierase'] = self.exec_saorierase
        self.kis_commands['callsaori'] = self.exec_callsaori
        self.kis_commands['callsaorix'] = self.exec_callsaorix

    def use_saori(self, saori):
        self.saori = saori

    def load(self, kawari_dir):
        self.kawari_dir = kawari_dir
        pathlist = [None]
        rdictlist = [{}]
        kdictlist = [{}]
        self.saori_ini = {}
        for file_type, path in list_dict(kawari_dir, self.saori_ini, self.debug):
            pathlist.append(path)
            if file_type == INI_FILE:
                rdict, kdict = create_dict(read_ini(path), self.debug)
            elif is_local_script(path):
                rdict, kdict = read_local_script(path)
            else:
                rdict, kdict = create_dict(read_dict(path, self.debug),
                                           self.debug)
            rdictlist.append(rdict)
            kdictlist.append(kdict)
        Kawari.__init__(self, kawari_dir, pathlist, rdictlist, kdictlist,
                        self.debug)
        for value in self.saori_ini.itervalues():
            if value[1] == 'preload':
                head, tail = os.path.split(value[0].replace('\\', '/'))
                self.saori_load(value[0], os.path.join(self.kawari_dir, head))
        return 1

    def unload(self):
        Kawari.finalize(self)
        for name in self.saori_list.keys():
            self.saori_list[name].unload()
            del self.saori_list[name]
        global charset
        charset = 'Shift_JIS' # reset

    def find(self, dir, dll_name):
        result = 0
        if list_dict(dir):
            result = 200
        global charset
        charset = 'Shift_JIS' # reset
        return result

    def show_description(self):
        sys.stdout.write(
            'Shiori: KAWARI compatible module for ninix\n'
            '        Copyright (C) 2001, 2002 by Tamito KAJIYAMA\n'
            '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
            '        Copyright (C) 2002-2009 by Shyouzou Sugitani\n'
            '        Copyright (C) 2003 by Shun-ichi TAHARA\n')

    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        req_header = {}
        line = header.readline()
        if line:
            line = line.strip()
            req_list = line.split()
            if len(req_list) >= 2:
                command = req_list[0].strip()
                protocol = req_list[1].strip()
            for line in header:
                line = line.strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                try:
                    value = int(value)
                except:
                    value = str(value)
                req_header[key] = value
        result = ''
        to = None
        if 'ID' in req_header:
            if req_header['ID'] == 'charset':
                result = charset
            elif req_header['ID'] == 'dms':
                result = self.getdms()
            elif req_header['ID'] == 'OnAITalk':
                result = self.getaistringrandom()
            elif req_header['ID'] in ['\\ms', '\\mz', '\\ml', '\\mc', '\\mh', \
                                      '\\mt', '\\me', '\\mp']:
                result = self.getword(req_header['ID'][1:])
            elif req_header['ID'] == '\\m?':
                result = self.getword('m')
            elif req_header['ID'] == 'otherghostname':
                otherghost = []
                for n in range(128):
                    key = ''.join(('Reference', str(n)))
                    if key in req_header:
                        otherghost.append(req_header[key])
                result = self.otherghostname(otherghost)
            elif req_header['ID'] == 'OnTeach':
                if 'Reference0' in req_header:
                    self.teach(req_header['Reference0'])
            else:
                result = self.getstring(req_header['ID'])
                if not result:
                    ref = []
                    for n in range(8):
                        key = ''.join(('Reference', str(n)))
                        if key in req_header:
                            ref.append(req_header[key])
                        else:
                            ref.append(None)
                    ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7 = ref
                    result = self.get_event_response(
                        req_header['ID'], ref0, ref1, ref2, ref3, ref4,
                        ref5, ref6, ref7)
            if result is None:
                result = ''
            to = self.communicate_to()
        result = 'SHIORI/3.0 200 OK\r\n' \
                 'Sender: Kawari\r\n' \
                 'Value: %s\r\n' % result
        if to is not None:
            result = ''.join((result, 'Reference0: %s\r\n' % to))
        result = ''.join((result, '\r\n'))
        return result            

    def exec_saoriregist(self, kawari, argv):
        filename = self.expand(argv[1])
        alias = self.expand(argv[2])
        if len(argv) == 4:
            option = self.expand(argv[3])
        else:
            option = 'loadoncall'
        self.saori_ini[alias] = [filename, option]
        if self.saori_ini[alias][1] == 'preload':
            head, tail = os.path.split(
                self.saori_ini[alias][0].replace('\\', '/'))
            self.saori_load(self.saori_ini[alias][0],
                            os.path.join(self.kawari_dir, head))
        return ''

    def exec_saorierase(self, kawari, argv):
        alias = self.expand(argv[1])
        if alias in self.saori_ini:
            self.saori_unload(self.saori_ini[alias][0])
        return ''

    def exec_callsaori(self, kawari, argv):
        alias = self.expand(argv[1])
        if alias not in self.saori_ini:
            return ''
        if self.saori_ini[alias][0] not in self.saori_list:
            if self.saori_ini[alias][1] == 'preload':
                return ''
            else:
                head, tail = os.path.split(
                    self.saori_ini[alias][0].replace('\\', '/'))
                self.saori_load(self.saori_ini[alias][0],
                                os.path.join(self.kawari_dir, head))
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: KAWARI\r\n' \
              'SecurityLevel: local\r\n' \
              'Charset: %s\r\n' % charset
        for i in range(2, len(argv)):
            req = ''.join((req,
                           'Argument%s: %s\r\n' % (i - 2, self.expand(argv[i]))))
        req = ''.join((req, '\r\n'))
        response = self.saori_request(self.saori_ini[alias][0],
                                      req.encode(charset, 'ignore'))
        header = StringIO.StringIO(response)
        line = header.readline()
        if line:
            line = line.strip()
            if ' ' in line:
                saori_protocol, saori_statuscode = [x.strip() for x in line.split(' ', 1)]
            for line in header:
                line = line.strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                if key:
                    saori_header.append(key)
                    saori_value[key] = value
        if 'Result' in saori_value:
            result = saori_value['Result']
        else:
            result =  ''
        if self.saori_ini[alias][1] == 'noresident':
            self.saori_unload(self.saori_ini[alias][0])
        return result

    def exec_callsaorix(self, kawari, argv):
        alias = self.expand(argv[1])
        entry = self.expand(argv[2])
        if alias not in self.saori_ini:
            return ''
        if self.saori_ini[alias][0] not in self.saori_list:
            if self.saori_ini[alias][1] == 'preload':
                return ''
            else:
                head, tail = os.path.split(
                    self.saori_ini[alias][0].replace('\\', '/'))
                self.saori_load(self.saori_ini[alias][0],
                                os.path.join(self.kawari_dir, head))
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: KAWARI\r\n' \
              'SecurityLevel: local\r\n'
        for i in range(3, len(argv)):
            req = ''.join((req,
                           'Argument%s: %s\r\n' % (i - 3, self.expand(argv[i]))))
        req = ''.join((req, '\r\n'))
        response = self.saori_request(self.saori_ini[alias][0],
                                      req.encode(charset, 'ignore'))
        header = StringIO.StringIO(response)
        line = header.readline()
        if line:
            line = line.strip()
            if ' ' in line:
                saori_protocol, saori_statuscode = [x.strip() for x in line.split(' ', 1)]
            for line in header:
                line = line.strip()
                if not line:
                    continue
                if ':' not in line:
                    continue
                key, value = [x.strip() for x in line.split(':', 1)]
                if key:
                    saori_header.append(key)
                    saori_value[key] = value
        result = {}
        for key, value in saori_value.iteritems():
            if key.startswith('Value'):
                result[key] = value
        for key, value in result.iteritems():
            self.set(''.join((entry, '.', key)), value)
        if self.saori_ini[alias][1] == 'noresident':
            self.saori_unload(self.saori_ini[alias][0])
        return len(result)

    def saori_load(self, saori, path):
        result = 0
        if saori in self.saori_list.keys():
            result = self.saori_list[saori].load(path)
        else:
            module = self.saori.request(saori)
            if module:
                self.saori_list[saori] = module
                result = self.saori_list[saori].load(path)
        return result

    def saori_unload(self, saori):
        result = 0
        if saori in self.saori_list.keys():
            result = self.saori_list[saori].unload()
        return result

    def saori_request(self, saori, req):
        result = 'SAORI/1.0 500 Internal Server Error'
        if saori in self.saori_list:
            result = self.saori_list[saori].request(req)
        return result


def open(kawari_dir, debug=0):
    pathlist = [None]
    rdictlist = [{}]
    kdictlist = [{}]
    for file_type, path in list_dict(kawari_dir, debug=debug):
        pathlist.append(path)
        if file_type == INI_FILE:
            rdict, kdict = create_dict(read_ini(path), debug)
        elif is_local_script(path):
            rdict, kdict = read_local_script(path)
        else:
            rdict, kdict = create_dict(read_dict(path, debug), debug)
        rdictlist.append(rdict)
        kdictlist.append(kdict)
    return Kawari(kawari_dir, pathlist, rdictlist, kdictlist, debug)

def is_local_script(path):
    line = __builtin__.open(path).readline()
    return line.startswith('[SAKURA]')

###   TEST   ###

def test():
    import getopt
    try:
        options, argv = getopt.getopt(sys.argv[1:], 'd:', ['debug='])
    except getopt.error, e:
        raise SystemExit, ''.join((str(e), '\n'))
    if argv:
        kawari_dir = argv[0]
    else:
        kawari_dir = os.curdir
    debug = 4 + 8 + 16 + 32 + 64 + 128
    for opt, val in options:
        if opt in ['-d', '--debug']:
            debug = int(val)
    print 'reading kawari.ini...'
    kawari = open(kawari_dir, debug)
    for rdict in kawari.rdictlist:
        for k, v in rdict.iteritems():
            print k.encode('UTF-8', 'ignore')
            for p in v:
                print ''.join(('\t', ''.join(p).encode('UTF-8', 'ignore')))
    for kdict in kawari.kdictlist:
        for k, v in kdict.iteritems():
            print ''.join(
                ('[ "', '", "'.join(k).encode('UTF-8', 'ignore'), '" ]'))
            for p in v:
                print ''.join(('\t', ''.join(p).encode('UTF-8', 'ignore')))
    while 1:
        print '=' * 40
        s = kawari.getaistringrandom()
        print '-' * 40
        print unicode(s, charset, 'ignore').encode('UTF-8', 'ignore')
        try:
            raw_input()
        except (EOFError, KeyboardInterrupt):
            break

if __name__ == '__main__':
    test()
