#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import optparse
import urlparse
import urllib
import os
import signal
import socket
import sys
import shutil
import random
import gettext
import traceback
import cStringIO
import fcntl

gettext.install('ninix')

if 'DISPLAY' in os.environ:
    if 'gtk' not in sys.modules:
        try:
            import pygtk
            pygtk.require('2.0')
        except ImportError:
            pass
    import gtk
    import glib
    import pango
    import ninix.pix

import ninix.home
import ninix.prefs
import ninix.sakura
import ninix.sstp
import ninix.communicate
import ninix.ngm
import ninix.plugin

import ninix.nekodorif
import ninix.kinoko

USAGE = 'Usage: ninix [options]'
parser = optparse.OptionParser(USAGE)
parser.add_option('-H', '--homedir', type='string', dest='homedir',
                  help='ninix home directory (default: ~/.ninix)')
parser.add_option('--sstp-port', type='int', dest='sstp_port',
                  help='additional port for listening SSTP requests')
parser.add_option('--debug', type='int', dest='debug', help='debug')

def _traceback(exception_type, value, tb):
    response_id = 1
    dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_NONE,
                               _('A ninix-aya error has been detected.'))
    dialog.set_title(_('Bug Detected'))
    dialog.set_position(gtk.WIN_POS_CENTER)
    dialog.set_gravity(gtk.gdk.GRAVITY_CENTER)
    button = dialog.add_button(_('Show Details'), response_id)
    dialog.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)
    textview = gtk.TextView()
    textview.set_editable(False)
    left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
    width = scrn_w / 2
    height = scrn_h / 4
    textview.set_size_request(width, height)
    textview.show()
    sw = gtk.ScrolledWindow()
    sw.show()
    sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.add(textview)
    frame = gtk.Frame()
    frame.set_shadow_type(gtk.SHADOW_IN)
    frame.add(sw)
    frame.set_border_width(7)
    dialog.vbox.add(frame)
    stringio = cStringIO.StringIO()
    traceback.print_exception(exception_type, value, tb, None, stringio)
    textbuffer = textview.get_buffer()
    textbuffer.set_text(stringio.getvalue())
    while 1:
        if dialog.run() == response_id:
            frame.show()
            button.set_sensitive(0)
        else: # close button
            break
    dialog.destroy()
    raise SystemExit

if 'DISPLAY' in os.environ:
    if 'gtk' in sys.modules:
        sys.excepthook = _traceback

def main():
    if gtk.pygtk_version < (2,10,0):
        print "PyGtk 2.10.0 or later required"
        raise SystemExit
    # reopen stderr to avoid IOError
    sys.stderr = os.fdopen(sys.stderr.fileno(), 'w')
    # check if X is running
    if 'DISPLAY' not in os.environ:
        raise SystemExit, 'Error: cannot open display (abort)\n'
    # parse command line arguments
    (options, rest) = parser.parse_args()
    if rest:
        parser.error('Unknown option(s)')
    sstp_port = [9801, 11000]
    # check environment variables
    val = os.environ.get('NINIX_SSTP_PORT')
    if val:
        port = []
        for num in val.split(','):
            try:
                num = int(num)
                if num < 1024:
                    raise ValueError
                port.append(num)
            except ValueError:
                sys.stderr.write('Invalid NINIX_SSTP_PORT number (ignored)\n')
                break
        else:
            sstp_port.extend(port)
    # parse command line options
    home_dir = options.homedir if options.homedir is not None else \
        os.environ.get('NINIX_HOME')
    if options.sstp_port is not None:
        if options.sstp_port < 1024:
            sys.stderr.write('Invalid --sstp-port number (ignored)\n')
        else:
            sstp_port.append(options.sstp_port)
    debug = options.debug if options.debug is not None else 0
    if home_dir:
        if not os.path.isabs(os.path.expanduser(home_dir)):
            home_dir = os.path.join(os.getcwd(), home_dir)
        ninix.home.NINIX_HOME = home_dir
    home_dir = ninix.home.get_ninix_home()
    if not os.path.exists(home_dir):
        try:
            os.makedirs(home_dir)
        except:
            raise SystemExit, 'Cannot create Home directory (abort)\n'
    # aquire Inter Process Mutex (not Global Mutex)
    lockfile = open(os.path.join(ninix.home.get_ninix_home(), '.lock'), 'w')
    try:
        fcntl.flock(lockfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except:
        raise SystemExit, 'ninix-aya is already running'
    # start
    sys.stdout.write('loading...')
    sys.stdout.flush()
    app = Application(sstp_port, debug)
    sys.stdout.write('done.\n')
    app.run()
    try:
        fcntl.flock(lockfile, fcntl.LOCK_UN)
    except:
        pass
    lockfile.close()


class SSTPControler:

    def __init__(self, callback, sstp_port):
        self.callback = callback
        self.sstp_port = sstp_port
        self.sstp_servers = []
        self.__sstp_queue = []
        self.__sstp_flag = 0
        self.__current_sender = None

    def enqueue_script_if_ghost(self, if_ghost, script, sender, handle,
                                address, show_sstp_marker,
                                use_translator, entry_db):
        self.__sstp_queue.append(
            (if_ghost, script, sender, handle, address, show_sstp_marker,
             use_translator, entry_db))

    def check_request_queue(self, sender):
        count = 0
        for request in self.__sstp_queue:
            if request[2].split(' / ')[0] == sender.split(' / ')[0]:
                count += 1
        if self.__sstp_flag and \
           self.__current_sender.split(' / ')[0] == sender.split(' / ')[0]:
            count += 1
        return str(count), str(len(self.__sstp_queue))

    def set_sstp_flag(self, sender):
        self.__sstp_flag = 1
        self.__current_sender = sender

    def reset_sstp_flag(self):
        self.__sstp_flag = 0
        self.__current_sender = None        

    def handle_sstp_queue(self):
        if self.__sstp_flag or not self.__sstp_queue:
            return
        if_ghost, script, sender, handle, address, show_sstp_marker, \
                  use_translator, entry_db = self.__sstp_queue.pop(0)
        if not self.callback['if_ghost'](if_ghost) and \
                not self.callback['get_preference']('allowembryo'): # Refuse
            return
        self.callback['select_current_sakura'](if_ghost)
        self.set_sstp_flag(sender)
        self.callback['enqueue_script'](
            script, sender, handle, address,
            show_sstp_marker, use_translator, entry_db, temp_mode=True)

    def receive_sstp_request(self):
        try:
            for sstp_server in self.sstp_servers:
                sstp_server.handle_request()
        except socket.error, (code, message):
            sys.stderr.write('socket.error: %s (%d)\n' % (message, code))
        except ValueError: # may happen when ninix is terminated
            return

    def get_sstp_port(self):
        if not self.sstp_servers:
            return None
        return self.sstp_servers[0].server_address[1]

    def quit(self):
        for server in self.sstp_servers:
            server.close()

    def start_servers(self):
        for port in self.sstp_port:
            sstp_callback = {
                'enqueue_script_if_ghost': self.enqueue_script_if_ghost,
                'check_request_queue': self.check_request_queue,
                }
            for key in ['get_sakura_cantalk', 'get_event_response',
                        'if_ghost', 'keep_silence',
                        'get_ghost_names', 'get_ghost_name',
                        'enqueue_event', 'enqueue_script']:
                sstp_callback[key] = self.callback[key]
            try:
                server = ninix.sstp.SSTPServer(('', port), sstp_callback)
            except socket.error, (code, message):
                sys.stderr.write('Port %d: %s (ignored)\n' % (port, message))
                continue
            self.sstp_servers.append(server)
            print 'Serving SSTP on port', port


class Application(object):

    def __init__(self, sstp_port=[9801, 11000], debug=0):
        self.loaded = False
        self.confirmed = False
        self.console = Console(self)
        self.debug = debug
        # create preference dialog
        prefs_callback = {
            'notify_preference_changed': self.notify_preference_changed,
            'get_balloon_list': self.get_balloon_list,
            }
        self.prefs = ninix.prefs.PreferenceDialog(prefs_callback)
        sstp_controler_callback = {
            'get_sakura_cantalk': self.get_sakura_cantalk,
            'get_event_response': self.get_event_response,
            'if_ghost': self.if_ghost,
            'keep_silence': self.keep_silence,
            'get_ghost_names': self.get_ghost_names,
            'get_ghost_name': self.get_ghost_name,
            'enqueue_event': self.enqueue_event,
            'enqueue_script': self.enqueue_script,
            'select_current_sakura': self.select_current_sakura,
            'get_preference': self.prefs.get,
            }
        self.sstp_controler = SSTPControler(sstp_controler_callback, sstp_port)
        plugin_callback = {
            'get_sstp_port': self.sstp_controler.get_sstp_port,
            }
        self.plugin_controler = ninix.plugin.PluginControler(plugin_callback)
        # create usage dialog
        self.usage_dialog = UsageDialog()
        self.communicate = ninix.communicate.Communicate()
        # create ghost manager
        ngm_callback = {
            'update_sakura': self.update_sakura,
            }
        self.__ngm = ninix.ngm.NGM(ngm_callback)
        self._current_sakura = None

    def _set_current_sakura(self, item):
        self._current_sakura = item
        self.prefs.set_current_sakura(*self.get_current_sakura_name())

    def _get_current_sakura(self):
        return self._current_sakura

    current_sakura = property(_get_current_sakura, _set_current_sakura)

    def create_ghost(self, data, debug):
        default_path = os.path.join(os.environ['PYTHONPATH'], 'ninix/dll')
        sakura_callback = {
            'change_sakura': self.change_sakura,
            'close_all': self.close_all_ghosts,
            'close_ghost': self.close_ghost,
            'edit_preferences': self.prefs.edit_preferences,
            'find_balloon_by_name': self.find_balloon_by_name,
            'find_balloon_by_subdir': self.find_balloon_by_subdir,
            'get_preference': self.prefs.get,
            'get_balloon_list': self.get_balloon_list,
            'get_balloon_description': self.get_balloon_description,
            'get_ghost_list': self.get_ghost_list,
            'get_kinoko_list': self.get_kinoko_list,
            'get_nekodorif_list': self.get_nekodorif_list,
            'get_plugin_list': self.get_plugin_list,
            'get_otherghostname': self.communicate.get_otherghostname,
            'open_ghost_manager': self.open_ghost_manager,
            'rebuild_ghostdb': self.communicate.rebuild_ghostdb,
            'reset_sstp_flag': self.sstp_controler.reset_sstp_flag,
            'show_usage': self.show_usage,
            'select_plugin': self.select_plugin,
            'select_kinoko': self.select_kinoko,
            'select_nekodorif': self.select_nekodorif,
            'select_ghost': self.select_ghost,
            'select_ghost_by_name': self.select_ghost_by_name,
            'start_sakura': self.start_sakura_cb,
            'stop_sakura': self.stop_sakura,
            'reload_current_sakura': self.reload_current_sakura,
            'select_current_sakura': self.select_current_sakura,
            'send_message': self.communicate.send_message,
            'update_sakura': self.update_sakura,
            'vanish_sakura': self.vanish_sakura,
            }
        return ninix.sakura.Sakura(data, default_path, sakura_callback, debug)

    def get_sakura_cantalk(self):
        sakura = self.get_current_sakura_instance()
        return sakura.cantalk

    def get_event_response(self, event, *arglist, **argdict):
        sakura = self.get_current_sakura_instance()
        return sakura.get_event_response(*event)

    def keep_silence(self, quiet):
        sakura = self.get_current_sakura_instance()
        sakura.keep_silence(quiet)

    def get_ghost_name(self):
        sakura = self.get_current_sakura_instance()
        return '%s,%s' % (sakura.get_selfname(), sakura.get_keroname())

    def enqueue_event(self, event, *arglist, **argdict):
        sakura = self.get_current_sakura_instance()
        sakura.enqueue_event(event, *arglist, **argdict)

    def enqueue_script(self, script, sender, handle,
                       host, show_sstp_marker, use_translator,
                       db=None, request_handler=None, temp_mode=False):
        sakura = self.get_current_sakura_instance()
        if temp_mode:
            sakura.enter_temp_mode()
        sakura.enqueue_script(script, sender, handle,
                              host, show_sstp_marker, use_translator,
                              db, request_handler)

    def get_working_ghost(self, cantalk=0):
        working_list = [item['instance'] for item in self.ghost_list \
                        if item is not None and \
                           item['instance'] is not None and \
                           item['instance'].is_running()]
        if cantalk:
            working_list = [sakura for sakura in working_list if sakura.cantalk]
        return working_list

    def get_ghost_list(self):
        return self.ghost_list

    def range_ghosts(self):
        return [i for i, sakura in enumerate(self.ghosts) if sakura is not None]

    def __create_ghost_list_item(self, i): ## FIXME: id
        if i >= len(self.ghosts):
            return None
        if self.ghosts[i] is None:
            return None
        set_type = 'g'
        desc = self.ghosts[i][0]
        shiori_dir = self.ghosts[i][1]
        icon = desc.get('icon', None)
        if icon is not None:
            icon_path = os.path.join(shiori_dir, icon)
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        name = desc.get('name',
                        ''.join((unicode(_('Ghost'), 'utf-8'),
                                 '#%d' % (i + 1))))
        shell_list = []
        surface_set = self.ghosts[i][3]
        for j in range(len(surface_set)):
            value = (set_type, i, j)
            shell_name = surface_set[j][0]
            shell_list.append((shell_name, value))
        item = {}
        item['name'] = name
        item['icon'] = icon_path
        item['shell'] = shell_list
        item['instance'] = None
        return item

    def update_ghost_list(self, i, sakura):
        self.ghost_list[i] = self.__create_ghost_list_item(i)
        if self.ghost_list[i] is not None:
            self.ghost_list[i]['instance'] = sakura
        else:
            pass ## FIXME: ghost instance should be deleted

    def create_ghost_list(self): ## FIXME
        ghost_list = []
        for i in range(len(self.ghosts)):
            item = self.__create_ghost_list_item(i)
            ghost_list.append(item)
        self.ghost_list = ghost_list

    def get_balloon_list(self):
        balloon_list = []
        balloons = ninix.home.search_balloons()
        for desc, balloon in balloons:
            subdir = balloon['balloon_dir'][0]
            name = desc.get('name', subdir)
            balloon_list.append((name, subdir))
        return balloon_list

    def get_plugin_list(self): ## FIXME
        return self.plugin_controler.get_plugin_list()

    def get_nekodorif_list(self):
        nekodorif_list = []
        nekoninni = ninix.home.search_nekoninni()
        for nekoninni_name, nekoninni_dir in nekoninni:
            if not nekoninni_name:
                continue
            item = {}
            item['name'] = nekoninni_name
            item['dir'] = nekoninni_dir
            nekodorif_list.append(item)
        return nekodorif_list

    def get_kinoko_list(self):
        self.kinoko = ninix.home.search_kinoko()
        return self.kinoko

    def load(self):
        self.create_ghost_list()
        # load user preferences
        self.prefs.load()
        # choose default ghost/shell
        name = self.prefs.get('sakura_name')
        surface = self.prefs.get('sakura_surface', 0)
        default_sakura = self.find_ghost_by_name(name, surface) or \
                         self.choose_default_sakura()
        # load ghost
        self.current_sakura = default_sakura
        for i in range(len(self.ghosts)): ## FIXME
            sakura = self.create_ghost(self.ghosts[i], self.debug)
            self.ghost_list[i]['instance'] = sakura
        ##names = self.get_ghost_names()
        ##for i in range(len(names)):
        ##    print 'GHOST(%d): %s' % (i, names[i].encode('utf-8', 'ignore'))
        self.start_sakura(self.current_sakura, init=1)

    def find_ghost_by_name(self, name, surface):
        for i in range(len(self.ghosts)): ## FIXME
            if self.ghosts[i] is None:
                continue
            desc = self.ghosts[i][0]
            try:
                if desc.get('name') == name:
                    surface_set = self.ghosts[i][3]
                    if not surface_set or surface < len(surface_set):
                        return ('g', i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None

    def choose_default_sakura(self):
        for i in range(len(self.ghosts)): ## FIXME
            if self.ghosts[i] is None:
                continue
            if self.ghosts[i][3]: # surface_set
                return ('g', i, 0)
        return ('s', 0, 0)

    def find_balloon_by_name(self, name):
        balloons = ninix.home.search_balloons()
        for desc, balloon in balloons:
            try:
                if desc.get('name') == name:
                    return desc, balloon
                if balloon['balloon_dir'][0] == name.encode('utf-8').lower(): # XXX
                    return desc, balloon
            except: # old preferences(EUC-JP)
                pass
        return None

    def find_balloon_by_subdir(self, subdir):
        balloons = ninix.home.search_balloons()
        for desc, balloon in balloons:
            try:
                if balloon['balloon_dir'][0] == subdir:
                    return desc, balloon
                if desc.get('name').encode('utf-8').lower == subdir: # XXX
                    return desc, balloon
            except: # old preferences(EUC-JP)
                pass
        return None

    def run(self): ## FIXME
#        self.load()
#        # set SIGCHLD handler
#        signal.signal(signal.SIGCHLD, self.plugin_controler.terminate_plugin)
#        # start SSTP server
#        self.sstp_controler.start_servers()
#        # start plugins
#        self.plugin_controler.start_plugins()
        self.set_timeout()
        gtk.main()

    def get_current_sakura_instance(self):
        return self.ghost_list[self.current_sakura[1]]['instance']

    def get_ghost_names(self): ## FIXME
        name_list = []
        for item in self.ghost_list: ## FIXME
            sakura = item['instance']
            name = sakura.get_selfname()
            name_list.append(name)
        return name_list

    def get_ghost(self, if_ghost):
        for item in self.ghost_list:
            if item is None:
                continue
            sakura = item['instance']
            assert sakura is not None
            if sakura.get_selfname() == if_ghost:
                return sakura
        else:
            return None

    def if_ghost(self, if_ghost):
        for item in self.ghost_list:
            if item is None:
                continue
            sakura = item['instance']
            assert sakura is not None
            if sakura.get_selfname() == if_ghost:
                return 1
        else:
            return 0

    def update_sakura(self, name, sender):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        sakura = self.select_ghost_from_list(item)
        if not sakura.is_running():
            self.start_sakura(item, init=1)
        sakura.enqueue_script('\![updatebymyself]\e', sender,
                              None, None, 0, 0, None)

    def select_current_sakura(self, ifghost=None):
        if ifghost is not None:
            for item in self.ghost_list:
                if item is None:
                    continue
                sakura = item['instance']
                assert sakura is not None
                names = '%s,%s' % (sakura.get_selfname(),
                                   sakura.get_keroname())
                name =  '%s' % sakura.get_selfname()
                if ifghost in [name, names]:
                    if not sakura.is_running():
                        self.current_sakura = item['shell'][0][1] ## FIXME
                        self.start_sakura(self.current_sakura, init=1, temp=1) ## FIXME
                    else:
                        self.current_sakura = sakura.current
                    break
                else:
                    pass
            else:
                return
        else:
            working_list = self.get_working_ghost(cantalk=1)
            if working_list:
                self.current_sakura = random.choice(working_list).current
            else:
                return ## FIXME

    def close_ghost(self, sakura):
        if not self.get_working_ghost():
            self.quit()
        elif self.current_sakura == sakura.current:
            self.select_current_sakura()

    def close_all_ghosts(self):
        for sakura in self.get_working_ghost():
            sakura.close()

    def quit(self):
        glib.source_remove(self.timeout_id)
        self.usage_dialog.close()
        self.sstp_controler.quit() ## FIXME
        self.save_preferences()
        gtk.main_quit()
        # stop plugins
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        os.kill(0, signal.SIGHUP)

    def get_current_sakura_name(self): ## FIXME
        # last ghost
        set_type, i, j = self.current_sakura
        assert set_type == 'g'
        desc = self.ghosts[i][0]
        name = desc.get('name', '')
        return name, str(j) # name, surface

    def save_preferences(self):
        try:
            self.prefs.save()
        except IOError:
            sys.stderr.write('Cannot write preferences to file (ignored).\n')
        except:
            pass ## FIXME

    def select_ghost(self, sakura, sequential, event=1, vanished=0):
        if len(self.range_ghosts()) < 2:
            return
        # select another ghost
        set_type, i, j = sakura.current
        if set_type == 's':
            i = 0
        if sequential: ## FIXME
            i += 1
            if i == len(self.ghosts): ## FIXME
                i = 0
        else:
            index_list = self.range_ghosts()
            index_list.remove(i)
            i = random.choice(index_list)
        if self.current_sakura == sakura.current: # XXX
            self.current_sakura = ('g', i, 0)
        self.change_sakura(sakura, ('g', i, 0), 'automatic', event, vanished)

    def select_ghost_by_name(self, sakura, name, event=1):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        self.change_sakura(sakura, item, 'automatic', event)

    def change_sakura(self, sakura, item, method, event=1, vanished=0):
        set_type, i, j = item
        assert self.ghosts[i] is not None
        if sakura.current == item: # XXX: needs reloading?
            return
        assert sakura.current[1] != i # XXX: not shell change
        assert set_type == 'g'
        desc, shiori_dir, use_makoto, surface_set, prefix, \
            shiori_dll, shiori_name = self.ghosts[i]
        assert surface_set and len(surface_set) > j
        name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips = \
            surface_set[j]
        def proc(self=self, item=item):
            self.stop_sakura(sakura, self.start_sakura, item, sakura.current)
        if vanished:
            sakura.finalize()
            self.start_sakura(item, sakura.current, vanished)
            self.close_ghost(sakura)
        elif not event:
            proc()
        else:
            name = surface_desc.get('sakura.name', desc.get('sakura.name'))
            sakura.enqueue_event('OnGhostChanging', name, method, proc=proc)

    def stop_sakura(self, sakura, starter=None, *args):
        sakura.finalize()
        if starter is not None:
            starter(*args)
        self.close_ghost(sakura)

    def select_ghost_from_list(self, item):
        set_type, i, j = item
        assert self.ghosts[i] is not None
        assert self.ghost_list[i]['instance'] is not None
        return self.ghost_list[i]['instance']

    def start_sakura(self, item, prev=None, vanished=0, init=0, temp=0):
        set_type, i, j = item
        sakura = self.select_ghost_from_list(item)
        if prev is not None:
            assert self.ghosts[prev[1]] is not None
            assert self.ghost_list[prev[1]]['instance'] is not None
        if init:
            ghost_changed = 0
        else:
            assert prev is not None ## FIXME
            if prev[0] == set_type and prev[1] == i:
                ghost_changed = 0
            else:
                ghost_changed = 1
        if ghost_changed:
            name = self.ghost_list[prev[1]]['instance'].get_selfname()
        else:
            name = None
        sakura.notify_preference_changed()
        sakura.start(item, init, temp, vanished, ghost_changed, name)

    def notify_preference_changed(self):
        for sakura in self.get_working_ghost():
            sakura.notify_preference_changed()

    def start_sakura_cb(self, event, item):
        self.start_sakura(item, init=1)

    def get_balloon_description(self, subdir):
        balloon = self.find_balloon_by_subdir(subdir)
        if balloon is None:
            ##print 'Balloon %s not found.' % name
            default_balloon = self.prefs.get('default_balloon')
            balloon = self.find_balloon_by_subdir(default_balloon)
        if balloon is None:
            balloons = ninix.home.search_balloons()
            assert balloons
            balloon = balloons[0]
        return balloon

    def reload_current_sakura(self, sakura):
        self.save_preferences()
        set_type, i, j = sakura.current
        if set_type == 's':
            i = 0
        ghost_dir = sakura.get_prefix()
        ghost_conf = ninix.home.search_ghosts([ghost_dir])
        if ghost_conf:
            self.ghosts[i] = ghost_conf[0]
            sakura.new(*ghost_conf[0]) # reset
        else:
            self.ghosts[i] = None ## FIXME
        self.update_ghost_list(i, sakura)
        item = (set_type, i, j)
        self.start_sakura(item, item, init=1) 

    def vanish_sakura(self, sakura):
        # remove ghost
        prefix = sakura.get_prefix()
        for filename in os.listdir(prefix):
            if os.path.isfile(os.path.join(prefix, filename)):
                if filename != 'HISTORY':
                    try:
                        os.remove(os.path.join(prefix, filename))
                    except:
                        print '*** REMOVE FAILED *** :', filename
            else: # dir
                try:
                    shutil.rmtree(os.path.join(prefix, filename))
                except:
                    print '*** REMOVE FAILED *** :', filename
        set_type, i, j = sakura.current
        self.select_ghost(sakura, 0, vanished=1)
        self.ghosts[i] = None
        self.update_ghost_list(i, sakura)

    def select_plugin(self, event, item): ## FIXME
        self.plugin_controler.select_plugin(event, item)

    def select_nekodorif(self, nekodorif_dir, target):
        ninix.nekodorif.Nekoninni().load(nekodorif_dir,
                                         ninix.home.search_katochan(), target)

    def select_kinoko(self, data, target):
        ninix.kinoko.Kinoko(self.kinoko).load(data, target)

    def open_ghost_manager(self):
        self.__ngm.show_dialog()

    def show_usage(self):
        for sakura in self.get_working_ghost():
            sakura.save_history()
        history = {}
        for i in range(len(self.ghosts)):
            if self.ghosts[i] is None:
                continue
            desc = self.ghosts[i][0]
            name = desc.get('name',
                            ''.join((unicode(_('Ghost'), 'utf-8'),
                                     '#%d' % (i + 1))))
            ghost_time = 0
            prefix = self.ghosts[i][5]
            path = os.path.join(prefix, 'HISTORY')
            if os.path.exists(path):
                try:
                    f = open(path, 'r')
                except IOError, (code, message):
                    sys.stderr.write('cannot read %s\n' % path)
                else:
                    for line in f:
                        if ',' not in line:
                            continue
                        key, value = [x.strip() for x in line.split(',', 1)]
                        if key == 'time':
                            try:
                                ghost_time = int(value)
                            except:
                                pass
                f.close()
            ai_list = []
            dirlist = os.listdir(os.path.join(prefix, 'shell'))
            for subdir in dirlist:
                path = os.path.join(prefix, 'shell', subdir, 'ai.png')
                if os.path.exists(path):
                    ai_list.append(path)
            history[name] = (ghost_time, ai_list)
        self.usage_dialog.open(history)

    def set_timeout(self):
        self.timeout_id = glib.timeout_add(100, self.do_idle_tasks) # 100ms

    def search_ghosts(self): ## FIXME
        self.ghosts = ninix.home.search_ghosts()
        balloons = self.get_balloon_list()
        if len(self.ghosts) > 0 and len(balloons) > 0:
            self.confirmed = True
        return len(self.ghosts), len(balloons)

    def do_idle_tasks(self):
        if not self.confirmed:
            self.console.open(exception=True)
        else:
            if not self.loaded:
                self.load()
                # set SIGCHLD handler
                signal.signal(signal.SIGCHLD, self.plugin_controler.terminate_plugin)
                # start SSTP server
                self.sstp_controler.start_servers()
                # start plugins
                self.plugin_controler.start_plugins()
                self.loaded = True
            else:
                self.sstp_controler.handle_sstp_queue()
                self.sstp_controler.receive_sstp_request()
        self.set_timeout()


class Console:

    # DnD data types
    dnd_targets = [
        ('text/plain', 0, 0),
        ]

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('drag_data_received', self.drag_data_received)
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.size = (330, 110) ## FIXME
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.opened = 0

    def open(self, exception=False): ## FIXME
        if self.opened:
            return
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        if not self.app.confirmed: ## FIXME
            self.app.quit()
        return True

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def draw_message(self, text): ## FIXME
        pass

    def redraw(self, darea, event):
        ghosts, balloons = self.app.search_ghosts() # XXX
        if ghosts > 0 and balloons > 0:
            self.window.set_title(_('Console'))
        else:
            self.window.set_title(_('Nanntokashitekudasai.'))
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        w, h = self.size
        cr.set_source_rgb(0.0, 0.0, 0.0) # black
        cr.paint()
        layout.set_text('Ghosts: %d' % ghosts)
        if ghosts == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15)
        cr.show_layout(layout)
        w, h = layout.get_pixel_size()
        layout.set_text('Balloons: %d' % balloons)
        if balloons == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15 + h)
        cr.show_layout(layout)
        del cr

    def drag_data_received(self, widget, context, x, y, data, info, time):
        print 'Content-type:', data.type
        print 'Content-length:', data.get_length()
        ##print repr(data.data)
        if str(data.type) == 'text/plain':
            filelist = []
            for line in data.data.split('\r\n'):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                pathname = urllib.url2pathname(path)
                if scheme == 'file' and os.path.exists(pathname):
                    filelist.append(pathname)
            for filename in filelist:
                os.system('ninix-install %s' % filename) ## FIXME
        self.darea.queue_draw()
        return True


class UsageDialog:

    def __init__(self):
        self.window = gtk.Dialog()
        self.window.set_title('Usage')
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.size = (550, 330)
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.opened = 0

    def open(self, history):
        if self.opened:
            return
        self.history = history
        self.items = []
        for name, (clock, path) in self.history.iteritems():
            self.items.append((name, clock, path))
        self.items[:] = [(x[1], x) for x in self.items]
        self.items.sort()
        self.items[:] = [x for x_1, x in self.items]
        self.items.reverse()
        ai_list = self.items[0][2]
        if ai_list:
            path = random.choice(ai_list)
            assert os.path.exists(path)
            self.pixbuf = ninix.pix.create_pixbuf_from_file(
                path, is_pnr=False)
            self.pixbuf.saturate_and_pixelate(self.pixbuf, 1.0, True)
        else:
            self.pixbuf = None
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        return True

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def redraw(self, darea, event):
        if not self.items:
            return # should not reach here
        total = float(0)
        for name, clock, path in self.items:
            total += clock
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        # redraw graph
        w, h = self.size
        cr.set_source_rgb(1.0, 1.0, 1.0) # white
        cr.paint()
        # ai.png
        if self.pixbuf:
            cr.set_source_pixbuf(self.pixbuf, 16, 32) # XXX
            cr.paint()
        w3 = w4 = 0
        rows = []
        for name, clock, path in self.items[:14]:
            layout.set_text(name)
            name_w, name_h = layout.get_pixel_size()
            rate = '%.1f%%' % (clock / total * 100)
            layout.set_text(rate)
            rate_w, rate_h = layout.get_pixel_size()
            w3 = max(rate_w, w3)
            time = '%d:%02d' % divmod(clock / 60, 60)
            layout.set_text(time)
            time_w, time_h = layout.get_pixel_size()
            w4 = max(time_w, w4)
            rows.append((clock, name, name_w, name_h, rate, rate_w, rate_h,
                         time, time_w, time_h))
        w1 = 280
        w2 = w - w1 - w3 - w4 - 70
        x = 20
        y = 15
        x += w1 + 10
        label = 'name'
        layout.set_text(label)
        label_name_w, label_name_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x, y)
        cr.show_layout(layout)
        x = x + w2 + 10
        label = 'rate'
        layout.set_text(label)
        label_rate_w, label_rate_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w3 - label_rate_w, y)
        cr.show_layout(layout)
        x += w3 + 10
        label = 'time'
        layout.set_text(label)
        label_time_w, label_time_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w4 - label_time_w, y)
        cr.show_layout(layout)
        y += max([label_name_h, label_rate_h, label_time_h]) + 4
        for clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, \
                time_h  in rows:
            x = 20
            bw = int(clock / total * w1)
            bh = max([name_h, rate_h, time_h]) - 1
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
            cr.rectangle(x + 1, y + 1, bw, bh)
            cr.stroke()
            cr.set_source_rgb(1.0, 1.0, 1.0) # white
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            x += w1 + 10
            layout.set_text(name)
            end = len(name)
            while end > 0:
                w, h = layout.get_pixel_size()
                if w > 168:
                    end -= 1
                    layout.set_text(''.join((name[:end], u'...')))
                else:
                    break
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x, y)
            cr.show_layout(layout)
            x += w2 + 10
            layout.set_text(rate)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w3 - rate_w, y)
            cr.show_layout(layout)
            x += w3 + 10
            layout.set_text(time)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w4 - time_w, y)
            cr.show_layout(layout)
            y += max([name_h, rate_h, time_h]) + 4
        del cr


if __name__ == '__main__':
    main()
