<!-- section: Why use this? -->
<details>
  <summary><strong>Why use this?</strong></summary>

<br>

Although it's easy to split on a string:

```js
console.log('a.b.c'.split('.'));
//=> ['a', 'b', 'c']
```

It's more challenging to split a string whilst respecting escaped or quoted characters.

**Bad**

```js
console.log('a\\.b.c'.split('.'));
//=> ['a\\', 'b', 'c']

console.log('"a.b.c".d'.split('.'));
//=> ['"a', 'b', 'c"', 'd']
```

**Good**

```js
var split = require('{%= name %}');
console.log(split('a\\.b.c'));
//=> ['a.b', 'c']

console.log(split('"a.b.c".d'));
//=> ['a.b.c', 'd']
```

See the [options](#options) to learn how to choose the separator or retain quotes or escaping.

<br>

</details>


## Usage

```js
var split = require('{%= name %}');

split('a.b.c');
//=> ['a', 'b', 'c']

// respects escaped characters
split('a.b.c\\.d');
//=> ['a', 'b', 'c.d']

// respects double-quoted strings
split('a."b.c.d".e');
//=> ['a', 'b.c.d', 'e']
```

**Brackets**

Also respects brackets [unless disabled](#optionsbrackets):

```js
split('a (b c d) e', ' ');
//=> ['a', '(b c d)', 'e']
```

## Options

### options.brackets

**Type**: `object|boolean` 

**Default**: `undefined`

**Description**

If enabled, split-string will not split inside brackets. The following brackets types are supported when `options.brackets` is `true`,

```js
{
  '<': '>',
  '(': ')',
  '[': ']',
  '{': '}'
}
```

Or, if object of brackets must be passed, each property on the object must be a bracket type, where the property key is the opening delimiter and property value is the closing delimiter.

**Examples**

```js
// no bracket support by default
split('a.{b.c}');
//=> [ 'a', '{b', 'c}' ]

// support all basic bracket types: "<>{}[]()"
split('a.{b.c}', {brackets: true});
//=> [ 'a', '{b.c}' ]

// also supports nested brackets 
split('a.{b.{c.d}.e}.f', {brackets: true});
//=> [ 'a', '{b.{c.d}.e}', 'f' ]

// support only the specified brackets
split('[a.b].(c.d)', {brackets: {'[': ']'}});
//=> [ '[a.b]', '(c', 'd)' ]
```

### options.sep

**Type**: `string`

**Default**: `.`

The separator/character to split on.

**Example**

```js
split('a.b,c', {sep: ','});
//=> ['a.b', 'c']

// you can also pass the separator as string as the last argument
split('a.b,c', ',');
//=> ['a.b', 'c']
```

### options.keepEscaping

**Type**: `boolean`

**Default**: `undefined`

Keep backslashes in the result.

**Example**

```js
split('a.b\\.c');
//=> ['a', 'b.c']

split('a.b.\\c', {keepEscaping: true});
//=> ['a', 'b\.c']
```

### options.keepQuotes

**Type**: `boolean`

**Default**: `undefined`

Keep single- or double-quotes in the result.

**Example**

```js
split('a."b.c.d".e');
//=> ['a', 'b.c.d', 'e']

split('a."b.c.d".e', {keepQuotes: true});
//=> ['a', '"b.c.d"', 'e']

split('a.\'b.c.d\'.e', {keepQuotes: true});
//=> ['a', '\'b.c.d\'', 'e']
```

### options.keepDoubleQuotes

**Type**: `boolean`

**Default**: `undefined`

Keep double-quotes in the result.

**Example**

```js
split('a."b.c.d".e');
//=> ['a', 'b.c.d', 'e']

split('a."b.c.d".e', {keepDoubleQuotes: true});
//=> ['a', '"b.c.d"', 'e']
```

### options.keepSingleQuotes

**Type**: `boolean`

**Default**: `undefined`

Keep single-quotes in the result.

**Example**

```js
split('a.\'b.c.d\'.e');
//=> ['a', 'b.c.d', 'e']

split('a.\'b.c.d\'.e', {keepSingleQuotes: true});
//=> ['a', '\'b.c.d\'', 'e']
```


## Customizer

**Type**: `function`

**Default**: `undefined`

Pass a function as the last argument to customize how tokens are added to the array.

**Example**

```js
var arr = split('a.b', function(tok) {
  if (tok.arr[tok.arr.length - 1] === 'a') {
    tok.split = false;
  }
});
console.log(arr);
//=> ['a.b']
```

**Properties**

The `tok` object has the following properties:

- `tok.val` (string) The current value about to be pushed onto the result array
- `tok.idx` (number) the current index in the string
- `tok.str` (string) the entire string
- `tok.arr` (array) the result array


## Release history

### v3.0.0 - 2017-06-17

**Added**

- adds support for brackets
