#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fstream>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <signal.h>
#include <errno.h>
#include <glibmm/fileutils.h>
#include <glibmm/miscutils.h>

#include <arc/ArcConfig.h>
#include <arc/IniConfig.h>
#include <arc/ArcLocation.h>
#include <arc/Logger.h>
#include <arc/XMLNode.h>
#include <arc/StringConv.h>
#include <arc/message/MCCLoader.h>

#include "daemon.h"
#include "../options.h"

static Arc::Daemon *main_daemon = NULL;
static Arc::Config config;
static Arc::MCCLoader *loader = NULL;
static Arc::Logger& logger = Arc::Logger::rootLogger;
static int exit_code = 0;
static bool req_shutdown = false;

static void sig_shutdown(int)
{
    if(req_shutdown) _exit(exit_code);
    req_shutdown = true;
}

static void do_shutdown(void)
{
    logger.msg(Arc::VERBOSE, "shutdown");
    if(loader) delete loader;
    if(main_daemon) delete main_daemon;
    logger.msg(Arc::DEBUG, "exit");
    _exit(exit_code);
}

static Arc::LogFile* sighup_dest = NULL;

static void sighup_handler(int) {
    int old_errno = errno;
    if(main_daemon) main_daemon->logreopen();
    if(sighup_dest) {
        sighup_dest->setReopen(true);
        sighup_dest->setReopen(false);
    }
    errno = old_errno;
}

static void glib_exception_handler()
{
  std::cerr << "Glib exception thrown" << std::endl;
  try {
    throw;
  }
  catch(const Glib::Error& err) {
    std::cerr << "Glib exception caught: " << err.what() << std::endl;
  }
  catch(const std::exception &err) {
    std::cerr << "Exception caught: " << err.what() << std::endl;
  }
  catch(...) {
    std::cerr << "Unknown exception caught" << std::endl;
  }
}

static void merge_options_and_config(Arc::Config& cfg, Arc::ServerOptions& opt)
{
    Arc::XMLNode srv = cfg["Server"];
    if (!(bool)srv) {
      logger.msg(Arc::ERROR, "No server config part of config file");
      return;
    }

    if (opt.pid_file != "") {
        if (!(bool)srv["PidFile"]) {
           srv.NewChild("PidFile")=opt.pid_file;
        } else {
            srv["PidFile"] = opt.pid_file;
        }
    }

    if (opt.foreground == true) {
        if (!(bool)srv["Foreground"]) {
            srv.NewChild("Foreground");
        }
    }

    if (opt.user != "") {
        if (!(bool)srv["User"]) {
            srv.NewChild("User") = opt.user;
        } else {
            srv["User"] = opt.user;
        }
    }

    if (opt.group != "") {
        if (!(bool)srv["Group"]) {
            srv.NewChild("Group") = opt.group;
        } else {
            srv["Group"] = opt.group;
        }
    }

    if (!opt.log_file.empty()) {
        if (!(bool)srv["Logger"]["File"]) {
            srv.NewChild("Logger").NewChild("File") = opt.log_file;
        } else {
            srv["Logger"]["File"] = opt.log_file;
        }
    }
}

static std::string init_logger(Arc::XMLNode log, bool foreground)
{
    /* setup root logger */
    Arc::LogFile* sd = NULL;
    Arc::XMLNode xlevel = log["Level"];

    Arc::Logger::rootLogger.setThreshold(Arc::WARNING);
    for(;(bool)xlevel;++xlevel) {
      std::string domain = xlevel.Attribute("Domain");
      Arc::LogLevel level = Arc::WARNING;
      if(!string_to_level((std::string)xlevel, level)) {
        logger.msg(Arc::WARNING, "Unknown log level %s", (std::string)xlevel);
      } else {
        Arc::Logger::setThresholdForDomain(level,domain);
      }
    }

    std::string log_file = (log["File"] ? (std::string)log["File"] : "/var/log/arc/arched.log");
    sd = new Arc::LogFile(log_file);
    if((!sd) || (!(*sd))) {
      logger.msg(Arc::ERROR, "Failed to open log file: %s", log_file);
      _exit(1);
    }
    if(log["Backups"]) {
      int backups;
      if(Arc::stringto((std::string)log["Backups"], backups)) {
	sd->setBackups(backups);
      }
    }
    if(log["Maxsize"]) {
      int maxsize;
      if(Arc::stringto((std::string)log["Maxsize"], maxsize)) {
	sd->setMaxSize(maxsize);
      }
    }
    bool reopen_b = false;
    if(log["Reopen"]) {
      std::string reopen = (std::string)(log["Reopen"]);
      if((reopen == "true") || (reopen == "1")) reopen_b = true;
      sd->setReopen(reopen_b);
    }
    if(!reopen_b) sighup_dest = sd;
    Arc::Logger::rootLogger.removeDestinations();
    Arc::Logger::rootLogger.addDestination(*sd);
    if (foreground) {
      logger.msg(Arc::INFO, "Start foreground");
      Arc::LogStream *err = new Arc::LogStream(std::cerr);
      Arc::Logger::rootLogger.addDestination(*err);
    }
    if(reopen_b) return "";
    return log_file;
}

static uid_t get_uid(const std::string &name)
{
    struct passwd *ent;
    if (name[0] == '#') {
        return (atoi(&(name.c_str()[1])));
    }
    if (!(ent = getpwnam(name.c_str()))) {
        std::cerr << "Bad user name" << std::endl;
        exit(1);
    }
    return (ent->pw_uid);
}

static gid_t get_gid(uid_t uid)
{
    struct passwd *ent;
    if (!(ent = getpwuid(uid))) {
        std::cerr << "Bad user id" << std::endl;
        exit(1);
    }
    return (ent->pw_gid);
}

static gid_t get_gid(const std::string &name)
{
    struct group *ent;
    if (name[0] == '#') {
        return (atoi(&(name.c_str()[1])));
    }
    if (!(ent = getgrnam(name.c_str()))) {
        std::cerr << "Bad user name" << std::endl;
        exit(1);
    }
    return (ent->gr_gid);
}

static void init_config(const Arc::ServerOptions &options)
{
    if (!options.xml_config_file.empty()) {
        if (Glib::file_test(options.xml_config_file,
            Glib::FILE_TEST_EXISTS) == false) {
            logger.msg(Arc::ERROR, "XML config file %s does not exist", options.xml_config_file);
            exit(1);
        }
        if(!config.parse(options.xml_config_file.c_str())) {
            logger.msg(Arc::ERROR, "Failed to load service configuration from file %s", options.xml_config_file);
            exit(1);
        }
    } else if (!options.ini_config_file.empty()) {
        if (Glib::file_test(options.ini_config_file,
            Glib::FILE_TEST_EXISTS) == false) {
            logger.msg(Arc::ERROR, "INI config file %s does not exist", options.xml_config_file);
            exit(1);
        }
        Arc::IniConfig ini_parser(options.ini_config_file);
        if (ini_parser.Evaluate(config) == false) {
            logger.msg(Arc::ERROR, "Error evaluating profile");
            exit(1);
        }
        if (!config) {
            logger.msg(Arc::ERROR, "Failed to load service configuration from file %s", options.ini_config_file);
            exit(1);
        }
    } else {
        std::string ini_config_file = "/etc/arc/service.ini";
        if (Glib::file_test(ini_config_file,
            Glib::FILE_TEST_EXISTS) == false) {
                std::string xml_config_file = "/etc/arc/service.xml";
                if (Glib::file_test(xml_config_file,
                    Glib::FILE_TEST_EXISTS) == false) {
                }
                if(!config.parse(xml_config_file.c_str())) {
                    logger.msg(Arc::ERROR, "Error loading generated configuration");
                    exit(1);
                }
        } else {
            Arc::IniConfig ini_parser(ini_config_file);
            if (ini_parser.Evaluate(config) == false) {
                logger.msg(Arc::ERROR, "Error evaulating profile");
                exit(1);
            }
        }
        if (config.Size() == 0) {
            logger.msg(Arc::ERROR, "Failed to load service configuration from any default config file");
            exit(1);
        }
    }
}

int main(int argc, char **argv)
{
    // Ignore some signals
    signal(SIGTTOU,SIG_IGN);
    signal(SIGPIPE,SIG_IGN);
    signal(SIGHUP,&sighup_handler);
    // Set up Glib exception handler
    Glib::add_exception_handler(sigc::ptr_fun(&glib_exception_handler));
    // Temporary stderr destination for error messages
    Arc::LogStream logcerr(std::cerr);
    Arc::Logger::getRootLogger().addDestination(logcerr);
    /* Create options parser */
    Arc::ServerOptions options;

    if((argc>0) && (argv[0])) Arc::ArcLocation::Init(argv[0]);

    try {
        std::list<std::string> params = options.Parse(argc, argv);
        if (params.empty()) {
            if (options.version) {
                std::cout << Arc::IString("%s version %s", "arched", VERSION) << std::endl;
                exit(0);
            }

            /* Load and parse config file */
            init_config(options);

            // schema validation
            if (!options.schema_file.empty()) {
                std::string err_msg;
                bool ret = config.Validate(options.schema_file, err_msg);
                if (ret == false) {
                    logger.msg(Arc::ERROR, "Schema validation error");
                    logger.msg(Arc::ERROR, err_msg);
                    exit(1);
                }
            }

            // dump config if it was requested
            if (options.config_dump == true) {
                std::string str;
                config.GetXML(str, true);
                std::cout << Arc::strip(str) << std::endl;
                exit(0);
            }

            if(!MatchXMLName(config,"ArcConfig")) {
                logger.msg(Arc::ERROR, "Configuration root element is not <ArcConfig>");
                exit(1);
            }

            /* overwrite config variables by cmdline options */
            merge_options_and_config(config, options);
            std::string pid_file = (config["Server"]["PidFile"] ? (std::string)config["Server"]["PidFile"] : "/var/run/arched.pid");
            std::string user = (std::string)config["Server"]["User"];
            std::string group = (std::string)config["Server"]["Group"];
            // set signal handlers
            signal(SIGTERM, sig_shutdown);
            signal(SIGINT, sig_shutdown);

            // switch user
            if (getuid() == 0) { // are we root?
                /* switch group it is specified */
                if (!group.empty()) {
                    gid_t g = get_gid(group);
                    if (setgid(g) != 0) {
                        logger.msg(Arc::ERROR, "Cannot switch to group (%s)", group);
                        exit(1);
                    }
                }
                /* switch user if it is specied */
                if (!user.empty()) {
                    uid_t u = get_uid(user);
                    if (group.empty()) {
                        gid_t g = get_gid(u);
                        if (setgid(g) != 0) {
                            logger.msg(Arc::ERROR, "Cannot switch to primary group for user (%s)", user);
                            exit(1);
                        }
                    }
                    if (setuid(u) != 0) {
                        logger.msg(Arc::ERROR, "Cannot switch to user (%s)", user);
                        exit(1);
                    }
                }
            }
            /* initalize logger infrastucture */
            std::string root_log_file = init_logger(config["Server"]["Logger"], config["Server"]["Foreground"]);
            // demonize if the foreground options was not set
            if (!(bool)(config)["Server"]["Foreground"]) {
                main_daemon = new Arc::Daemon(pid_file, root_log_file);
            }

            // bootstrap
            loader = new Arc::MCCLoader(config);
            if(!*loader) {
                logger.msg(Arc::ERROR, "Failed to load service side MCCs");
            } else {
                logger.msg(Arc::INFO, "Service side MCCs are loaded");
                // sleep forever
                for (;!req_shutdown;) {
                    sleep(1);
                }
            }
        } else {
            logger.msg(Arc::ERROR, "Unexpected arguments supplied");
        }
    } catch (const Glib::Error& error) {
      logger.msg(Arc::ERROR, error.what());
    }

    exit_code = -1;
    do_shutdown();
    return exit_code;
}
