/* Reimplemented NoSQL column operator
 * Copyright (c) 2003 Micah Stetson <micah@cnm-vra.com>
 * 
 * Permission is hereby granted, free of charge, to any
 * person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the
 * Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do
 * so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice
 * shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF
 * ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "util.h"

#ifndef HELPDIR
#define HELPDIR "/usr/local/nosql/help"
#endif
#ifndef HELPFILE
#define HELPFILE (HELPDIR "/column.txt")
#endif

void
usage(void)
{
	fprintf(stderr, "Usage: %s [-i input] [-o output] [-lNh] [column ...]\n", argv0);
	exit(1);
}

int
main(int argc, char **argv)
{
	char **f;
	char *l;
	int *fields;
	int ncol, nf;
	int line, i, j;
	int last = 0;
	int header = 1;

	ARGBEGIN{
	case 'i':
		efreopen(EARGF(usage()), "r", stdin);
		break;
	case 'o':
		efreopen(EARGF(usage()), "w", stdout);
		break;
	case 'l':
		last = 1;
		break;
	case 'h':
		execlp("grep", "grep", "-v", "^#", HELPFILE, NULL);
		fatal("Couldn't exec() grep: %s", strerror(errno));
		break;
	case 'n':
	case 'N':
		header=0;
		break;
	default:
		usage();
	}ARGEND

	if (argc == 0)
		exit(0);

	if (header) {
		for (i=0;argv[i] != NULL; i++)
			fprintf(stdout, "%s\001%s", i > 0 ? "\t": "", argv[i]);
		putc('\n', stdout);
	}

	fields = emalloc(argc * sizeof(int));

	l = frdstr(stdin, '\n', 1);
	if (l == NULL || l[0] != '\001')
		fatal("not a valid table");

	for (i=0,ncol=0; l[i] != '\0'; i++)
		if (l[i] == '\001')
			ncol++;

	/* extra column for detecting corrupt rows */
	f = emalloc((ncol+1) * sizeof(char *));

	nf = getfields(l, f, ncol+1, 0, "\t");
	if (nf != ncol)
		fatal("corrupt header line");
	
	for (i=0; i<argc; i++) {
		fields[i] = -1;
		for (j=0; j<ncol; j++) {
			if (strcmp(argv[i], f[j]+1) == 0) {	/* +1 skips \001 */
				fields[i] = j;
				if (!last)
					break;
			}
		}
	}

	free(l);

	for (line=2; (l = frdstr(stdin, '\n', 1)) != NULL; free(l), line++) {
		/* We look for an extra column in case the table is corrupt. */
		nf = getfields(l, f, ncol+1, 0, "\t");
		if (nf != ncol) {
			warn("wrong number of columns on line %d, skipping", line);
			continue;
		}
		for (i=0; i < argc; i++) {
			if (i > 0)
				putc('\t', stdout);
			if (fields[i] >= 0)
				fputs(f[fields[i]], stdout);
		}
		putc('\n', stdout);
	}

	return 0;
}
