////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Main application window implementation
////////////////////////////////////////////////////////////////////////////

#include "config.h"
#include <string>
#include <gdk/gdkkeysyms.h>

#include "support.h"
#include "MainWnd.h"
#include "TreeView.h"
#include "TextView.h"
#include "mru.h"
#include "callbacks.h"
#include "lib/debug.h"
#include "lib/NoteDocument.h"
#include "lib/IniFile.h"
#include "PortableTrayIcon.h"
#include "PasswordDialog.h"
#include "../res/notecase.xpm"

void ShowBusyCursor();
void HideBusyCursor();

extern NoteDocument g_doc;
extern MRU g_objMRU;
extern GtkWidget *window1;
extern GtkWidget *g_menuitem5;
extern TreeView g_tree;
extern TextView g_text;
extern bool g_bMinimizeToTray;
extern MainWindow g_wnd;

PortableTrayIcon g_tray;
int g_nTitleBarTextSize = 10; //default size

void register_extensions_check();
static GtkWidget* create_main_win ();
static GtkWidget* create_toolbar ();
static void create_menu (GtkWidget *window1, GtkWidget *menubar1, GtkAccelGroup *accel_group);
static gboolean window_state_event (GtkWidget *widget, GdkEventWindowState *event);
static const char *PasswordCallback(const char *szFile);
static gboolean on_focus_event (GtkWidget *widget, GtkDirectionType arg1, gpointer user_data);

MainWindow::MainWindow()
{
	m_pWidget = NULL;
}

MainWindow::~MainWindow()
{
}

void MainWindow::Create()
{
	m_pWidget = create_main_win();
	OnCreate();
}

void MainWindow::OnCreate()
{
	//check startup options
	IniFile file;
	file.Load(MRU::getfilename());

	window1 = g_wnd.m_pWidget;
	g_signal_connect (window1, "destroy", G_CALLBACK (on_quit1_activate), NULL); 
	g_signal_connect (window1, "delete_event", G_CALLBACK (on_window_delete_event), NULL);

	GtkWidget *treeview = lookup_widget(window1, "treeview1");
	GtkWidget *textview = lookup_widget(window1, "textview1");
	GtkWidget *label1   = lookup_widget(window1, "label1");

	//set initial fonts
	std::string strFont1;
	file.GetValue("Display", "TreeFont", strFont1, "");
	if(!strFont1.empty()){
		PangoFontDescription *desc1 = pango_font_description_from_string(strFont1.c_str());
		gtk_widget_modify_font(treeview, desc1);
		pango_font_description_free(desc1);
	}

	std::string strFont2;
	file.GetValue("Display", "TextFont", strFont2, "");
	if(!strFont2.empty()){
		PangoFontDescription *desc1 = pango_font_description_from_string(strFont2.c_str());
		gtk_widget_modify_font(textview, desc1);
		//calculate text size for node title bar as 1.5 the size of other text
		g_nTitleBarTextSize = (pango_font_description_get_size(desc1) * 15)/10;
		pango_font_description_free(desc1);
	}

	//set font size for node title bar (1.5 times of node text font)
	PangoContext *context = gtk_widget_get_pango_context (label1);
	PangoFontDescription *desc2 = pango_context_get_font_description(context);
	PangoFontDescription *desc3 = pango_font_description_copy(desc2);
#if GTK_CHECK_VERSION(2,6,0) //minimal version for pango_font_description_set_absolute_size	
	pango_font_description_set_absolute_size(desc3, g_nTitleBarTextSize);
#else	
	pango_font_description_set_size (desc3, (8*g_nTitleBarTextSize)/10);	//TOFIX convert from pixels to fractions?
#endif	 
	gtk_widget_modify_font(label1, desc3);
	pango_font_description_free(desc3);
	TRACE("Node title bar: Font size=%d\n", g_nTitleBarTextSize);


	bool bMaximize;
	if(file.GetValue("Startup", "Maximize", bMaximize))
		if(bMaximize)
			gtk_window_maximize(GTK_WINDOW(window1));

	//on demand check if we need to register file formats  
	bool bRegisterFormats;
	if(file.GetValue("Startup", "RegisterFormats", bRegisterFormats))
		if(bRegisterFormats)
			register_extensions_check();

	//restore text wrapping option
	bool bWrapText;
	if(file.GetValue("Display", "WrapText", bWrapText))
	if(bWrapText)
		set_wrap_activated(true);

	//restore "toolbar visible" option
	bool bShowToolbar;
	file.GetValue("Display", "ShowToolBar", bShowToolbar, 1);
		if(bShowToolbar)
			set_show_toolbar(true);

	//restore "node title visible" option
	bool bShowTitle;
	if(file.GetValue("Display", "NodeTitleBar", bShowTitle))
		if(bShowTitle)
			set_show_node_title(true);

	//restore "status bar visible" option
	bool bShowStatBar;
	file.GetValue("Display", "ShowStatusBar", bShowStatBar, 1);
	if(bShowStatBar)	// default is true, change only if false
		set_show_status_bar(true);

	//restore last position
	bool bRestoreLastPos;
	file.GetValue("Startup", "RestoreLastPosition", bRestoreLastPos, 1);
	if(bRestoreLastPos)
	{
		gint nPosLeft, nPosTop, nPosWidth, nPosHeight, nPosDivider;
		nPosLeft = nPosTop = nPosWidth = nPosHeight = nPosDivider = 0;

		file.GetValue("Startup", "LastPos_Left", nPosLeft);
		file.GetValue("Startup", "LastPos_Top", nPosTop);
		file.GetValue("Startup", "LastPos_Width", nPosWidth);
		file.GetValue("Startup", "LastPos_Height", nPosHeight);
		file.GetValue("Startup", "LastPos_Divider", nPosDivider);
		
		//apply some common sense checks/fixes
		if(nPosWidth  <= 0) nPosWidth   = 500;
		if(nPosHeight <= 0) nPosHeight  = 350;
		if(nPosLeft   <= 0) nPosLeft    = 0;
		if(nPosTop    <= 0) nPosTop     = 0;
		if(nPosDivider<= 0) nPosDivider = 150;

		gtk_window_move(GTK_WINDOW(window1), nPosLeft, nPosTop);
		gtk_window_set_default_size(GTK_WINDOW(window1), nPosWidth, nPosHeight);
		gtk_window_resize(GTK_WINDOW(window1), nPosWidth, nPosHeight);
		GtkWidget *divider = lookup_widget(window1, "hbox1");
		gtk_paned_set_position(GTK_PANED(divider), nPosDivider);
	}
	else
	{
		gtk_window_set_default_size (GTK_WINDOW (window1), 500, 350);
		GtkWidget *divider = lookup_widget(window1, "hbox1");
		gtk_paned_set_position (GTK_PANED(divider), 150);
	}
	gtk_widget_show (window1);

	//load global option
	file.GetValue("Display", "MinimizeToTray", g_bMinimizeToTray);
	restart_autosave();
}

//TOFIX make class member RefreshTitle() 
void RefreshMainTitle()
{
	std::string strTitle(APP_NAME_STR);
	strTitle += " ";
	strTitle += APP_VER_STR;
	
	if(g_doc.GetPath().size()>0)
	{
		strTitle += " - [";
		if(g_doc.IsModified())
			strTitle += "*";
		strTitle += g_doc.GetPath();
		strTitle += "]";
	}
	
	gtk_window_set_title (GTK_WINDOW (window1), strTitle.c_str());
}

GtkWidget* create_main_win ()
{
	GtkWidget *window1;
	GtkWidget *vbox1;
	GtkWidget *menubar1;
	GtkWidget *hbox1;
	GtkWidget *scrolledwindow1;
	GtkWidget *treeview1;
	GtkWidget *scrolledwindow2;
	GtkWidget *textview1;
	GtkWidget *statusbar1;
	GtkWidget *vbox2;
	GtkWidget *label1;
	GtkAccelGroup *accel_group;
	GtkWidget *toolbar1;
	
	accel_group = gtk_accel_group_new ();
	
	//prepare version string
	std::string strApp(APP_NAME_STR);
	strApp += " ";
	strApp += APP_VER_STR;

	window1 = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (window1), strApp.c_str());
	
	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_container_add (GTK_CONTAINER (window1), vbox1);
	
	menubar1 = gtk_menu_bar_new ();
	gtk_widget_show (menubar1);
	gtk_box_pack_start (GTK_BOX (vbox1), menubar1, FALSE, FALSE, 0);

	create_menu(window1, menubar1, accel_group);
	
	// create toolbar
	toolbar1 = create_toolbar();
	gtk_box_pack_start (GTK_BOX (vbox1), toolbar1, FALSE, FALSE, 0);

	//
	hbox1 = gtk_hpaned_new ();
	gtk_widget_show (hbox1);
	gtk_box_pack_start (GTK_BOX (vbox1), hbox1, TRUE, TRUE, 0);
	gtk_paned_set_position (GTK_PANED (hbox1), 0);
	
	scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow1);
	gtk_paned_pack1(GTK_PANED(hbox1), scrolledwindow1, FALSE, TRUE);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow1), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		
	//create tree view
	g_tree.Create();
	treeview1 = g_tree.m_pWidget;
	gtk_container_add (GTK_CONTAINER (scrolledwindow1), treeview1);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox2);
	gtk_paned_pack2 (GTK_PANED (hbox1), vbox2, TRUE, TRUE); 
	
	label1 = gtk_label_new (_("Untitled"));
	gtk_misc_set_alignment(GTK_MISC(label1), 0.0f, 0.7f);  // horizontal: left aligned, vertically centered
	gtk_misc_set_padding(GTK_MISC(label1), 0, 2);
	gtk_box_pack_start (GTK_BOX (vbox2), label1, FALSE, FALSE, 0); 
	
	scrolledwindow2 = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolledwindow2);
	gtk_box_pack_start (GTK_BOX (vbox2), scrolledwindow2, TRUE, TRUE, 0); 
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow2), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	
	//create text view
	g_text.Create();
	textview1 = g_text.m_pWidget;
	gtk_container_add (GTK_CONTAINER (scrolledwindow2), textview1);
		
	statusbar1 = gtk_statusbar_new ();
	gtk_box_pack_start (GTK_BOX (vbox1), statusbar1, FALSE, FALSE, 0);

	g_signal_connect (GTK_OBJECT (window1), "window_state_event", GTK_SIGNAL_FUNC (window_state_event),  NULL);
	
	// Store pointers to all widgets, for use by lookup_widget()
	GLADE_HOOKUP_OBJECT_NO_REF (window1, window1, "window1");
	GLADE_HOOKUP_OBJECT (window1, vbox1, "vbox1");
	GLADE_HOOKUP_OBJECT (window1, menubar1, "menubar1");
	GLADE_HOOKUP_OBJECT (window1, hbox1, "hbox1");
	GLADE_HOOKUP_OBJECT (window1, scrolledwindow1, "scrolledwindow1");
	GLADE_HOOKUP_OBJECT (window1, treeview1, "treeview1");
	GLADE_HOOKUP_OBJECT (window1, scrolledwindow2, "scrolledwindow2");
	GLADE_HOOKUP_OBJECT (window1, textview1, "textview1");
	GLADE_HOOKUP_OBJECT (window1, statusbar1, "statusbar1");
	GLADE_HOOKUP_OBJECT (window1, vbox2, "vbox2");
	GLADE_HOOKUP_OBJECT (window1, label1, "label1");
	GLADE_HOOKUP_OBJECT (window1, toolbar1, "toolbar1");
	
	//attach accelerators
	GClosure *closure;
	
	closure = g_cclosure_new (G_CALLBACK (on_help1_activate), NULL, NULL);
	gtk_accel_group_connect (accel_group, GDK_F1, (GdkModifierType)0, GTK_ACCEL_VISIBLE, closure);
	g_closure_unref (closure);

	closure = g_cclosure_new (G_CALLBACK (on_find1_activate), NULL, NULL);
	gtk_accel_group_connect (accel_group, GDK_F, (GdkModifierType)GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE, closure);
	g_closure_unref (closure);

	closure = g_cclosure_new (G_CALLBACK (on_wrap_activate), NULL, NULL);
	gtk_accel_group_connect (accel_group, GDK_W, (GdkModifierType)GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE, closure);
	g_closure_unref (closure);

	closure = g_cclosure_new (G_CALLBACK (on_find2_activate), NULL, NULL);
	gtk_accel_group_connect (accel_group, GDK_F3, (GdkModifierType)0, GTK_ACCEL_VISIBLE, closure);
	g_closure_unref (closure);
	
	closure = g_cclosure_new (G_CALLBACK (on_options1_activate), NULL, NULL);
	gtk_accel_group_connect (accel_group, GDK_F7, (GdkModifierType)0, GTK_ACCEL_VISIBLE, closure);
	g_closure_unref (closure);
	
	gtk_window_add_accel_group (GTK_WINDOW (window1), accel_group);
	
	g_doc.SetPassCallback( PasswordCallback );

	
#ifdef _WIN32
#else
	//set window/application icon
	GdkPixbuf *pixbuf = gdk_pixbuf_new_from_xpm_data ((const char **)&notecase_xpm);
	gtk_window_set_icon(GTK_WINDOW(window1), pixbuf);
	g_object_unref(pixbuf);
#endif
	
	return window1;
}

GtkWidget* create_toolbar()
{
	GtkWidget *toolitem1;
	GtkWidget *button3; 
	GtkIconSize tmp_toolbar_icon_size;

	GtkWidget *toolbar1 = gtk_toolbar_new ();
	//gtk_widget_show (toolbar1);  //initialy invisible (set by .ini value)

	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar1), GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_icon_size (GTK_TOOLBAR (toolbar1), GTK_ICON_SIZE_SMALL_TOOLBAR); 
	tmp_toolbar_icon_size = gtk_toolbar_get_icon_size (GTK_TOOLBAR (toolbar1)); 
	
	g_signal_connect (GTK_OBJECT (toolbar1), "focus", GTK_SIGNAL_FUNC (on_focus_event), NULL);

	gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar1), TRUE );
	GtkTooltips *tips = gtk_tooltips_new ();

	button3 = gtk_image_new_from_stock (GTK_STOCK_NEW, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("New"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("New"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_new1_activate), NULL);


	button3 = gtk_image_new_from_stock (GTK_STOCK_OPEN, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Open"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Open"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_open1_activate), NULL);


	button3 = gtk_image_new_from_stock (GTK_STOCK_SAVE, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Save"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Save"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_save1_activate), NULL);


	button3 = gtk_image_new_from_stock (GTK_STOCK_SAVE_AS, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Save As"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Save As"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_save_as1_activate), NULL);

	//append space
	toolitem1 = (GtkWidget*) gtk_separator_tool_item_new();
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);


	button3 = gtk_image_new_from_stock (GTK_STOCK_GO_BACK, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Undo"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Undo"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_undo_activate), NULL);
	gtk_widget_set_sensitive(toolitem1, FALSE);
	//GLADE_HOOKUP_OBJECT (toolbar1, button3, "tbr_undo");


	button3 = gtk_image_new_from_stock (GTK_STOCK_GO_FORWARD, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Redo"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Redo"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_redo_activate), NULL);
	gtk_widget_set_sensitive(toolitem1, FALSE);
	//GLADE_HOOKUP_OBJECT (toolbar1, button3, "tbr_redo");


	button3 = gtk_image_new_from_stock (GTK_STOCK_CUT, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Cut"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Cut"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_cut1_activate), NULL);


	button3 = gtk_image_new_from_stock (GTK_STOCK_COPY, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Copy"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Copy"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_copy1_activate), NULL);


	button3 = gtk_image_new_from_stock (GTK_STOCK_PASTE, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Paste"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Paste"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_paste1_activate), NULL);

	
	button3 = gtk_image_new_from_stock (GTK_STOCK_DELETE, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Delete"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Delete"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_delete1_activate), NULL);

	//append space
	toolitem1 = (GtkWidget*) gtk_separator_tool_item_new();
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);

	button3 = gtk_image_new_from_stock (GTK_STOCK_HELP, tmp_toolbar_icon_size);
	gtk_widget_show (button3);

	toolitem1 = (GtkWidget*) gtk_tool_button_new (button3, _("Help"));
	gtk_widget_show (toolitem1);
	gtk_container_add (GTK_CONTAINER (toolbar1), toolitem1);
	gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(toolitem1), tips, _("Help"), NULL);
	g_signal_connect (GTK_OBJECT (toolitem1), "clicked", GTK_SIGNAL_FUNC (on_help1_activate), NULL);


	return toolbar1;
}

gboolean window_state_event (GtkWidget           *widget,
							GdkEventWindowState *event)
{
	if(event->changed_mask & GDK_WINDOW_STATE_ICONIFIED)
	{
		if(event->new_window_state & GDK_WINDOW_STATE_ICONIFIED)
		{
			//on window minimized, show tray icon
			if(g_bMinimizeToTray)
			{
				g_tray.Show();
				g_tray.SetTooltip("NoteCase"); //important: call this after Show
				gtk_widget_hide(window1);
			}
		}
	}
	return FALSE;
}

const char *PasswordCallback(const char *szFile)
{
	static char szPass[1024] = "";
	szPass[0] = '\0';
	
	HideBusyCursor();	//restore the normal cursor for the moment

	//ask user for password
	PasswordDialog dlg;

	//set label	
	std::string strLabel = _("Please enter the password for file\n");
	strLabel += szFile;
	dlg.SetLabel(strLabel.c_str());

	gtk_grab_add( dlg.GetDialog() );	//block all other widgets in the app
			
	if(GTK_RESPONSE_OK == dlg.ShowModal())
	{
		//get result
		GtkWidget *entry1 = lookup_widget(dlg.GetDialog(), "entry1");
		const char *szText = gtk_entry_get_text (GTK_ENTRY(entry1));
		if(szText)
			strcpy(szPass, szText);
	}
	
	gtk_grab_remove(dlg.GetDialog());	//unblock
	gtk_widget_destroy (dlg.GetDialog());
	
	ShowBusyCursor(); //keep showing busy 

	//return password
	if(strlen(szPass)<1)
		return NULL;
	return szPass;
}

void create_menu (GtkWidget *window1, GtkWidget *menubar1, GtkAccelGroup *accel_group)
{
	GtkWidget *menuitem1;
	GtkWidget *menu1;
	GtkWidget *new1;
	GtkWidget *open1;
	GtkWidget *save1;
	GtkWidget *save_as1;
	GtkWidget *reload1;
	GtkWidget *separatormenuitem1;
	GtkWidget *quit1;
	GtkWidget *menuitem2;
	GtkWidget *menu2;
	GtkWidget *cut1;
	GtkWidget *copy1;
	GtkWidget *paste1;
	GtkWidget *delete1;
	GtkWidget *find1;
	GtkWidget *find2;
	GtkWidget *title1;
	GtkWidget *statbar1;
	GtkWidget *menutool1;
	GtkWidget *wrap1;
	GtkWidget *options1;
	GtkWidget *undo1;
	GtkWidget *redo1;
	GtkWidget *menuitem3;
	GtkWidget *menu3;
	GtkWidget *menuitem4;
	GtkWidget *menu4;
	GtkWidget *about1;
	GtkWidget *help1;
	GtkWidget *icon;
	GtkWidget *menu_item;

	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_File"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menu1 = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menu1);
	
	new1 = gtk_image_menu_item_new_with_mnemonic (_("_New"));
	gtk_widget_show (new1);
	gtk_widget_add_accelerator (new1, "activate", accel_group, 'N', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu1), new1);
	icon = gtk_image_new_from_stock (GTK_STOCK_NEW, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (new1), icon);
	
	open1 = gtk_image_menu_item_new_with_mnemonic (_("_Open"));
	gtk_widget_show (open1);
	gtk_widget_add_accelerator (open1, "activate", accel_group, 'O', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu1), open1);
	icon = gtk_image_new_from_stock (GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (open1), icon);

	save1 = gtk_image_menu_item_new_with_mnemonic (_("_Save"));
	gtk_widget_show (save1);
	gtk_widget_add_accelerator (save1, "activate", accel_group, 'S', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu1), save1);
	icon = gtk_image_new_from_stock (GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (save1), icon);

	save_as1 = gtk_image_menu_item_new_with_mnemonic (_("Save _As"));
	gtk_widget_show (save_as1);
	gtk_container_add (GTK_CONTAINER (menu1), save_as1);
	icon = gtk_image_new_from_stock (GTK_STOCK_SAVE_AS, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (save_as1), icon);
	
	reload1 = gtk_menu_item_new_with_mnemonic (_("Re_load"));
	gtk_widget_show (reload1);
	gtk_widget_add_accelerator (reload1, "activate", accel_group, 'R', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu1), reload1);

	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu1), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);

	menu_item = gtk_menu_item_new_with_label(_("Import"));
	gtk_widget_show (menu_item);  // Show the widget
	gtk_widget_add_accelerator (menu_item, "activate", accel_group, 'I', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	g_signal_connect(menu_item, "activate",	G_CALLBACK (on_menu_import), NULL);
	gtk_menu_append(menu1, menu_item);
	
	menu_item = gtk_menu_item_new_with_label(_("Export"));
	gtk_widget_show (menu_item);  // Show the widget 
	gtk_widget_add_accelerator (menu_item, "activate", accel_group, 'E', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	g_signal_connect(menu_item, "activate",	G_CALLBACK (on_menu_export), NULL);
	gtk_menu_append(menu1, menu_item);

	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu1), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);
		
	//
	//  MRU (most recently used) list
	//
	
	//create MRU item
	g_menuitem5 = gtk_menu_item_new_with_mnemonic (_("_Recent"));
	gtk_widget_show (g_menuitem5);
	gtk_container_add (GTK_CONTAINER (menu1), g_menuitem5);
	
	g_objMRU.Load();
	g_objMRU.RebuildMenu();
	
	//add separator
	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu1), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);
	
	//continue creating menus
	quit1 = gtk_image_menu_item_new_with_mnemonic (_("_Quit"));
	gtk_widget_show (quit1);
	gtk_widget_add_accelerator (quit1, "activate", accel_group, 'Q', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu1), quit1);
	icon = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (quit1), icon);

	menuitem2 = gtk_menu_item_new_with_mnemonic (_("_Edit"));
	gtk_widget_show (menuitem2);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem2);
	
	menu2 = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem2), menu2);

	undo1 = gtk_image_menu_item_new_with_mnemonic (_("_Undo"));
	gtk_widget_show (undo1);
	gtk_widget_set_sensitive(undo1, FALSE);
	gtk_widget_add_accelerator (undo1, "activate", accel_group, 'Z', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), undo1);
	icon = gtk_image_new_from_stock (GTK_STOCK_UNDO, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (undo1), icon);

	redo1 = gtk_image_menu_item_new_with_mnemonic (_("_Redo"));
	gtk_widget_show (redo1);
	gtk_widget_set_sensitive(redo1, FALSE);
	gtk_widget_add_accelerator (redo1, "activate", accel_group, 'Y', (GdkModifierType)(GDK_CONTROL_MASK), (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), redo1);
	icon = gtk_image_new_from_stock (GTK_STOCK_REDO, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (redo1), icon);

	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu2), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);
	
	cut1 = gtk_image_menu_item_new_with_mnemonic (_("Cu_t"));
	gtk_widget_show (cut1);
	gtk_widget_add_accelerator (cut1, "activate", accel_group, 'X', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), cut1);
	icon = gtk_image_new_from_stock (GTK_STOCK_CUT, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (cut1), icon);

	copy1 = gtk_image_menu_item_new_with_mnemonic (_("_Copy"));
	gtk_widget_show (copy1);
	gtk_widget_add_accelerator (copy1, "activate", accel_group, 'C', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), copy1);
	icon = gtk_image_new_from_stock (GTK_STOCK_COPY, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (copy1), icon);
	
	paste1 = gtk_image_menu_item_new_with_mnemonic (_("_Paste"));
	gtk_widget_show (paste1);
	gtk_widget_add_accelerator (paste1, "activate", accel_group, 'V', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), paste1);
	icon = gtk_image_new_from_stock (GTK_STOCK_PASTE, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (paste1), icon);

	delete1 = gtk_image_menu_item_new_with_mnemonic (_("_Delete"));
	gtk_widget_show (delete1);
	gtk_container_add (GTK_CONTAINER (menu2), delete1);
	icon = gtk_image_new_from_stock (GTK_STOCK_DELETE, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (delete1), icon);
	
	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu2), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);

	find1 = gtk_image_menu_item_new_with_mnemonic (_("_Find"));
	gtk_widget_show (find1);
	gtk_widget_add_accelerator (find1, "activate", accel_group, 'F', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), find1);
	icon = gtk_image_new_from_stock (GTK_STOCK_FIND, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (find1), icon);

	find2 = gtk_menu_item_new_with_mnemonic (_("Find _Next"));
	gtk_widget_show (find2);
	gtk_widget_add_accelerator (find2, "activate", accel_group, GDK_F3, (GdkModifierType)0, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), find2);

	options1 = gtk_image_menu_item_new_with_mnemonic (_("_Options"));
	gtk_widget_show (options1);
	gtk_widget_add_accelerator (options1, "activate", accel_group, GDK_F7, (GdkModifierType)0, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu2), options1);
	icon = gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (options1), icon);

	menuitem3 = gtk_menu_item_new_with_mnemonic (_("_View"));
	gtk_widget_show (menuitem3);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem3);
	
	menu3 = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem3), menu3);

	menutool1 = gtk_check_menu_item_new_with_label (_("Tool Bar"));
	gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(menutool1), TRUE);	//show check mark always
	gtk_widget_show (menutool1);
	gtk_container_add (GTK_CONTAINER (menu3), menutool1);

	title1 = gtk_check_menu_item_new_with_label (_("Node Title Bar"));
	gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(title1), TRUE);	//show check mark always
	gtk_widget_show (title1);
	gtk_container_add (GTK_CONTAINER (menu3), title1);

	statbar1 = gtk_check_menu_item_new_with_label (_("Status Bar"));
	gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(statbar1), TRUE);	//show check mark always
	gtk_widget_show (statbar1);
	gtk_container_add (GTK_CONTAINER (menu3), statbar1);

	wrap1 = gtk_check_menu_item_new_with_label (_("Wrap"));
	gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(wrap1), TRUE);	//show check mark always
	gtk_widget_show (wrap1);
	gtk_widget_add_accelerator (wrap1, "activate", accel_group, 'W', (GdkModifierType)GDK_CONTROL_MASK, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu3), wrap1);
		
	menuitem4 = gtk_menu_item_new_with_mnemonic (_("_Help"));
	gtk_widget_show (menuitem4);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem4);

	menu4 = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem4), menu4);
	
	help1 = gtk_image_menu_item_new_with_mnemonic (_("_Show help"));
	gtk_widget_show (help1);
	gtk_widget_add_accelerator (help1, "activate", accel_group, GDK_F1, (GdkModifierType)0, (GtkAccelFlags)(GTK_ACCEL_VISIBLE));
	gtk_container_add (GTK_CONTAINER (menu4), help1);
	icon = gtk_image_new_from_stock (GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
    gtk_widget_show (icon);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (help1), icon);
	
	separatormenuitem1 = gtk_separator_menu_item_new ();
	gtk_widget_show (separatormenuitem1);
	gtk_container_add (GTK_CONTAINER (menu4), separatormenuitem1);
	gtk_widget_set_sensitive (separatormenuitem1, FALSE);
	
	about1 = gtk_menu_item_new_with_mnemonic (_("_About"));
	gtk_widget_show (about1);
	gtk_container_add (GTK_CONTAINER (menu4), about1);

	g_signal_connect (GTK_OBJECT (new1),    "activate", GTK_SIGNAL_FUNC (on_new1_activate),  NULL);
	g_signal_connect (GTK_OBJECT (open1),   "activate", GTK_SIGNAL_FUNC (on_open1_activate), NULL);
	g_signal_connect (GTK_OBJECT (save1),   "activate", GTK_SIGNAL_FUNC (on_save1_activate), NULL);
	g_signal_connect (GTK_OBJECT (save_as1),"activate", GTK_SIGNAL_FUNC (on_save_as1_activate), NULL);
	g_signal_connect (GTK_OBJECT (reload1), "activate", GTK_SIGNAL_FUNC (on_reload1_activate), NULL);
	g_signal_connect (GTK_OBJECT (quit1),   "activate", GTK_SIGNAL_FUNC (on_quit1_activate), NULL);
	g_signal_connect (GTK_OBJECT (cut1),    "activate", GTK_SIGNAL_FUNC (on_cut1_activate), NULL);
	g_signal_connect (GTK_OBJECT (copy1),   "activate", GTK_SIGNAL_FUNC (on_copy1_activate), NULL);
	g_signal_connect (GTK_OBJECT (paste1),  "activate", GTK_SIGNAL_FUNC (on_paste1_activate), NULL);
	g_signal_connect (GTK_OBJECT (delete1), "activate", GTK_SIGNAL_FUNC (on_delete1_activate), NULL);
	g_signal_connect (GTK_OBJECT (about1),  "activate", GTK_SIGNAL_FUNC (on_about1_activate), NULL);
	g_signal_connect (GTK_OBJECT (help1),   "activate", GTK_SIGNAL_FUNC (on_help1_activate), NULL);
	g_signal_connect (GTK_OBJECT (options1),"activate", GTK_SIGNAL_FUNC (on_options1_activate), NULL);
	g_signal_connect (GTK_OBJECT (find1),	"activate", GTK_SIGNAL_FUNC (on_find1_activate), NULL);
	g_signal_connect (GTK_OBJECT (find2),	"activate", GTK_SIGNAL_FUNC (on_find2_activate), NULL);
	g_signal_connect (GTK_OBJECT (wrap1),	"activate", GTK_SIGNAL_FUNC (on_wrap_activate), NULL);
	g_signal_connect (GTK_OBJECT (title1),	"activate", GTK_SIGNAL_FUNC (on_show_node_titlebar_activate), NULL);
	g_signal_connect (GTK_OBJECT (statbar1),"activate", GTK_SIGNAL_FUNC (on_show_status_bar_activate), NULL);
	g_signal_connect (GTK_OBJECT (menutool1),"activate", GTK_SIGNAL_FUNC (on_show_toolbar_activate), NULL);
	g_signal_connect (GTK_OBJECT (undo1),	"activate", GTK_SIGNAL_FUNC (on_undo_activate), NULL);
	g_signal_connect (GTK_OBJECT (redo1),	"activate", GTK_SIGNAL_FUNC (on_redo_activate), NULL);

	GLADE_HOOKUP_OBJECT (window1, menuitem1, "menuitem1");
	GLADE_HOOKUP_OBJECT (window1, menu1, "menu1");
	GLADE_HOOKUP_OBJECT (window1, new1, "new1");
	GLADE_HOOKUP_OBJECT (window1, open1, "open1");
	GLADE_HOOKUP_OBJECT (window1, wrap1, "wrap1");
	GLADE_HOOKUP_OBJECT (window1, title1, "title1");
	GLADE_HOOKUP_OBJECT (window1, menutool1, "menutool1");
	GLADE_HOOKUP_OBJECT (window1, find1, "find1");
	GLADE_HOOKUP_OBJECT (window1, find2, "find2");
	GLADE_HOOKUP_OBJECT (window1, save1, "save1");
	GLADE_HOOKUP_OBJECT (window1, save_as1, "save_as1");
	GLADE_HOOKUP_OBJECT (window1, reload1, "reload1");
	GLADE_HOOKUP_OBJECT (window1, separatormenuitem1, "separatormenuitem1");
	GLADE_HOOKUP_OBJECT (window1, quit1, "quit1");
	GLADE_HOOKUP_OBJECT (window1, menuitem2, "menuitem2");
	GLADE_HOOKUP_OBJECT (window1, menu2, "menu2");
	GLADE_HOOKUP_OBJECT (window1, cut1, "cut1");
	GLADE_HOOKUP_OBJECT (window1, copy1, "copy1");
	GLADE_HOOKUP_OBJECT (window1, paste1, "paste1");
	GLADE_HOOKUP_OBJECT (window1, delete1, "delete1");
	GLADE_HOOKUP_OBJECT (window1, menuitem4, "menuitem4");
	GLADE_HOOKUP_OBJECT (window1, menu4, "menu4");
	GLADE_HOOKUP_OBJECT (window1, about1, "about1");
	GLADE_HOOKUP_OBJECT (window1, help1, "help1");
	GLADE_HOOKUP_OBJECT (window1, statbar1, "statbar1");
	GLADE_HOOKUP_OBJECT (window1, undo1, "undo1");
	GLADE_HOOKUP_OBJECT (window1, redo1, "redo1");	
}

gboolean on_focus_event (GtkWidget *widget, GtkDirectionType arg1, gpointer user_data)
{
	//FIX: prevent toolbar from getting focus by keyboard
	static bool bRecurse = false;
	if(!bRecurse)
	{
		bRecurse = true;
		gtk_widget_child_focus (widget, (GtkDirectionType)GTK_DIR_TAB_FORWARD);
		bRecurse = false;
	}
	
	return FALSE;
}

