////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: GTK+ TextView wrapper class
////////////////////////////////////////////////////////////////////////////

#include "TextView.h"

//declare signal handlers
void on_textview_edited(GtkTextBuffer *, gpointer data);
void on_delete_text (GtkTextBuffer *textbuffer, GtkTextIter *arg1, GtkTextIter *arg2, gpointer user_data);
void on_insert_text (GtkTextBuffer *textbuffer, GtkTextIter *arg1, gchar *arg2, gint arg3, gpointer user_data);
gint textview_keyboard_handler(GtkWidget *widget, GdkEventKey *event);

TextView::TextView()
{
	m_pWidget = NULL;
}

TextView::~TextView()
{
}

void TextView::Create()
{
	GtkWidget *textview1;

	GtkTextBuffer *buffer = gtk_text_buffer_new (NULL);
	textview1 = gtk_text_view_new_with_buffer (buffer);
	gtk_widget_show (textview1);

	g_signal_connect (G_OBJECT (textview1), "key_press_event", GTK_SIGNAL_FUNC (textview_keyboard_handler), NULL);
	g_signal_connect(buffer, "changed", G_CALLBACK(on_textview_edited),  NULL); //track buffer changes

	//events to handle undo in textview
	g_signal_connect (buffer, "insert-text", G_CALLBACK(on_insert_text), NULL);
	g_signal_connect (buffer, "delete-range", G_CALLBACK(on_delete_text), NULL);

	m_pWidget = textview1;
}

void TextView::Clear()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	//delete previous content
	GtkTextIter itStart, itEnd;
	gtk_text_buffer_get_iter_at_offset(buffer1, &itStart, 0);
	gtk_text_buffer_get_iter_at_offset(buffer1, &itEnd, gtk_text_buffer_get_char_count(buffer1));

	g_signal_handlers_block_by_func(buffer1, (void *)on_delete_text, 0);
	g_signal_handlers_block_by_func(buffer1, (void *)on_textview_edited, 0);
	gtk_text_buffer_delete(buffer1, &itStart, &itEnd);
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_delete_text, 0);
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_textview_edited, 0);
}

gchar* TextView::GetText()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	GtkTextIter itStart, itEnd;
	gtk_text_buffer_get_iter_at_offset(buffer1, &itStart, 0);
	gtk_text_buffer_get_iter_at_offset(buffer1, &itEnd, gtk_text_buffer_get_char_count(buffer1));
	
	return gtk_text_buffer_get_text(buffer1, &itStart, &itEnd, FALSE);
}

void TextView::SetText(const gchar *szText)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	g_signal_handlers_block_by_func(buffer1, (void *)on_textview_edited, 0);
	g_signal_handlers_block_by_func(buffer1, (void *)on_insert_text, 0);

	gtk_text_buffer_insert_at_cursor(buffer1, szText, -1);
	gtk_text_view_place_cursor_onscreen(GTK_TEXT_VIEW(textview));
	
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_textview_edited, 0);
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_insert_text, 0);
}

void TextView::InsertText(int nOffset, const gchar *szText)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	GtkTextIter itStart;
	gtk_text_buffer_get_iter_at_offset(buffer1, &itStart, nOffset);

	g_signal_handlers_block_by_func(buffer1, (void *)on_textview_edited, 0);
	g_signal_handlers_block_by_func(buffer1, (void *)on_insert_text, 0);

	gtk_text_buffer_insert(buffer1, &itStart, szText, -1);
	gtk_text_view_place_cursor_onscreen(GTK_TEXT_VIEW(textview));

	g_signal_handlers_unblock_by_func(buffer1, (void *)on_insert_text, 0);
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_textview_edited, 0);
}

void TextView::DeleteText(int nOffset, int nLength)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	
	GtkTextIter itStart, itEnd;
	gtk_text_buffer_get_iter_at_offset(buffer1, &itStart, nOffset);
	gtk_text_buffer_get_iter_at_offset(buffer1, &itEnd, nOffset+nLength);

	g_signal_handlers_block_by_func(buffer1, (void *)on_textview_edited, 0);
	g_signal_handlers_block_by_func(buffer1, (void *)on_delete_text, 0);

	gtk_text_buffer_delete(buffer1, &itStart, &itEnd);

	g_signal_handlers_unblock_by_func(buffer1, (void *)on_delete_text, 0);
	g_signal_handlers_unblock_by_func(buffer1, (void *)on_textview_edited, 0);
}

void TextView::ClipboardCut()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	// get the clipboard object
	GtkClipboard *clipboard = gtk_widget_get_clipboard (m_pWidget, GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_cut_clipboard (buffer1, clipboard, true);
}

void TextView::ClipboardCopy()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	// get the clipboard object
	GtkClipboard *clipboard = gtk_widget_get_clipboard (m_pWidget, GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_copy_clipboard (buffer1, clipboard);
}

void TextView::ClipboardPaste()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	// get the clipboard object
	GtkClipboard *clipboard = gtk_widget_get_clipboard (m_pWidget, GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_paste_clipboard (buffer1, clipboard, NULL, true);
}

void TextView::ClipboardDelete()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	gtk_text_buffer_delete_selection (buffer1, false, true);
}

int TextView::GetSelectionEnd()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	GtkTextIter end;
	if(gtk_text_buffer_get_selection_bounds(buffer1, NULL, &end))
	{
		return gtk_text_iter_get_offset(&end);
	}
	return -1; //no selection
}

void TextView::SelectRange(int nStart, int nEnd)
{
	//select found text in a text view
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	GtkTextIter iterStart, iterEnd;
	gtk_text_buffer_get_iter_at_offset(buffer1, &iterStart, nStart);
	gtk_text_buffer_get_iter_at_offset(buffer1, &iterEnd, nEnd);
#if GTK_CHECK_VERSION(2,4,0) //define minimal version for this api
	gtk_text_buffer_select_range(buffer1, &iterStart, &iterEnd);
#endif	
}

void TextView::EnsureVisible(int nOffset)
{
	//scroll text view to ensure text at this offset is visible
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);

	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_offset(buffer1, &iter, nOffset);

	gtk_text_view_scroll_to_iter(textview, &iter, 0.0, TRUE, 0.0, 1.0);
}

bool TextView::IsWrapped()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	return (GTK_WRAP_NONE != gtk_text_view_get_wrap_mode(GTK_TEXT_VIEW(textview)));
}

bool TextView::GetModified()
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	return gtk_text_buffer_get_modified(buffer1);
}

void TextView::SetModified(bool bModified)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	gtk_text_buffer_set_modified(buffer1, (bModified)? TRUE : FALSE);
}

void TextView::SetFocus()
{
 	gtk_widget_grab_focus( m_pWidget );

 	//view cursor in a TextView
 	gtk_text_view_scroll_mark_onscreen (
 		GTK_TEXT_VIEW(m_pWidget),
		gtk_text_buffer_get_insert( gtk_text_view_get_buffer( GTK_TEXT_VIEW(m_pWidget) ) ) 
	);
}

//
// cursor will be set to cursor position and selection to the other end of the selection.
// If there is not selection, selection will be set to same value as the cursor
//
void TextView::GetSelectionBounds(unsigned int& cursor, unsigned int& selection)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	GtkTextIter iter;
	
	gtk_text_buffer_get_iter_at_mark (buffer1, &iter, gtk_text_buffer_get_insert (buffer1) );
	cursor = (unsigned int)gtk_text_iter_get_offset(&iter);

	gtk_text_buffer_get_iter_at_mark (buffer1, &iter, gtk_text_buffer_get_selection_bound (buffer1) );
	selection = (unsigned int)gtk_text_iter_get_offset(&iter);
}

void TextView::RestoreSelectionBounds(unsigned int cursor, unsigned int selection)
{
	GtkTextView *textview =  (GtkTextView *)m_pWidget;
	GtkTextBuffer* buffer1 = gtk_text_view_get_buffer(textview);
	GtkTextIter cursIter, boundIter;

	gtk_text_buffer_get_iter_at_offset( buffer1, &cursIter, gint(cursor) );
	gtk_text_buffer_get_iter_at_offset( buffer1, &boundIter, gint(selection) );
	
	gtk_text_buffer_select_range ( buffer1, &cursIter, &boundIter );
}
