////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Implementations of menu action handlers
////////////////////////////////////////////////////////////////////////////

#include "config.h"
#include "lib/debug.h"

#ifdef _WIN32
  #pragma warning(disable:4786)
#endif

#include "lib/NoteDocument.h"
#include "lib/DocumentIterator.h"
#include "lib/FilePath.h"
#include "lib/TextSearch.h"
#include "lib/IniFile.h"
#include "lib/DocActionManager.h"
#include "DocAction.h"
#include "mru.h"

#include "../res/internal_blank.xpm"
#include "../res/internal_folder.xpm"
#include "../res/internal_help.xpm"
#include "../res/internal_lock.xpm"
#include "../res/internal_new_dir.xpm"
#include "../res/internal_recycle.xpm"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <pango/pango.h>

#ifdef _WIN32
 #include <io.h> //access
 #define access _access
#endif 

#include "gui/FileDialog.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "PasswordDialog.h"
#include "OptionsDialog.h"
#include "ExportDialog.h"
#include "FindDialog.h"
#include "gui/GuiLanguage.h"
#include "TreeView.h"
#include "TextView.h"
#include "NodePropertiesDlg.h"

int  load_file(const char *filename);
bool save_file(const char *filename, bool bRememberPath = true, bool bAutosave = false, const char *szPassword = NULL);
bool PathFromNodeIdx(int nIdx, GtkTreePath *&path1);
int gtkMessageBox(const char *szText, int nButtons = GTK_BUTTONS_OK, int nIcon = GTK_MESSAGE_INFO);
void on_textview_edited(GtkTextBuffer *, gpointer data);
int GetSelectedNodeIdx();
bool MoveNodeLeft(int nIdx, NoteDocument &doc);
void text_find();
void ShowBusyCursor();
void HideBusyCursor();
gboolean autosave_timer(gpointer data);
const char *calculate_autosave_filename();
const char *calculate_autosave_filename1();
const char *GetDocumentErrorString(int nErrCode);
const char **InternalIcon_GetFromIdx(int nIdx);
int InternalIcon_Name2Index(const char *szName);
void UpdateNodeIcon(int nIdx);
void RememberSelectionBounds(int nodeIdx);
void set_title_bar(const char *szText);

extern bool g_bMinimizeToTray;
extern GuiLanguage g_lang;
extern GtkWidget *window1;
extern MRU g_objMRU;
extern TreeView g_tree;
extern TextView g_text;

int g_nActiveNodeIdx = -1;	//TOFIX move to be memeber of document
int g_nAutosaveTimer = 0;
NoteDocument g_doc;
DocActionManager g_undoManager(UNDO_LIST_SIZE);

//// find dialog settings ////////
std::string g_strFindText;
bool g_bFindSensitive = true;
bool g_bFindDirectionDown = true;
int  g_nFindStartNodeRecursiveIdx = -1;
int  g_nFindCurNodeRecursiveIdx = -1;
int  g_nFindBufferPos = 0;
//////////////////////////////////

void on_new1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//check to save current document
	UpdateTextFromScreen();

	if(g_doc.IsModified())
	{
		//ask to save the document
		gint result = gtkMessageBox(_("Document has been modified! Do you want to save it?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES == result)
		{
			bool bSaved = false;			
			on_save1_activate(NULL, &bSaved);

			if(!bSaved)	return;	//quit action if user cancels old document saving
		}
	}

	//
	// clear the content (both screen and memory)
	//
	g_tree.Clear(); //clear tree view
	g_text.Clear(); //clear edit view
	g_doc.Close();  //close the document

	RefreshMainTitle();
}

void on_open1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	FileDialog dlg(true);
	dlg.SetTitle(_("Open document"));

	//define filters
	dlg.AddFilter(_("All supported formats (*.ncd,*.nce,*.hnc)"), "*.ncd|*.nce|*.hnc");
	dlg.AddFilter(_("NoteCase document (*.ncd)"), "*.ncd");
	dlg.AddFilter(_("NoteCase encrypted document (*.nce)"), "*.nce");
	dlg.AddFilter(_("NoteCenter document (*.hnc)"), "*.hnc");	
	dlg.AddFilter(_("All files (*)"), "*");

	//TOFIX set initial directory from INI (store last used)
	//dlg.SetDirectory(const char *szPath);

	
	if(dlg.DoModal())
	{
		const gchar *filename = dlg.GetFilename();
		dlg.Close();
		
		int nResult = load_file(filename);
		
		//do not delete MRU for file that exists, but failed to open
		g_objMRU.Change(filename, (DOC_LOAD_NOT_FOUND != nResult));	
	}
}

void on_save1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	if( g_doc.GetPath().size() > 0 ){
		bool bDone = save_file(g_doc.GetPath().c_str());
		if(NULL != user_data)
			*((bool *)user_data) = bDone;	//store result
		
		RefreshMainTitle();
	}
	else
		on_save_as1_activate(menuitem, user_data);
}

void on_save_as1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	FileDialog dlg(false);
	dlg.SetTitle(_("Save document"));

	//set filters
	dlg.AddFilter(_("NoteCase document (*.ncd)"), "*.ncd");
	dlg.AddFilter(_("NoteCase encrypted document (*.nce)"), "*.nce");
	dlg.AddFilter(_("NoteCenter document (*.hnc)"), "*.hnc");	

	//calculate default document name
	std::string strDefaultName;
	if( g_doc.GetPath().size() > 0 )
		strDefaultName = g_doc.GetPath();
	else
		strDefaultName = "untitled.ncd";

	dlg.SetFilename(strDefaultName.c_str());

	//get last used document folder
	std::string strDir;
	IniFile file;
	file.Load(MRU::getfilename());
	file.GetValue("Cache", "LastSaveDir", strDir, "");
	if( strDir.size() > 0.&&
		0 == access(strDir.c_str(), 00))
	{
		dlg.SetDirectory(strDir.c_str());
	}

	//assume failure until success
	if(NULL != user_data)
		*((bool *)user_data) = false;	//store result

	//start dialog
	if(dlg.DoModal())
	{
		const gchar *filename = dlg.GetFilename();
		std::string strFile(filename);

		//force valid file extension using selected filter settings
		const gchar *filtername = dlg.GetCurrentFilterName();
		if(filtername)
		{
			std::string strExt;
			if(NULL != strstr(filtername, "*.ncd"))
				strExt = ".ncd";
			else if(NULL != strstr(filtername, "*.nce"))
				strExt = ".nce";
			else if(NULL != strstr(filtername, "*.hnc"))
				strExt = ".hnc";

			std::string strCurExt = strFile.substr(strFile.size() - 4, 1000); //all my extensions have fixed size = 4

			//if no extension set, select the one set by current filter
			if( strCurExt != ".ncd" &&
				strCurExt != ".nce" &&
				strCurExt != ".hnc" &&
				strExt.size()>0)
			{
				strFile += strExt;
			}
		}

		//cache current directory before closing
		strDir = dlg.GetDirectory();

		dlg.Close();

		//if selected file name already exists on the disk
		if(0 == access(strFile.c_str(), 0))
		{
			char szBuffer[1000];
			sprintf(szBuffer, _("File %s already exists! Do you want to overwrite it?"), strFile.c_str());

			//ask permission to overwrite
			gint result = gtkMessageBox(szBuffer, GTK_BUTTONS_YES_NO);
			if(GTK_RESPONSE_NO == result)
				return;
		}

		//if switching from unencrpypted to encrypted format
		//offer to delete original file (unencrpyted)
		bool bEncSwitchedOn = false;
		std::string strOrigFile;
		if( g_doc.GetPath().size() > 0 &&
			!g_doc.IsEncrypted())
		{
			strOrigFile = g_doc.GetPath();
			bEncSwitchedOn = true;
		}

		//save the file to disk
		if(save_file(strFile.c_str()))
		{
			//report success
			if(NULL != user_data)
				*((bool *)user_data) = true;	//store result

			if( bEncSwitchedOn &&
				g_doc.IsEncrypted())
			{
				//ask to delete original file
				gint result = gtkMessageBox(_("You have saved file in encrypted format! Do you want to delete the original (unencrypted) file?"), GTK_BUTTONS_YES_NO);
				if(GTK_RESPONSE_YES == result)
				{
					remove(strOrigFile.c_str());
					g_objMRU.Change(strOrigFile.c_str(), false);
				}
			}
		
			RefreshMainTitle();
			g_objMRU.Change(strFile.c_str(), true);
		}

		//store last save directory
		file.SetValue("Cache", "LastSaveDir", strDir.c_str());
		file.Save();
	}
}

//on main window destruction
gboolean on_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	UpdateTextFromScreen();

	if(g_doc.IsModified())
	{
		//ask to save the document
		gint result = gtkMessageBox(_("Document has been modified! Do you want to save it?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES  == result)
			on_save1_activate(NULL, NULL);
	}

	//calculate position variables
	gint nPosLeft, nPosTop, nPosWidth, nPosHeight, nPosDivider;
	gtk_window_get_position(GTK_WINDOW(window1), &nPosLeft, &nPosTop);
	gtk_window_get_size(GTK_WINDOW(window1), &nPosWidth, &nPosHeight);
	GtkWidget *divider = lookup_widget(window1, "hbox1");
	nPosDivider = gtk_paned_get_position(GTK_PANED(divider));

	//save some options
	IniFile file;
	file.Load(MRU::getfilename());
	file.SetValue("Display", "NodeTitleBar", get_node_title_set());
	file.SetValue("Display", "ShowToolBar", get_show_toolbar());
	file.SetValue("Display", "ShowStatusBar", get_show_status_bar());
	file.SetValue("Display", "MinimizeToTray", g_bMinimizeToTray);
	file.SetValue("Startup", "LastPos_Left", nPosLeft);
	file.SetValue("Startup", "LastPos_Top", nPosTop);
	file.SetValue("Startup", "LastPos_Width", nPosWidth);
	file.SetValue("Startup", "LastPos_Height", nPosHeight);
	file.SetValue("Startup", "LastPos_Divider", nPosDivider);
	file.Save();

	return FALSE;	//allows GTK to destroy window
}

void on_quit1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	gtk_main_quit(); //quit application
}

void on_cut1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//if focus correct, apply command
	if(g_text.m_pWidget == gtk_window_get_focus(GTK_WINDOW(window1)))
		g_text.ClipboardCut();
}

void on_copy1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//if focus correct, apply command
	if(g_text.m_pWidget == gtk_window_get_focus(GTK_WINDOW(window1)))
		g_text.ClipboardCopy();
}

void on_paste1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//if focus correct, apply command
	if(g_text.m_pWidget == gtk_window_get_focus(GTK_WINDOW(window1)))
		g_text.ClipboardPaste();
}


void on_delete1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//if focus correct, apply command
	if(g_text.m_pWidget == gtk_window_get_focus(GTK_WINDOW(window1)))
		g_text.ClipboardDelete();
}

void on_about1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//prepare version string
	std::string strMsg(APP_NAME_STR);
	strMsg += " v.";
	strMsg += APP_VER_STR;

	//append web address string
	strMsg += "\nhttp://notecase.sourceforge.net/";

	//append current locale settings
	strMsg += "\n\n(locale: ";
	strMsg += g_lang.GetLocale();
	strMsg += ")";

	//display message
	gtkMessageBox(strMsg.c_str());
}

std::string GetHelpFile()
{
	//prepare help document path
	std::string strHelp;

#ifndef _WIN32
	strHelp = INSTALL_PREFIX;
	strHelp += HELP_FILE_PATH;
#endif

	//if file not defined in a fixed location or not found calc path dynamically
	if( strHelp.empty() ||
	    0 != access(strHelp.c_str(), 0)) 
	{
		//Win32 and Linux debug version
		strHelp = GetAppPath();
		strHelp = GetParentDir(strHelp.c_str());
		strHelp += "help.ncd";
	}
	
	return strHelp;
}

void on_help1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//skip if help already loaded and not modified 
	if( g_doc.GetPath().size() > 0 && 
		0 == strcmp(GetHelpFile().c_str(), g_doc.GetPath().c_str()) &&
		!g_doc.IsModified() )
	{
		return;
	}

	//if some document loaded (and not help document) or document was changed
	if( (g_doc.GetPath().size() > 0 && 0 != strcmp(GetHelpFile().c_str(), g_doc.GetPath().c_str()) ) ||
		g_doc.IsModified())
	{
		//user must confirm current document closing (to load help)
		gint result = gtkMessageBox(_("Current document has to be closed in order to open help document!\nDo you wish to proceed?\n\n(Note that if you choose to proceed you'll be prompted to save current document changes)"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_NO == result)
			return;
	}

	load_file(GetHelpFile().c_str());
}

int load_file(const char *filename)
{
	//TOFIX preserve old document content if loading fails
	//(requires additional doc object, and = operator) 

	//check if the current document has been modified
	if(g_doc.IsModified())
	{
		//ask to save the document
		gint result = gtkMessageBox(_("Document has been modified! Do you want to save it?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES == result)
		{
			bool bSaved = false;
			on_save1_activate(NULL, &bSaved);
			
			if(!bSaved)	return DOC_LOAD_ABORTED;	//quit action if user cancels old document saving
		}
	}

	ShowBusyCursor();

	g_nActiveNodeIdx = -1;

	//trying to load new content
	int nResult = g_doc.Load(filename);
	
	if(DOC_LOAD_OK == nResult)
	{
		//clear old content
		g_tree.Clear();
		g_text.Clear();

		//rebuild treeview
		//recursively add nodes into the tree
		add_child_nodes(NULL);

		//select initial node (as written in document)
		DocumentIterator it(g_doc);
		int nIdx = it.RecursiveIdx2NodeIdx(g_doc.m_nCurrentNode);
		SelectNodeByIdx(nIdx);

		HideBusyCursor();
	}
	else
	{
		//clear old content (g_doc is in empty state)
		g_tree.Clear();
		g_text.Clear();

		HideBusyCursor();

		//error message
		gtkMessageBox(GetDocumentErrorString(nResult), GTK_BUTTONS_OK, GTK_MESSAGE_ERROR);
	}

	g_undoManager.Clear();
	UpdateUndoRedoMenus();
	
	RefreshMainTitle();
	return nResult;
}

//recursively builds document tree
void add_child_nodes(GtkTreeIter *iter, int nParentID, int nFirstSibling)
{
	GtkWidget *treeview = lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);
	
	GtkTreeIter  child;
	DocumentIterator itDoc(g_doc);

	int nSiblingIdx = nFirstSibling;
	int nIdx = itDoc.GetChildIdx(nParentID, nSiblingIdx); //first child
	while (nIdx >= 0)
	{
		if(nParentID >= 0 && iter != 0)
			gtk_tree_store_append(GTK_TREE_STORE(model), &child, iter);  /* Acquire a child iterator */
		else
			gtk_tree_store_append(GTK_TREE_STORE(model), &child, NULL);   /* Acquire a child iterator */

		//set data
		std::string strData = itDoc.GetNodeByIdx(nIdx).GetTitle();
		gtk_tree_store_set (GTK_TREE_STORE(model), &child, STORE_IDX_TEXT, strData.c_str(), -1);

		UpdateNodeIcon(nIdx);

		//pump event messages to repaint correctly for big documents
		while (g_main_iteration(FALSE));

		//recursively add child's children
		add_child_nodes(&child, itDoc.GetNodeByIdx(nIdx).m_nID);

		//go to the next sibling node
		nSiblingIdx ++;
		nIdx = itDoc.GetChildIdx(nParentID, nSiblingIdx);
	}
}

void on_tree_row_click (GtkTreeView *treeview, gpointer user_data)
{
	//get iterator to selected node
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter  child;
	if(!gtk_tree_selection_get_selected(treesel, &model, &child))
		return;

	//get document node index from GUI tree iterator
	GtkTreePath *path1;
	path1 = gtk_tree_model_get_path(model, &child);
	//char * szText = gtk_tree_path_to_string (path1);

	int nIdx = NodeIdxFromPath(path1);
	ASSERT(nIdx >= 0);

	//skip action if this node is already selected (preserves scroll position)
	if(g_nActiveNodeIdx == nIdx){
		gtk_tree_path_free(path1);
		return;
	}

	// update the text of previously selected node
	UpdateTextFromScreen();
	
	//store selection (to be restored when we get back to this node)
	RememberSelectionBounds(g_nActiveNodeIdx);

	//delete previous content
	g_text.Clear();

	//
	//load the text for the selected node
	//
	g_nActiveNodeIdx = nIdx;
	gtk_tree_path_free(path1);

	if(nIdx > -1){
		//set new text
		NoteNode &myNode = g_doc.GetNodeByIdx(nIdx);
		g_text.SetText(myNode.GetText().c_str());
		
		//update node title label TOFIX separate method (used multiple times)
		set_title_bar(myNode.GetTitle().c_str());

		//restore cursor position and selection bound
		g_text.RestoreSelectionBounds( myNode.GetSelStart(), myNode.GetSelEnd() );
	}

	g_text.SetModified(false);
}

bool save_file(const char *filename, bool bRememberPath, bool bAutosave, const char *szPassword)
{
	//remember focus
	bool bTextFocused = gtk_widget_is_focus(g_text.m_pWidget) > 0;

	if(!bAutosave)
		ShowBusyCursor();

	UpdateTextFromScreen();

	bool bSuccess = true;

	//refresh selected node depth-first index before save
	DocumentIterator it(g_doc);
	g_doc.m_nCurrentNode = it.NodeIdx2RecursiveIdx(g_nActiveNodeIdx);

	// save to file
	if(!g_doc.Save(filename, bRememberPath, szPassword))
	{
		//error message
		gtkMessageBox(_("Failed to save the file!"), GTK_BUTTONS_OK, GTK_MESSAGE_ERROR);
		bSuccess = false;
	}

	HideBusyCursor();

	//restore focus
	if(!bAutosave){
		if(bTextFocused)
			gtk_widget_grab_focus(g_text.m_pWidget);
		else
			gtk_widget_grab_focus(g_tree.m_pWidget);
	}

	return bSuccess;
}

void UpdateTextFromScreen()
{
	//
	// update the text of previously selected node
	//
	if( g_text.GetModified() )
	{
		if(g_nActiveNodeIdx >= 0)
		{
			//refresh text
			gchar *szText = g_text.GetText();
			g_doc.GetNodeByIdx(g_nActiveNodeIdx).SetText(szText);
			g_free (szText);
			
			g_doc.SetModified(true);
			g_text.SetModified(false);
		}
	}
}

//insert as child of the selected node
void on_menu_insert_child_node (GtkMenuItem *menuitem, gpointer user_data)
{	
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter  iter, iternew;

	int nIdx = -1;
	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//insert node as child
		gtk_tree_store_append(GTK_TREE_STORE(model), &iternew, &iter);  /* Acquire a child iterator */

		//TOFIX method to map from GUI tree iter to document node ID and back 
		GtkTreePath *path1;
		path1 = gtk_tree_model_get_path(model, &iter);
		nIdx = NodeIdxFromPath(path1);

		gtk_tree_view_expand_row(treeview, path1, FALSE);	//expand new parent
		gtk_tree_path_free(path1);
	}
	else
	{
		//insert node under root
		gtk_tree_store_append(GTK_TREE_STORE(model), &iternew, NULL);   /* Acquire a child iterator */
	}	
	
	std::string strName = _("New node");
	gtk_tree_store_set (GTK_TREE_STORE(model), &iternew, STORE_IDX_TEXT, strName.c_str(), -1);


	//now add the node into the document
	if(nIdx >= 0)
		g_doc.NodeInsert(g_doc.GetNodeByIdx(nIdx).m_nID, -1);
	else
		g_doc.NodeInsert(-1, -1);

	//set default node name to the latest note
	int nNewIdx = g_doc.GetNodeCount()-1;
	g_doc.GetNodeByIdx(nNewIdx).SetTitle(strName.c_str());

	//select node in a tree
	SelectNodeByIdx(nNewIdx);
	
	RefreshMainTitle();
	
	//push document change into undo/redo manager
	DocAction *pAction = new DocAction;
	pAction->SetType(ACT_NODE_INSERT);
	pAction->SetDoc(g_doc);
	pAction->m_nNodeIndex   = nNewIdx; //TOFIX recursive index
	pAction->m_nNodeID      = g_doc.GetNodeByIdx(nNewIdx).m_nID;
	pAction->m_nNodePID     = g_doc.GetNodeByIdx(nNewIdx).m_nParentID;
	pAction->m_nNodeSibling = g_doc.GetNodeByIdx(nNewIdx).m_nSiblingIdx;
	pAction->m_objSubTree.AssignSubtree(g_doc, nNewIdx);
	
	g_undoManager.AddAction(pAction);
	UpdateUndoRedoMenus();

	g_tree.EditLabel();	//trigger label edit mode
}

void on_menu_insert_node (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);
	GtkTreeIter  iternew;

	int nIdx = -1;
	//insert node under root
	gtk_tree_store_append(GTK_TREE_STORE(model), &iternew, NULL);   /* Acquire a child iterator */
	
	std::string strName = _("New node");
	gtk_tree_store_set (GTK_TREE_STORE(model), &iternew, STORE_IDX_TEXT, strName.c_str(), -1);

	//now add the node into the document
	if(nIdx >= 0)
		g_doc.NodeInsert(g_doc.GetNodeByIdx(nIdx).m_nID, -1);
	else
		g_doc.NodeInsert(-1, -1);

	//set default node name to the latest note
	int nNewIdx = g_doc.GetNodeCount()-1;
	g_doc.GetNodeByIdx(nNewIdx).SetTitle(strName.c_str());

	//select node in a tree
	SelectNodeByIdx(nNewIdx);
	
	RefreshMainTitle();
	
	//push document change into undo/redo manager
	DocAction *pAction = new DocAction;
	pAction->SetType(ACT_NODE_INSERT);
	pAction->SetDoc(g_doc);
	pAction->m_nNodeIndex   = nNewIdx; //TOFIX recursive index
	pAction->m_nNodeID      = g_doc.GetNodeByIdx(nNewIdx).m_nID;
	pAction->m_nNodePID     = g_doc.GetNodeByIdx(nNewIdx).m_nParentID;
	pAction->m_nNodeSibling = g_doc.GetNodeByIdx(nNewIdx).m_nSiblingIdx;
	pAction->m_objSubTree.AssignSubtree(g_doc, nNewIdx);
	
	g_undoManager.AddAction(pAction);
	UpdateUndoRedoMenus();

	g_tree.EditLabel();	//trigger label edit mode
}

void on_menu_delete_node (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter  iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//recursively delete node and all its children
		
		//TOFIX method to map from GUI tree iter to document node ID and back 
		GtkTreePath *path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);

		//prepare path to show to the next selection candidate
		int nNextSelectedID = -1;
		if(g_tree.GetPreviousVisibleNode(&path1)){
			int nNextIdx = NodeIdxFromPath(path1);
			if(nNextIdx)
				nNextSelectedID = g_doc.GetNodeByIdx(nNextIdx).m_nID;
		}

		gtk_tree_path_free(path1);

		//prepare question
		std::string strMsg = _("Are you sure to delete node \"%s\"?");
		char szBuffer[2024];
		sprintf(szBuffer, strMsg.c_str(), g_doc.GetNodeByIdx(nIdx).GetTitle().c_str());
		strMsg = szBuffer;

		//ask for confirmation
		gint result = gtkMessageBox(strMsg.c_str(), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES != result)
			return;

		UpdateTextFromScreen();	//proper node contents neded for undo

		//push document change into undo/redo manager
		DocAction *pAction = new DocAction;
		pAction->SetType(ACT_TREE_DELETE);
		pAction->SetDoc(g_doc);
		pAction->m_objSubTree.AssignSubtree(g_doc, nIdx);
		pAction->m_nNodeIndex   = nIdx;
		pAction->m_nNodeID      = g_doc.GetNodeByIdx(nIdx).m_nID;
		pAction->m_nNodePID     = g_doc.GetNodeByIdx(nIdx).m_nParentID;
		pAction->m_nNodeSibling = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;
		
		//remove node info (recursive)
		if(nIdx >= 0)
			g_doc.NodeDelete(g_doc.GetNodeByIdx(nIdx).m_nID);

		g_nActiveNodeIdx = -1;

		//remove GUI tree node
		gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 

		//clear edit belonging to selected node
		g_text.Clear();

		//push document change into undo/redo manager
		g_undoManager.AddAction(pAction);
		UpdateUndoRedoMenus();

		//update node title label
		set_title_bar(_("Untitled"));

		RefreshMainTitle();
		
		//select another node after selected node deleted
		if(nNextSelectedID >= 0){
			int nNewIdx = g_doc.GetIdxFromID(nNextSelectedID);
			SelectNodeByIdx(nNewIdx);
		}
		else {
			//no previous node, select the first visible node
			GtkTreePath *path = g_tree.GetFirstVisibleNode();
			if(path){
				int nNewIdx = NodeIdxFromPath(path);
				SelectNodeByIdx(nNewIdx);
				gtk_tree_path_free(path);
			}
		}

	}
}

void on_menu_rename_node (GtkMenuItem *menuitem, gpointer user_data)
{
	g_tree.EditLabel();	//enter edit mode
	RefreshMainTitle();
}

void on_menu_move_up (GtkMenuItem *menuitem, gpointer user_data)
{
	do_node_move_up();
}

void on_menu_move_down (GtkMenuItem *menuitem, gpointer user_data)
{
	do_node_move_down();
}

void on_menu_move_left (GtkMenuItem *menuitem, gpointer user_data)
{
	do_node_move_left();
}

void on_menu_move_right (GtkMenuItem *menuitem, gpointer user_data)
{
	do_node_move_right();
}

gint treeview_keyboard_handler(GtkWidget *widget, GdkEventKey *event)
{
	if( event->keyval == GDK_Insert ) 
	{
		if( event->state & GDK_SHIFT_MASK )
			on_menu_insert_child_node(NULL, 0);
		else
			on_menu_insert_node(NULL, 0);
		return FALSE;
	}
	else if( event->keyval == GDK_Delete ) 
	{
		on_menu_delete_node(NULL, 0);
		return FALSE;
	}
	else if( event->keyval == GDK_F2 )
	{
		on_menu_rename_node(NULL, 0);
		return FALSE;
	}
	else if( event->keyval == GDK_Down )
	{
		if( event->state & GDK_SHIFT_MASK )
			on_menu_move_down(NULL, 0); //move item one place down
		else
			g_tree.SelectionDown(); //select next item below current
	}
	else if( event->keyval == GDK_Up )
	{
		if( event->state & GDK_SHIFT_MASK )
			on_menu_move_up(NULL, 0); //move item one place up
		else
			g_tree.SelectionUp(); //select previous item above current
	}
	else if( event->keyval == GDK_Left )
	{
		if( event->state & GDK_SHIFT_MASK )
			on_menu_move_left(NULL, 0); //move item one place left
		else
			g_tree.SelectionLevelUp(); //collapse selected node OR select parent node
	}
	else if( event->keyval == GDK_Right )
	{
		if( event->state & GDK_SHIFT_MASK )
			on_menu_move_right(NULL, 0); //move item one place right
		else
			g_tree.SelectionLevelDown(); //expand selected node OR select the first child (if already expanded)
	}
	else if( event->keyval == GDK_Home )
	{
		g_tree.SelectionHome();	//select first tree item
	}
	else if( event->keyval == GDK_End )
	{
		g_tree.SelectionEnd(); //select last item (only expanded counts)
	}
	else if( event->keyval == GDK_Page_Up )
	{
		g_tree.SelectionPageUp(); //moves one page up (no nodes are being expanded/collapsed)
	}
	else if( event->keyval == GDK_Page_Down )
	{
		g_tree.SelectionPageDown();	//moves one page up (no nodes are being expanded/collapsed)
	}
	else if( event->keyval == GDK_F5 )
 	{
		g_text.SetFocus();
		return FALSE;
	}
	else if( event->keyval == GDK_Tab )
 	{
		g_text.SetFocus();
		return TRUE;
	}
	else if( event->keyval == GDK_Return )
 	{
		if( event->state & GDK_CONTROL_MASK )
		{
			on_menu_node_properties(NULL, 0);
			return FALSE;
		}
	}

	return TRUE;
}

//
// remember actual selection and cursor position in textView and 
// set corresponding values in nodeIdx's noteNode 
//
void RememberSelectionBounds( int nodeIdx )
{
	if( nodeIdx >= 0 ) {
		unsigned int cursor, selection;
		g_text.GetSelectionBounds( cursor, selection );
		NoteNode &myNode = g_doc.GetNodeByIdx(nodeIdx);
		myNode.SetSelStart( cursor );
		myNode.SetSelEnd( selection );
	}
}

//TOFIX move to DocTreeNavigation or similar class ?
void SelectNodeByIdx(int nIdx)	//bool bExpand
{
	if(nIdx < 0)
		return;	//invalid index

	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreePath *path2 = NULL;
	if(PathFromNodeIdx(nIdx, path2))
	{
		GtkTreeViewColumn *column1 = gtk_tree_view_get_column(treeview, 0);
		gtk_tree_view_expand_to_path(treeview, path2);
		gtk_tree_view_set_cursor(treeview, path2, column1, FALSE);	//refresh node selection
		gtk_tree_path_free(path2);
	}
}

int gtkMessageBox(const char *szText, int nButtons, int nIcon)
{
	GtkWidget* msgbox;
	msgbox = gtk_message_dialog_new ( (GtkWindow*)window1,
		GTK_DIALOG_DESTROY_WITH_PARENT,
		(GtkMessageType)nIcon,
		(GtkButtonsType)nButtons,
		szText);
	gint result = gtk_dialog_run (GTK_DIALOG (msgbox));
	gtk_widget_destroy (msgbox);
	return result;
}

void on_menu_import (GtkMenuItem *menuitem, gpointer user_data)
{
	FileDialog dlg(true);
	dlg.SetTitle(_("Import from file"));

	//define filters
	dlg.AddFilter(_("All supported formats (*.ncd,*.nce,*.hnc,*.gjots2,*.xml)"), "*.ncd|*.nce|*.hnc|*.gjots2|*.xml");
	dlg.AddFilter(_("NoteCase document (*.ncd)"), "*.ncd");
	dlg.AddFilter(_("NoteCase encrypted document (*.nce)"), "*.nce");
	dlg.AddFilter(_("NoteCenter document (*.hnc)"), "*.hnc");	
	dlg.AddFilter(_("Gjots2 document (*.gjots2)"), "*.gjots2");
	dlg.AddFilter(_("Sticky Notes document (*.xml)"), "*.xml");
	dlg.AddFilter(_("All files (*)"), "*");

	//TOFIX set initial directory from INI (store last used)
	//dlg.SetDirectory(const char *szPath);

	if(dlg.DoModal())
	{
		const gchar *filename = dlg.GetFilename();
		dlg.Close();

		NoteDocument doc;
		doc.Load(filename);	//TOFIX handle errors

		//IMPORTANT: get the facts before merge
		DocumentIterator it(g_doc);
		int nCnt   = it.GetChildCount(-1);
		int nTotal = it.GetNodeCount();

		//TOFIX ask user for import details and merge with current document
		g_doc.Merge(doc);

		//refresh tree starting from new content
		add_child_nodes(NULL, -1, nCnt);

		//push document change into undo/redo manager
		DocAction *pAction = new DocAction;
		pAction->SetType(ACT_TREE_IMPORT);
		pAction->SetDoc(g_doc);
		pAction->m_objSubTree = doc;	//copy content
		pAction->m_nNodeIndex = nTotal;	//first node to be removed

		g_undoManager.AddAction(pAction);
		UpdateUndoRedoMenus();
	}
}

void on_menu_export (GtkMenuItem *menuitem, gpointer user_data)
{
	//ask for export details
	int nExportMode = 0;	//0-curent node only, 1-cur node and children, 2-all document
	
	ExportDialog dlgExp;
	
	if(GTK_RESPONSE_OK == dlgExp.ShowModal())
	{
		//get result
		GtkWidget *button = lookup_widget(dlgExp.GetDialog(), "radiobutton1");
		if(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(button)))
			nExportMode = 0;
		else{
			button = lookup_widget(dlgExp.GetDialog(), "radiobutton2");
			if(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(button)))
				nExportMode = 1;
			else
			{
				button = lookup_widget(dlgExp.GetDialog(), "radiobutton3");
				if(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(button)))
					nExportMode = 2;
			}
		}
		gtk_widget_destroy (dlgExp.GetDialog());
	}
	else
	{
		gtk_widget_destroy (dlgExp.GetDialog());
		return;
	}
	

	//select export file name
	FileDialog dlg(false);
	dlg.SetTitle(_("Export to file"));

	//define filters
	if(0 == nExportMode)
	{
		dlg.AddFilter(_("Text file (*.txt)"), "*.txt");
	}
	else
	{
		dlg.AddFilter(_("All supported formats (*.ncd,*.nce,*.hnc,*.html)"), "*.ncd|*.nce|*.hnc|*.html");
		dlg.AddFilter(_("NoteCase document (*.ncd)"), "*.ncd");
		dlg.AddFilter(_("NoteCase encrypted document (*.nce)"), "*.nce");
		dlg.AddFilter(_("NoteCenter document (*.hnc)"), "*.hnc");	
		dlg.AddFilter(_("HTML file (*.html)"), "*.html");
	}
	dlg.AddFilter(_("All files (*)"), "*");

	//TOFIX set initial directory from INI (store last used)
	//dlg.SetDirectory(const char *szPath);

	if(dlg.DoModal())
	{
		const gchar *filename = dlg.GetFilename();
		dlg.Close();
		
		NoteDocument doc(g_doc);
		
		//TOFIX remove some branches based on export mode
		switch(nExportMode){
		case 0:
			{
				int nIdx = GetSelectedNodeIdx();
				if(nIdx >= 0)
				{
					//write note text into the file
					DocumentIterator it(doc);
					FILE *pFile = fopen(filename,"w");
					if(pFile)
					{
						fprintf(pFile, "%s", it.GetNodeByIdx(nIdx).GetText().c_str());
						fclose(pFile);
					}
				}
				break;
			}

		case 1:
			{
				int nIdx = GetSelectedNodeIdx();
				if(nIdx >= 0)
				{
					//TOFIX convert to use AssignSubtree
					//remove all notes from document not being children of this node
					//STEP 1: move this node to be the root node
					while(MoveNodeLeft(nIdx, doc))
					{
					}	

					//STEP 2: delete all other root nodes
					DocumentIterator it(doc);
					int nID = doc.GetNodeByIdx(nIdx).m_nID;
					int nSibling  = 0;
					int nChildIdx = it.GetChildIdx(-1, nSibling);
					while(nChildIdx >= 0)
					{
						int nChildID = doc.GetNodeByIdx(nChildIdx).m_nID; 
						if(nChildID == nID)
							nSibling ++;
						else
							doc.NodeDelete(nChildID);

						nChildIdx = it.GetChildIdx(-1, nSibling);	//keep looping
					}

					doc.Save(filename);
				}
				break;
			}

		case 2:
			doc.Save(filename);
			break;
		}
	}
}

//TOFIX use this wrapper more often
//TOFIX move to DocTreeNavigation or similar class ?
int GetSelectedNodeIdx()
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter  iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//get document node index from GUI tree iterator
		GtkTreePath* path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);
		gtk_tree_path_free(path1);
		return nIdx;
	}

	return -1;
}

//TOFIX move to DocTreeNavigation or similar class ?
bool MoveNodeLeft(int nIdx, NoteDocument &doc)
{
	int nParentID = doc.GetNodeByIdx(nIdx).m_nParentID;
	if(nParentID > -1)	//parent exists
	{
		int nIdx2 = doc.GetIdxFromID(nParentID);

		//change node position in the tree
		int nSIB = doc.GetNodeByIdx(nIdx).m_nSiblingIdx;
		int nNewParentID = doc.GetNodeByIdx(nIdx2).m_nParentID;
		doc.GetNodeByIdx(nIdx).m_nParentID = nNewParentID;
		DocumentIterator itDoc(doc);
		
		//position new node as first sibling after its former parent!
		int nNewSIB = doc.GetNodeByIdx(nIdx2).m_nSiblingIdx + 1;

		//refresh indexes of new siblings
		int i;
		for(i=0; i<doc.GetNodeCount(); i++)
		{
			if( doc.GetNodeByIdx(i).m_nParentID == nNewParentID &&
				doc.GetNodeByIdx(i).m_nSiblingIdx >= nNewSIB) 
			{
				doc.GetNodeByIdx(i).m_nSiblingIdx ++;
			}
		}

		//set new sibling idx for the node
		doc.GetNodeByIdx(nIdx).m_nSiblingIdx = nNewSIB;
		
		//refresh indexes of previous siblings
		for(i=0; i<doc.GetNodeCount(); i++)
		{
			if( doc.GetNodeByIdx(i).m_nParentID == nParentID &&
				doc.GetNodeByIdx(i).m_nSiblingIdx > nSIB) 
			{
				doc.GetNodeByIdx(i).m_nSiblingIdx --;
			}
		}

		return true;
	}

	return false;
}

void on_options1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	OptionsDialog dlg;
	dlg.ShowModal();
}

void on_find1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//find dialog
	FindDialog dlg;
	
	GtkWidget *casesensitive_chk = lookup_widget(dlg.GetDialog(), "casesensitive_chk");
	GtkWidget *entry1 = lookup_widget(dlg.GetDialog(), "entry1");

	//initialize from current settings
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(casesensitive_chk), g_bFindSensitive);
	gtk_entry_set_text(GTK_ENTRY(entry1), g_strFindText.c_str());

	if(GTK_RESPONSE_OK == dlg.ShowModal())
	{
		TextSearch search;

		if(gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(casesensitive_chk)))
			g_bFindSensitive = true;
		else
			g_bFindSensitive = false;

		g_strFindText = gtk_entry_get_text (GTK_ENTRY(entry1));
		if(g_strFindText.empty())
		{
			gtk_widget_destroy (dlg.GetDialog());
			gtkMessageBox(_("Error: Search text is empty!"));
			return;
		}

		//start searching from the current node
		int nIdx = GetSelectedNodeIdx();
		if(nIdx < 0 && g_doc.GetNodeCount() > 0)
			nIdx = 0;

		DocumentIterator it(g_doc);
		g_nFindCurNodeRecursiveIdx   = it.NodeIdx2RecursiveIdx(nIdx);
		g_nFindStartNodeRecursiveIdx = g_nFindCurNodeRecursiveIdx;
		g_nFindBufferPos = 0; //start from begining of the buffer

		//start find
		text_find();
	}
	gtk_widget_destroy (dlg.GetDialog());
}

void on_find2_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//refresh current node in case user moved in the document
	//or done some editing after the find action was started
	int nIdx = GetSelectedNodeIdx();
	if(nIdx < 0 && g_doc.GetNodeCount() > 0)
		nIdx = 0;

	DocumentIterator it(g_doc);
	g_nFindCurNodeRecursiveIdx  = it.NodeIdx2RecursiveIdx(nIdx);
	g_nFindBufferPos            = g_text.GetSelectionEnd(); //start searching from current caret position?
	
	//find next
	if(!g_strFindText.empty())
		text_find();
}

void text_find()
{
	DocumentIterator it(g_doc);
	TextSearch search;

	//set search parameters
	search.SetSearchPattern(g_strFindText.c_str());
	if(!g_bFindSensitive)
		search.SetScanStyle(FS_CASE_INSENSITIVE);
	
	//start searching the current buffer
	int nCurIdx = GetSelectedNodeIdx();

	//search all nodes in a depth first order
	while(g_nFindCurNodeRecursiveIdx >= 0)
	{
		int nIdx = it.RecursiveIdx2NodeIdx(g_nFindCurNodeRecursiveIdx);
		if(nIdx < 0){
			gtkMessageBox(_("No more results found!"));
			return;	//no more
		}

		//TOFIX find in title

		//find in node
		const char *szBuf = g_doc.GetNodeByIdx(nIdx).GetText().c_str();
		search.SetScanBuffer(szBuf, strlen(szBuf));

		int nPos = search.Search(g_nFindBufferPos);
		if(nPos >= 0){
			//result found!!!

			//if not active node, activate it!
			if(nIdx != nCurIdx){
				SelectNodeByIdx(nIdx);
				nCurIdx = nIdx;
			}

			//select pattern in a text view
			g_text.SelectRange(nPos, nPos+g_strFindText.size());
			g_text.EnsureVisible(nPos);

			//restore focus
			GtkWidget *textview = lookup_widget(window1, "textview1");
			gtk_window_set_focus(GTK_WINDOW(window1), textview);

			g_nFindBufferPos = nPos+g_strFindText.size();
			break;
		}

		//TOFIX support direction up

		g_nFindCurNodeRecursiveIdx ++;
		
		//TOFIX reached to the end of document, search form top!
		//if(g_nFindCurNodeRecursiveIdx >= it.GetNodeCount())
		//	if(g_nFindStartNodeRecursiveIdx)

		g_nFindBufferPos = 0;
	}
}

void set_wrap_activated(bool bActivate)
{
	bool bIsActive = g_text.IsWrapped();

	if((bActivate && !bIsActive) || (!bActivate && bIsActive))
		refresh_wrap_menu(bActivate); //this will trigger the signal
}

void refresh_wrap_menu (bool bActivate)
{
	GtkWidget *wrap1	= lookup_widget(window1, "wrap1");
	if(bActivate)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(wrap1), TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(wrap1), FALSE);
}

void on_wrap_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *textview = lookup_widget(window1, "textview1");

	//swap wrapping state of text view control
	if(!g_text.IsWrapped())
		gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
	else
		gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_NONE);
}

void on_show_node_titlebar_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *title1 = lookup_widget(window1, "title1");
	GtkWidget *label1 = lookup_widget(window1, "label1");

	if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(title1)))
		gtk_widget_show(label1);
	else
		gtk_widget_hide(label1);
}

bool get_node_title_set()
{
	GtkWidget *label1 = lookup_widget(window1, "label1");
	return GTK_WIDGET_VISIBLE(label1);
}

void refresh_nodetitle_menu(bool bActivate)
{
	GtkWidget *title1	= lookup_widget(window1, "title1");
	if(bActivate)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(title1), TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(title1), FALSE);
}

void set_show_node_title(bool bShow)
{
	bool bIsVisible = get_node_title_set();

	if((bShow && !bIsVisible) || (!bShow && bIsVisible))
		refresh_nodetitle_menu(bShow); //this will trigger the signal
}

void on_show_toolbar_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *menutool1 = lookup_widget(window1, "menutool1");
	GtkWidget *toolbar1 = lookup_widget(window1, "toolbar1");

	if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menutool1)))
		gtk_widget_show(toolbar1);
	else
		gtk_widget_hide(toolbar1);
}

void refresh_toolbar_menu(bool bActivate)
{
	GtkWidget *menutool1 = lookup_widget(window1, "menutool1");
	if(bActivate)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menutool1), TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menutool1), FALSE);
}

void set_show_toolbar(bool bShow)
{
	bool bIsVisible = get_show_toolbar();

	if((bShow && !bIsVisible) || (!bShow && bIsVisible))
		refresh_toolbar_menu(bShow); //this will trigger the signal
}

bool get_show_toolbar()
{
	GtkWidget *toolbar1 = lookup_widget(window1, "toolbar1");
	return GTK_WIDGET_VISIBLE(toolbar1);
}

void on_undo_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	g_undoManager.Undo();
	UpdateUndoRedoMenus();
}

void on_redo_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	g_undoManager.Redo();
	UpdateUndoRedoMenus();
}

void UpdateUndoRedoMenus()
{
	// update undo/redo menu/toolbar state
	GtkWidget *undo1 = lookup_widget(window1, "undo1");
	GtkWidget *redo1 = lookup_widget(window1, "redo1");

	GtkWidget *toolbar1 = lookup_widget(window1, "toolbar1");
	//GtkWidget *tbr_undo = lookup_widget(toolbar1, "tbr_undo");
	//GtkWidget *tbr_redo = lookup_widget(toolbar1, "tbr_redo");
	GtkWidget *tbr_undo = (GtkWidget *)gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar1), 5);
	GtkWidget *tbr_redo = (GtkWidget *)gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar1), 6);
	

	if(g_undoManager.CanUndo())
	{
		gtk_widget_set_sensitive(undo1, TRUE);
		gtk_widget_set_sensitive(tbr_undo, TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(undo1, FALSE);
		gtk_widget_set_sensitive(tbr_undo, FALSE);
	}

	if(g_undoManager.CanRedo())
	{
		gtk_widget_set_sensitive(redo1, TRUE);
		gtk_widget_set_sensitive(tbr_redo, TRUE);
	}
	else
	{
		gtk_widget_set_sensitive(redo1, FALSE);
		gtk_widget_set_sensitive(tbr_redo, FALSE);
	}
}

void SetNodeTitle(int nIdx, const char *szTitle)
{
	//TOFIX recursive index
	g_doc.GetNodeByIdx(nIdx).SetTitle(szTitle);
	
	//store new text into the tree store for given cell
	GtkTreePath *path1 = NULL;
	if(PathFromNodeIdx(nIdx, path1))
	{
		GtkWidget *treeview = lookup_widget(window1, "treeview1");
		GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);
		
		GtkTreeIter iter;
		gtk_tree_model_get_iter(model, &iter, path1);
		gtk_tree_store_set (GTK_TREE_STORE(model), &iter, STORE_IDX_TEXT, szTitle, -1);
		gtk_tree_path_free(path1);
	}

	//update node title label
	if(g_nActiveNodeIdx == nIdx){
		set_title_bar(szTitle);
	}
}

void InsertNodeText(int nIdx, int nOffset, const char *szText)
{
	//select node if needed
	if(nIdx != GetSelectedNodeIdx())
		SelectNodeByIdx(nIdx);

	g_text.InsertText(nOffset, szText);
}

void DeleteNodeText(int nIdx, int nOffset, int nLength)
{
	//select node if needed
	if(nIdx != GetSelectedNodeIdx())
		SelectNodeByIdx(nIdx);

	g_text.DeleteText(nOffset, nLength);
}

void on_insert_text (GtkTextBuffer *textbuffer,
                     GtkTextIter *arg1,
                     gchar *arg2,
                     gint arg3,
                     gpointer user_data)
{
	unsigned int nOffset = gtk_text_iter_get_offset(arg1);
	int nIdx = GetSelectedNodeIdx();

	DocAction *pCurAction = (DocAction *)g_undoManager.GetCurrentAction();
	if(pCurAction && 
	   pCurAction->GetType() == ACT_TEXT_INSERT &&
	   pCurAction->m_nNodeIndex == nIdx &&
	   nOffset == (pCurAction->m_nTextStartPos + pCurAction->m_strNodeText.size()))
	{
		//join multiple typed characters into single DocAction change
		pCurAction->m_strNodeText += arg2;
	}
	else{
		//push document change into undo/redo manager
		DocAction *pAction = new DocAction;
		pAction->SetType(ACT_TEXT_INSERT);
		pAction->SetDoc(g_doc);
		pAction->m_nNodeIndex = nIdx; //TOFIX recursive index
		pAction->m_nTextStartPos = nOffset;
		pAction->m_strNodeText = arg2;
	
		g_undoManager.AddAction(pAction);
		UpdateUndoRedoMenus();
	}
}

void on_delete_text (GtkTextBuffer *textbuffer,
                     GtkTextIter *arg1,
                     GtkTextIter *arg2,
                     gpointer user_data)
{
	unsigned int nOffset = gtk_text_iter_get_offset(arg1);
	int nIdx = GetSelectedNodeIdx();

	//refresh text
	gchar *szText = gtk_text_buffer_get_text(textbuffer, arg1, arg2, FALSE);
	int nTxtLen   = strlen(szText);

	DocAction *pCurAction = (DocAction *)g_undoManager.GetCurrentAction();
	if(pCurAction && 
	   pCurAction->GetType() == ACT_TEXT_DELETE &&
	   pCurAction->m_nNodeIndex == nIdx &&
	   nOffset == (unsigned int)pCurAction->m_nTextStartPos)
	{
		//join multiple deleted characters into single DocAction change
		//deletion using Delete key
		pCurAction->m_strNodeText += szText;
	}
	else if(pCurAction && 
	   pCurAction->GetType() == ACT_TEXT_DELETE &&
	   pCurAction->m_nNodeIndex == nIdx &&
	   nOffset == (unsigned int)(pCurAction->m_nTextStartPos - nTxtLen))
	{
		//join multiple deleted characters into single DocAction change
		//deletion using Backspace key
		pCurAction->m_strNodeText = szText + pCurAction->m_strNodeText;
		pCurAction->m_nTextStartPos -= nTxtLen;
	}
	else{
		//push document change into undo/redo manager
		//new deletion action
		DocAction *pAction = new DocAction;
		pAction->SetType(ACT_TEXT_DELETE);
		pAction->SetDoc(g_doc);
		pAction->m_nNodeIndex = nIdx; //TOFIX recursive index
		pAction->m_nTextStartPos = nOffset;
		pAction->m_strNodeText = szText;
		
		g_undoManager.AddAction(pAction);
		UpdateUndoRedoMenus();
	}
	
	g_free (szText);
}

void TreeIterFromID(GtkTreeIter  &iter, int nNodeID)
{
	int nIdx = g_doc.GetIdxFromID(nNodeID);
	IteratorFromNodeIdx(nIdx, iter);
}

void do_node_move_up (bool bStoreUndo)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//get document node index from GUI tree iterator
		GtkTreePath* path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);

		if(nIdx > -1)
		{
			int nParentID = g_doc.GetNodeByIdx(nIdx).m_nParentID;
			int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

			//find previous sibling child if exists
			DocumentIterator itDoc(g_doc);
			int nIdx2 = itDoc.GetChildIdx(nParentID, nSIB-1);
			if(-1 == nIdx2)
			{
				//TOFIX is it allowed to change node level with this fn.
				//no previous sibling, move one level up
				if(nParentID >= 0)
				{
					do_node_move_left (false);
					SelectNodeByIdx(nIdx);
					do_node_move_up (false);
				}
			}
			else if(nIdx2 > -1)
			{
				//find previous sibling
				if(!gtk_tree_path_prev(path1)){
					gtk_tree_path_free(path1);
					return;
				}

				//iter from path
				GtkTreeIter itPrev;
				gtk_tree_model_get_iter(model, &itPrev, path1);

				//swap nodes (with all child nodes)
				//gtk_tree_store_swap(GTK_TREE_STORE(model), &iter, &itPrev); //BUG crashes!
				gtk_tree_store_move_before(GTK_TREE_STORE(model), &iter, &itPrev);
	
				int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

				//swap sibling indexes
				g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx  --; 
				g_doc.GetNodeByIdx(nIdx2).m_nSiblingIdx ++;

				g_doc.SetModified(true);
				RefreshMainTitle();
				//TOFIX refresh node selection and text, update active ID, ...
				
				if(bStoreUndo)
				{
					//push document change into undo/redo manager
					DocAction *pAction = new DocAction;
					pAction->SetType(ACT_TREE_MOVE);
					pAction->SetDoc(g_doc);
					pAction->m_nMoveDirection = MOVE_UP;
					pAction->m_nNodeIndex = nIdx; //TOFIX recursive index
					pAction->m_nNodeSibling = nSIB;

					g_undoManager.AddAction(pAction);
					UpdateUndoRedoMenus();
				}
			}
		}
		gtk_tree_path_free(path1);
	}	
}

void do_node_move_down(bool bStoreUndo)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//get document node index from GUI tree iterator
		GtkTreePath* path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);

		if(nIdx > -1)
		{
			int nParentID = g_doc.GetNodeByIdx(nIdx).m_nParentID;
			int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

			//find next sibling child if exists
			DocumentIterator itDoc(g_doc);
			int nIdx2 = itDoc.GetChildIdx(nParentID, nSIB+1);
			if(nIdx2 > -1)
			{
				//find next sibling
				//iter from path
				GtkTreeIter *itNext = gtk_tree_iter_copy(&iter);
				if(!gtk_tree_model_iter_next(model, itNext))
				{
					gtk_tree_iter_free(itNext);
					gtk_tree_path_free(path1);
					return;
				}
				
				//swap nodes (with all child nodes)
				gtk_tree_store_move_after(GTK_TREE_STORE(model), &iter, itNext);
				gtk_tree_iter_free(itNext);
	
				int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

				//swap sibling indexes
				g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx  ++; 
				g_doc.GetNodeByIdx(nIdx2).m_nSiblingIdx --;

				g_doc.SetModified(true);
				RefreshMainTitle();

				if(bStoreUndo)
				{
					//push document change into undo/redo manager
					DocAction *pAction = new DocAction;
					pAction->SetType(ACT_TREE_MOVE);
					pAction->SetDoc(g_doc);
					pAction->m_nMoveDirection = MOVE_DOWN;
					pAction->m_nNodeIndex = nIdx; //TOFIX recursive index
					pAction->m_nNodeSibling = nSIB;

					g_undoManager.AddAction(pAction);
					UpdateUndoRedoMenus();
				}
			}
		}
		gtk_tree_path_free(path1);
	}	
}

void do_node_move_left(bool bStoreUndo)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//get document node index from GUI tree iterator
		GtkTreePath* path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);

		if(nIdx > -1)
		{
			int nParentID = g_doc.GetNodeByIdx(nIdx).m_nParentID;
			if(nParentID > -1)	//parent exists
			{
				int nIdx2 = g_doc.GetIdxFromID(nParentID);

				//get parent's parent path
				if(!gtk_tree_path_up(path1)){
					gtk_tree_path_free(path1);
					return;
				}
				if(!gtk_tree_path_up(path1)){
					gtk_tree_path_free(path1);
					return;
				}	
				//iter from path
				GtkTreeIter *itParent = NULL;
				GtkTreeIter itUp;
				if(gtk_tree_model_get_iter(model, &itUp, path1))
					itParent = &itUp;
				
				//delete current node
				gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 
				
				int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

				int nNewParentID = g_doc.GetNodeByIdx(nIdx2).m_nParentID;
				MoveNodeLeft(nIdx, g_doc);

				//attach as new child of a new parent (including subnodes)
				while(gtk_tree_model_iter_children(model, &iter, itParent))
					gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); //delete all children
				add_child_nodes(&itUp, nNewParentID);				     //reattach children with new chld	
	
				g_doc.SetModified(true);
				RefreshMainTitle();
				
				gtk_tree_view_expand_row(treeview, path1, FALSE);	//expand new parent
				SelectNodeByIdx(nIdx);
				
				if(bStoreUndo)
				{
					//push document change into undo/redo manager
					DocAction *pAction = new DocAction;
					pAction->SetType(ACT_TREE_MOVE);
					pAction->SetDoc(g_doc);
					pAction->m_nMoveDirection = MOVE_LEFT;
					pAction->m_nNodeIndex = nIdx; //TOFIX recursive index
					pAction->m_nNodeSibling = nSIB;

					g_undoManager.AddAction(pAction);
					UpdateUndoRedoMenus();
				}
			}
		}
		gtk_tree_path_free(path1);
	}
}

void do_node_move_right(bool bStoreUndo)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//get document node index from GUI tree iterator
		GtkTreePath* path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);
		if(nIdx > -1)
		{
			//find previous sibling
			if(!gtk_tree_path_prev(path1)){
				gtk_tree_path_free(path1);
				return;
			}

			//iter from path
			GtkTreeIter itPrev;
			gtk_tree_model_get_iter(model, &itPrev, path1);

			//delete current node
			gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 
					
			//change node position in the tree
			int nParentID = g_doc.GetNodeByIdx(nIdx).m_nParentID;
			int nSIB = g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx;

			DocumentIterator itDoc(g_doc);
			int nNewParentIdx = itDoc.GetChildIdx(nParentID, nSIB-1);
			if(nNewParentIdx < 0){
				gtk_tree_path_free(path1);
				return;
			}
			int nNewParentID = g_doc.GetNodeByIdx(nNewParentIdx).m_nID;

			//change parent
			g_doc.GetNodeByIdx(nIdx).m_nParentID = nNewParentID;
			
			//recalc sibling
			g_doc.GetNodeByIdx(nIdx).m_nSiblingIdx = itDoc.GetChildCount(nNewParentID) - 1;
			
			//refresh indexes of previous siblings
			for(int i=0; i<g_doc.GetNodeCount(); i++)
			{
				if( g_doc.GetNodeByIdx(i).m_nParentID == nParentID &&
					g_doc.GetNodeByIdx(i).m_nSiblingIdx > nSIB) 
				{
					g_doc.GetNodeByIdx(i).m_nSiblingIdx --;
				}
			}

			//attach as new child of a new parent (including subnodes)
			while(gtk_tree_model_iter_children(model, &iter, &itPrev))
				gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); //delete all children
			add_child_nodes(&itPrev, nNewParentID);				     //reattach children with new chld	

			g_doc.SetModified(true);
			RefreshMainTitle();
			
			gtk_tree_view_expand_row(treeview, path1, FALSE);	//expand new parent

			GtkTreePath *path2 = NULL;
			if(PathFromNodeIdx(nIdx, path2)){
				GtkTreeViewColumn *column1 = gtk_tree_view_get_column(treeview, 0);
				gtk_tree_view_set_cursor(treeview, path2, column1, FALSE);	//refresh node selection
				gtk_tree_path_free(path2);
			}
			
			if(bStoreUndo)
			{
				//push document change into undo/redo manager
				DocAction *pAction = new DocAction;
				pAction->SetType(ACT_TREE_MOVE);
				pAction->SetDoc(g_doc);
				pAction->m_nMoveDirection = MOVE_RIGHT;
				pAction->m_nNodeIndex = nIdx; //TOFIX recursive index ????
				pAction->m_nNodeSibling = nSIB;

				g_undoManager.AddAction(pAction);
				UpdateUndoRedoMenus();
			}
		}
		gtk_tree_path_free(path1);
	}
}

void on_reload1_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	if( g_doc.GetPath().size() > 0 )
	{
		gint result = gtkMessageBox(_("Do you want to reload the document and lose your possible changes?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES == result)
		{
			std::string strFile = g_doc.GetPath();

			int nResult = load_file(strFile.c_str());

			//do not delete MRU for file that exists, but failed to open
			g_objMRU.Change(strFile.c_str(), (DOC_LOAD_NOT_FOUND != nResult));	
		}
	}
	else
		gtkMessageBox(_("Document has not been saved yet!"));
}

void on_menu_expand_all (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	gtk_tree_view_expand_all(treeview);
}

void on_menu_collapse_all (GtkMenuItem *menuitem, gpointer user_data)
{
	//TOFIX fix tree selection losing when node is collapsed and becomes invisible -> set empty screen active?
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	gtk_tree_view_collapse_all(treeview);
}

void on_menu_node_properties (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//get iterator to selected node
	GtkTreeSelection* treesel = gtk_tree_view_get_selection (treeview);
	GtkTreeIter iter;

	if(gtk_tree_selection_get_selected(treesel, &model, &iter))
	{
		//TOFIX method to map from GUI tree iter to document node ID and back 
		GtkTreePath *path1 = gtk_tree_model_get_path(model, &iter);
		int nIdx = NodeIdxFromPath(path1);
		gtk_tree_path_free(path1);

		NodePropertiesDlg dlg;
		dlg.m_strTitle = g_doc.GetNodeByIdx(nIdx).GetTitle();
		dlg.Create();
		dlg.SetIconType(g_doc.GetNodeByIdx(nIdx).m_nIconType);
		if(ICON_CUSTOM == g_doc.GetNodeByIdx(nIdx).m_nIconType)
			dlg.SetIconValue(g_doc.GetNodeByIdx(nIdx).m_strIconFile.c_str());

		if(GTK_RESPONSE_OK == dlg.ShowModal())
		{
			if(0 != strcmp(g_doc.GetNodeByIdx(nIdx).GetTitle().c_str(), dlg.GetNodeTitle()))
			{
				SetNodeTitle(nIdx, dlg.GetNodeTitle());
				g_doc.SetModified(true);
				RefreshMainTitle();
			}

			//set icon
			g_doc.GetNodeByIdx(nIdx).m_nIconType = dlg.GetIconType();
			g_doc.GetNodeByIdx(nIdx).m_strIconFile = dlg.GetIconValue();
			if(dlg.GetIconType() == ICON_INTERNAL_FIRST)
			{
				g_doc.GetNodeByIdx(nIdx).m_nIconType += InternalIcon_Name2Index(dlg.GetIconValue());
			}
			UpdateNodeIcon(nIdx);
			g_doc.SetModified(true); //TOFIX detect if icon changed
			RefreshMainTitle();
		}
	}
	else
		gtkMessageBox(_("No Node selected"));
}

void ShowBusyCursor()
{
	// set busy curosr and disable the window
	GdkCursor *cursor=gdk_cursor_new(GDK_WATCH);
	gtk_widget_set_sensitive(GTK_WIDGET(window1),FALSE);

	gdk_pointer_grab(window1->window, 
			TRUE,
			(GdkEventMask)(GDK_POINTER_MOTION_MASK|GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK),
			NULL, 
			cursor,
			GDK_CURRENT_TIME);
	gdk_cursor_unref (cursor);
}

void HideBusyCursor()
{
	// restore the cursor and enable the window
	gdk_pointer_ungrab(GDK_CURRENT_TIME);

	gtk_widget_set_sensitive(GTK_WIDGET(window1),TRUE);
}

void on_show_status_bar_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *menu = lookup_widget(window1, "statbar1");
	GtkWidget *statusbar1 = lookup_widget(window1, "statusbar1");

	if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menu)))
		gtk_widget_show(statusbar1);
	else
		gtk_widget_hide(statusbar1);
}

void refresh_statbar_menu(bool bActivate)
{
	GtkWidget *menutool1 = lookup_widget(window1, "statbar1");
	if(bActivate)
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menutool1), TRUE);
	else
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menutool1), FALSE);
}

void set_show_status_bar(bool bShow)
{
	bool bIsVisible = get_show_status_bar();

	if((bShow && !bIsVisible) || (!bShow && bIsVisible))
		refresh_statbar_menu(bShow); //this will trigger the signal
}

bool get_show_status_bar()
{
	GtkWidget *toolbar1 = lookup_widget(window1, "statusbar1");
	return GTK_WIDGET_VISIBLE(toolbar1);
}

void stop_autosave_timer()
{
	if(g_nAutosaveTimer > 0)
		g_source_remove (g_nAutosaveTimer);
	g_nAutosaveTimer = 0;
}

void restart_autosave()
{
	stop_autosave_timer();	//ensure timer is stopped

	//read ini settings
	IniFile file;
	file.Load(MRU::getfilename());

	bool bAutosave = false;
	file.GetValue("Other", "UseAutosave", bAutosave);
	if(bAutosave)
	{
		int nAutosaveTimer;
		file.GetValue("Other", "AutosaveTimer", nAutosaveTimer, 30);
		g_nAutosaveTimer = g_timeout_add (nAutosaveTimer*1000, autosave_timer, NULL);  
	}
}

gboolean autosave_timer(gpointer data)
{
	TRACE("Starting autosave timer\n");

	//save the current document to the special/reserved file/place
	if(g_doc.IsEncrypted())
		save_file(calculate_autosave_filename1(), false, true, g_doc.GetPassword());
	else
		save_file(calculate_autosave_filename(), false, true);

	return TRUE;
}

bool autosave_check_crash()
{
	if(0 == access(calculate_autosave_filename(), 0))
	{
		//crash file exists, ask the user for decision
		int nRes = gtkMessageBox(_("Autosave file found! Do you want to load it?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES == nRes)
			load_file(calculate_autosave_filename());

		//now, important thing, delete the autosave file
		remove(calculate_autosave_filename());
		return true;
	}

	//check for encrpyted format
	if(0 == access(calculate_autosave_filename1(), 0))
	{
		//crash file exists, ask the user for decision
		int nRes = gtkMessageBox(_("Autosave file found! Do you want to load it?"), GTK_BUTTONS_YES_NO);
		if(GTK_RESPONSE_YES == nRes)
			load_file(calculate_autosave_filename1());

		//now, important thing, delete the autosave file
		remove(calculate_autosave_filename1());
		return true;
	}
		
	return false;
}

void autosave_shutdown()
{
	stop_autosave_timer();	//ensure timer is stopped
	remove(calculate_autosave_filename());
	remove(calculate_autosave_filename1());
}

const char *calculate_autosave_filename()
{
	//there exists one autosave file per user
	static std::string strDir;
	strDir  = GetHomeDir();
#ifdef _WIN32
	strDir += "\\.notecase\\autosave.ncd";
#else
	strDir += "/.notecase/autosave.ncd";
#endif

	return strDir.c_str();
}

//this is a version for encrypted format
const char *calculate_autosave_filename1()
{
	//there exists one autosave file per user
	static std::string strDir;
	strDir  = GetHomeDir();
#ifdef _WIN32
	strDir += "\\.notecase\\autosave.nce";
#else
	strDir += "/.notecase/autosave.nce";
#endif

	return strDir.c_str();
}

const char *GetDocumentErrorString(int nErrCode)
{
	switch(nErrCode)
	{
		case DOC_LOAD_OK:
			return _("OK!");
		case DOC_LOAD_ABORTED:
			return _("Loading aborted by user!");
		case DOC_LOAD_NOT_FOUND:
			return _("File not found!");
		case DOC_LOAD_WRONG_PASSWORD:
			return _("Invalid document password!");
		case DOC_LOAD_WRONG_FORMAT:
			return _("Unsupported document format!");
		case DOC_LOAD_FORMAT_ERROR:
			return _("Error when parsing document (bad formatting)!");
		case DOC_LOAD_ERROR:
			return _("Failed to load the file!");
	}

	return _("Unknown error!");	//should never get this
}

gint textview_keyboard_handler(GtkWidget *widget, GdkEventKey *event)
{
	if( event->keyval == GDK_F5 )
	{
		g_tree.SetFocus();
		return FALSE;
	}
	return FALSE;
}

const char **InternalIcon_GetFromIdx(int nIdx)
{
	switch (nIdx)
	{
		case 0:	return (const char **)&blank_xpm;
		case 1:	return (const char **)&folder_xpm;
		case 2:	return (const char **)&help_xpm;
		case 3:	return (const char **)&lock_xpm;
		case 4:	return (const char **)&new_dir_xpm;
		case 5:	return (const char **)&recycle_xpm;
		default: return NULL;
	}
}

void UpdateNodeIcon(int nIdx)
{
	//get iterator to given node
	GtkWidget *treeview = lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);
	DocumentIterator itDoc(g_doc);

	//store new text into the tree store for given cell
	GtkTreePath *path1 = NULL;
	if(!PathFromNodeIdx(nIdx, path1))
		return;

	GtkTreeIter iter;
	gtk_tree_model_get_iter(model, &iter, path1);

	if(ICON_NONE != itDoc.GetNodeByIdx(nIdx).m_nIconType)
	{
		if(ICON_CUSTOM == itDoc.GetNodeByIdx(nIdx).m_nIconType)
		{
			//external xpm icon file
			GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_size(itDoc.GetNodeByIdx(nIdx).m_strIconFile.c_str(), 16, 16, NULL);
			if(pixbuf)
			{
				gtk_tree_store_set (GTK_TREE_STORE(model), &iter, STORE_IDX_ICON, pixbuf, -1);
				g_object_unref (G_OBJECT (pixbuf));
			}
		}
		else if(itDoc.GetNodeByIdx(nIdx).m_nIconType > ICON_CUSTOM)
		{
			//one of the internal icons
			const char **szIconData = InternalIcon_GetFromIdx(itDoc.GetNodeByIdx(nIdx).m_nIconType);
			GdkPixbuf *pixbuf = gdk_pixbuf_new_from_xpm_data (szIconData);
			GdkPixbuf *destpix = gdk_pixbuf_scale_simple(pixbuf, 16, 16, GDK_INTERP_NEAREST);
			g_object_unref (G_OBJECT (pixbuf));
			gtk_tree_store_set (GTK_TREE_STORE(model), &iter, STORE_IDX_ICON, destpix, -1);
			g_object_unref (G_OBJECT (destpix));
		}
	}
	else
		gtk_tree_store_set (GTK_TREE_STORE(model), &iter, STORE_IDX_ICON, NULL, -1);	//clear icon

	gtk_tree_path_free(path1);
}

void on_dnd_row_deleted (GtkTreeModel *treemodel, GtkTreePath *arg1, gpointer user_data)
{
	if(g_tree.m_bDND)	// handles only DND
	{
		//TOFIX add support for UNDO in DND
		int nIdx = NodeIdxFromPath(arg1);
		if(nIdx >= 0)
		{
			//debugging
			TRACE("Dnd: Deleting node idx=%d at path [%s]\n", nIdx, gtk_tree_path_to_string(arg1));

			//on dragged subtree root node being deleted, first restore all the properties
			//of its node and subnodes at the new position (before deleting the nodes in document)
			int nNewRootIdx = g_tree.m_nFirstDroppedIdx;

			//is this the first node processed in this DnD operation ?
			//(if so, than it is the one being dragged, the top of the branch)
			if(-1 == g_tree.m_nFirstDraggedIdx)
				g_tree.m_nFirstDraggedIdx = nIdx;

			//copy node contents into new nodes 
			//(map old subtree starting at nIdx to new subtree starting at nNewRootIdx)
			if(nNewRootIdx >= 0)
			{
				DocumentIterator it(g_doc);

				int nNewPID = it.GetNodeByIdx(nNewRootIdx).m_nParentID;
				int nNewSID = it.GetNodeByIdx(nNewRootIdx).m_nSiblingIdx;

				//copy undo data right now (on first/top node being deleted)
				//push document change into undo/redo manager (menu is updated at the end of DnD)
				DocAction *pAction = new DocAction;
				pAction->SetType(ACT_TREE_DND);
				pAction->SetDoc(g_doc);
				pAction->m_nNodeIndex = nIdx;
				pAction->m_objSubTree.AssignSubtree(g_doc, nIdx);
				pAction->m_nNodeID         = it.GetNodeByIdx(nIdx).m_nID;
				pAction->m_nNodePID        = it.GetNodeByIdx(nIdx).m_nParentID;
				pAction->m_nNodeSibling    = it.GetNodeByIdx(nIdx).m_nSiblingIdx;
				pAction->m_nNewNodeID      = pAction->m_nNodeID;
				pAction->m_nNodeNewPID     = nNewPID;
				pAction->m_nNodeNewSibling = nNewSID;

				g_undoManager.AddAction(pAction);
			
				//move old node subtree to new place
				g_doc.NodeDelete(it.GetNodeByIdx(nNewRootIdx).m_nID); //was a placeholder
				g_doc.NodeMove(g_doc.GetNodeByIdx(nIdx).m_nID, nNewPID, nNewSID);
			}
		}

		//mark document changed
		g_text.SetModified(true);
		RefreshMainTitle();
	}
}

void on_dnd_row_inserted (GtkTreeModel *treemodel, GtkTreePath *arg1, GtkTreeIter *arg2, gpointer user_data)
{
	if(g_tree.m_bDND)	// handles only DND
	{
		//
		// make a new node at drop point (copying from selected node)
		//

		//calc parent node
		int nPID = -1;
		GtkTreePath *parent = gtk_tree_path_copy(arg1);
		if(gtk_tree_path_up(parent)){
			int nParentIdx = NodeIdxFromPath(parent);
			if(nParentIdx >= 0){
				ASSERT(0 <= nParentIdx && nParentIdx < g_doc.GetNodeCount());
				nPID = g_doc.GetNodeByIdx(nParentIdx).m_nID;
			}
		}

		//calc sibling position of a new node
		gint* arrIndices = gtk_tree_path_get_indices(arg1);
		int nMax = gtk_tree_path_get_depth(arg1);
		int nSID = arrIndices[nMax-1];

		DocumentIterator it(g_doc);
		int nPrevSiblingIdx = it.GetChildIdx(nPID, nSID);
		int nPrevSiblingID  = -1;	//sibling index of previous sibling node
		if(nPrevSiblingIdx >= 0)
			nPrevSiblingID  = g_doc.GetNodeByIdx(nPrevSiblingIdx).m_nSiblingIdx;

		// insert new node into document
		g_doc.NodeInsert(nPID, nPrevSiblingID);

		//remeber the new node index (data copy will be handled on deleting old node)
		int nNewIdx = g_doc.GetNodeCount()-1;

		//debugging
	#ifdef _DEBUG
		char *szPath1 = gtk_tree_path_to_string(arg1);
		char *szPath2 = gtk_tree_path_to_string(parent);
		TRACE("Dnd: Inserted new node idx=%d at path %s\n", nNewIdx, szPath1);
		TRACE("Dnd: Parent path [%s], Parent ID = %d, SID=%d, prevSibIdx=%d, prevSIB=%d\n", szPath2, nPID, nSID, nPrevSiblingIdx, nPrevSiblingID);
		g_free(szPath1);
		g_free(szPath2);
	#endif

		if(g_tree.m_nFirstDroppedIdx == -1)
			g_tree.m_nFirstDroppedIdx = nNewIdx;

		g_nActiveNodeIdx = -1;

	#ifdef _DEBUG
		g_doc.Dump();
	#endif
		gtk_tree_path_free(parent);
	}
}

