////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Class that stores/manipulates NoteCase document contents 
////////////////////////////////////////////////////////////////////////////

#include "NoteDocument.h"
#include "FormatHtml.h"
#include "FormatEncHtml.h"
#include "FormatGjots2.h"
#include "FormatStickyNotes.h"
#include "DocumentIterator.h"
#include "debug.h"

#ifdef _WIN32
 #include <io.h>
#else
 #include <sys/types.h>	//chmod
 #include <sys/stat.h>
#endif

NoteDocument::NoteDocument()
{
	m_pObjLoader = NULL;
	m_nIDGenerator = 0;	//first valid node ID
	m_nCurrentNode = -1;
	m_pfnPassCallback = NULL;
}

NoteDocument::~NoteDocument()
{
	Close();
}

NoteDocument::NoteDocument(const NoteDocument &doc)
{
	operator = (doc);
}

void NoteDocument::operator = (const NoteDocument &doc)
{
	//m_pObjLoader = doc.m_pObjLoader;	//TOFIX create new loader like this!
	m_pObjLoader	= NULL;

	m_nIDGenerator    = doc.m_nIDGenerator;
	m_nCurrentNode    = doc.m_nCurrentNode;
	m_pfnPassCallback = doc.m_pfnPassCallback;
	m_lstNodes        = doc.m_lstNodes;
	m_bModified       = doc.m_bModified;	
	m_strPath         = doc.m_strPath;
	m_strPassword     = doc.m_strPassword;
}

void NoteDocument::Close()
{
	Clear();
	
	if(m_pObjLoader)
		delete m_pObjLoader;
	m_pObjLoader = NULL;	
	
	m_strPath = "";	//forget path
	m_bModified = false;
}

void NoteDocument::Clear()
{
	m_lstNodes.erase(m_lstNodes.begin(), m_lstNodes.end());
}

bool NoteDocument::IsEncrypted()
{ 
	if(m_pObjLoader)
		return (FORMAT_HTML_ENC == m_pObjLoader->GetFormat()); 

	return false;
}
	
int NoteDocument::Load(const char *szPath)
{
	Close();

#ifdef _WIN32
 #define access _access
#endif

	//check if file exists
	if(0 != access(szPath, 0))
		return DOC_LOAD_NOT_FOUND;

	//TOFIX special factory fn.

	//calculate extension to determine format
	char *szExt = strrchr(szPath, '.');
	if(NULL == szExt)
		return DOC_LOAD_WRONG_FORMAT;

	m_strPassword = ""; //forget password
	
	if( 0 == strcmp(szExt, ".ncd"))
	{
		m_pObjLoader = new FormatHTML;
	}
	else if(0 == strcmp(szExt, ".hnc"))
	{
		m_pObjLoader = new FormatHTML;

		//set special flag for .hnc compatibility
		((FormatHTML *)m_pObjLoader)->m_bNoteCenterMode = true;
	}
	else if(0 == strcmp(szExt, ".nce"))
	{
		//password required to load encrypted file format
		if(NULL == m_pfnPassCallback)
			return false;
		const char *szPass = m_pfnPassCallback(szPath);
		if(NULL == szPass)
			return false;	//Cancel

		m_pObjLoader = new FormatEncHTML;
		
		//set password to the loader
		((FormatEncHTML *)m_pObjLoader)->SetPassword(szPass);

		m_strPassword = szPass;	//store password
	}
	else if(0 == strcmp(szExt, ".gjots2"))
	{
		m_pObjLoader = new FormatGjots2;
	}
	else if(0 == strcmp(szExt, ".xml"))
	{
		m_pObjLoader = new FormatStickyNotes;
	}
	else
		return DOC_LOAD_WRONG_FORMAT;

	int nResult = m_pObjLoader->Load(szPath, *this);
	if(DOC_LOAD_OK == nResult)
	{
		m_strPath = szPath;	//store path
		SetModified(false);	//just loaded
	}
	else
		m_strPassword = ""; //forget password

	return nResult;
}

bool NoteDocument::Save(const char *szPath, bool bRemberPath, const char *szPassword)
{
	//TOFIX special factory fn.

	//close previous loader object to prevent memory leaks
	if(m_pObjLoader)
		delete m_pObjLoader;
	m_pObjLoader = NULL;	
	
	//quick FIX: delete file if it already exists
	//TOFIX more precise fix would be to test/fix file open flags when saving
#ifdef _WIN32
	::DeleteFile(szPath);
#else
	remove(szPath);
#endif

	//calculate extension to determine format
	char *szExt = strrchr(szPath, '.');
	if(NULL == szExt)
		return false;
	
	 if( 0 == strcmp(szExt, ".ncd"))
	{
		m_pObjLoader = new FormatHTML;
	}
	else if( 0 == strcmp(szExt, ".html"))
	{
		m_pObjLoader = new FormatHTML;
		
		//set special flag for .html export
		((FormatHTML *)m_pObjLoader)->m_bHtmlExport = true;
	}
	else if(0 == strcmp(szExt, ".hnc"))
	{
		m_pObjLoader = new FormatHTML;

		//set special flag for .hnc compatibility
		((FormatHTML *)m_pObjLoader)->m_bNoteCenterMode = true;
	}
	else if(0 == strcmp(szExt, ".nce"))
	{
		//password required to load encrypted file format
		if(NULL == m_pfnPassCallback)
			return false;

		if( 0 == strcmp(szPath, m_strPath.c_str()) &&
			m_strPassword.size() > 0)
		{
			//password already exists for thsi path (stored)
		}
		else
		{
			if(NULL != szPassword)
				m_strPassword = szPassword;
			else
			{
				//ask for password
				const char *szPass = m_pfnPassCallback(szPath);
				if(NULL == szPass)
					return false;	//Cancel
				m_strPassword = szPass;	//store password
			}
		}

		m_pObjLoader = new FormatEncHTML;
		
		//set password to the loader
		((FormatEncHTML *)m_pObjLoader)->SetPassword(m_strPassword.c_str());
	}
	else
		return false;
		
	bool bSuccess = m_pObjLoader->Save(szPath, *this);
	if(bSuccess){
#ifndef _WIN32
		//set default file permission
		mode_t mode = S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH; //0x0644
		chmod(szPath, mode); 
#endif

		if(bRemberPath)
		{
			m_strPath = szPath;	//store path
			SetModified(false);	//just saved to known location
		}
	}
	else
		m_strPassword = ""; //forget password

	return bSuccess;
}

bool NoteDocument::IsOpen()
{
	return (GetPath().size() > 0);
}

bool NoteDocument::IsEmpty()
{
	return (m_lstNodes.size() > 0);
}

bool NoteDocument::IsModified()
{
	return m_bModified;
}

void NoteDocument::SetModified(bool bModified)
{
	m_bModified = bModified;
}

int  NoteDocument::GetFormat()
{
	if(m_pObjLoader)
		return m_pObjLoader->GetFormat();
	return FORMAT_NONE;	
}

NoteNode &NoteDocument::GetNodeByIdx(int nIdx)
{
	return m_lstNodes[nIdx];
}

int NoteDocument::GetNodeCount() const
{
	return m_lstNodes.size();
}

bool NoteDocument::NodeInsert(int nParentID, int nSiblingIdx)
{
	DocumentIterator it(*this);

#ifdef _DEBUG
	int nCountBefore = GetNodeCount();
#endif

	NoteNode note;
	note.m_nID = m_nIDGenerator;
	note.m_nParentID = nParentID;
	if(nSiblingIdx < 0)
		note.m_nSiblingIdx = it.GetChildCount(nParentID);
	else
	{
		note.m_nSiblingIdx = nSiblingIdx;	
		
		//refresh sibling indexes of the other sibling nodes
		for(int i=0; i<GetNodeCount(); i++)
		{
			if( m_lstNodes[i].m_nParentID == nParentID &&
				m_lstNodes[i].m_nSiblingIdx >= nSiblingIdx)
				m_lstNodes[i].m_nSiblingIdx ++;
		}
	}

	m_lstNodes.push_back(note);
	ASSERT(GetNodeCount() == nCountBefore + 1);

	m_nIDGenerator ++;
	m_bModified = true;
	return true;
}

bool NoteDocument::NodeRename(int nID, const char *szTitle)
{
	//TOFIX
	m_bModified = true;
	return true;
}

bool NoteDocument::NodeDelete(int nID)
{
	int nIdx = GetIdxFromID(nID);
	if(nIdx < 0)
		return false;

	// store sibling and parent ID
	int nParentID   = GetNodeByIdx(nIdx).m_nParentID;
	int nSiblingIdx = GetNodeByIdx(nIdx).m_nSiblingIdx;

	//actual deletion
	NodeDeleteRecursive(nID);

	//refresh all top-level parent siblings
	for(int i=0; i<GetNodeCount(); i++)
	{
		if( GetNodeByIdx(i).m_nParentID == nParentID &&
		    GetNodeByIdx(i).m_nSiblingIdx > nSiblingIdx)
		{
			GetNodeByIdx(i).m_nSiblingIdx --;
		}
	}

	m_bModified = true;
	return true;
}

void NoteDocument::NodeDeleteRecursive(int nID)
{
	int nIdx = GetIdxFromID(nID);
	if(nIdx < 0)
		return;

	//erase the node
	m_lstNodes.erase(m_lstNodes.begin()+nIdx);

	//delete all its children nodes
	int i = GetNodeCount()-1;
	while(i>=0)
	{
		if(GetNodeByIdx(i).m_nParentID == nID)
		{
			NodeDelete(GetNodeByIdx(i).m_nID);	//recurse into children node
			i = GetNodeCount()-1;	//scan again all over (index might have been changed)
		}
		else
			i--;
	}
}

bool NoteDocument::NodeMove(int nID, int nNewParentID, int nSiblingIdx)
{
#ifdef _DEBUG
	TRACE("Move node ID=%d to PID=%d, SID=%d\n", nID, nNewParentID, nSiblingIdx);
	Dump();
#endif
	int nIdx = GetIdxFromID(nID);
	if(nIdx >= 0){
		int nOldPID = GetNodeByIdx(nIdx).m_nParentID;
		int nOldSID = GetNodeByIdx(nIdx).m_nSiblingIdx;

		//redirect node position data
		GetNodeByIdx(nIdx).m_nParentID = nNewParentID;
		GetNodeByIdx(nIdx).m_nSiblingIdx = nSiblingIdx;

		if(nNewParentID == nOldPID)
		{	
			//SIB algorithm when moving node within the same branch (a<b):
			//1. moving from a->b (to higher pos) -- where sib between a & b
			//2. moving from b->a (to lower  pos) ++ where sib between a & b
			bool bToHigherPos = nOldSID < nSiblingIdx;
			int i=0;

			if(bToHigherPos)
			{
				//decrement target sibling by one (because we deleted old position in lower pos than new)
				GetNodeByIdx(nIdx).m_nSiblingIdx --;
				nSiblingIdx --;

				for(i=0; i<GetNodeCount(); i++)
				{
					if(i != nIdx && GetNodeByIdx(i).m_nParentID == nOldPID)
						if(GetNodeByIdx(i).m_nSiblingIdx >= nOldSID && GetNodeByIdx(i).m_nSiblingIdx <= nSiblingIdx)
							GetNodeByIdx(i).m_nSiblingIdx --;
				}
			}
			else
			{
				for(i=0; i<GetNodeCount(); i++)
				{
					if(i != nIdx && GetNodeByIdx(i).m_nParentID == nOldPID)
						if(GetNodeByIdx(i).m_nSiblingIdx >= nSiblingIdx && GetNodeByIdx(i).m_nSiblingIdx <= nOldSID)
							GetNodeByIdx(i).m_nSiblingIdx ++;
				}
			}
		}
		else
		{
			//fix other sibling indexes for old position
			int i=0;
			for(i=0; i<GetNodeCount(); i++)
			{
				if(i != nIdx && GetNodeByIdx(i).m_nParentID == nOldPID)
					if(GetNodeByIdx(i).m_nSiblingIdx >= nOldSID)
						GetNodeByIdx(i).m_nSiblingIdx --;
			}

			//fix other sibling indexes for new position
			for(i=0; i<GetNodeCount(); i++)
			{
				if(i != nIdx && GetNodeByIdx(i).m_nParentID == nNewParentID)
					if(GetNodeByIdx(i).m_nSiblingIdx >= nSiblingIdx)
						GetNodeByIdx(i).m_nSiblingIdx ++;
			}
		}
	}

#ifdef _DEBUG
	TRACE("After node moving\n");
	Dump();
#endif
	
	m_bModified = true;
	return true;
}

int NoteDocument::GetIdxFromID(int nID)
{
	for(unsigned int i=0; i<m_lstNodes.size(); i++)
	{
		if(m_lstNodes[i].m_nID == nID)
			return i;
	}

	return -1;
}

bool NoteDocument::Merge(NoteDocument &doc, int nParentID, int nSiblingIdx, bool bKeepIDs)
{
	DocumentIterator it(*this);

	int nIdOffset    = (bKeepIDs)? 0 : m_nIDGenerator;
	int nParentNodes = it.GetChildCount(nParentID);

	//append other object at the end of our
	int nCountExt = doc.GetNodeCount();
	for(int i=0; i<nCountExt; i++)
	{
		//insert empty node in current doc
		NodeInsert(-1, -1);
		int nIdx = GetNodeCount()-1;

		if(bKeepIDs)
			GetNodeByIdx(nIdx).m_nID = doc.GetNodeByIdx(i).m_nID;

		//initialize it with other doc's data
		GetNodeByIdx(nIdx).SetText(doc.GetNodeByIdx(i).GetText().c_str());
		GetNodeByIdx(nIdx).SetTitle(doc.GetNodeByIdx(i).GetTitle().c_str());

		//copy positioning info
		if(doc.GetNodeByIdx(i).m_nParentID >= 0)
			GetNodeByIdx(nIdx).m_nParentID = doc.GetNodeByIdx(i).m_nParentID + nIdOffset;
		else
			GetNodeByIdx(nIdx).m_nParentID = nParentID;

		//calculate new sibling index for the new node
		if(GetNodeByIdx(nIdx).m_nParentID == nParentID)
		{
			if(nSiblingIdx >= 0)	//specific starting sibling position requested
			{
				int nNewSID = doc.GetNodeByIdx(i).m_nSiblingIdx + nSiblingIdx;
				GetNodeByIdx(nIdx).m_nSiblingIdx = nNewSID;

				TRACE("Merge: NodeIdx=%d, SibIdx=%d\n", nIdx, nNewSID);

				//update sibling for all other sibling nodes in this position and higher
				for(int j=0; j<GetNodeCount(); j++)
				{
					if(j != nIdx && GetNodeByIdx(j).m_nParentID == nParentID)
					{
						if(GetNodeByIdx(j).m_nSiblingIdx >= nNewSID)
							GetNodeByIdx(j).m_nSiblingIdx ++;
					}
				}
			}
			else
				GetNodeByIdx(nIdx).m_nSiblingIdx = doc.GetNodeByIdx(i).m_nSiblingIdx + nParentNodes;
		}
		else
			GetNodeByIdx(nIdx).m_nSiblingIdx = doc.GetNodeByIdx(i).m_nSiblingIdx;
	}

	return true;
}

void NoteDocument::AssignSubtree(NoteDocument &doc, int nIdx)
{
	ASSERT(this != &doc);

	Clear(); //erase existing contents

	DocumentIterator it(doc);

	int nIdOffset = 0;
	int nSibOffset = - it.GetNodeByIdx(nIdx).m_nSiblingIdx;

	//copy other object's subtree (starting at nIdx) to this object
	int nCount = doc.GetNodeCount();
	for(int i=0; i<nCount; i++)
	{
		if(!(nIdx==i || it.IsAncestorByIdx(nIdx, i)))
			continue;

		//insert empty node in current doc
		NodeInsert(-1, -1);
		int nIdxNew = GetNodeCount()-1;

		//initialize it with other doc's data
		GetNodeByIdx(nIdxNew).m_nID = doc.GetNodeByIdx(i).m_nID;
		GetNodeByIdx(nIdxNew).SetText(doc.GetNodeByIdx(i).GetText().c_str());
		GetNodeByIdx(nIdxNew).SetTitle(doc.GetNodeByIdx(i).GetTitle().c_str());

		//copy positioning info
		if(doc.GetNodeByIdx(i).m_nParentID >= 0)
			GetNodeByIdx(nIdxNew).m_nParentID = doc.GetNodeByIdx(i).m_nParentID + nIdOffset;
		else
			GetNodeByIdx(nIdxNew).m_nParentID = -1;

		if(GetNodeByIdx(nIdxNew).m_nParentID >= 0)
			GetNodeByIdx(nIdxNew).m_nSiblingIdx = doc.GetNodeByIdx(i).m_nSiblingIdx;
		else
			GetNodeByIdx(nIdxNew).m_nSiblingIdx = doc.GetNodeByIdx(i).m_nSiblingIdx + nSibOffset;
	}
}

//used for debugging
void NoteDocument::Dump()
{
	TRACE("ID:  PID, SIB :Title\n");

	for(int i=0; i<GetNodeCount(); i++)
	{
		TRACE("%3d: %3d, %3d :%s\n",
			GetNodeByIdx(i).m_nID,
			GetNodeByIdx(i).m_nParentID,
			GetNodeByIdx(i).m_nSiblingIdx,
			GetNodeByIdx(i).GetTitle().c_str());
	}
}

