/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Ubuntu.Unity.Action 1.0 as UnityActions
import NotesApp.Plugins 0.1
import "Components"
import "Models"

MainView {
    id: notesMainWindow
    applicationName: "notes-app"
    width: units.gu(40)
    height: units.gu(71)
    automaticOrientation: true

    actions: [
        UnityActions.Action {
            text: i18n.tr("Add")
            keywords: i18n.tr("Create New Note")
            onTriggered: notesMainWindow.createNewNote()
        },
        UnityActions.Action {
            text: i18n.tr("Delete")
            keywords: i18n.tr("Trash;Erase Note")
            enabled: noteList.currentIndex !== -1
            onTriggered: dataModel.deleteNote(noteList.currentIndex)
        }
    ]

    NotesApplication {
        id: application

        /* We need to set this because we are being run through qmlscene, but
           our custom CachingProvider image provider needs to reserve a directory
           to save its data, and Qt will reserve it based on application name */
        applicationName: "notes-app"

        onAboutToQuit: {
            if (noteList.currentIndex !== -1)
                noteList.commitNote(noteList.currentIndex, true);
        }
    }

    Page {
        id: page

        anchors.fill: parent
        title: i18n.tr("Notes")

        tools: ToolbarItems {
            ToolbarButton {
                action: Action {
                    text: i18n.tr("Add")
                    iconSource: Qt.resolvedUrl("Components/graphics/add.png")
                    onTriggered: {
                        page.tools.opened = false
                        notesMainWindow.createNewNote()
                    }
                }
            }
        }

        // model to access the database of Notes
        DataModel {
            id: dataModel
        }

        // background
        Rectangle {
            anchors.fill: parent
            color: "#fff6e5"
        }

        NoteList {
            id: noteList
            anchors.fill: parent
            anchors.topMargin: units.gu(1.5)
            focus: true
            model: dataModel
        }

        Connections {
            target: Qt.inputMethod
            onVisibleChanged: {
                if (!Qt.inputMethod.visible) {
                    noteList.focus = true;
                }
            }
        }

    }

    function createNewNote() {
        // Don't create a new note if there's only one and it's empty already,
        // but ensure it's activated so that the user can start typing
        if (dataModel.count === 1 &&
            noteList.commitNote(0, false).length === 0) {
        } else {
            dataModel.newNote();
        }
        noteList.setActiveIndex(0);
    }

    KeyboardRectangle {
    }
}
