/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#ifndef INSTRUMENT_TABLE_H

#define INSTRUMENT_TABLE_H
#include <gtk/gtk.h>

class NedTableInstrument;
class NedInstrumentTable;
class NedMainWindow;
class NedChordOrRest;

class NedTableTuplet {
	public:
		NedTableTuplet(GList *members, unsigned long long t0, unsigned long long t2) :
			m_members(members), m_t0(t0), m_t2(t2), m_bad(false) {}
		bool overlaps(unsigned long long t0, unsigned long long t2);
		GList *m_members;
		unsigned long long m_t0, m_t2;
		bool m_bad;
};

class NedTableTimesig {
	public:
		NedTableTimesig(unsigned long long midi_time, int numerator, int denominator) :
			m_midi_time(midi_time), m_numerator(numerator), m_denominator(denominator) {}
		static int compare(NedTableTimesig *t1, NedTableTimesig *t2);
		unsigned int getDuration();
		unsigned long long m_midi_time;
		int m_numerator;
		int m_denominator;
		unsigned int m_measure_number;
};

class NedTableTemposig {
	public:
		NedTableTemposig(unsigned long long midi_time, int tempo) :
			m_midi_time(midi_time), m_tempo(tempo), m_used(false) {}
		static int compare(NedTableTemposig *t1, NedTableTemposig *t2);
		unsigned long long m_midi_time;
		int m_tempo;
		bool m_used;
};


class NedTableNote {
	public:
		NedTableNote(unsigned long long start, unsigned long long stop, int pitch, int volume);
		NedTableNote(bool is_rest, unsigned long long start, unsigned long long stop, int pitch, int volume);
		NedTableNote *clone(unsigned long long measure_start);
		NedTableNote *clone_with_different_length(unsigned long long start, unsigned long long end);
		void clone_chord(NedTableNote *pattern);
		NedTableNote *m_tie_forward, *m_tie_backward;
		GList *getGroup(int *av_midi_pitch);
		void tripletRecognition(unsigned long long  base);
		bool snapWouldFit();
		bool overlapping(NedTableNote *other);
		void print();
		unsigned long long m_start, m_stop;
		unsigned long long m_ori_start, m_ori_stop;
		unsigned long long m_triplet_start, m_triplet_stop;
		int m_midi_pitch;
		int m_volume;
		int m_line;
		unsigned int m_state;
		bool m_start_fix, m_end_fix;
		bool m_is_triplet, m_tr_start, m_tr_stop;
		bool m_takenForTuplet;
		bool m_handledAsTuplet;
		bool m_triplet_split_note;
		unsigned int getDuration();
		static int compare(NedTableNote *n1, NedTableNote *n2);
		static GList *my_insert_sorted(GList *list, NedTableNote *nnote,
			 gint (*cfunc)(const void*, const void*), bool allow_doubles);
		NedTableNote *snapNoteToMeasure(unsigned long long measure_start, unsigned long long measure_end,
				bool *changed);
		void snapNote(unsigned long long next_start);
		bool m_is_rest;
		unsigned long long m_measure_start, m_measure_end;
		NedTableTuplet *m_tuplet_ptr;
		GList *m_chord;
		bool m_closed;
		bool m_is_chord_member;
};

class NedTableInstrument {
	public:
		NedTableInstrument(int midi_nr);
		~NedTableInstrument();
		void addNote(NedTableNote *note);
		bool closeNote(int pitch, unsigned long long end_time);
		void distributeOverlapping(int voice_num);
		int getMidiNr() {return m_midi_nr;}
		GList *splitRange(bool force_piano, bool with_alto_clef);
		void setBestClef(bool with_alto_clef);
		void snapNotesToMeasures(bool onlyRest, NedInstrumentTable *table);
		void setInstrumentName(char *name);
		void resort();
		void determineLines();
		void tripletRecognition(unsigned long long  base);
		void combinePossibleTripletChords();
		void buildTriplets(unsigned long long base, bool allow_single_notes);
		void fillUpUnwantedRests();
		void handleTuplets(unsigned long long  base);
		void freeUnhandledTuplets();
		void insertRests();
		void buildChords();
		void snapNotes();
		char *getInstrumentName();
		GList *convertToNtEdNotes(int voice_nr, NedInstrumentTable *table, bool place_tempo_sigs);
		int getClef() {return m_clef;}
		int getOctaveShift() {return 12 * m_octave_shift;}
		void addKeySig(int keysig) {m_keysig = keysig;}
		void testUnclosed(int track);
		void detectClashes();
		void detectClashes2(char *m);
		void testForMeasureCrossingNotes(NedInstrumentTable *table);
		void testSort();
		void testForRests();
		void print();
		int getMidiPgm();
		GList *m_tuplet_list;
		int m_midi_channel;
		int m_midi_volume;
		int m_keysig;
	private:
		bool searchForTupletMembers(GList **table_tuplet, GList *lptr, unsigned long long t1, unsigned long long t2, unsigned long long measure_start);
		bool overlapping(GList *table_tuplet, NedTableNote *tabletNote);
		void determineRange(bool determine_volume = false);
		void determineBestClef(bool with_alto_clef);
		void testNewVolume(NedChordOrRest *chord, int volume, int *current_volume, int volume_change_density);
		GList *m_note_list;
		int m_midi_nr;
		int m_min_pitch, m_max_pitch;
		int m_clef;
		int m_octave_shift;
		int m_staves;
		int m_av_pitch;
		char m_instrument_name[512];
		bool m_slots[4];
		bool m_piano_slots[2];
		static int m_splits[5];
		static int m_piano_splits[3];
};

class NedInstrumentTable {
	public:
		NedInstrumentTable(bool with_triplets, bool with_single_triplets, bool with_alto_clef, int voice_num,
			int tempo_change_density, int volume_change_density, bool force_piano, bool force_one_staff);
		~NedInstrumentTable();
		int getNumerator();
		int getDenominator();
		int getStartTempo();
		void getMeasureDurationAt(unsigned long long midi_time, unsigned long long *measure_start, unsigned long long *measure_end);
		void addNote(NedTableNote *note, int midi_nr);
		void closeNote(int pitch, unsigned long long end_time, int midi_nr);
		void print();
		void prepareNotes();
		void computeTimeSigMeasureNumbers(NedMainWindow *main_window);
		int getPartCount();
		GList *convertToNtEdNotes(int voice_nr);
		int getClef(int staff_nr);
		int getOctaveShift(int staff_nr);
		int getMidiPgm(int staff_nr);
		int getChannel(int staff_nr);
		int getVolume(int staff_nr);
		int getKeySig(int staff_nr);
		char* getInstrumentName(int staff_nr);
		void setInstrumentName(char *name);
		void addKeySig(unsigned long long midi_time, int keysig);
		void addTimeSig(NedTableTimesig *timesig);
		void addTempoSig(NedTableTemposig *temposig);
		NedTableTemposig *getCurrentTempo(int *tempo_so_far, unsigned long long midi_time);
		GList *convertToNtEdNotes(int staff_nr, int voice_nr);
		void removeLastInserted() {m_last_inserted = NULL;}
		double m_tempo_inverse;
		int m_tempo_change_density;
		int m_volume_change_density;
		bool m_force_piano;
		bool m_force_one_staff;
	private:
		GList *m_instrument_list;
		GList *timesig_list;
		GList *temposig_list;
		int m_pending_keysig;
		char m_pending_instrument_name[512];
		NedTableInstrument *m_last_inserted;
		bool m_with_triplets;
		bool m_with_single_triplets;
		bool m_with_alto_clef;
		int m_voice_num;
};

#endif /* INSTRUMENT_TABLE_H */
