program pi;
(* file exemples/pascal/pi.p: compute some digits of pi
 +-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                    Calcul de Pi, formule de Ramanujan                 |
 |                                                                       |
 +-----------------------------------------------------------------------*)

(* cf. "The Caml Numbers Reference Manual", Inria, RT-0141 *)
(* annexe A, pp. 115 et suivantes.                         *)

uses _name_;

                       (* +--------------------------+
                          |  Sommation dichotomique  |
                          +--------------------------+ *)

const maxprof = 32; (* profondeur de rcursion maximale *)
   
procedure somme(prec:longint; var num,den:xint);
var etapes : longint;                       (* nombre de termes  calculer *)
    pile   : array[0..3*maxprof-1] of xint; (* pile de rcusrion *)
    sp     : longint;                       (* pointeur de pile  *)
    i,j    : longint;
    c,p,alpha,beta,gamma,delta,eps,t,u,x : xint;

   (* constantes  *)
   const a = 13591409;
   const b = 545140134;
   
begin

   c     := of_string('10939058860032000');
   p     := of_int(0);            (* index srie *)
   alpha := of_int(1);            (* 2p + 1      *)
   beta  := of_int(1);            (* 6p + 1      *)
   gamma := of_int(5);            (* 6p + 5      *)
   delta := of_int(53360);        (* c*p^3       *)
   eps   := of_int(a);            (* a + bp      *)
   t     := xnew();               (* scratch     *)
   u     := xnew();               (* scratch     *)

   etapes := (prec+197) div 94;
   sp := 0;

   (* initialise la pile *)
   for i := 0 to 3*maxprof-1 do pile[i] := xnew();

   for i:=1 to etapes do begin

      (* calcule et retranche les termes de rangs p et p+1 *)
      mul  (t,          alpha,  beta);
      mul  (pile[sp],   t,      gamma);
      copy (pile[sp+1], delta); 
      copy (pile[sp+2], eps);   
                                
      add_1(p,          p,          1);
      add_1(alpha,      alpha,      2);
      add_1(beta,       beta,       6);
      add_1(gamma,      gamma,      6);
      sqr  (t,          p);
      mul  (u,          c,          p);
      mul  (delta,      t,          u);
      add_1(eps,        eps,        b);
                             
      mul  (t,          delta,      pile[sp+2]);
      mul  (u,          pile[sp],   eps);
      sub  (pile[sp+2], t,          u);
      mul  (t,          alpha,      beta);
      mul  (u,          pile[sp],   gamma);
      mul  (pile[sp],   t,          u);
      mul  (pile[sp+1], pile[sp+1], delta);
                             
      add_1(p,          p,          1);
      add_1(alpha,      alpha,      2);
      add_1(beta,       beta,       6);
      add_1(gamma,      gamma,      6);
      sqr  (t,          p);
      mul  (u,          c,          p);
      mul  (delta,      t,          u);
      add_1(eps,        eps,        b);
                             
      sp := sp+3;
    
      (* combine avec les calculs prcdents *)
      j:=1; while (j and i) = 0 do begin
         sp := sp - 3;

         mul(t,           pile[sp+1],  pile[sp-1]);
         mul(pile[sp-1],  pile[sp-3],  pile[sp+2]);
         add(pile[sp-1],  pile[sp-1],  t);
         mul(pile[sp-3],  pile[sp-3],  pile[sp]);
         mul(pile[sp-2],  pile[sp-2],  pile[sp+1]);

         j := 2*j;
      end;
   end;

  (* termine les calculs en instance *)
   sp := sp - 3;
   while sp <> 0 do begin
      sp := sp - 3;

      mul(t,           pile[sp+4],  pile[sp+2]);
      mul(pile[sp+2],  pile[sp],    pile[sp+5]);
      add(pile[sp+2],  pile[sp+2],  t);
      mul(pile[sp+1],  pile[sp+1],  pile[sp+4]);
   end;

   (* nettoie les variables locales et retourne la fraction *)
   x := num; num := pile[1]; pile[1] := x;
   x := den; den := pile[2]; pile[2] := x;
   for i := 0 to 3*maxprof-1 do xfree(pile[i]);
   xfree(c);
   xfree(p);
   xfree(alpha);
   xfree(beta);
   xfree(gamma);
   xfree(delta);
   xfree(eps);
   xfree(t);
   xfree(u);
   
end;


                 (* +--------------------------------------+
                    |  Calcule pi avec digits+2 dcimales  |
                    +--------------------------------------+ *)

procedure calc_pi(digits:longint; pgcd,print,skip,debug,test:boolean; cmd,true_pi:ansistring);
var prec, i,j,l,l1,l2  : longint;
    num,den,t,x1,x2,x3 : xint;
    s,ss      : ansistring;

begin
   
   if debug then chrono('dbut');
   num := xnew();
   den := xnew();
   t   := xnew();
   x1  := xnew();
   x2  := xnew();
   x3  := xnew();

   (* t <- 5^(digits+2) *)
   copy_int(t, 5);
   pow     (t, t, digits+2);
   if debug then chrono('puiss-5');

   (* t <- floor( sqrt(640320) * 10^(digits+2) ) *)
   prec := nbits(t) + digits;
   sqr   (t, t);
   mul_1 (t, t, 640320);
   shiftl(t, t, 2*digits+4);
   sqrt  (t, t);
   if debug then chrono('sqrt');

   (* num/den <- somme de la srie  env. 10^(-digits-2) prs *)
   somme(prec,num,den);
   if debug then begin
      str(nbits(num),s);
      ss := 'srie lb=' + s;
      chrono(pchar(ss));
   end;

   (* simplifie la fraction si demand (ceci ne vaut pas le coup, le
      temps de calcul du pgcd est trs suprieur au temps de calcul
      de la division sans simplification) *)
   if pgcd then begin
      cfrac(x1,x2,x3,num,den,num,den);
      if debug then begin
         str(nbits(num),s);
         ss := 'pgcd lb=' + s;
         chrono(pchar(ss));
      end;
   end;

   (* t <- sqrt(640320)*num/den * 10^digits+2) *)
   mul  (t, num, t);
   quo  (t, t, den);
   if debug then chrono('quotient');

   (* on n'a plus besoin de num,den *)
   xfree(num);
   xfree(den);

   (* conversion en dcimal *)
   if print then begin
      ss := string_of(t);
      if debug then chrono('conversion');

      writeln(ss[1],'.');
      i := 2;
      while ss[i] <> #0 do begin
         write(ss[i]);
         if      (i mod 250) = 1 then begin writeln; writeln; end
         else if (i mod 50)  = 1 then writeln
         else if (i mod 10)  = 1 then write('  ')
         else if (i mod 5)   = 1 then write(' ');
         if skip and ((i mod 50) = 1) then begin
            j := (length(ss)-i) div 50 - 1;
            if j > 0 then begin
               writeln('... (',j,' lignes omises)');
               i := i + 50*j;
            end;
         end;
         i := i+1;
      end;
      if (i mod 50) <> 2 then writeln;
   end

   else if test then begin

      ss := string_of(t);
      l1 := length(ss);
      l2 := length(true_pi);
      if l1 < l2 then l := l1 else l := l2;
      if system.copy(ss,1,l) <> system.copy(true_pi,1,l)
         then writeln('error in the ',cmd,' test')
         else writeln(cmd,#9'test ok');
   end;

   (* termin *)
   xfree(t);
   xfree(x1);
   xfree(x2);
   xfree(x3);

end;

                        (* +-----------------------+
                           |  Programme principal  |
                           +-----------------------+ *)


var digits,i : longint;
    pgcd,print,skip,debug,help,test : boolean;
    cmd,true_pi : ansistring;
    c : word;
   
begin

   (* les 1000 premiers chiffres de pi pour contrle *)
   true_pi := '';
   true_pi := true_pi + '31415926535897932384626433832795028841971693993751';
   true_pi := true_pi + '05820974944592307816406286208998628034825342117067';
   true_pi := true_pi + '98214808651328230664709384460955058223172535940812';
   true_pi := true_pi + '84811174502841027019385211055596446229489549303819';
   true_pi := true_pi + '64428810975665933446128475648233786783165271201909';
   true_pi := true_pi + '14564856692346034861045432664821339360726024914127';
   true_pi := true_pi + '37245870066063155881748815209209628292540917153643';
   true_pi := true_pi + '67892590360011330530548820466521384146951941511609';
   true_pi := true_pi + '43305727036575959195309218611738193261179310511854';
   true_pi := true_pi + '80744623799627495673518857527248912279381830119491';
   true_pi := true_pi + '29833673362440656643086021394946395224737190702179';
   true_pi := true_pi + '86094370277053921717629317675238467481846766940513';
   true_pi := true_pi + '20005681271452635608277857713427577896091736371787';
   true_pi := true_pi + '21468440901224953430146549585371050792279689258923';
   true_pi := true_pi + '54201995611212902196086403441815981362977477130996';
   true_pi := true_pi + '05187072113499999983729780499510597317328160963185';
   true_pi := true_pi + '95024459455346908302642522308253344685035261931188';
   true_pi := true_pi + '17101000313783875288658753320838142061717766914730';
   true_pi := true_pi + '35982534904287554687311595628638823537875937519577';
   true_pi := true_pi + '81857780532171226806613001927876611195909216420198';
   
   digits := 100;
   pgcd:=false; print:=true; skip:=false; debug:=false; help:=false; test:=false;
   cmd := paramstr(0);

   for i:=1 to paramcount do begin
      if      paramstr(i) = '-h'       then help  := true
      else if paramstr(i) = '-d'       then debug := true
      else if paramstr(i) = '-noprint' then print := false
      else if paramstr(i) = '-skip'    then skip  := true
      else if paramstr(i) = '-gcd'     then pgcd  := true
      else if paramstr(i) = '-test'    then begin
         digits := 1000;
         print  := false;
         skip   := false;
         debug  := false;
         pgcd   := false;
         test   := true;
      end
      else val(paramstr(i),digits,c);
   end;

  if help then writeln('usage: ',cmd,' [digits] [-d] [-noprint] [-skip] [-gcd]')
  else calc_pi(digits-2,pgcd,print,skip,debug,test,cmd,true_pi);

end.
