/* blazer.c - model specific routines for Blazer UPS models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
                 2002  Phil Hutton <blazer-driver@hutton.sh>
                 2003  Arnaud Quette <arnaud.quette@free.fr>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"

#define ENDCHAR  13	/* replies end with CR */
#define UPSDELAY  5
#define MAXTRIES 10

static	float	lowvolt = 0, highvolt = 0, voltrange = 0;
static	int	poll_failures = 0;
static	int	inverted_bypass_bit = 0;


int instcmd(const char *cmdname, const char *extra)
{
	/* Stop battery test */
	if (!strcasecmp(cmdname, "test.battery.stop")) {
			upssendchar('C');
			upssendchar('T');
			upssendchar(13);
			return STAT_INSTCMD_HANDLED;
	}

	/* Start battery test */
	if (!strcasecmp(cmdname, "test.battery.start")) {
			upssendchar('T');
			upssendchar(13);
			return STAT_INSTCMD_HANDLED;
	}

  upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

void upsdrv_initinfo(void)
{
	dstate_setinfo("ups.mfr", "Centralion");
	dstate_setinfo("ups.model", "Blazer");
	dstate_setinfo("input.voltage", "");
	dstate_setinfo("input.frequency", "");
	dstate_setinfo("output.voltage", "");
	dstate_setinfo("battery.charge", "");
	dstate_setinfo("battery.voltage", "");
	dstate_setinfo("ups.load", "");
  
	dstate_addcmd("test.battery.start");
	dstate_addcmd("test.battery.stop");

	printf("Detected UPS on %s\n", device_path);

	upsh.new_instcmd = instcmd;

	/* paranoia - cancel any shutdown that might already be running */
	upssendchar('C');
	upssendchar(13);
}

static void setup_serial(void)
{
	struct	termios	tio;

	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");

	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0; 

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B2400);
	cfsetospeed(&tio, B2400);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
}

void ups_sync(void)
{
	char	buf[256];
	int	tries = 0, ret;

	printf("Syncing with UPS: ");
	fflush(stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES)
			fatalx("\nFailed - giving up...");

		printf(".");
		fflush(stdout);
		upssend("%s", "\r");
		upssend("%s", "Q1");
		upssend("%s", "\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

		if ((ret > 0) && (buf[0] == '('))
			break;
	}

	printf(" done\n");
}

void upsdrv_shutdown(void)
{
	upssend("%s", "S01R0001");
	upssend("%s", "\r");	/* end sequence */
}

void ups_ident(void)
{
	char	buf[256];
	int	tries = 0, ret;
	int	ratecurrent;
	float	ratevolt, ratefreq;

	printf("Identifying UPS: ");
	fflush (stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			upsdebugx(2, "Failed - giving up...");
			exit (1);
		}

		printf(".");
		fflush(stdout);
		upssend("%s", "\r");
		upssend("%s", "F");
		upssend("%s", "\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

		if ((ret > 0) && (buf[0] == '#') && (strlen(buf) == 21))
			break;
	}

	printf(" done\n");

	sscanf(buf, "%*c %f %d %f %f", &ratevolt, &ratecurrent, 
		&lowvolt, &ratefreq);
	upsdebugx(2, "UPS is rated at %.2fV, %dA, %.2fHz.\n",
    ratevolt, ratecurrent, ratefreq);

	/* Not sure exactly how the battery voltage from the "F" command
	 * should be handled.  The lowest voltage I've seen before the
	 * UPS warns is 22V, so we'll try subtracting 2 from lowvolt.*/
	lowvolt -= 2;

	highvolt = 27.4;

	voltrange = highvolt - lowvolt;
}

void pollfail(char *why)
{
	poll_failures++;

	/* ignore the first few since these UPSes tend to drop characters */
	if (poll_failures == 3)
		upslogx(LOG_ERR, why);

	return;
}

void upsdrv_updateinfo(void)
{
	char	utility[16], outvolt[16], loadpct[16], acfreq[16], 
		battvolt[16], upstemp[16], stat[16], buf[256];
	float	bvoltp;
	int	ret;

	upssend("%s", "\rQ1");
	upssend("%s", "\r");
	sleep(UPSDELAY); 

	ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

	if (strlen(buf) < 46) {
		pollfail("Poll failed: short read from UPS");
    dstate_datastale();
		return;
	}

	if (strlen(buf) > 46) {
		pollfail("Poll failed: oversized read from UPS");
    dstate_datastale();
		return;
	}

	if (buf[0] != '(') {
		pollfail("Poll failed: invalid start character");
		return;
	}

	/* only say this if it got high enough to log a failure note */
	if (poll_failures >= 3)
		upslogx(LOG_NOTICE, "UPS poll succeeded");

	poll_failures = 0;

	sscanf(buf, "%*c%s %*s %s %s %s %s %s %s", utility, outvolt, 
		loadpct, acfreq, battvolt, upstemp, stat);
	
	bvoltp = ((atof (battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

  dstate_setinfo("input.voltage", "%s", utility);
	dstate_setinfo("input.frequency", "%s", acfreq);
	dstate_setinfo("output.voltage", "%s", outvolt);
	dstate_setinfo("battery.charge", "%02.1f", bvoltp);
	dstate_setinfo("battery.voltage", "%s", battvolt);
	dstate_setinfo("ups.load", "%s", loadpct);

	status_init();

	if (stat[0] == '0') {
		status_set("OL");		/* on line */

		/* only allow these when OL since they're bogus when OB */

		if (stat[2] == (inverted_bypass_bit ? '0' : '1')) {
			/* boost or trim in effect */
			if (atof(utility) < atof(outvolt))
				status_set("BOOST");

			if (atof(utility) > atof(outvolt))
				status_set("TRIM");
		}

	} else {
		status_set("OB");		/* on battery */
	}

	if (stat[1] == '1')
		status_set("LB");		/* low battery */

	status_commit();
  dstate_dataok();
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Blazer UPS driver 0.06 (%s)\n", UPS_VERSION);
}

void upsdrv_initups(void)
{
	upssend_delay = 100000;
	open_serial(device_path, B2400);
	setup_serial();

	/* don't let upscommon warn about it since it happens way too often */
	/* this driver does its own checking with better handling */
	flag_timeoutfailure = -1;

	ups_sync();
	ups_ident();
}

void upsdrv_cleanup(void)
{
}

