/* fentonups.c - model specific routines for Fenton Technologies units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"
#include "fentonups.h"

#define ENDCHAR 13	/* replies end with CR */

#define	DRV_VERSION "1.11"

static	int	cap_upstemp = 0;
static	float	lowvolt = 0, voltrange;
static	int	lownorm, highnorm, poll_failures = 0;

/* handle devices which don't give a properly formatted I string */
static int check_mtab2(const char *raw)
{
	int	i;
	char	*cooked;

	/* trim off the leading # and any trailing spaces */
	cooked = xstrdup(&raw[1]);
	rtrim(cooked, ' ');

	for (i = 0; mtab2[i].id != NULL; i++) {
		if (!strcmp(cooked, mtab2[i].id)) {

			/* found it - load the data */
			dstate_setinfo("ups.mfr", mtab2[i].mfr);
			dstate_setinfo("ups.model", mtab2[i].model);
			lowvolt = mtab2[i].lowvolt;
			voltrange = mtab2[i].voltrange;
			cap_upstemp = mtab2[i].has_temp;

			dstate_setinfo("input.transfer.low", "%d",
				mtab2[i].lowxfer);
			dstate_setinfo("input.transfer.high", "%d",
				mtab2[i].highxfer);

			lownorm = mtab2[i].lownorm;
			highnorm = mtab2[i].highnorm;

			free(cooked);
			return 1;	/* found */
		}
	}

	free(cooked);
	return 0;	/* not found */
}

static void guessmodel(const char *raw)
{
	char	mch, mstr[256];

	/* first see if it's in the mtab2 */
	if (check_mtab2(raw))
		return;

	mch = raw[17];

	strlcpy(mstr, &raw[18], sizeof(mstr));
	mstr[10] = '\0';	/* 10 chars max, per the protocol */

	/* trim whitespace */
	rtrim(mstr, ' ');

	/* use Fenton model-chars to attempt UPS detection */

	switch (mch) {
		case 'L':
			dstate_setinfo("ups.model", "PowerPal %s", mstr);
			cap_upstemp = 0;
			break;

		case 'H':
			dstate_setinfo("ups.model", "PowerOn %s", mstr);
			cap_upstemp = 1;
			break;

		case 'M':
			dstate_setinfo("ups.model", "PowerPure %s", mstr);
			cap_upstemp = 1;
			break;

		default: 
			dstate_setinfo("ups.model", "Unknown %s", mstr);
			upslogx(LOG_ERR, "Unknown ups - "
				"please report this ID string: %s", raw);
			break;
	}
}

int instcmd(const char *cmdname, const char *extra)
{
	if (!strcasecmp(cmdname, "test.battery.start")) {
		upssend("T\r");
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "test.battery.stop")) {
		upssend("CT\r");
		return STAT_INSTCMD_HANDLED;
	}

	upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

static char *get_id(void)
{
	int	i, ret;
	char	temp[SMALLBUF];

	/* try to get the initial UPS data */
	for (i = 0; i < 5; i++) {
		upssend("I\r");
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, "");

		if (ret < 1) {
			upslog(LOG_ERR, "Short read during UPS id sequence");
			continue;
		}

		if (temp[i] != '#') {
			upslogx(LOG_ERR, "Bad UPS info start character [%c]",
				temp[i]);
			continue;
		}

		/* got what we need.. */
		return xstrdup(temp);
	}

	fatalx("Unable to detect a Fenton or Megatec protocol UPS");

	/* NOTREACHED */
	return NULL;
}

void upsdrv_initinfo(void)
{
	char	temp[256], model[32], *raw;
	int	modelnum, i;

	/* dummy read attempt to sync - throw it out */
	upssend("I\r");
	upsrecv(temp, sizeof(temp), ENDCHAR, "");

	raw = get_id();

	if (!raw)	/* shouldn't happen */
		fatalx("Didn't get a usable ID string");

	snprintf(temp, sizeof(temp), "%s", raw);

	temp[11] = 0;
	temp[27] = 0;

	/* manufacturer */
	rtrim(&temp[1], ' ');
	dstate_setinfo("ups.mfr", &temp[1], 0, 0);

	/* L660A = PowerPal (L) @ 660 VA, American (A) version (115V) */

	/* grab full model string */
	rtrim(&temp[17], ' ');
	snprintf(model, sizeof(model), "%s", &temp[17]);

	modelnum = -1;

	/* figure out official model name and voltage info from table */
	for (i = 0; modeltab[i].mtext != NULL; i++) {
		if (!strcmp(modeltab[i].mtext, model)) {
			modelnum = i;
			lowvolt = modeltab[i].lowvolt;
			voltrange = modeltab[i].voltrange;
			cap_upstemp = modeltab[i].has_temp;
			break;
		}
	}

	/* table lookup fails -> guess */
	if (modelnum == -1)
		guessmodel(raw);
	else {
		dstate_setinfo("ups.model", "%s", modeltab[modelnum].desc);

		dstate_setinfo("input.transfer.low", "%d", 
			modeltab[modelnum].lowxfer);

		dstate_setinfo("input.transfer.high", "%d",
			modeltab[modelnum].highxfer);

		lownorm = modeltab[modelnum].lownorm;
		highnorm = modeltab[modelnum].highnorm;
	}

	/* now add instant command support info */
	dstate_addcmd("test.battery.start");
	dstate_addcmd("test.battery.stop");

	printf("Detected %s on %s\n", dstate_getinfo("ups.model"), device_path);
	free(raw);

	/* paranoia - cancel any shutdown that might already be running */
	upssend("C\r");

	upsh.new_instcmd = instcmd;

	dstate_setinfo("driver.version.internal", "%s", DRV_VERSION);
}

static void pollfail(char *why)
{
	poll_failures++;

	/* don't spew into the syslog forever */
	if (poll_failures < 3)
		upslogx(LOG_ERR, why);

	return;
}

void upsdrv_updateinfo(void)
{
	char	temp[256], involt[16], loadpct[16], acfreq[16], battvolt[16],
		upstemp[16], stat[16], outvolt[16];
	int	util, ret;
	double	bvoltp;

	upssend("Q1\r");

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, "");

	/* sanity checks for poll data */
	if (strlen(temp) < 46) {
		pollfail("Poll failed: short read from UPS");
		dstate_datastale();
		return;
	}

	if (strlen(temp) > 46) {
		pollfail("Poll failed: oversized read from UPS");
		dstate_datastale();
		return;
	}

	if (temp[0] != '(') {
		pollfail("Poll failed: invalid start character");
		dstate_datastale();
		return;
	}

	if (poll_failures > 0)
		upslogx(LOG_NOTICE, "UPS poll succeeded");

	poll_failures = 0;		

	/* (MMM.M NNN.N PPP.P QQQ RR.R S.SS TT.T  b7b6b5b4b3b2b1b0<cr>
	 *
	 * MMM.M : input voltage (involt)
	 * NNN.N : fault voltage (ignored)
	 * PPP.P : output voltage
	 */

	sscanf(temp, "%*c%s %*s %s %s %s %s %s %s", involt, outvolt, loadpct,
		acfreq, battvolt, upstemp, stat);

	dstate_setinfo("input.voltage", "%s", involt);
	dstate_setinfo("output.voltage", "%s", outvolt);
	dstate_setinfo("battery.voltage", "%s", battvolt);

	bvoltp = ((atof(battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	dstate_setinfo("battery.charge", "%02.1f", bvoltp);

	status_init();

	util = atoi(involt);

	if (stat[0] == '0') {
		status_set("OL");		/* on line */

		/* only allow these when OL since they're bogus when OB */
		if (stat[2] == '1') {		/* boost or trim in effect */
			if (util < lownorm)
				status_set("BOOST");

			if (util > highnorm)
				status_set("TRIM");
		}

	} else {
		status_set("OB");		/* on battery */
	}

	if (stat[1] == '1')
		status_set("LB");		/* low battery */

	status_commit();

	if (cap_upstemp == 1)
		dstate_setinfo("ups.temperature", "%s", upstemp);

	dstate_setinfo("input.frequency", "%s", acfreq);
	dstate_setinfo("ups.load", "%s", loadpct);

	dstate_dataok();
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	char	temp[256], stat[32];

	/* basic idea: find out line status and send appropriate command */

	upssend("Q1\r");
	upsrecv(temp, sizeof(temp), ENDCHAR, "");
	sscanf(temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0003<cr>, ups will cycle and return soon */

	upssend("S01");

	if (stat[0] == '0') {			/* on line */
		printf("On line, sending shutdown+return command...\n");
		upssend("R0003");
	} else
		printf("On battery, sending normal shutdown command...\n");

	upssendchar(13);	/* end sequence */
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Fenton UPS driver %s (%s)\n", 
		DRV_VERSION, UPS_VERSION);
}

void upsdrv_initups(void)
{
	open_serial(device_path, B2400);
}

void upsdrv_cleanup(void)
{
}
